///
// Copyright (C) 2003, 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "imagefilesel.h"
#include <sstream>
#include "wmisc.h"
#include <gtkmm/alignment.h>
#include <gtkmm/frame.h>
#include "util/stringutil.h"

const int ImageFilesel::maxwidth = 100;
const int ImageFilesel::maxheight = 100;

ImageFilesel::ImageFilesel(Gtk::Window &parent, const ustring& title,
			     float default_res, 
			   const ustring& unit_name, float unit_factor)
  : Filesel(parent, title), res(0, true), 
    show_preview("Show _preview", true),
    no_preview("No preview available", 0.5, 0.5)
{
  unit.name = unit_name;  unit.factor = unit_factor;
  // *** The preview image ***
  // Find the right HBox
  Gtk::Box::BoxList::iterator boxiter = get_main_vbox()->children().end();
  boxiter--; boxiter--; boxiter--; boxiter--; // the fourth last widget
  Gtk::Box *list_box = 
    dynamic_cast<Gtk::Box*>(boxiter->get_widget());
  if(!list_box)
    list_box = get_main_vbox(); // safeguard
      
  Gtk::Frame *frame = manage(new Gtk::Frame);
  Gtk::Box *box = manage(new Gtk::VBox(false, double_space));
  Gtk::Alignment *alignment = manage(new Gtk::Alignment());
  alignment->set_size_request(maxwidth, maxheight);
  Gtk::Box *inner_box = manage(new Gtk::VBox());
  alignment->add(*inner_box);
  frame->add(*alignment);
  box->pack_end(show_preview, Gtk::PACK_SHRINK);
  box->pack_end(info, Gtk::PACK_SHRINK);
  box->pack_end(*frame, Gtk::PACK_SHRINK);
  inner_box->pack_start(preview);
  inner_box->pack_start(no_preview, Gtk::PACK_EXPAND_WIDGET);      
  no_preview.set_line_wrap(); no_preview.set_sensitive(false);
  no_preview.set_size_request(maxwidth, maxheight);
  no_preview.set_justify(Gtk::JUSTIFY_CENTER);
  list_box->pack_end(*box, Gtk::PACK_SHRINK);
  box->show_all(); no_preview.hide(); preview.hide();

  // *** The image resolution
  res.set(default_res);
  box = manage(new Gtk::HBox(false, double_space));
  box->pack_start(*manage(new Gtk::Label("Image resolution:")),
		  Gtk::PACK_SHRINK);
  box->pack_start(res, Gtk::PACK_SHRINK);
  box->pack_start(*manage(new Gtk::Label("ppi")),
		  Gtk::PACK_SHRINK);
  get_main_vbox()->pack_start(*box, Gtk::PACK_SHRINK, single_space);
  box->show_all();

  get_selection_entry()->signal_changed().connect
    (slot(*this, &ImageFilesel::update_preview));
  show_preview.signal_clicked().connect
    (slot(*this, &ImageFilesel::update_preview));
  res.signal_value_changed().connect
    (slot(*this, &ImageFilesel::update_size));
}

void ImageFilesel::update_size() {
  if(!show_preview.get_active())
    return;
  int w = image_size.width;  int h = image_size.height;
  float resolution = res.get() / 72.0; // pixels per point
  std::ostringstream text;
  text << w  << " x " << h << " pixels";
  text.precision(3);
  if(!unit.name.empty())
    text << '\n' 
	 << w / resolution / unit.factor << " x " 
	 << h / resolution / unit.factor  << " " << std::string(unit.name);
  text << std::flush;
  info.set_text(text.str());
}

void ImageFilesel::update_preview() {
  if(!show_preview.get_active()) {
    info.set_text("");
    preview.hide();
    no_preview.hide();
    return;
  }
  try {	
    Glib::RefPtr<Gdk::Pixbuf> pixbuf = 
      Gdk::Pixbuf::create_from_file(get_filename());
    // scale to appropriate size
    int w = pixbuf->get_width();  int h = pixbuf->get_height();
    image_size.width = w;  image_size.height = h;
    update_size();
    float fw = float(maxwidth) / w;  float fh = float(maxheight) / h;
    float f = std::min(fw, fh);
    w = int(f * w);  h = int(f * h);
    pixbuf = pixbuf->scale_simple(w, h, Gdk::INTERP_NEAREST);
    preview.set(pixbuf);
    preview.show();
    no_preview.hide();
  } 
  catch (...) {
    // failed to load image, possibly because the file was not an
    // image file
    info.set_text("");
    preview.hide();
    no_preview.show();
  }
}
