/*****************************************************************************
 *                                                                           *
 * Program:   paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Modul:     matchpic.c                                                     *
 *            Funktions to move images until they match each other           *
 * Author:    Andreas Tille                                                  *
 * Date:      08.06.1998                                                     *
 * Copyright: Andreas Tille, 1999; GNU Public License                        *
 *                                                                           *
 *****************************************************************************/

#include <gtkimmov.h>

#include "paul.h" 
#include "callback.h"
#include "names.h"

static int CutMovedImage(PICTURE *bild, GtkImMovPic *im, char *what)
/* perform cutting after obtaining cut coordinates
 * --- Parameter: ---
 * PICTURE     *bild : image to cut
 * GtkImMovPic *im   : gtkimmove image with cut coordinates
 * char        *what : moved against what
 */
{
  char        *desc;

  if ( CropImage(bild, im->x1, im->y1, im->x2 - im->x1, im->y2 - im->y1) != RET_OK ) {
    gtk_immov_free_struct(im);
    return RET_ERR;
  }

  desc = g_strdup_printf("%s moved against %s: (%i,%i)-(%i,%i)", bild->file, what, 
                                                                 im->x1, im->y1, im->x2, im->y2);
  ImgChunksUpdate(bild, TypMove, desc, APPMOVE, MATCHPICTURES);
  FREE(desc);
  gtk_immov_free_struct(im);

  return RET_OK;
}

static int MovedImages(GtkWidget *widget, GtkImMov *immov)
/*
 * --- Parameter: ---
 * GtkWidget *widget : OK-button inside GtkImReg widget
 *                     GList containing image to cut as user_data
 * GtkImReg *imreg   : GtkImReg widget, PAUL *p set as user_data
 */
{
  PAUL        *p;
  GtkImMovPic *im;
  GList       *movl, *pl;
  PICTURE     *bild;
  int          ret = 0;

  g_return_val_if_fail ( GTK_IS_IMMOV(immov), RET_ERR ) ;
  g_return_val_if_fail ( (p = gtk_object_get_user_data(GTK_OBJECT(immov)) ), RET_ERR ) ;
  g_return_val_if_fail ( IS_PAUL(p), RET_ERR ) ;
  g_return_val_if_fail ( GTK_IS_BUTTON(widget), RET_ERR ) ;

  for ( im = (movl = immov->iml)->data, bild = BILD(pl = p->piclist); 
        movl && (im = movl->data) && pl; movl = movl->next, bild = BILD(pl = pl->next) ) {
    ret += CutMovedImage(bild, im, p->op->file);
  }
  ret += CutMovedImage(p->op, immov->fix, "other images");

  gtk_widget_destroy(GTK_WIDGET(immov));

  for ( pl = p->piclist; pl; pl = pl->next ) RefreshImage(GTK_CLIST(p->filelist), pl);
  SelectAndUpdateInfo(p);
  ApplyPicture(p->opi->window, p->op);

  return ret ? RET_ERR : RET_OK;
}

int MakeMove(PAUL *p)
/* Call gtkimmov to get matching images
 * --- Parameter: ---
 * PAUL *p         : image structure containing list of images
 * --- return: ---
 * p->piclist      : new list of images containing differences
 * int  MakeMove() : RET_ERR or RET_OK
 */
{
  GList       *pl, *movl = NULL;
  GtkImMovPic *fix;
  GtkWidget   *immov;
  PICTURE     *bild;
   
  g_return_val_if_fail ( IS_PAUL(p), RET_ERR ) ;
  if ( !(p->piclist) ) return RET_ERR;
  g_return_val_if_fail ( CheckPicList(p->piclist), RET_ERR ) ;
  if ( !CheckOperating(p) ) return RET_ERR;
  
  fix = gtk_immov_set_struct(p->op->W, p->op->H, p->op->DATA, p->op->file);
  for ( bild = BILD(pl = p->piclist); pl; bild = BILD(pl = pl->next) ) 
    movl = g_list_append(movl, gtk_immov_set_struct(bild->W, bild->H, bild->DATA, bild->file) );
  
  immov = gtk_immov_new(NULL, fix, movl, 0);

  gtk_window_position(GTK_WINDOW(immov), GTK_WIN_POS_MOUSE);
  gtk_signal_connect(GTK_OBJECT(immov), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &immov);

  gtk_object_set_user_data(GTK_OBJECT(immov), p);
  gtk_signal_connect(GTK_OBJECT(GTK_IMMOV(immov)->ok_button), "clicked", 
                     GTK_SIGNAL_FUNC(MovedImages), immov);
  gtk_signal_connect_object(GTK_OBJECT(GTK_IMMOV(immov)->cancel_button), "clicked", 
                            GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(immov));
  
  gtk_widget_show(immov);
  return RET_OK;
}


int AutoMatchPictures(PAUL *p)
/* Automatically moving pictures by minimizing image differences.
 * The first picture is used as reference to which the difference with
 * all other images will be minimized.
 * --- Parameter: ---
 * PAUL *p                   : image sources
 * --- Return: ---
 * int   AutoMatchPictures() : RET_ERR or RET_OK
 */
{ 
  GList       *movl = NULL, *pl;
  PICTURE     *bild, *op;
  GtkImMovPic *fix, *im;
  int          ret = 0;

  g_return_val_if_fail ( IS_PAUL(p), RET_ERR );
  g_return_val_if_fail ( p->piclist, RET_ERR) ;
  g_return_val_if_fail ( BILD(p->piclist), RET_ERR) ;
  
  if ( NBILDER(p->piclist) < 2 ) return RET_OK;

  op = bild = BILD(p->piclist);
  fix = gtk_immov_set_struct(bild->W, bild->H, bild->DATA, bild->file);
  for ( bild = BILD(pl = p->piclist->next); pl; bild = BILD(pl = pl->next) ) 
    movl = g_list_append(movl, gtk_immov_set_struct(bild->W, bild->H, bild->DATA, bild->file) );

  immov_automatch(movl, fix);
  if ( immov_accept_images(movl, fix) ) {
    g_warning(_("No images moved at all."));
    return RET_SPECIAL;
  }
  
  for ( im = movl->data, bild = BILD(pl = p->piclist->next); 
        movl && (im = movl->data) && pl && bild; movl = movl->next, bild = BILD(pl = pl->next) ) {
    ret += CutMovedImage(bild, im, op->file);
  }
  ret += CutMovedImage(op, fix, "other images");

  p->opt->f &= ~MATCHPICTURES;
  return ret ? RET_SPECIAL : RET_OK;
}






