/* PSK31  -- receiver classs
 */

#ifndef PSK31RECEIVER_H
#define PSK31RECEIVER_H                                                         

#define FIR1TAPS 64
#define FIR2TAPS 64
#define CONFLEN  300

#include <qobject.h>
#include "psk31_coder.h"

/* receiver channel interface class */
class Psk31Receiver : public QObject{

Q_OBJECT

private:
	int qpsk;        /* QPSK mode on */
	int lsb;         /* inverted mode */
	float _rxfreq;
	int rxfreq;
	int rxphase;
	int dcd;         /* dcd flag */
	int usedcd;      /* stop decoding when dcd is off */
	int afc;         /* AFC flag */
	int noiselevel;  /* test only: simulated addition noise */
	int dofft;       /* freq scan via fft activation */
	int lastdelta;
    int strength;    /* strength of signal 0 -> 40 TJW */
    int txtrack;     /* tx freq tracks rx freq TJW */
    int _confidence;  /* rx confidence accumulator */
    int confidenceLevel;
	psk31_coder coder;

	/* fft stuff */
	float fft_result;
	float fftbufI[1024], fftbufQ[1024];
	int fftind;

	/* receive filters */
	float ibuf1_I[FIR1TAPS];
	float ibuf1_Q[FIR1TAPS];
	float ibuf2_I[FIR2TAPS];
	float ibuf2_Q[FIR2TAPS];
	int amp_buf[16];
	int ibuf1_cur, ibuf2_cur; /*index of oldest sample/place for new one */
	int lastphase; /* phase of last rx symbol... */
	int bitclk;    /* phase synchronisation */

	int stat;        /* cyclic state counter for receiver */
	char *decode_buffer;

	float do_fir(float *base, int cur, float *coeff, int len);

	int process_500();
	int process_rx_symbol(complex rxsymb);
	void dodcd(int diffphase);
	void doafc(int symb);
	void pdistr(float *I, float *Q);
	void perform_fft(float I, float Q);
	int diffphase2symbol(int diffphase);

private slots:
    void resetConfidence( void );

signals:
    void dcdOn( void );
    void dcdOff( void );
    void signalStrength( void );

public:
    int confidence_A[ CONFLEN ];	
    const static int confidenceLen;
	
	Psk31Receiver() {
		stat=0; ibuf1_cur=ibuf2_cur=0; rxphase=0,_confidence=0;
		for(int i=0; i<16; i++) amp_buf[i]=0;
		set_freq(1250); dofft=0; lastdelta=0;
		for(int i=0; i<FIR1TAPS; i++) ibuf1_I[i]=ibuf1_Q[i]=0;
		for(int i=0; i<FIR2TAPS; i++) ibuf2_I[i]=ibuf2_Q[i]=0;
		for(int i=0; i<CONFLEN;  i++) confidence_A[i]=0;
		coder.prime_decoder();
	}


    Psk31Receiver::~Psk31Receiver(){ }


	int set_mode(int qpsk_flag, int lsb_flag=0) { 
		qpsk=qpsk_flag; lsb=lsb_flag;
		coder.setmode(qpsk, lsb);
		return 0;
	}
	void set_afc(int afc_flag) { afc=afc_flag; }
	void set_usedcd(int dcd_flag) { usedcd=dcd_flag; }
	void set_dcd(int dcd_flag) { dcd=dcd_flag; }
	void set_freq(float f) { _rxfreq=f; rxfreq=(int)(65536.0/8000*f); }
    void set_txtrack(int t) { txtrack = t; }         /*TJW*/
    int  get_txtrack() { return txtrack; }           /*TJW*/
	void get_info(int *q, int *l, float *f, int *used, int *a, int *d,
		      int *stat, int *mag) {
		if(q) *q=qpsk; if(l) *l=lsb; if(f) *f=_rxfreq;
		if(used) *used=usedcd; if(a) *a=afc; if(d) *d=dcd;
		if(stat) *stat=lastdelta;
		if(mag) *mag=strength;
	}
	float get_freq() { return _rxfreq; }
	void enable_fft(int f) { dofft=f; }
	float get_fft_result() { return fft_result; }
	int process_rx_sample(int sample); /* return: 0=no new char; 1=new char
					              -1=buffer full */
	int get_decoded_bytes(char *buffer, int nmax); /* return: # of chars */
};

#endif 







