#
# FILENAME
#	Tclpp.tcl
#
# RCSID
# 	$Id: tclpp.tcl.in,v 1.3 1998/05/18 20:16:27 stefan Exp $
#
# DESCRIPTION
#	This file implements the Object Oriented Programming extension
#	to Tcl. This extension requires Tcl 8.0 or higher.
#
# AUTHOR
#	Stefan Sinnige <ssinnige@geocities.com>
#
# COPYRIGHT, LICENSE AND DISCLAIMER
#
#	Copyright (C) 1997-1998, Stefan Sinnige.
#
#	This program is free software; you can redistribute it and/or modify
#	it under the terms of the GNU General Public License as published by
#	the Free Software Foundation; either version 2 of the License, or
#	(at your option) any later version.
#
#	This program is distributed in the hope that it will be useful,
#	but WITHOUT ANY WARRANTY; without even the implied warranty of
#	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#	GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with this program; if not, write to the Free Software
#	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#       For the full GNU General Public License, see the 'LICENSE' file.
#

######################################################################
#
# Check if the correct Tcl package is loaded. We need 8.0 or higher
# for this version. If that is correct, then present our package
# and use the revision number as Tclpp package version.
#

if [catch {package require Tcl 8.0} errorString] {
    puts $errorString
    exit
}
package provide Tclpp {1.1.1}

######################################################################
#
# Procedure: class <classname>  <args>
#
# This procedure evaluates the class definition. It will register the 
# class name, alter the functions and setup the entire namespace.
# Furthermore, it defines a new command, equal to the class name, to
# instantiate objects of this type.
#

proc class {classname args} {
    # Get the list of baseclasses and the classdefinition
    set baseclasses     [list]
    set classdefinition [lindex [lrange $args end end] 0]
    if {[lindex $args 0] == ":"} {
        set baseclasses [lrange $args 1 [expr [llength $args] - 2]]
    }

    # Register the class and all its base classes.
    Tclpp::registerClass $classname 
    foreach baseclass $baseclasses {
        Tclpp::registerBaseClass $classname $baseclass
    }

    # == START OF SUBSTITUTION ==
    #
    # In order to insert extra code to the procedures of the class
    # and to keep track of variables, the functions 'proc' and
    # 'variable' are overridden. Furthermore, all the class syntax
    # tokens (eg. the 'virtual' keyword) are defined as well. The
    # class definition is then evaluated to create a new variable
    # containing the substituted class definition.

    rename proc _proc
    rename variable _variable

    # Variables are not defined in the class. Their names will
    # only be registered such that they can be created when the
    # class is instantiated.
    _proc variable {args} [subst {
        Tclpp::registerVariable $classname \$args
    }]

    # Each procedure is altered as follows:
    #	- An argument 'this' is appended to the list of arguments.
    #	- The first function statement is a call to Tclpp's
    #	  enterFunction procedure to bind the object namespace
    #	  variables to the class's representatives.
    _proc proc {function arguments body} [subst {
	::Tclpp::registerFunction $classname \$function no
        upvar 1 _classdefinition_ _classdefinition_
        append _classdefinition_  \[subst -nocommands {
	    if {(\[namespace which -command \$function\] != "") &&
	        (\[namespace which -command \$function\] != "::\$function")} {
	        error "Command '\$function' in '$classname' already defined\
		       as \[namespace origin \$function\]"
	    }
	    proc \$function {this \$arguments} {
	        Tclpp::enterFunction $classname \[set this\] 
	        \$body
	    }
	}\]
    }]

    # The 'virtual' keyword makes it possible to call an overridden
    # function of a base class while explicitely calling that base
    # class function with an object of the derived class as 'this'.
    # This is accomplished as follows:
    #	- Get the full name of this function, eg. ::Base::func
    #	- Get the full name of this function from the object's point
    #	  of view, eg. ::Derived::func.
    #   - If they are not equal, we have to call the derived function,
    #	  otherwise, the the function is not overridden and we continue
    #	  exeuting this function definition.
    _proc virtual {proc function arguments body} [subst {
	::Tclpp::registerFunction $classname \$function yes
        set pars \[list\]
	foreach par \$arguments {
	    lappend pars \[subst -novariables $\[subst \$par\]\] 
	}
        upvar 1 _classdefinition_ _classdefinition_
        append _classdefinition_  \[subst -nocommands {
	    proc \$function {this \$arguments} {
	        Tclpp::enterFunction $classname \[set this\] 
		set thisFunc \[namespace which -command \
				\$function\]
		set dervFunc \[namespace inscope ::\[::\[set this\]::isA\] {
				namespace origin \$function}\]
		if {\[set thisFunc\] != \[set dervFunc\]} {
		    set cmd "\[set dervFunc\] \[set this\] \$pars"
		    return \[eval \[set cmd\]\]
		} 
	        \$body
	    }
	}\]
    }]

    # Substitute the class definition into the _classdefinition_
    # variable. This substituted definition will be evaluated in
    # the class's namespace later. Any errors are catched and
    # only be thrown again after we restored the overridden global
    # procedure names.
    set _classdefinition_ ""
    set result [catch {eval $classdefinition} errorString]

    rename proc {}
    rename _proc proc
    rename variable {}
    rename _variable variable

    if $result {
        error $errorString
    }

    # == END OF SUBSTITUTION ==
    #
    # Create a new namespace which represents the class. The
    # name of the namespace equals the name of the class. First,
    # all the functions defined in all the base classes are imported 
    # such that their commands become available. Virtual functions are
    # forced to ensure the function overriding.
    # Then the isA and isKindOf functions are defined which will invoke 
    # the object defined isA and isKindOf functions.
    # Finally, the expanded class-definition is evaluated in the
    # namespace.
    namespace eval $classname [subst {
        namespace export *
	foreach base \[Tclpp::classInfo $classname allhierarchy\] {
	    foreach func \[Tclpp::classInfo \$base virtualfunctions\] {
	        namespace import -force ::\${base}::\${func}
	    }
	    foreach func \[Tclpp::classInfo \$base functions\] {
	        namespace import ::\${base}::\${func}
	    }
	}
	proc isA {this} {
	    return \[::\${this}::isA\]
	}
	proc isKindOf {this class} {
	    return \[::\${this}::isKindOf \$class\]
	}
        $_classdefinition_ 
    }]

    # Create a new function to instantiate new instances of this class. 
    # This will be a global function for convenience and intuitive 
    # object instantiation. It will create the instance and initialize
    # all its base classes and the class itself.
    proc $classname {instance args} [subst {
        Tclpp::createInstance $classname \$instance
	foreach base \[Tclpp::classInfo $classname allhierarchy\] {
	    Tclpp::initializeInstance \$base \$instance
	}
	if \[llength \$args\] {
            Tclpp::initializeInstance $classname \$instance \$args
	} else {
            Tclpp::initializeInstance $classname \$instance
	}
	}]
}

######################################################################
#
# Namespace: Tclpp
#
# This namespace encapsulates all the internal Tclpp procedures, 
# ranging from class and variable registration, function call wrappers
# and object instantiation.
#
#     registerClass <classname>
#	  Register the class <classname> to Tclpp.
#
#     registerBaseClass <classname> <baseclass>
#	  Register the base class <baseclass> as the direct ascendant
#	  of <classname> to Tclpp.
#
#     registerVariable <classname> <variabledefinition>
#	  Register the <variabledefinition> of <classname>. The variable
#	  is defined by <classname> directly.
#
#     registerFunction <classname> <functionname> <virtual ? yes : no>
#	  Register the <function> of <classname>. Only functions defined
#	  in the <classname> are registered, not the imported ones.
#
#     classInfo <classname> <property>
# 	  Retrieve information of <classname>. The <property> can be one of:
#	      variables	   - Class defined variables
#	      allvariables - Class and all its base classes defined variables
#	      hierarchy	   - Class direct base class hierarchy
#	      allhierarchy - Class and all its base classes
#	      functions    - Class defined functions
#
#     createInstance <classname> <instance>
#	  Create instance named <instance> of class <classname>.
#
#     initializeInstance <classname> <instance> <args>
#	  Initialize the <instance> of <classname> by initializing all its
#	  variables (well, only the class variables) and calling the con-
#	  structor if it is defined. Only the toplevel constructor can 
#	  accept <args>.
#
#     deleteInstance <classname> <instance>
#	  Delete the <instance> of <classname> completely and call the
#	  destructor of <classname> and all its base classes.
#
#     enterFunction <classname> <instance>
# 	  This function is called as the very first statement of a class
# 	  member function. It will setup the 'this' and links each registered
# 	  variable to the local variable with the same name.
#
#     callInstanceFunction <classname> <instance> <args>
# 	   Call the class's member function and let it operate on the
# 	   specified instance. Catch the special function 'delete' which
# 	   deletes the instance, deletes it's instance namespace and
# 	   removes the instance function.
#

namespace eval Tclpp {
    variable ClassInfo

    ###################################################
    # 
    # Procedure: registerClass <classname> <baseclasses>
    #
    ###################################################

    proc registerClass {classname} {
        variable ClassInfo

	# Check if class already exists
	if [info exists ClassInfo($classname,hierarchy)] {
	    error "Class '$classname' already defined"
	}
	
	# Define the ClassInfo variables
	set ClassInfo($classname,hierarchy)        [list]
	set ClassInfo($classname,allhierarchy)     [list]
	set ClassInfo($classname,variables)        [list]
	set ClassInfo($classname,allvariables)     [list]
	set ClassInfo($classname,functions)        [list]
	set ClassInfo($classname,virtualfunctions) [list]
    }

    ###################################################
    # 
    # Procedure: registerBaseClass <classname> <baseclass>
    #
    ###################################################

    proc registerBaseClass {classname baseclass} {
        variable ClassInfo

	# Check if base class exists
	if {! [info exists ClassInfo($baseclass,variables)]} {
	    error "Base '$baseclass' for '$classname' unknown"
	}

        # Check if direct base class already exists. 
	if {[lsearch $ClassInfo($classname,hierarchy) $baseclass] != -1} {
	    error "Base '$baseclass' for '$classname' already defined"
	}
	lappend ClassInfo($classname,hierarchy) $baseclass

	# Check if base class already exists in the entire hiearchy. If so,
	# we have the diamond shape in some form. In that case we just return.
	if {[lsearch $ClassInfo($classname,allhierarchy) $baseclass] != -1} {
	    return
	}

	# Append the base class to the entire class hierarchy and also all its
	# baseclasses without inserting duplicates. 
	foreach base $ClassInfo($baseclass,allhierarchy) {
	    if {[lsearch $ClassInfo($classname,allhierarchy) $base] == -1} {
	        lappend ClassInfo($classname,allhierarchy) $base
	    }
	}
	lappend ClassInfo($classname,allhierarchy) $baseclass

	# Set the list of all variables to all the base class defined
	# variables of all the baseclasses. Check for redefinitions.
	set ClassInfo($classname,allvariables) [list]
	foreach base $ClassInfo($classname,allhierarchy) {
	    foreach basevardef $ClassInfo($base,variables) {
	        set basevartype [lindex $basevardef 0]
	        set basevarname [lindex $basevardef 1]
	        foreach vardef $ClassInfo($classname,allvariables) {
	            set vartype [lindex $vardef 0]
	            set varname [lindex $vardef 1]
		    set varbase [lindex $vardef 2]
		    if {$basevarname == $varname} {
		        error "Variable '$varname' defined by base '$base'\
		               already defined by base '$varbase' (ambiguity)"
		    }
		}
	        lappend ClassInfo($classname,allvariables) $basevardef
	    }
	}
    }

    ###################################################
    # 
    # Procedure: registerVariable <classname> <variabledefinition>
    #
    ###################################################
    
    proc registerVariable {classname variabledefinition} {
        if {[info commands _variable] != ""} {
            _variable ClassInfo
	} else {
	    variable ClassInfo
	}

	# Check if the variable definition already exists. This includes the
	# ones defined by the base-classes.
	foreach vardef $ClassInfo($classname,allvariables) {
	    set type [lindex $vardef 0]
	    set name [lindex $vardef 1]
	    set base [lindex $vardef 2]

	    if {$name == [lindex $variabledefinition 1]} {
	        error "Variable '$name' in class '$classname' already defined\
		       by '$base' (ambiguity)"
	    }
	}

	# Append it to the list of direct and all variable definitions
	lappend variabledefinition                 $classname
	lappend ClassInfo($classname,variables)    $variabledefinition
	lappend ClassInfo($classname,allvariables) $variabledefinition
    }

    ###################################################
    #
    # Procedure: registerFunction <classname> <functionname> <virtual>
    #
    ###################################################

    proc registerFunction {classname functionname virtual} {
        if {[info commands _variable] != ""} {
            _variable ClassInfo
	} else {
	    variable ClassInfo
	}

	# Check if one of the reserved functions is registered.
	if {[lsearch "delete isA isKindOf" $functionname] != -1} {
	    error "Reserved function '$functionname' redefined in '$classname'."
	}

	# Add it to the list of functions.
	if {$virtual == "yes"} {
	    lappend ClassInfo($classname,virtualfunctions) $functionname
	} else {
	    lappend ClassInfo($classname,functions) $functionname
	}
    }

    ###################################################
    #
    # Procedure: classInfo <classname> <property>
    #
    ###################################################

    proc classInfo {classname property} {
        variable ClassInfo
	return $ClassInfo($classname,$property)
    }

    ###################################################
    #
    # Procedure: createInstance <classname> <instance>
    #
    ###################################################

    proc createInstance {classname instance} {
        variable ClassInfo

	# Check if class is defined
        if {! [info exists ClassInfo($classname,variables)]} {
	    error "Class '$classname' unknown"
	}

	# Create the object namespace and create the variables
        uplevel 1 [subst {
	    namespace eval ${instance} {
		proc isA {} {
		    return $classname
		}
		proc isKindOf {class} {
		    set allclasses \[Tclpp::classInfo $classname allhierarchy\]
		    lappend allclasses $classname
		    if {\[lsearch \$allclasses \$class\] == -1} {
			return 0
		    } else {
		        return 1
		    }
		}
	        foreach vardef \[Tclpp::classInfo $classname allvariables\] {
		    set type \[lindex \$vardef 0\]
		    set name \[lindex \$vardef 1\]

		    switch \$type {
		        scalar {
	                    eval "variable \$name {}"
			}
			array {
	                    eval "variable \$name"
			}
			default {
			    Tclpp::createInstance \$type ::${instance}::\$name
			}
		    }
	        }
	    }
	}]
        proc ::$instance args [subst {
	    Tclpp::callInstanceFunction $classname $instance \$args
	}]
    }

    ###################################################
    #
    # Procedure: initializeInstance <classname> <instance> <args>
    #
    ###################################################

    proc initializeInstance {classname instance args} {
	foreach vardef [Tclpp::classInfo $classname variables] {
	    set type [lindex $vardef 0]
	    set name [lindex $vardef 1]
	    switch $type {
	        scalar {
		}
		array {
		}
		default {
		    Tclpp::initializeInstance $type ${instance}::$name
		}
	    }
	}
	if {[namespace which -command ${classname}::${classname}] != ""} {
	    if [llength $args] {
	        eval ${classname}::${classname} $instance [join $args]
	    } else {
	        ${classname}::${classname} $instance
	    }
	}
    }

    ###################################################
    #
    # Procedure: deleteInstance <classname> <instance>
    #
    ###################################################

    proc deleteInstance {classname instance} {
	foreach vardef [Tclpp::classInfo $classname variables] {
	    set type [lindex $vardef 0]
	    set name [lindex $vardef 1]
	    switch $type {
	        scalar {
		}
		array {
		}
		default {
		    Tclpp::deleteInstance $type ${instance}::$name
		}
	    }
	}
	foreach base [Tclpp::classInfo $classname hierarchy] {
	    Tclpp::deleteInstance $base $instance
	}
	if {[namespace which -command ${classname}::~${classname}] != ""} {
	    ${classname}::~${classname} $instance
	}
    }

    ###################################################
    #
    # Procedure: enterFunction <classname> <instance>
    #
    ###################################################

    proc enterFunction {classname instance} {
        #
	# Setup the 'this' reference to the instance name of the class. 
	#

	upvar this this
	set this $instance

	#
	# For each class variable, make a link between these variable
	# names and the actual storage space (the object name space).
	#

        variable ClassInfo
	foreach variabledefinition [Tclpp::classInfo $classname allvariables] {
	    set type [lindex $variabledefinition 0]
	    set name [lindex $variabledefinition 1]
	    switch $type {
	        scalar {
	            uplevel 1 [subst {
		        upvar 1 ${instance}::$name $name
		    }]
		}
		array {
	            uplevel 1 [subst {
		        upvar 1 ${instance}::$name $name
		    }]
		}
		default {
	            uplevel 1 [subst {
		        proc $name {args} {
			    upvar this this
			    set function \[lindex   \$args 0\]
			    set args     \[lreplace \$args 0 0\]
			    eval "${type}::\$function \${this}::$name \$args"
			}
		    }]
		}
	    }
	}
    }

    ###################################################
    #
    # Procedure: callInstanceFunction <classname> <instance> <args>
    #
    ###################################################

    proc callInstanceFunction {classname instance args} {
        set args     [lindex   $args 0]
	set function [lindex   $args 0]
	set args     [lreplace $args 0 0]

	if {$function == "delete"} {
	    Tclpp::deleteInstance $classname $instance
	    namespace delete ::$instance
	    rename ::$instance {}
	} else {
            eval "${classname}::$function $instance $args"
	}
    }

	set ::Tclpp::objNum -1
	proc genSym { { base obj } } {
		incr ::Tclpp::objNum
		
		return $base$::Tclpp::objNum
	}
}

proc new { what args } {
	set name [::Tclpp::genSym]
	eval $what $name $args
	return $name
}
