#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/file.h>
#include <sys/fcntl.h>
#include <signal.h>

#include <version.h>
#include <config.h>
#include <support.h>
#include <xcio.h>

#include "option.h"
#include "console.h"
#include "log.h"
#include "phase.h"
#include "mbuf.h"
#include "frame.h"
#include "dev/device.h"

char *ifName;
int ifNum=-1;

extern struct mbuf *queueHead, *queueTail;
extern int devFd;

static struct select_s {
    const char *name;
    int *fdp;
    int (*read)();
    void (*close)();
    bool_t autoclose;
    struct select_s *next;
} *slHead;

static fd_set orgReadFds;
fd_set orgWriteFds;
static int maxFd, ifFd;
struct timeval *timeOutp;
static char *procTitle;
static int lenTitle;

void
MagicInit()
{
    long s;
    struct timeval tv;

    gettimeofday(&tv, NULL);
#if 1
    s = gethostid() ^ tv.tv_sec ^ tv.tv_usec ^ getpid();
#else
    s = tv.tv_sec ^ tv.tv_usec ^ getpid();
#endif
    SRAND(s);
}

struct select_s *
NewSelectLink(const char *name, int *fdp, int (*rfunc)(),
	      void (*cfunc)(), bool_t autoclose, bool_t high)
{
    struct select_s *sp;

    sp = TALLOC(struct select_s);
    sp->name = name;
    sp->fdp = fdp;
    sp->read = rfunc;
    sp->close = cfunc;
    sp->autoclose = autoclose;
    if (high) {
	sp->next = slHead;
	slHead = sp;
    } else {
	struct select_s *sp1;

	if ((sp1 = slHead) == NULL)
	    slHead = sp;
	else while (sp1) {
	    if (sp1->next) sp1 = sp1->next;
	    else {
		sp1->next = sp;
		break;
	    }
	}
	sp->next = NULL;
    }
    if (*fdp >= 0) {
	FD_SET(*fdp, &orgReadFds);
	if (*fdp > maxFd - 1) maxFd = *fdp + 1;
    }
    return(sp);
}

static void
RemoveSelectLink(struct select_s *sp)
{
    struct select_s *p0, *p1;

    p0 = NULL;
    p1 = slHead;
    while (p1) {
	if (p1 == sp) {
	    if (p0) p0->next = p1->next;
	    else slHead = p1->next;
	    if (*sp->fdp >= 0) FD_CLR(*sp->fdp, &orgReadFds);
	    Free(sp);
	    break;
	}
	p0 = p1;
	p1 = p1->next;
    }
}

void
RemoveSelectFdp(int *fdp)
{
    struct select_s *sp;

    sp = slHead;
    while (sp) {
	if (sp->fdp == fdp) {
	    RemoveSelectLink(sp);
	    return;
	}
	sp = sp->next;
    }
}

void
RemoveSelectFd(int fd)
{
    struct select_s *sp;

    sp = slHead;
    while (sp) {
	if (*sp->fdp == fd) {
	    RemoveSelectLink(sp);
	    return;
	}
	sp = sp->next;
    }
}

void
CloseSelectFds(int *efds)
{
    struct select_s *sp;
    int *ep;

    sp = slHead;
    while (sp) {
	for (ep = efds; *ep >= 0; ep ++)
	    if (*ep == *sp->fdp) break;
	if (*ep < 0) close(*sp->fdp);
	sp = sp->next;
    }
}

void
PauseSelectLink(int fd, bool_t sw)
{
    if (fd < 0) return;
    if (sw) FD_CLR(fd, &orgReadFds);
    else {
	FD_SET(fd, &orgReadFds);
	if (fd > maxFd - 1) maxFd = fd + 1;
    }
}

inline static int
ReadSelectLink(fd_set *fdsp, int n)
{
    struct select_s *sp;
    int k;

    sp = slHead;
    while (sp && n > 0) {
	if (*sp->fdp >= 0 && FD_ISSET(*sp->fdp, fdsp)) {
	    if ((k = sp->read(*sp->fdp)) <= 0
		&& errno != EINTR && sp->autoclose) {
		Logf(LOG_ERROR, "closed %s (%d) by peer "
		     "(read=%d,errno=%d)\n", sp->name, *sp->fdp,
		     k, errno);
		if (sp->close) {
		    sp->close(*sp->fdp);
		    if (*sp->fdp >= 0) FD_CLR(*sp->fdp, &orgReadFds);
		    if (sp->fdp == &devFd) {
			phaseShift = PHASE_DOWN;
			if (devFd >= 0) FD_CLR(devFd, &orgWriteFds);
		    }
		    *sp->fdp = -1;
		}
		RemoveSelectLink(sp);
	    }
	    n --;
	}
	sp = sp->next;
    }
    return(0);
}

inline static bool_t
FlushNext()
{
    struct mbuf *qp=queueHead;

    if (!(pppInfo.l_stat & LSTAT_NLINK)
	&& queueHead->m_pri < PRI_NPMAX) return(FALSE);
#ifdef	QUEUE_ID
    if (ISLOG(LOG_DUMP))
	Logf(LOG_DUMP, "flush qid=%d/%d\n", qp->id, qp->m_pri);
#endif
    DevWrite(devFd, qp->m_data, qp->m_len, qp->m_proto);
    queueHead = qp->m_next;
    Free(qp->m_data);
    Free(qp);
    if (queueHead) queueHead->m_prev = NULL;
    return(queueHead ? TRUE: FALSE);
}

static void
SigHup()
{
    extern int CmdDisconnect();

#ifdef	ONESHOT_SIGNAL
    signal(SIGHUP, SigHup);
#endif
    if (ISLOG(LOG_OS)) Logf(LOG_OS, "received SIGHUP\n");
    if (pppInfo.l_stat & LSTAT_PPP) CmdDisconnect(0, NULL);
}

inline static void
Daemon()
{
    int fd;
    pid_t parent, child;

    parent = getpid();
    if ((child = fork()) == -1) return;
    if (child) {
	static void QuitParent(int sig) {
	    exit(0);
	}
	signal(SIGTERM, QuitParent);
	pause();
    }
    setsid();

    fd = open("/dev/null", O_RDWR, 0644);

    if (fd >= 0) {
	dup2(fd, fileno(stderr));
	dup2(fd, fileno(stdout));
	Close(fileno(stdin));
	Close(fd);
    }
    kill(parent, SIGTERM);
}

inline static int
Setups(int argc, char *argv[], char *env[])
{
    int a, n;
    char *p, *line=NULL;
    bool_t daemon=TRUE;
    runmode_t mode=RUN_MAX;
    extern void EnvSetup(), PathsInit(), TcpSetup(), PtySetup();
    extern void SioSetup(), LogSetup(), AuthSetup(), IfSetup();
    extern void XcSetup(), CommandSetup(), FrameSetup();
    extern void IpSetup(), IdleTimerSetup();
    extern int SysIfOpen();
    int dargc = 2;
    char *dargv[] = { "ppxp", "/ppxprc", NULL };

    procTitle = argv[0];
    p = argv[argc - 1] + strlen(argv[argc - 1]);
    lenTitle = p - procTitle;

    if ((p = strrchr(argv[0], '/')) != NULL) p ++;
    else p = argv[0];
    for (n = a = 1; a < argc; a ++) {
	if (*argv[a] == '-') {
	    if (!strcmp(argv[a], "-debug")) daemon = FALSE;
	    else if (!strcmp(argv[a], "-direct")) {
		daemon = FALSE;
		mode = RUN_DIRECT;
	    } else if (!strcmp(argv[a], "-getty")) {
		if ((a + 1) < argc && *argv[a + 1] == '/') {
		    a ++;
		    line = Strdup(argv[a]);
		}
		daemon = FALSE;
		mode = RUN_GETTY;
	    } else {
		printf("PPxP version "VERSION" ("CONFIGURED")\n");
		if (strcmp(argv[a], "-version")) {
		    printf("  -debug    Debug mode\n"
			   "  -getty    Getty mode\n"
			   "  -direct   Direct mode\n"
			   "  -version  Show version\n");
		}
		exit(0);
	    }
	} else {
	    argv[n] = argv[a];
	    n ++;
	}
    }
    argc = n;
    EnvSetup(VERSION);
    PathsInit();
    TcpSetup();
    PtySetup();
    SioSetup();
    LogSetup(p);
    AuthSetup();
    IfSetup();
    if ((ifFd = SysIfOpen(ifName, &ifNum)) < 0) {
	perror("SysIfOpen");
	return(-1);
    }
    XcSetup(ifName);
    CommandSetup();

    if (mode == RUN_MAX) printf("\rinterface: %s\n", ifName);
    if (daemon) Daemon();
    signal(SIGHUP, SigHup);

    FrameSetup();
    IpSetup();
    IdleTimerSetup();

    LoadOption(dargc, dargv);
    if (argc > 1) {
	LoadOption(argc, argv);
	/*
	pppOpt.name = Strdup(argv[1]);
	pppOpt.a_entry = Strdup(argv[1]);
	for (n = 1; n < argc; n ++) LoadScriptFile(argv[n]);
	*/
	if (mode != RUN_MAX) pppOpt.mode = mode;
	if (line) {
	    char *av[]={"line", line};

	    EnvLine(2, av, NULL);
	    Free(line);
	}
    }
    return(0);
}

void
SetProcTitle(char *title)
{
    strncpy(procTitle, title, lenTitle);
}

int
main(int argc, char *argv[], char *env[])
{
    extern int IfRead();
    int n;
    fd_set rfds, wfds;

    FD_ZERO(&orgReadFds);
    FD_ZERO(&orgWriteFds);
    MagicInit();
    if (Setups(argc, argv, env) < 0) exit(0);
    NewSelectLink("I/F", &ifFd, IfRead, NULL, FALSE, TRUE);
    SuperPrivilege(FALSE);
    Logf(LOG_OS, "PPxP version "VERSION" using %s\n", ifName);
    if (pppOpt.mode == RUN_DIRECT) {
	pppInfo.l_stat = LSTAT_TTY;
	devFd = 0;
	DirectSetup();
	NewSelectLink("DEVICE", &devFd, DevRead, DevClose, TRUE, TRUE);
	LcpMode(pppOpt.mode);
	PhaseUp();
    } else if (pppOpt.mode == RUN_GETTY) {
	char *av[]={"c", "getty", NULL};

	CmdConnect(2, av);
    }
    while (1) {
	/*	FlushQueue();*/
	rfds = orgReadFds;
	wfds = orgWriteFds;
	n = select(maxFd, &rfds, &wfds, NULL, NULL);
	if (n < 0 && errno == EINTR) {
	    errno = 0;
	    continue;
	}
	if (devFd >= 0 && FD_ISSET(devFd, &wfds)) {
	    if (!FlushNext()) FD_CLR(devFd, &orgWriteFds);
	    if (n == 1) continue;
	}
	ReadSelectLink(&rfds, n);
	if (phaseShift != PHASE_KEEP) switch(phaseShift) {
	case PHASE_UP:
	    PhaseUp();
	    break;
	case PHASE_DOWN:
	    PhaseDown();
	    break;
	case PHASE_PREV:
	    PhasePrev();
	    break;
	}
    }
    return(0);
}
