/*****
*
* Copyright (C) 2001, 2002, 2003 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <ctype.h>
#include <errno.h>
#include <assert.h>

#include "rules.h"

#include <libprelude/list.h>
#include <libprelude/prelude-log.h>
#include <libprelude/extract.h>

#include "string-matching.h"
#include "rules-default.h"
#include "rules-operations.h"
#include "rules-default.h"
#include "rules-parsing.h"
#include "rules-type.h"
#include "optparse.h"



#define MIN(x, y) ((x) < (y)) ? (x) : (y)

        

typedef struct {
	uint16_t depth;
	uint16_t offset;
	uint8_t nocase;
        uint16_t within;
        uint16_t distance;
	bm_string_t *s;
} string_t;


static string_t *string_ptr = NULL;
static uint16_t detect_offset_end = 0;


/*
 * Packet matching related function.
 */

int signature_match_content(unsigned char *pdata, unsigned int dlen, void *pattern) 
{
        int ret;
	string_t *str = pattern;
        uint32_t distance_adjustment = 0;
        
        if ( str->offset >= dlen )
		return -1;

        dlen -= str->offset;
	pdata += str->offset;

        /*
         * If a "depth" was provided for this content test,
         * set data len to this depth in order to only tests what
         * interest us.
         *
         * if not, depth is 65535, so len will never be greater than.
         */
	dlen = MIN(dlen, str->offset + str->depth);

	if ( str->distance || str->within ) {
        	distance_adjustment = detect_offset_end + str->distance;
                
        	if ( distance_adjustment >= dlen )
                	goto err;
        	else {
                	dlen -= distance_adjustment;
                	pdata = pdata + distance_adjustment;
        	}
                
                /*
                 * if the 'within' attribute is specified,
                 * we need to find the pattern within X specified first bytes
                 * of data.
                 */
                if ( str->within )
                        dlen = MIN(dlen, str->within);
	}
        
        if ( dlen < str->s->length )
                goto err;
        
	if ( str->nocase )
		ret = BoyerMoore_CI_StringMatching(pdata, dlen, str->s->pattern, str->s->length,
                                                   str->s->bad_character_shift, str->s->good_suffix_shift);
	else
		ret = BoyerMoore_StringMatching(pdata, dlen, str->s->pattern, str->s->length,
                                                str->s->bad_character_shift, str->s->good_suffix_shift);
       
        if ( ret < 0 )
                goto err;

        detect_offset_end = distance_adjustment + ret + str->offset + str->s->length;
        
        return 0;

  err:
        detect_offset_end = 0;
        return -1;
}




int signature_match_packet_content(packet_container_t *packet, void *pattern) 
{
        unsigned int dlen;
        unsigned char *pdata;
        int depth = packet->application_layer_depth;
        
        if ( depth < 0 )
                return -1;

        assert(packet->packet[depth].proto == p_data);
        
        dlen = packet->packet[depth].len;
        pdata = packet->packet[depth].p.data;

        return signature_match_content(pdata, dlen, pattern);
}




int signature_match_ipsrc(packet_container_t *pc, ip_t *test) 
{
        struct in_addr src;
        iphdr_t *ip = pc->packet[pc->network_layer_depth].p.ip;

        src = extract_ipv4_addr(&ip->ip_src);
        
        if ( (src.s_addr & test->netmask) == test->ip ) 
                return 0;
        
        return -1;
}



int signature_match_ipdst(packet_container_t *pc, ip_t *test) 
{
        struct in_addr dst;
        iphdr_t *ip = pc->packet[pc->network_layer_depth].p.ip;

        dst = extract_ipv4_addr(&ip->ip_dst);
        
        if ( (dst.s_addr & test->netmask) == test->ip ) 
                return 0;
        
        return -1;
}



int signature_match_sameip(packet_container_t *pc, void *data) 
{
        iphdr_t *hdr;
        struct in_addr src, dst;
        
        hdr = pc->packet[pc->network_layer_depth].p.ip;

        src = extract_ipv4_addr(&hdr->ip_src);
        dst = extract_ipv4_addr(&hdr->ip_dst);
        
        return (src.s_addr == dst.s_addr) ? 0 : -1;
}



int signature_match_ipproto(packet_container_t *pc, integer_t *integer) 
{
        iphdr_t *ip = pc->packet[pc->network_layer_depth].p.ip;

        if ( integer->num == ip->ip_p )
                return 0;

        return -1;
}



int signature_match_portsrc(packet_container_t *pc, segment_t *port) 
{
        int depth;
        uint16_t sport;
        
        depth = pc->transport_layer_depth;
        if ( depth < 0 )
                return -1;

	switch ( pc->packet[depth].proto ) {

        case p_tcp:
                sport = extract_uint16(&pc->packet[depth].p.tcp->th_sport);
		break;

	case p_udp:
                sport = extract_uint16(&pc->packet[depth].p.udp_hdr->uh_sport);
		break;

	default:
		return -1; 
	}

        if ( sport >= port->lo && sport <= port->hi )
                return 0;

        return -1;
}




int signature_match_portdst(packet_container_t *pc, segment_t *port) 
{
	int depth;
        uint16_t dport;
        
        depth = pc->transport_layer_depth;
        if ( depth < 0 )
                return -1;
        
	switch( pc->packet[depth].proto ) {
                
	case p_tcp:
                dport = extract_uint16(&pc->packet[depth].p.tcp->th_dport);
		break;

	case p_udp:
                dport = extract_uint16(&pc->packet[depth].p.udp_hdr->uh_dport);
		break;

	default:
		return -1; 
	}
        
	if ( dport >= port->lo && dport <= port->hi ) 
                return 0;

        return -1;
}



int signature_match_tcpflags(packet_container_t *pc, flags_t *flags) 
{
        int depth = pc->transport_layer_depth;
        
	if ( depth < 0 || pc->packet[depth].proto != p_tcp )
		return -1; 

	if ( (pc->packet[depth].p.tcp->th_flags & flags->mask) == flags->flags )
                return 0;

        return -1;
}



int signature_match_fragbits(packet_container_t *pc, flags_t *fragbits) 
{
        uint16_t offset;
        int depth = pc->network_layer_depth;

        offset = extract_uint16(&pc->packet[depth].p.ip->ip_off);
        offset &= ~IP_OFFMASK;
        
	if ( (offset & fragbits->mask) == fragbits->flags )
                return 0;

        return -1;
}



int signature_match_fragoffset(packet_container_t *pc, integer_t *integer) 
{
        uint16_t offset;
        int depth = pc->network_layer_depth;

        offset = extract_uint16(&pc->packet[depth].p.ip->ip_off);

        return ( (offset & IP_OFFMASK) == integer->num ) ? 0 : -1;
}



int signature_match_ttl(packet_container_t *pc, integer_t *ttl) 
{
        int depth = pc->network_layer_depth;
        
	if ( pc->packet[depth].p.ip->ip_ttl == ttl->num )
                return 0;

        return -1;
}



int signature_match_tos(packet_container_t *pc, integer_t *tos) 
{
        int depth = pc->network_layer_depth;
        
	if ( pc->packet[depth].p.ip->ip_tos == tos->num )
                return 0;

        return -1;
}



int signature_match_id(packet_container_t *pc, integer_t *id) 
{
        uint16_t ip_id;
        int depth = pc->network_layer_depth;

        ip_id = extract_uint16(&pc->packet[depth].p.ip->ip_id);
	if ( ip_id == id->num )
                return 0;

        return -1;
}



int signature_match_datasize(packet_container_t *pc, segment_t *dsize) 
{
        int len;
        int depth = pc->application_layer_depth;
        
	if ( depth < 0 || pc->packet[depth].proto != p_data )
                len = 0;
        else
                len = pc->packet[depth].len;
        
	if ( len >= dsize->lo && len <= dsize->hi )
                return 0;

        return -1;
}



int signature_match_win(packet_container_t *pc, integer_t *win) 
{
        uint16_t th_win;
        int depth = pc->transport_layer_depth;

        if ( depth < 0 || pc->packet[depth].proto != p_tcp )
                return -1;

        th_win = extract_uint16(&pc->packet[depth].p.tcp->th_win);
        if ( th_win == win->num )
                return 0;
        
        return -1;
}



int signature_match_seq(packet_container_t *pc, integer_t *seq) 
{
        uint32_t th_seq;
        int depth = pc->transport_layer_depth;
        
	if ( depth < 0 || pc->packet[depth].proto != p_tcp )
		return -1;

        th_seq = extract_uint32(&pc->packet[depth].p.tcp->th_seq);
	if ( th_seq == seq->num )
                return 0;

        return -1;
}



int signature_match_ack(packet_container_t *pc, integer_t *ack) 
{
        uint32_t th_ack;
        int depth = pc->transport_layer_depth;
        
	if ( depth < 0 || pc->packet[depth].proto != p_tcp )
		return -1;

        th_ack = extract_uint32(&pc->packet[depth].p.tcp->th_ack);
	if ( th_ack == ack->num )
                return 0;

        return -1;
}



int signature_match_icmptype(packet_container_t *pc, integer_t *itype) 
{
        int depth = pc->network_layer_depth + 1;
        
	if ( depth < 0 || pc->packet[depth].proto != p_icmp )
		return -1;

	if ( pc->packet[depth].p.icmp_hdr->icmp_type == itype->num )
                return 0;

        return -1;
}



int signature_match_icmpcode(packet_container_t *pc, integer_t *icode) 
{
        int depth = pc->network_layer_depth + 1;
        
	if ( depth < 0 || pc->packet[depth].proto != p_icmp )
		return -1;
        
	if ( pc->packet[depth].p.icmp_hdr->icmp_code == icode->num )
                return 0;

        return -1;
}



int signature_match_icmpid(packet_container_t *pc, integer_t *test) 
{
        uint16_t id;
        int depth = pc->network_layer_depth + 1;
        
	if ( depth < 0 || pc->packet[depth].proto != p_icmp )
		return -1;

        id = extract_uint16(&pc->packet[depth].p.icmp_hdr->icmp_id);
	if ( id == test->num )
                return 0;

        return -1;
}



int signature_match_icmpseq(packet_container_t *pc, integer_t *test) 
{
        uint16_t seq;
        int depth = pc->network_layer_depth + 1;
        
	if ( depth < 0 || pc->packet[depth].proto != p_icmp )
		return -1;

        seq = extract_uint16(&pc->packet[depth].p.icmp_hdr->icmp_seq);
	if ( seq == test->num )
                return 0;

        return -1;
}



int signature_match_ipopts(packet_container_t *pc, void *data)
{
        flags_t *ipopts = data;
	int depth = pc->network_layer_depth + 1;
        
	if ( pc->packet[depth].proto != p_ipopts )
		return -1;

	return option_is_set(ipopts->flags, 
                             pc->packet[depth].p.opts, 
                             pc->packet[depth].len);
}




/*
 * Parsing function.
 *
 * TODO :
 * - session:  ?
 * - content-list: 
 * - regex: 
 * - uricontent: (HTTP protocol plugin)
 */



int signature_parse_nothing(parameters_t *parameters, rules_t **rules) 
{
        *rules = NULL;
        return 0;
}




int signature_parse_sameip(parameters_t *parameters, rules_t **rules) 
{
        rule_t *rule;
        
        rule = make_new_rule(0, NULL);
	add_rule_leaf_match(rule, 0, NULL, &signature_match_sameip);
	*rules = make_new_rules(rule, NULL);

        return 0;
}



int signature_parse_nocase(parameters_t *parameters, rules_t **rules)
{
	if ( parameters ) {
		signature_parser_set_error("nocase keyword don't take parameters\n");
		return -1;
	}

	if ( ! string_ptr ) {
		signature_parser_set_error("nocase keyword must be used after the content key\n");
		return -1;
	}

	string_ptr->nocase = 1;
	BoyerMoore_MakeCaseInsensitive(string_ptr->s);

        return 0;
}



int signature_parse_offset(char *str, rules_t **rules) 
{
        if ( ! string_ptr ) {
		signature_parser_set_error("offset keyword must be used after the content key\n");
		return -1;
        }

	string_ptr->offset = atoi(str);
        return 0;
}



int signature_parse_depth(char *str, rules_t **rules) 
{
        if ( ! string_ptr ) {
		signature_parser_set_error("depth keyword must be used after the content key\n");
		return -1;
        }

        string_ptr->depth = atoi(str);
	return 0;
}



int signature_parse_within(char *str, rules_t **rules) 
{
        if ( ! string_ptr ) {
                signature_parser_set_error("within keyword must be used after the content key\n");
		return -1;
        }

        string_ptr->within = atoi(str);

        return 0;
}



int signature_parse_distance(char *str, rules_t **rules) 
{
        if ( ! string_ptr ) {
                signature_parser_set_error("distance keyword must be used after the content key\n");
		return -1;
        }

        string_ptr->distance = atoi(str);

        return 0;
}



/*
 * Contents functions are public, because there is several place
 * that might use them.
 */

rule_t *signature_get_content_rule(char *str, rules_t **rules) 
{
        enum { literal, hexa, normal } mode = normal;
	int hexadata = 0;
	int hexalast = 0;
	int cur = 0;
	char s[2048];
	rule_t *rule;
        
	if ( strlen(str) == 0 ) {
		signature_parser_set_error("Missing value for content option");
		return NULL;
	}
        
        while ( *str ) {

		if ( cur >= sizeof(s) - 1 ) {
			signature_parser_set_error("Content string too long\n");
			return NULL;
		}

                if ( mode == normal && *str == '\\' ) {
                        if ( mode == hexa ) {
                                 signature_parser_set_error("Invalid \\ in hexa mode");
                                 return NULL;
                        }

                        mode = literal;
                }

                else if ( mode == literal ) {
                        mode = normal;
                        s[cur++] = *str;
                }
                
                else if ( *str == '|' ) 
                        mode = (mode == hexa) ? normal : hexa;
                
                else if ( mode == normal ) 
                        s[cur++] = *str;

                else if ( mode == hexa ) {
                        
                        switch (*str) {

                        case ' ':
                                break;
                                
                        case '0' ... '9':
                                if (hexalast) 
                                        s[cur++] = hexadata * 16 + (*str) - '0';
                                else
                                        hexadata = (*str) - '0';
                                hexalast = !hexalast;
                                break;
                                
			case 'a' ... 'f':
                                if (hexalast) 
                                        s[cur++] = hexadata * 16 + (*str) - 'a' + 10;
                                else
                                        hexadata = (*str) - 'a' + 10;
                                hexalast = !hexalast;
                                break;
                                
			case 'A' ... 'F':
                                if (hexalast) 
                                        s[ cur++ ] = hexadata * 16 + (*str) -'A' + 10;
                                else
                                        hexadata = (*str) - 'A' + 10;
                                hexalast = !hexalast;
                                break;

                        default:
                                signature_parser_set_error("Invalid character in hexa buffer '%c'", *str);
                                return NULL;
                        }
                }

                str++;
        }

	s[cur] = '\0';

	string_ptr = calloc(1, sizeof(string_t));
	if ( ! string_ptr ) {
		log(LOG_ERR, "memory exhausted.\n");
		return NULL;
	}
	
	string_ptr->s = BoyerMoore_Init(s, cur);
	if ( ! string_ptr->s )
		return NULL;

        string_ptr->depth = 65535;

	rule = make_new_rule(0, NULL);
	add_rule_leaf_match(rule, RULE_CONTENT, string_ptr, &signature_match_packet_content);
        //*rules = make_new_rules(rule, NULL);

        return rule;
}



int signature_parse_content(char *str, rules_t **rules)
{
	rule_t *rule;
        
        rule = signature_get_content_rule(str, rules);
        if ( ! rule )
                return -1;
        
        *rules = make_new_rules(rule, NULL);
        
        return 0;
}



static int set_nocase(rules_t *rules) 
{
	string_ptr = NULL;
	
	return 0;
}




static rules_node_t *ip_root, *tcp_root, *udp_root, *icmp_root;



void signature_engine_init(void)
{
	/*
         * add default post processing function
         */
        signature_parser_add_post_processing(set_nocase);
        signature_parser_add_post_processing(validate_rules);
        
	/*
         * make tree
         */
        ip_root = make_rules_root();
	tcp_root = make_rules_root();
	udp_root = make_rules_root();
	icmp_root = make_rules_root();
}



/*
 * get build-in trees
 */
rules_node_t *signature_engine_get_ip_root(void)
{
        return ip_root;
}


rules_node_t *signature_engine_get_tcp_root(void)
{
        return tcp_root;
}


rules_node_t *signature_engine_get_udp_root(void)
{
        return udp_root;
}


rules_node_t *signature_engine_get_icmp_root(void)
{
        return icmp_root;
}



void signature_engine_start_new_match(void)
{
        detect_offset_end = 0;
}



uint16_t signature_engine_get_last_matched_offset(void)
{
        return detect_offset_end;
}


void signature_engine_set_last_matched_offset(uint16_t off)
{
        detect_offset_end = off;
}
