#!/usr/bin/perl -w

##############################################################################
#
# Print billing management system - web admin tools, version 4.1.1
#
# Copyright (C) 2000, 2001, 2002 Daniel Franklin
#
# This program is distributed under the terms of the GNU General Public
# License Version 2.
#
# This CGI script allows administrators to check printer stats and manage
# the print quota system via the web.
#
##############################################################################

use Printbill::PTDB_File;
use POSIX;
use Printbill::printbill_pcfg;
use strict;

my $request_method = $ENV{'REQUEST_METHOD'};

my ($config, %params);

# Attempt to read and parse the configuration file

$config = "/etc/printbill/printbillrc";
%params = pcfg ($config);

print "Content-type: text/html\n\n";
print `/bin/cat $params{'web_header'}`;

if ($? >> 8) {
	&return_error (500, "CGI Error",
		"$0: cannot read web header file $params{'web_header'}\n");
}

if ($request_method eq "POST") {
	&handle_form();
} else {
	&setup_admin_check ();
}

print `/bin/cat $params{'web_footer'}`;

if ($? >> 8) {
	&return_error (500, "CGI Error",
		"$0: cannot read web footer file $params{'web_footer'}\n");
}

exit 0;

sub handle_form
{
	my (%formdata);
	
	&parse_form_data (\%formdata);

	if ($formdata{'ftype'} eq "admincheck") {
		&check_admin (\%formdata);
	} elsif ($formdata{'ftype'} eq "setup_editquota_page") {
		&setup_editquota_page (\%formdata);
	} elsif ($formdata{'ftype'} eq "deluser") {
		&deluser (\%formdata);
	} elsif ($formdata{'ftype'} eq "setup_adduser_page") {
		&setup_adduser_page (\%formdata);
	} elsif ($formdata{'ftype'} eq "adduser") {
		&adduser (\%formdata);
	} elsif ($formdata{'ftype'} eq "editquota") {
		&editquota (\%formdata);
	} elsif ($formdata{'ftype'} eq "setup_printer_page") {
		&setup_printer_page (\%formdata);
	} elsif ($formdata{'ftype'} eq "reset_pages") {
		&reset_pages (\%formdata);
	} elsif ($formdata{'ftype'} eq "reset_black") {
		&reset_black (\%formdata);
	} elsif ($formdata{'ftype'} eq "reset_colour") {
		&reset_colour (\%formdata);
	} elsif ($formdata{'ftype'} eq "stats") {
		&stats (\%formdata);
	} elsif ($formdata{'ftype'} eq "setup_change_password_page") {
		&setup_change_password_page (\%formdata);
	} elsif ($formdata{'ftype'} eq "change_password") {
		&change_password (\%formdata);
	}
	
	if ($params{'admin_form_type'} eq "SHORT") {
		if ($formdata{'ftype'} eq "setup_admin_page" || $formdata{'ftype'} eq "setup_short_admin_page1") {
			&setup_short_admin_page1 (\%formdata);
		} elsif ($formdata{'ftype'} eq "setup_short_admin_page2") {
			&setup_short_admin_page2 (\%formdata);
		} elsif ($formdata{'ftype'} eq "cancel") {
			&setup_short_admin_page1 (\%formdata);
		} elsif ($formdata{'ftype'} eq "setup_long_admin_page") {
			&setup_long_admin_page (\%formdata);
		}
	} else {
		if ($formdata{'ftype'} eq "setup_admin_page" || $formdata{'ftype'} eq "setup_long_admin_page" || $formdata{'ftype'} eq "cancel") {
			&setup_long_admin_page (\%formdata);
		}
	}
}

sub check_cookie_file
{
	my ($cookie) = @_;
	my ($file);

	$file = $params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $cookie;

	open (FILE, "<$file") or
		&return_error (500, "CGI Error",
			"$0: cannot open cookie file \"$file\"for reading: $!");

	close (FILE) or
		&return_error (500, "CGI Error",
			"$0: cannot close cookie file: $!");
}

sub parse_form_data
{
	my ($form_data) = @_;
	
	my ($post_info, @key_value_pairs, $key_value, $key, $value);
	
	read (STDIN, $post_info, $ENV{'CONTENT_LENGTH'});
	
	@key_value_pairs = split (/&/, $post_info);
	
	foreach $key_value (@key_value_pairs) {
		($key, $value) = split (/=/, $key_value);
		
		$value =~ tr/+/ /;
		$value =~ s/%([\dA-Fa-f][\dA-Fa-f])/pack ("C", hex ($1))/eg;
		$value =~ s/;//g;

# Escape any single quotes. Other dodgy characters are then encased in
# single quotes throughout the document. I think it should be reasonably
# secure (famous last words!)

		$value =~ s/\'/\\\'/g;

		$$form_data{$key} = $value;
	}
}

sub return_error
{
	my ($status, $keyword, $message) = @_;

	print "<title>CGI Program - Unexpected Error</title>";
	print "<h1>$keyword</h1><hr>$message<hr>";
	print "Status: $status $keyword\n\n";
	print "</body></html>";

	exit 1;
}

sub setup_admin_check
{
	my ($args) = @_;
	my ($cookie, $cookiefile);
	
	# setup header information
	print `/bin/cat $params{'web_message_text'}`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	$cookie = time;
	$cookiefile = "$ENV{'REMOTE_ADDR'}_$cookie";

	`/bin/rm -f $params{'cookieroot'}/$ENV{'REMOTE_ADDR'}*`;
		
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot remove old cookie(s)");
	}

	open (FILE, ">$params{'cookieroot'}/$cookiefile") or
		&return_error (500, "CGI Job Administration Error",
			"$0: cannot create cookie file");

# Stick something in there

	print FILE $$;
	close (FILE);

	print "<center><h2>Print Quota Administration System</h2></center>\n";

	print "<hr>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "Please enter your username and password:<br><br>\n";
	print "<center><table border=0>\n";
	print "<tr><td width=100 align=left>Username:</td><td width=100 align=right><input type=\"text\" name=\"username\" size=\"10\"></td></tr>\n";
	print "<tr><td align=left>Password:</td><td align=right><input type=\"password\" name=\"password\" size=\"10\"></td></tr>\n";
	print "<tr><td align=left><input type=\"submit\" value=\"Submit\"></td>\n";
	print "<td align=right><input type=\"reset\" value=\"Reset\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"admincheck\"></td></tr>\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$cookie\"></td></tr>\n";
	print "</table></center></form>\n";
	print "<p><hr>\n";
}

sub check_admin
{
	my ($args) = @_;
	my ($username, $password, $newcookie, $command, %adminhash, $cookie);
	
	$username = $$args{'username'};
	$password = $$args{'password'};
	$cookie = $$args{'cookie'};
	
	tie %adminhash, "Printbill::PTDB_File", "$params{'db_home'}/web_admin.db", "TRUE"
		or &return_error (500, "CGI Error", 
			"$0: cannot open file $params{'db_home'}/web_admin.db: $!\n");

# Sleep a while - if some bastard is trying to brute-force us, this should
# slow him/her down, at least long enough for a sysadmin to notice. Admins
# shouldn't object to a 2 second delay.

	sleep (2);

	if (!defined ($adminhash{$username}) or crypt ($password, substr ($adminhash{$username}, 0, 2)) ne $adminhash{$username}) {
		print `/bin/cat $params{'web_message_text'}`;
		
		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot read web message text file $params{'web_message_text'}");
		}

		print "<p>Invalid password for $username!\n";
	} else {
		my $file = $params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $cookie;
	
		open COOKIE, ">$file";

# Now the username is in the cookie... this can be used later.
	
		print COOKIE $username;
	
		close COOKIE;

		if ($params{'admin_form_type'} eq "SHORT") {
			&setup_short_admin_page1 ($args);
		} else {
			&setup_long_admin_page ($args);
		}
	}
	
	untie %adminhash;
}

sub setup_short_admin_page1
{
	my ($args) = @_;
	&check_cookie_file ($$args{'cookie'});

	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}

	print "<center><h2>Print Quota Administration Page<br>";
	print "This is a restricted system</h2></center>\n";
	
	print "<hr>\n";

	print "<center><table>\n";
	print "<tr><td colspan=5 align=center>\n";
	print "<table width=100%><tr><td width=50% align=center valign=middle>Username:</td>";
	
	print "<td width=50% align=center valign=middle><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"text\" name=\"user\" size=\"15\"></td></tr></table>\n";
	
	print "<tr><td valign=top align=left><input type=\"submit\" value=\"Submit\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page2\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></td>\n";
	print "</form>";
	
	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"View All\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_long_admin_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td>";
	
	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Add User\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_adduser_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td>\n";
	
	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"View Printers\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_printer_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td>\n";

	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Change Password\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_change_password_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td></tr>\n";
 
	print "</table></center>\n";
	print "<p><hr>\n";
}

sub setup_short_admin_page2
{
	my ($args) = @_;
	my ($referrer, $username, $pages, $paid, $spent, $quota, $answer);

	&check_cookie_file ($$args{'cookie'});

	$referrer = $$args{'referrer'};
	$username = $$args{'user'};
	
	print `/bin/cat $params{'web_message_text'}`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}

	print "<center><h2>Print Quota Administration Page<br>";
	print "This is a restricted system</h2></center>\n";

	if ($username ne "") {
		$answer = `$params{'admin_prog_path'}/pqm --display \'$username\' --web 2>&1`;
	
		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --display \'$username\' --web: $answer");
		} else {
			$quota = $answer;
		}
	}

	if ($? >> 8 or !defined ($quota)) {
		print "<p>Error: user \"$username\" is unknown.\n";
	} else {
		$answer = `$params{'admin_prog_path'}/pqm --paid --web 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --paid --web: $answer");
		} else {
			$paid = $answer;
		}

		$answer = `$params{'admin_prog_path'}/pqm --spent --web 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --spent --web: $answer");
		} else {
			$spent = $answer;
		}
	
		print "<hr>\n";
		print "<p>Global total money paid into the system: $paid";
		print "<p>Global total amount actually spent: $spent";
	
		print "<p><center><table cellpadding=2 cellspacing=2 border=1>";
		print "<thead><td><b>Username</b></td>\n";
		print "<td><b>Quota Remaining</b></td>\n";
		print "<td><b>Total Spent</b></td>\n";
		print "<td><b>Pages Printed</b></td>\n";
		print "<td colspan=2 align=center><b>Controls</b></td>\n</thead>\n";
	
		print "<tbody>";
	
		$pages = `$params{'admin_prog_path'}/pqm --pages \'$username\' --web 2>&1`;
		
		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --pages \'$username\' --web");
		}

		$spent = `$params{'admin_prog_path'}/pqm --spent \'$username\' --web 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --spent \'$username\' --web");
		}

		$quota = `$params{'admin_prog_path'}/pqm --display \'$username\' --web 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --display \'$username\' --web");
		}

# Attempt to make the resulting HTML a bit more readable...
		
		chop $pages;
		chop $spent;
		
		print "<tr><td>$username</td>\n<td>$quota</td>\n";
		print "<td>$spent</td>\n<td>$pages</td>\n";
		
		print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
		print "<input type=\"submit\" value=\"Change\">\n";
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_editquota_page\">\n";
		print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
		print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page2\">\n";
		print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></form></td>\n";

		print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
		print "<input type=\"submit\" value=\"Delete User\">\n";
		print "<input type=\"hidden\" name=\"ftype\" value=\"deluser\">\n";
		print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
		print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page2\">\n";
		print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
		print "</form></td></tr>\n";

		print "</tbody></table></center>\n";
	}
	
	print "<center><p><table>\n";
	
	print "<tr><td colspan=2><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Refresh\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page2\">\n";
	print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page2\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">";
	print "</form></td>";

	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Back\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page1\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_short_admin_page2\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td></tr>\n";

	print "</table></center>\n<p><hr>\n";
}

sub setup_long_admin_page
{
	my ($args) = @_;
	my ($username, $paid, $pages, $cyan, $magenta, $yellow, $black, $spent, @results, $listloop, $quota, %usernamehash);

	&check_cookie_file ($$args{'cookie'});

	$pages = `$params{'admin_prog_path'}/pqm --pages --web 2>&1`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --pages --web");
	}
	
	$paid = `$params{'admin_prog_path'}/pqm --paid --web 2>&1`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --paid --web");
	}

	$spent = `$params{'admin_prog_path'}/pqm --spent --web 2>&1`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --spent --web");
	}
	
	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}

	print "<center><h2>Print Quota Administration Page<br>";
	print "This is a restricted system</h2></center>\n";

	print "<hr>\n";
	print "<p>Global total money paid into the system: $paid";
	print "<p>Global total amount actually spent: $spent";
	
	print "<p><center><table cellpadding=2 cellspacing=2 border=1>";

	print "<thead><td><b>Username</b></td>\n";
	print "<td><b>Quota Remaining</b></td>\n";
	print "<td><b>Total Spent</b></td>\n";
	print "<td><b>Pages Printed</b></td>\n";
	print "<td colspan=2 align=center><b>Controls</b></td>\n</thead>\n";
	
	@results = `$params{'admin_prog_path'}/pqm --display --web 2>&1`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --display --web");
	}

# So that results may be presented in alphabetic order, we need to put them
# into a hash.
	
	for ($listloop = 0; $listloop <= $#results; $listloop++) {
		($username, $quota, $spent, $pages, $cyan, $magenta, $yellow, $black) = split (/&/, $results [$listloop]);
		chop $black;
		
# We currently do not display CMYK consumption, but the information is
# available if you actually care.
		
		$usernamehash{$username}{'quota'} = $quota;
		$usernamehash{$username}{'spent'} = $spent;
		$usernamehash{$username}{'pages'} = $pages;
#		$usernamehash{$username}{'yellow'} = $cyan;
#		$usernamehash{$username}{'magenta'} = $magenta;
#		$usernamehash{$username}{'cyan'} = $yellow;
#		$usernamehash{$username}{'black'} = $black;
	}
	
	print "<tbody>";
	
	foreach $username (sort keys %usernamehash) {
		print "<tr><td>$username</td>\n<td>$usernamehash{$username}{'quota'}</td>\n";
		print "<td>$usernamehash{$username}{'spent'}</td>\n<td>$usernamehash{$username}{'pages'}</td>\n";
		
		print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
		print "<input type=\"submit\" value=\"Change\">\n";
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_editquota_page\">\n";
		print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
		print "<input type=\"hidden\" name=\"referrer\" value=\"setup_long_admin_page\">\n";
		print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></form></td>\n";

		print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
		print "<input type=\"submit\" value=\"Delete User\">\n";
		print "<input type=\"hidden\" name=\"ftype\" value=\"deluser\">\n";
		print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
		print "<input type=\"hidden\" name=\"referrer\" value=\"setup_long_admin_page\">\n";
		print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
		print "</form></td></tr>\n";
	}
	
	print "</tbody></table>\n";
	
	print "<p><table>\n";
	
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<tr><td><input type=\"submit\" value=\"Add User\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_adduser_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_long_admin_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">";
	print "</form></td>\n";

	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"View Printers\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_printer_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_long_admin_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">";
	print "</form></td>\n";

	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Change Password\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_change_password_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_long_admin_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td>\n";

	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Refresh\">";
	
	if ($params{'admin_form_type'} eq "LONG") {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_admin_page\">\n";
	} else {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_long_admin_page\">\n";
	}
	
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">";
	print "</form>";

	if ($params{'admin_form_type'} eq "SHORT") {
		print "<td align=center><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
		print "<input type=\"submit\" value=\"Back\">\n";
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page1\">\n";
		print "<input type=\"hidden\" name=\"referrer\" value=\"setup_long_admin_page\">\n";
		print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
		print "</form></td></tr>\n";
	} else {
		print "</tr>\n";
	}

	print "</table></center>\n<p><hr>\n";
}

sub setup_editquota_page
{
	my ($args) = @_;
	my ($referrer, $username, $current_quota);
	
	&check_cookie_file ($$args{'cookie'});

	$referrer = $$args{'referrer'};
	$username = $$args{'user'};

	$current_quota = `$params{'prog_path'}/pqcheck --user \'$username\'`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'prog_path'}/pqcheck --user \'$username\'");
	}

	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}

	print "<center><h2>Change User Print Quota</h2></center>\n";
	print "<hr>\n";

	printf "<p>Change quota for user \"%s\", currently worth $params{'currency_symbol'}%.2f.", $username, $current_quota;

	print "<center><table>\n";
	print "<tr><td>Change by amount (negatives accepted):</td>\n";
	
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<td><input type=\"text\" name=\"deltaquota\" size=\"10\"></td></tr>\n";
	print "<tr><td valign=top align=left><input type=\"submit\" value=\"Submit\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"editquota\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></td>\n";
	print "<input type=\"hidden\" name=\"user\" value=\"$username\"></td>";
	print "</form>\n";
	
	print "<td align=right><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Cancel\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_editquota_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td></tr>\n";
	print "</table></center>\n";
	
	print "<hr>\n";
}

sub editquota
{
	my ($args) = @_;
	my ($username, $deltaquota, $answer, $absolute_deltaquota);

	&check_cookie_file ($$args{'cookie'});

	$username = $$args{'user'};
	$deltaquota = $$args{'deltaquota'};
	
	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}

	print "<center><h2>Change User Print Quota</h2></center>\n";
	print "<hr>\n";
	
	if (! isreal ($deltaquota)) {
		$answer = "Error: that isn't a real number!\n";
	} elsif ($deltaquota > 0.0) { 
		$answer = `$params{'admin_prog_path'}/pqm --inc \'$username\' --amount \'$deltaquota\' 2>&1`;
	
		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --inc \'$username\' --amount \'$deltaquota\': $answer");
		}
	} elsif ($deltaquota < 0.0) {
		$absolute_deltaquota = abs $deltaquota;
		$answer = `$params{'admin_prog_path'}/pqm --dec \'$username\' --amount \'$absolute_deltaquota\' 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --dec \'$username\' --amount \'$deltaquota\': $answer");
		}
	} else {
		$answer = "Well, that was easy... perhaps if you specified a value I could actually <b>do</b> something with it!\n";
	}

	print $answer;

	open (MAIL, "|$params{'mta'} $params{'admin_mail'}") or &return_error (500, "CGI Error",
		"$0: cannot open mail transport agent $params{'mta'}: $!\n");

	my $admin = &get_user ($params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'});

	print MAIL "Reply-to: $username\n";
	print MAIL "From: $username\n";
	print MAIL "Subject: Print Quota Changes\n\n";
	print MAIL "Changes by $admin:\n";
	print MAIL "------------------\n\n";
	print MAIL $answer;

	close MAIL;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";

	if ($params{'admin_form_type'} eq "SHORT") {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page2\">\n";
		print "<input type=\"hidden\" name=\"user\" value=\"$username\">\n";
	} else {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_admin_page\">\n";
	}
	
	print "<input type=\"hidden\" name=\"referrer\" value=\"editquota\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub setup_adduser_page
{
	my ($args) = @_;
	my ($referrer);

	&check_cookie_file ($$args{'cookie'});

	$referrer = $$args{'referrer'};

	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<center><h2>Add User to Print Quota System</h2></center>\n";
	print "<hr>\n";

	print "<center><table>\n";
	print "<tr><td>New user:</td>\n";
	
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<td><input type=\"text\" name=\"user\" size=\"10\"></td></tr>\n";
	print "<tr><td valign=top align=left><input type=\"submit\" value=\"Submit\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"adduser\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_adduser_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></td>\n";
	print "</form>\n";
	
	print "<td align=right><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Cancel\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_adduser_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td></tr>\n";
	print "</table></center>\n";
	print "<p><hr>\n";
}

sub adduser
{
	my ($args) = @_;
	my ($username, $answer);

	&check_cookie_file ($$args{'cookie'});
	
	$username = $$args{'user'};
	
	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";

	$answer = `$params{'admin_prog_path'}/pqm --add \'$username\' 2>&1`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --add \'$username\': $answer");
	}

	print $answer;

	open (MAIL, "|$params{'mta'} $params{'admin_mail'}") or &return_error (500, "CGI Error",
		"$0: cannot open mail transport agent $params{'mta'}: $!\n");

	my $admin = &get_user ($params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'});

	print MAIL "Reply-to: $username\n";
	print MAIL "From: $username\n";
	print MAIL "Subject: Print Quota System - addition of user $username\n\n";
	print MAIL "Changes by $admin:\n";
	print MAIL "------------------\n\n";
	print MAIL $answer;

	close MAIL;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";

	if ($params{'admin_form_type'} eq "SHORT") {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page1\">\n";
	} else {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_admin_page\">\n";
	}

	print "<input type=\"hidden\" name=\"referrer\" value=\"adduser\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub deluser
{
	my ($args) = @_;
	my ($username, $answer);

	&check_cookie_file ($$args{'cookie'});

	$username = $$args{'user'};

	$answer = `$params{'admin_prog_path'}/pqm --del \'$username\' 2>&1`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --del \'$username\': $answer");
	}
	
	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";
	print $answer;

	open (MAIL, "|$params{'mta'} $params{'admin_mail'}") or &return_error (500, "CGI Error",
		"$0: cannot open mail transport agent $params{'mta'}: $!\n");
	
	my $admin = &get_user ($params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'});

	print MAIL "Reply-to: $username\n";
	print MAIL "From: $username\n";
	print MAIL "Subject: Print Quota System - deletion of user $username\n\n";
	print MAIL "Changes by $admin:\n";
	print MAIL "------------------\n\n";
	print MAIL $answer;

	close MAIL;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";

	if ($params{'admin_form_type'} eq "SHORT") {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_short_admin_page1\">\n";
	} else {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_admin_page\">\n";
	}

	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"deluser\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub setup_printer_page
{
	my ($args) = @_;
	my ($printer, $pages, $colourspace, $used, @allused, $black, @printers, $referrer, $answer);

	&check_cookie_file ($$args{'cookie'});

	$referrer = $$args{'referrer'};

	$pages = `$params{'admin_prog_path'}/pqm --pages --web 2>&1`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --pages --web");
	}
	
	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}

	print "<center><h2>Printer Administration Page<br>";
	print "This is a restricted system<br>";
	print "</h2></center>\n";

	print "<hr>\n";
	print "<p>Global total number of pages printed: $pages";
	
	print "<p><center><table cellpadding=2 cellspacing=2 border=1>";

	print "<thead><td><b>Printer</b></td>\n";
	print "<td><b>Type</b></td>\n";
	print "<td colspan=2><b>Pages</b></td>\n";
	print "<td colspan=2><b>Black Used</b></td>\n";
	print "<td colspan=2><b>Colour (C, M, Y) Used</b></td>\n";
	print "<td></td></thead>\n";
	
	print "<tbody>";
	
	opendir (PRINTERS_DIR, "$params{'db_home'}/printers") or
		&return_error (500, "CGI Error",
			"$0: cannot open directory $params{'db_home'}/printers: $!");

	(@printers = grep { !/^\./ } readdir (PRINTERS_DIR)) or
		&return_error (500, "CGI Error",
			"$0: cannot read from directory $params{'db_home'}/printers: $!");

	closedir (PRINTERS_DIR) or
		&return_error (500, "CGI Error",
			"$0: cannot close directory $params{'db_home'}/printers: $!");
	
	foreach $printer (@printers) {
		$printer =~ s/\.db//;
		$answer = `$params{'admin_prog_path'}/pqm --ppages \'$printer\' --web 2>&1`;
		
		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --ppages \'$printer\'--web: $answer");
		} else {
			$pages = $answer;
		}

		$answer = `$params{'admin_prog_path'}/pqm --colourspace \'$printer\' --web 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --colourspace \'$printer\'--web: $answer");
		} else {
			$colourspace = $answer;
		}

		
		$answer = `$params{'admin_prog_path'}/pqm --used \'$printer\' --web 2>&1`;

		if ($? >> 8) {
			&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --used \'$printer\'--web: $answer");
		} else {
			$used = $answer;
		}
		
		chop $pages;
		chop $colourspace;
		chop $used;
		
		@allused = split (":", (split ("&", $used))[1]);
		
# Attempt to make the resulting HTML a bit more readable...
		
		print "<tr><td>$printer</td>\n<td>$colourspace</td>\n";
		print "<td>$pages</td>\n";
		
		print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
		print "<input type=\"submit\" value=\"Reset\">\n";
		print "<input type=\"hidden\" name=\"ftype\" value=\"reset_pages\">\n";
		print "<input type=\"hidden\" name=\"printer\" value=\"$printer\">\n";
		print "<input type=\"hidden\" name=\"referrer\" value=\"setup_printer_page\">\n";
		print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></form></td>\n";
		
		if ($colourspace ne "cmy") {
			print "<td>$allused[3]\%</td>\n";
			print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
			print "<input type=\"submit\" value=\"Reset\">\n";
			print "<input type=\"hidden\" name=\"ftype\" value=\"reset_black\">\n";
			print "<input type=\"hidden\" name=\"printer\" value=\"$printer\">\n";
			print "<input type=\"hidden\" name=\"referrer\" value=\"setup_printer_page\">\n";
			print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></form></td>\n";
		} else {
			print "<td colspan=2>N/A</td>\n";
		}

		if ($colourspace ne "mono") {
			print "<td>$allused[0]\%, $allused[1]\%, $allused[2]\%</td>\n";

			print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
			print "<input type=\"submit\" value=\"Reset\">\n";
			print "<input type=\"hidden\" name=\"ftype\" value=\"reset_colour\">\n";
			print "<input type=\"hidden\" name=\"printer\" value=\"$printer\">\n";
			print "<input type=\"hidden\" name=\"referrer\" value=\"setup_printer_page\">\n";
			print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></form></td>\n";
		} else {
			print "<td colspan=2>N/A</td>"
		}

		if (defined $params{'stats_path'}) {
			print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
			print "<input type=\"submit\" value=\"Stats\">\n";
			print "<input type=\"hidden\" name=\"ftype\" value=\"stats\">\n";
			print "<input type=\"hidden\" name=\"printer\" value=\"$printer\">\n";
			print "<input type=\"hidden\" name=\"referrer\" value=\"setup_printer_page\">\n";
			print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></form></td>\n";
		} else {
			print "<td>[disabled]</td>\n";
		}
	}

	print "</tbody></table>\n";
	
	print "<p><table>\n";
	
	print "<tr><td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Refresh\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"setup_printer_page\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">";
	print "</form></td>\n";

	print "<td><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Return to user page\">";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_printer_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">";
	print "</form></td></tr>";

	print "</table></center>\n<p><hr>\n";
}

sub reset_pages
{
	my ($args) = @_;
	my ($referrer, $printer, $answer);

	&check_cookie_file ($$args{'cookie'});
	
	$referrer = $$args{'referrer'};
	$printer = $$args{'printer'};

	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";

	$answer = `$params{'admin_prog_path'}/pqm --resetpages \'$printer\' 2>&1`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --resetpages \'$printer\': $answer");
	}

	print "<p>$answer";

	open (MAIL, "|$params{'mta'} $params{'admin_mail'}") or &return_error (500, "CGI Error",
		"$0: cannot open mail transport agent $params{'mta'}: $!\n");

	my $admin = &get_user ($params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'});

	print MAIL "Reply-to: $params{'admin_mail'}\n";
	print MAIL "From: $params{'admin_mail'}\n";
	print MAIL "Subject: Print Quota System - reset of page count for printer \"$printer\"\n\n";
	print MAIL "Changes by $admin:\n";
	print MAIL "------------------\n\n";
	print MAIL $answer;
	close MAIL;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"reset_pages\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub reset_black
{
	my ($args) = @_;
	my ($referrer, $printer, $answer);

	&check_cookie_file ($$args{'cookie'});
		$referrer = $$args{'referrer'};
	$printer = $$args{'printer'};

	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";

	$answer = `$params{'admin_prog_path'}/pqm --resetblack \'$printer\' 2>&1`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --resetblack \'$printer\': $answer");
	}

	print "<p>$answer";

	open (MAIL, "|$params{'mta'} $params{'admin_mail'}") or &return_error (500, "CGI Error",
		"$0: cannot open mail transport agent $params{'mta'}: $!\n");

	my $admin = &get_user ($params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'});

	print MAIL "Reply-to: $params{'admin_mail'}\n";
	print MAIL "From: $params{'admin_mail'}\n";
	print MAIL "Subject: Print Quota System - reset of black ink/toner levels for printer \"$printer\"\n\n";
	print MAIL "Changes by $admin:\n";
	print MAIL "------------------\n\n";
	print MAIL $answer;
	close MAIL;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"reset_black\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub reset_colour
{
	my ($args) = @_;
	my ($referrer, $printer, $answer);

	&check_cookie_file ($$args{'cookie'});
	
	$referrer = $$args{'referrer'};
	$printer = $$args{'printer'};
	
	print `/bin/cat $params{'web_message_text'}`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";

	$answer = `$params{'admin_prog_path'}/pqm --resetcolour \'$printer\' 2>&1`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'admin_prog_path'}/pqm --resetcolour \'$printer\': $answer");
	}

	print "<p>$answer";

	open (MAIL, "|$params{'mta'} $params{'admin_mail'}") or &return_error (500, "CGI Error",
		"$0: cannot open mail transport agent $params{'mta'}: $!\n");

	my $admin = &get_user ($params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'});

	print MAIL "Reply-to: $params{'admin_mail'}\n";
	print MAIL "From: $params{'admin_mail'}\n";
	print MAIL "Subject: Print Quota System - reset of colour ink/toner levels for printer \"$printer\"\n\n";
	print MAIL "Changes by $admin:\n";
	print MAIL "------------------\n\n";
	print MAIL $answer;
	close MAIL;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"reset_colour\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub stats
{
	my ($args) = @_;
	my ($referrer, $printer, $answer);

	&check_cookie_file ($$args{'cookie'});
	
	$referrer = $$args{'referrer'};
	$printer = $$args{'printer'};

	print `/bin/cat $params{'web_message_text'}`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";

	$answer = `$params{'prog_path'}/printbill_grapher --printer \'$printer\' --slices 40 --path $params{'graph_png_output_dir'} --png_url $params{'png_url'} --web`;
	
	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot run $params{'prog_path'}/printbill_grapher --printer \'$printer\' --slices 40 --path $params{'graph_png_output_dir'} --png_url $params{'png_url'} --web: $answer");
	}

	print "$answer";

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"stats\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

sub setup_change_password_page
{
	my ($args) = @_;
	my ($referrer, $file, $user);

	&check_cookie_file ($$args{'cookie'});

	$referrer = $$args{'referrer'};

	$file = $params{'cookieroot'} . '/' . $ENV{'REMOTE_ADDR'} . "_" . $$args{'cookie'};

	open COOKIE, "<$file";
	
	$user = <COOKIE>;
	
	close COOKIE;
	
	print `/bin/cat $params{'web_message_text'}`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<center><h2>Change Print Administration Password</h2></center>\n";
	print "<hr>\n";

	print "Changing password for user \"$user\":\n";

	print "<center><table>\n";
	
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<tr><td>Old Password:</td><td><input type=\"password\" name=\"oldpw\" size=\"10\"></td></tr>\n";
	print "<tr><td>New Password:</td><td><input type=\"password\" name=\"pw1\" size=\"10\"></td></tr>\n";
	print "<tr><td>Again:</td><td><input type=\"password\" name=\"pw2\" size=\"10\"></td></tr>\n";
	print "<tr><td valign=top align=left><input type=\"submit\" value=\"Submit\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"change_password\">\n";
	print "<input type=\"hidden\" name=\"username\" value=\"$user\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_admin_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\"></td>\n";
	print "</form>\n";
	
	print "<td align=right><form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"submit\" value=\"Cancel\">\n";
	print "<input type=\"hidden\" name=\"ftype\" value=\"$referrer\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"setup_adduser_page\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	print "</form></td></tr>\n";
	print "</table></center>\n";
	print "<p><hr>\n";
}

sub change_password
{
	my ($args) = @_;
	my ($referrer, $username, $oldpw, $pw1, $pw2, $answer, %adminhash);
	my $error = "FALSE";
	
	&check_cookie_file ($$args{'cookie'});
	
	$referrer = $$args{'referrer'};
	$username = $$args{'username'};
	$oldpw = $$args{'oldpw'};
	$pw1 = $$args{'pw1'};
	$pw2 = $$args{'pw2'};
	
	print `/bin/cat $params{'web_message_text'}`;

	if ($? >> 8) {
		&return_error (500, "CGI Error",
			"$0: cannot read web message text file $params{'web_message_text'}");
	}
	
	print "<hr>\n";

	tie %adminhash, "Printbill::PTDB_File", "$params{'db_home'}/web_admin.db", "TRUE"
		or &return_error (500, "CGI Error",
			"$0: cannot open file $params{'db_home'}/web_admin.db: $!\n");
	
# A pause for reflection (and to fix those brute-forcers)...

	sleep (2);
	
	if (!defined ($adminhash{$username}) || crypt ($oldpw, substr ($adminhash{$username}, 0, 2)) ne $adminhash{$username}) {
		print "<p>That is <b>not</b> your current password!\n";
		$error = "TRUE";
	} else {
		if ($pw1 eq $pw2) {
			$answer = `$params{'admin_prog_path'}/pqm --webnonintpasswd \'$username\' --passwd \'$pw1\' 2>&1`;
			
			if ($? >> 8) {
				&return_error (500, "CGI Error",
				"$0: cannot run $params{'admin_prog_path'}/pqm --webnonintpasswd \'$username\' --passwd \'********\': $answer");
			}
		} else {
			$error = "TRUE";
			$answer = "Error: passwords don't match. Nothing has been changed.";
		}

		print "<p>$answer";
	}
	
	untie %adminhash;

	print "<center>\n";
	print "<form action=\"$params{'admin_cgi'}\" method=\"post\">\n";
	print "<input type=\"hidden\" name=\"referrer\" value=\"change_password\">\n";
	print "<input type=\"hidden\" name=\"cookie\" value=\"$$args{'cookie'}\">\n";
	
	if ($error eq "FALSE") {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_admin_page\">\n";
	} else {
		print "<input type=\"hidden\" name=\"ftype\" value=\"setup_change_password_page\">\n";
	}

	print "<input type=\"submit\" value=\"OK\">\n";
	print "</form>\n";
	print "</center>\n";
	print "<hr>\n";
}

# tests whether a given number is real (e.g. 0, -123, 123.321, .123, 123.)
sub isreal {
	my ($arg) = shift;
	return ($arg =~ /^([+-]?)(?=\d|\.\d)\d*(\.\d*)?([Ee]([+-]?\d+))?$/);
}

sub get_user {
	my ($file) = @_;

	open COOKIE, "<$file";
	
	my $admin = <COOKIE>;
	chomp $admin;
	
	close COOKIE;

	return $admin;
}
