/***************************************************************************
                          ps_resolv.c  -  description
                             -------------------
    begin                : Sat Aug 24 2002
    copyright            : (C) 2002 by Jan Fernquist, Florian Boor
    email                : boor@unix-ag.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// This file is adapted code from ethereal. Thanks!

/* resolv.c
 * Routines for network object lookup
 *
 * $Id: ps_resolv.c,v 1.2 2003/09/27 16:01:00 florian Exp $
 *
 * Laurent Deniel <deniel@worldnet.fr>
 *
 * Ethereal - Network traffic analyzer
 * By Gerald Combs <gerald@ethereal.com>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <zlib.h>

#include "ps_resolv.h"

#define ENAME_MANUF		"manufacturers.dat.gz"

/*
 * Name of directory, under the user's home directory, in which
 * personal configuration files are stored.
 */
#define PF_DIR ".pstumbler"

#define MAXMANUFLEN	22	/* max vendor name length with ending '\0' */
#define HASHMANUFSIZE   256
#define HASHPORTSIZE	256


typedef struct hashname {
  guint			addr;
  guchar   		name[MAXNAMELEN];
  gboolean              is_dummy_entry;	/* name is IP address in dot format */
  struct hashname 	*next;
} hashname_t;


#define HASH_ETH_MANUF(addr) (((int)(addr)[2]) & (HASHMANUFSIZE - 1))

typedef struct hashmanuf {
  guint8 		addr[3];
  char 			name[MAXMANUFLEN];
  struct hashmanuf     	*next;
} hashmanuf_t;


/* internal ethernet type */

typedef struct _ether
{
  guint8		addr[6];
  char 			name[MAXNAMELEN];
} ether_t;

static hashmanuf_t 	*manuf_table[HASHMANUFSIZE];

static int 		eth_resolution_initialized = 0;

/*
 * Flag controlling what names to resolve.
 */
guint32 g_resolv_flags = RESOLV_MAC;


/*
 *  Miscellaneous functions
 */

/*
 * Get the directory in which our global configuration and data
 * files are stored.
 */
const char *
get_datafile_dir(void)
{
	static char* tmpenv;
	static char* datafile_dir;
	
	if (!access("/etc/" ENAME_MANUF,R_OK))
		return("/etc");
	
	tmpenv = getenv("HOME");
	datafile_dir = malloc(strlen(tmpenv)+1);
	datafile_dir = strcpy(datafile_dir,tmpenv);
	datafile_dir = realloc(datafile_dir,strlen(datafile_dir)+12);
	datafile_dir = strcat(datafile_dir,PF_DIR);
	return datafile_dir;
}

/* Wrapper for the most common case of asking
 * for a string using a colon as the hex-digit separator.
 */

gchar *
ether_to_str(const guint8 *ad)
{
	return ether_to_str_punct(ad, ':');
}


/* Places char punct in the string as the hex-digit separator.
 * If punct is '\0', no punctuation is applied (and thus
 * the resulting string is 5 bytes shorter)
 */
gchar *
ether_to_str_punct(const guint8 *ad, char punct) {
  static gchar  str[3][18];
  static gchar *cur;
  gchar        *p;
  int          i;
  guint32      octet;
  static const gchar hex_digits[16] = "0123456789abcdef";

  if (cur == &str[0][0]) {
    cur = &str[1][0];
  } else if (cur == &str[1][0]) {
    cur = &str[2][0];
  } else {
    cur = &str[0][0];
  }
  p = &cur[18];
  *--p = '\0';
  i = 5;
  for (;;) {
    octet = ad[i];
    *--p = hex_digits[octet&0xF];
    octet >>= 4;
    *--p = hex_digits[octet&0xF];
    if (i == 0)
      break;
    if (punct)
      *--p = punct;
    i--;
  }
  return p;
}



static int fgetline(char **buf, int *size, gzFile *fp)
{
  int len;
  int c;

  if (fp == NULL)
    return -1;

  if (*buf == NULL) {
    if (*size == 0)
      *size = BUFSIZ;

    if ((*buf = malloc(*size)) == NULL)
      return -1;
  }

  if (gzeof(fp))
    return -1;

  len = 0;
  while ((c = gzgetc(fp)) != EOF && c != '\n') {
    if (len+1 >= *size) {
      if ((*buf = realloc(*buf, *size += BUFSIZ)) == NULL)
	return -1;
    }
    (*buf)[len++] = c;
  }

  if (len == 0 && c == EOF)
    return -1;

  (*buf)[len] = '\0';

  return len;

} /* fgetline */


/*
 * Ethernet / manufacturer resolution
 *
 * The following functions implement ethernet address resolution and
 * ethers files parsing (see ethers(4)).
 *
 * The manuf file has the same format as ethers(4) except that names are
 * truncated to MAXMANUFLEN-1 characters and that an address contains
 * only 3 bytes (instead of 6).
 *
 * Notes:
 *
 * I decide to not use the existing functions (see ethers(3) on some
 * operating systems) for the following reasons:
 * - performance gains (use of hash tables and some other enhancements),
 * - use of two ethers files (system-wide and per user),
 * - avoid the use of NIS maps,
 * - lack of these functions on some systems.
 *
 * So the following functions do _not_ behave as the standard ones.
 *
 * -- Laurent.
 */


static int parse_ether_line(char *line, ether_t *eth, int six_bytes)
{
  /*
   *  See man ethers(4) for ethers file format
   *  (not available on all systems).
   *  We allow both ethernet address separators (':' and '-'),
   *  as well as Ethereal's '.' separator.
   */

  gchar *cp;
  int a0, a1, a2, a3, a4, a5;

  if ((cp = strchr(line, '#')))
    *cp = '\0';

  if ((cp = strtok(line, " \t\n")) == NULL)
    return -1;

  if (six_bytes) {
    if (sscanf(cp, "%x:%x:%x:%x:%x:%x", &a0, &a1, &a2, &a3, &a4, &a5) != 6) {
      if (sscanf(cp, "%x-%x-%x-%x-%x-%x", &a0, &a1, &a2, &a3, &a4, &a5) != 6) {
        if (sscanf(cp, "%x.%x.%x.%x.%x.%x", &a0, &a1, &a2, &a3, &a4, &a5) != 6)
	  return -1;
      }
    }
  } else {
    if (sscanf(cp, "%x:%x:%x", &a0, &a1, &a2) != 3) {
      if (sscanf(cp, "%x-%x-%x", &a0, &a1, &a2) != 3) {
        if (sscanf(cp, "%x.%x.%x", &a0, &a1, &a2) != 3)
	return -1;
      }
    }
  }

  if ((cp = strtok(NULL, " \t\n")) == NULL)
    return -1;

  eth->addr[0] = a0;
  eth->addr[1] = a1;
  eth->addr[2] = a2;
  if (six_bytes) {
    eth->addr[3] = a3;
    eth->addr[4] = a4;
    eth->addr[5] = a5;
  } else {
    eth->addr[3] = 0;
    eth->addr[4] = 0;
    eth->addr[5] = 0;
  }

  strncpy(eth->name, cp, MAXNAMELEN);
  eth->name[MAXNAMELEN-1] = '\0';

  return 0;

} /* parse_ether_line */

static gzFile *eth_p = NULL;

static void set_ethent(char *path)
{
  if (eth_p)
    gzrewind(eth_p);
  else
    eth_p = gzopen(path, "r");
}

static void end_ethent(void)
{
  if (eth_p) {
    gzclose(eth_p);
    eth_p = NULL;
  }
}

static ether_t *get_ethent(int six_bytes)
{

  static ether_t eth;
  static int     size = 0;
  static char   *buf = NULL;

  if (eth_p == NULL)
    return NULL;

  while (fgetline(&buf, &size, eth_p) >= 0) {
    if (parse_ether_line(buf, &eth, six_bytes) == 0) {
      return &eth;
    }
  }

  return NULL;

} /* get_ethent */



static void add_manuf_name(guint8 *addr, guchar *name)
{
  int hash_idx;
  hashmanuf_t *tp;

  hash_idx = HASH_ETH_MANUF(addr);

  tp = manuf_table[hash_idx];

  if( tp == NULL ) {
    tp = manuf_table[hash_idx] = (hashmanuf_t *)malloc(sizeof(hashmanuf_t));
  } else {
    while(1) {
      if (tp->next == NULL) {
	tp->next = (hashmanuf_t *)malloc(sizeof(hashmanuf_t));
	tp = tp->next;
	break;
      }
      tp = tp->next;
    }
  }

  memcpy(tp->addr, addr, sizeof(tp->addr));
  strncpy(tp->name, name, MAXMANUFLEN);
  tp->name[MAXMANUFLEN-1] = '\0';
  tp->next = NULL;

} /* add_manuf_name */

static hashmanuf_t *manuf_name_lookup(const guint8 *addr)
{
  int hash_idx;
  hashmanuf_t *tp;

  hash_idx = HASH_ETH_MANUF(addr);

  tp = manuf_table[hash_idx];

  while(tp != NULL) {
    if (memcmp(tp->addr, addr, sizeof(tp->addr)) == 0) {
      return tp;
    }
    tp = tp->next;
  }

  return NULL;

} /* manuf_name_lookup */


static void initialize_ethers(void)
{
  ether_t *eth;
  char *manuf_path;

  /* manuf hash table initialization */

  /* Compute the pathname of the manuf file */
  manuf_path = (gchar *) malloc(strlen(get_datafile_dir()) +
    strlen(ENAME_MANUF) + 2);
  sprintf(manuf_path, "%s" G_DIR_SEPARATOR_S "%s", get_datafile_dir(),
    ENAME_MANUF);
  /* Read it and initialize the hash table */
  set_ethent(manuf_path);

  while ((eth = get_ethent(0))) {
    add_manuf_name(eth->addr, eth->name);
  }

  end_ethent();

  free(manuf_path);

} /* initialize_ethers */



extern const guchar *get_manuf_name(const guint8 *addr)
{
  static gchar  str[3][MAXMANUFLEN];
  static gchar *cur;
  hashmanuf_t  *manufp;
  if ((g_resolv_flags & RESOLV_MAC) && !eth_resolution_initialized) {
    initialize_ethers();
    eth_resolution_initialized = 1;
  }

  if (!(g_resolv_flags & RESOLV_MAC) || ((manufp = manuf_name_lookup(addr)) == NULL)) {
    if (cur == &str[0][0]) {
      cur = &str[1][0];
    } else if (cur == &str[1][0]) {
      cur = &str[2][0];
    } else {
      cur = &str[0][0];
    }
    sprintf(cur, "<unknown> (%02x:%02x:%02x)", addr[0], addr[1], addr[2]);
    return cur;
  }

  return manufp->name;

} /* get_manuf_name */
