#ifndef _DTREE_PRIVATE_INCLUDED
#define _DTREE_PRIVATE_INCLUDED

/* 
 * Prospect: a developer's system profiler.
 * Digital Tree ADT.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Authors: Doug Baskins, HP
 *          Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: dtree_private.h,v 1.3 2004/01/09 20:29:27 type2 Exp $ */

/*
 * Sparce arrays implemented as a Digital Tree, or Trie.
 * See Knuth, Volume 3, pp. 492-507, 2nd Edition, 1998, for reference.
 * Note: Trie and Digital Tree are synonymous.
 *
 * This ADT is a quaternary Trie.  All references go 16 levels every
 * access.  Not the most efficient, however, it serves our needs now.
 * The leaf node stores a pointer that may be used to access your
 * sturcture, or may be cast to an unsigned long for simple storage.
 * Branches are how to get to flowers which is where the data is.
 *
 * This file: dtree_private.h
 *
 * Private definitions, constants, and macros for use by the dtree
 * ADT.  User code should not need to see this and should just include
 * the dtree.h file for the interface definition.
 */

/*
 *        Generic format of a Pointer word (quaternary) 32bit
 *
 *  |31|30|29|28|27|26|25|24|23|........|07|06|05|04|03|02|01|00|
 *  | 28 Bits of Pointer to 16-byte aligned object  | SlotId    |
 *
 *  Where: SlotId is the type, size and alignment of the object
 *
 *                     Pointer to next Branch
 *
 *  |31|30|29|28|27|26|25|24|23|........|07|06|05|04|03|02|01|00|
 *  | High 28 bits of 16-byte aligned pointer       |SlotId=1..8|
 *
 *  Where:
 *  SlotId == 1 is a pointer to  4 element branch, Level-=1
 *  SlotId == 2 is a pointer to 16 element branch, Level-=2
 *  SlotId == 3 is a pointer to 64 element branch, Level-=3
 *  ...
 *
 *  SlotId == cDT_BROWN_FLWR  is a pointer size 2 Slot object
 *  SlotId == cDT_RED_FLWR    is a pointer size 4 Slot object
 *  SlotId == cDT_ORANGE_FLWR is a pointer size 6 Slot object
 *  SlotId == cDT_YELLOW_FLWR is a pointer size 8 Slot object
 *
 *
 *    ImmediateIndex for Brown, Red, Orange, Yellow Flowers
 *
 *  Brown, Red, Yellow flowers are arranged in Slot pairs:
 *  Immediate Index, followed with associated Value
 *
 *  Brown  flower contains 1 Immediate Index  and 1 Value
 *  Red    flower contains 2 Immediate Indexs and 2 Values
 *  Orange flower contains 3 Immediate Indexs and 3 Values
 *  Yellow flower contains 4 Immediate Indexs and 4 Values
 *
 *  |31|30|29|28|27|26|25|24|23|.....|07|06|05|04|03|02|01|00|
 *  ==========================================================
 *  |         32 Index bits shifted left (15-Level)*2  | 0  0|
 *  |         Value associated with Index in above Slot      |
 *  ==========================================================
 *
 *  Level== 1..14, where the root pointer is Level = 16
 *
 *  LeafPointer must always point to Level == 0, therefore SlotId
 *  is always == to the current Level and is the Size of the Leaf
 *
 *  |31|30|29|28|27|26|25|24|23|........|07|06|05|04|03|02|01|00|
 *  | 28 bits of 16-byte-aligned pointer to Level=1 | SlotId    |
 *
 *                  NULL pointer = 0x0000000F 32bits, 0x1F 64bits
 *
 *  |31|30|29|28|27|26|25|24|23|........|07|06|05|04|03|02|01|00|
 *  | 0  0  0  0  0  0  0  0  0           0  0  0  0| 1  1  1  1|
 */

#include <unistd.h>

#define cDT_TRUE    1
#define cDT_FALSE   0

/*
 * Some types that dtree uses.
 */
#ifndef bool_t
typedef int bool_t;
#endif

#ifndef _ULONG_T
#define _ULONG_T
typedef unsigned long ulong_t;
#endif

#ifndef _UCHAR_T
#define _UCHAR_T
typedef unsigned char uchar_t;
#endif

#ifndef _PVOID_T
#define _PVOID_T
typedef void *Pvoid_t;
typedef void **PPvoid_t;
#endif

#ifndef _SLOT_T
#define _SLOT_T
typedef ulong_t Slot_t, *PSlot_t; /* 32 or 64 bits */
#endif


/******************************************************************************/
/*              Define MACRO's to structures used by dtree                    */
/******************************************************************************/

/* Bits per digit for a quaternary version of dtree */
#define cDT_BITS_PER_DIGIT       2L

/* Slots or Pointers per Narrow Branch */
#define cDT_SLOTS_PER_BRANCH    (1L << cDT_BITS_PER_DIGIT)

/* Constants */
#define cDT_BITS_PER_SLOT  (sizeof(ulong_t)*8L)  /* (32|64)  */
#define cDT_DGMSK  (cDT_SLOTS_PER_BRANCH-1L)     /* (3)  digit bit mask */

/* Definition of a Leaf in dtree */
#define cDT_ROOT_LEVEL  (cDT_BITS_PER_SLOT/cDT_BITS_PER_DIGIT)

/* Slot Type Fields: Bits to Mark the type of Slot.  */ 
#define cDT_SLOTIDMSK      ((sizeof(Slot_t)*cDT_SLOTS_PER_BRANCH)-1L)
#define cDT_NAR_BRANCH_ID  (0L)               /* 4 Slots long */
#define cDT_LEAF_ID        (0L)               /* 4 Slots long */

#define cDT_OD          (sizeof(Slot_t))   /* (0100) Odd Flower (address) Bit */
#define cDT_FLWR        (cDT_OD*2)         /* Bit identifies it as Flower */

#define cDT_BR          (1L)               /* Indexes in Brown Flower */
#define cDT_RD          (2L)               /* Indexes in Red Flower */
#define cDT_OR          (3L)               /* Indexes in Orange Flower */
#define cDT_YL          (4L)               /* Indexes in Yellow Flower */

#define cDT_MAX_FLWR    (4L)               /* Max Indexes in Flowers */

#define cDT_BR_FLWR_SLOTS    (cDT_BR * 2L) /* 2  */
#define cDT_RD_FLWR_SLOTS    (cDT_RD * 2L) /* 4 */
#define cDT_OR_FLWR_SLOTS    (cDT_OR * 2L) /* 6 */
#define cDT_YL_FLWR_SLOTS    (cDT_YL * 2L) /* 8 */

/* Maximum level change or size for a wide branch */
#define cDT_MAX_LEVEL  (cDT_BR_FLWR_ID)    /* 7(32bit) or 15(64bit) */

/* Slot Id word for 32 bit machines' Flowers */
#define cDT_BR_FLWR_EV (cDT_FLWR|(cDT_BR-1L))        /* (1000) 2 Slots */
#define cDT_BR_FLWR_ID (cDT_BR_FLWR_EV)              /* For allocations */
#define cDT_RD_FLWR_ID (cDT_FLWR|(cDT_RD-1L))        /* (1001) 4 Slots */
#define cDT_OR_FLWR_EV (cDT_FLWR|(cDT_OR-1L))        /* (1010) 6 Slots */
#define cDT_OR_FLWR_ID (cDT_OR_FLWR_EV)              /* For allocations */
#define cDT_YL_FLWR_ID (cDT_FLWR|(cDT_YL-1L))        /* (1011) 8 Slots */
#define cDT_BR_FLWR_OD (cDT_FLWR|(cDT_BR-1L)|cDT_OD) /* (1100) 2 Slots */
#define cDT_SPARE1                                   /* (1101) (Odd Red) */
#define cDT_OR_FLWR_OD (cDT_FLWR|(cDT_BR-1L)|cDT_OD) /* (1110) 6 Slots  */
#define cDT_NARROW_PT  (cDT_SLOTIDMSK)               /* (1111) Narrow Pointer */

#define cDT_NULL       ((Slot_t)(-1L))               /* Null Flower Pointer */

/* Get flower Index count from Flower Pointer */
#define DT_SLOTID_TO_FLWR_SIZE(SLOT) (((Slot_t)(SLOT) & 3L)+1L) 

/* Get normal pointer from Flower Pointer */
#define DT_FLWR_POINTER_TO_POINTER(PSLOT) \
        ((PSlot_t)(((Slot_t)(PSLOT) & cDT_OD) \
           ? (Slot_t)(PSLOT) & (-cDT_FLWR) \
           : (Slot_t)(PSLOT) & ~cDT_SLOTIDMSK))

/* Get Flower ID from Flower Pointer */
#define DT_FLWR_POINTER_TO_SLOTID(PSLOT) \
        (((Slot_t)(PSLOT) & cDT_SLOTIDMSK) & (~cDT_OD))

/* Get SlotId from a branch pointer */
#define DT_BRANCH_POINTER_TO_SLOTID(SLOT) ((SLOT) & cDT_SLOTIDMSK)

/* Get normal pointer from a branch pointer */
#define DT_BRANCH_POINTER_TO_POINTER(SLOT)  \
        ((PSlot_t)((Slot_t)(SLOT) & (~cDT_SLOTIDMSK)))

/* Check if is a Flower from Flower Pointer or Flower ID */
#define DT_IS_FLWR(PSLOT)       (((Slot_t)(PSLOT) & cDT_FLWR))
#define DT_IS_BRANCH_PTR(PSLOT) (!((Slot_t)(PSLOT) & cDT_FLWR))

/* Minimum memory allocation size using sbrk() */
#define cDT_PAGESZ_BYTES    (4096L)
#define cDT_CHUNKSIZE_SLOTS (64L)
#define cDT_CHUNKSIZE_BYTES (cDT_CHUNKSIZE_SLOTS * sizeof(Slot_t))

/* Calculate Bytes remaining on current page (page size must be power of 2) */
#define DT_REST_OF_PAGE(A) \
        ((cDT_PAGESZ_BYTES - (ulong_t)(A)) & (cDT_PAGESZ_BYTES-1))

/* NumbSlotsInBranch = DT_SLOTID_TO_BRANCHSLOTS(SltId) */
#define DT_SLOTID_TO_BRANCHSLOTS(SLOTID) \
        (1L << (((SLOTID)+1L) * cDT_BITS_PER_DIGIT))

/* Shift out digits according to level change */
#define DT_SFTIDX_DIGITS(SFTIDX, SLTID) \
        ((SFTIDX)<<=(cDT_BITS_PER_DIGIT*(SLTID+1L)))

/* Extract digit bits at current Level from SftIdx, given number of Levels */
#define DT_SFTIDX_TO_DIGIT(SFTIDX, LEVELS) \
        ((SFTIDX)>>(cDT_BITS_PER_SLOT-cDT_BITS_PER_DIGIT - \
                   ((LEVELS)*cDT_BITS_PER_DIGIT)))
    
/*
 * Define data structures used by dtree for measuring performance
 * and operating characteristics.
 */
ulong_t RamLookupReads[cDT_ROOT_LEVEL+1];
ulong_t SlotMoveCount;
ulong_t CascadeMoveCount;
ulong_t LevelsDecended;
ulong_t TotalIndexes;

/* dtree statistics structure */
typedef struct _DTREESTATS
{ 
    /* number of bytes allocated with sbrk() */
    ulong_t ss_TotalSbrkMem;
    
    /* Link list of Free Chunks (32 Slots each) */
    PSlot_t ss_AlcFreeChunks;
    
    /* Linked list of free branches/flowers */
    PSlot_t ss_AlcFreeBranch[cDT_ROOT_LEVEL];

    /* Numb of free branches/flowers in above list */
    ulong_t ss_FreeBranchCnt[cDT_ROOT_LEVEL];
    
    /* Total allocated */
    ulong_t ss_AlcBranchCnt [cDT_ROOT_LEVEL];
    
    /* Total allocated */
    ulong_t ss_Leaves;
} ss_t;

ss_t gDtreeStats;  /* Must be Initialized to zero */

/*
 * The following are only used internally by dtree
 */
void dtree_fill_branch_nulls(Slot_t);
Slot_t dtree_malloc(Slot_t, ulong_t);
void dtree_free_branch(PSlot_t, Slot_t, ulong_t);

#endif /* _DTREE_PRIVATE_INCLUDED */
