/* 
 * Prospect: a developer's system profiler.
 *
 * COPYRIGHT (C) 2001-2004 Hewlett-Packard Company
 *
 * Author: Alex Tsariounov, HP
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* $Id: ascii_report.c,v 1.47 2004/01/09 20:29:27 type2 Exp $ */

/*
 *******************************************************************************
 *
 *                            PROSPECT PROJECT
 *                   Linux ASCII Report Output Module
 *
 *******************************************************************************
 */

#ifndef __LINT__
static const char gRCSid[] = "@(#) $Id: ascii_report.c,v 1.47 2004/01/09 20:29:27 type2 Exp $";
#endif

/*
 * System Header Files
 */
#include <signal.h>
#include <string.h>
#include <time.h>
#include <math.h>

/*
 * Prospect Header Files
 */
#include "../dtree/dtree.h"
#include "prospect.h"
#include "incache.h"
#include "linux_model.h"
#include "linux_module.h"

/*
 * Static function prototypes
 */
static void pscr_header(void);
static void pscr_dash(void);
static void pscr_dash_div(void);
static void pscr_stats(void);
static void pscr_proc_table(void);
static void pscr_proc_summaries(void);
static void pscr_cpu_usage(char *title, unsigned long *clog, pid_t pid);
static void pscr_kernel_profiles(void);
static void print_kernel_profile(void*, unsigned int, name_list_t*);
static int  pscr_proc_header(process_t *p);
static void pscr_proc_profiles(process_t *p);
static void pscr_pc_profile(process_t *p, void *prof_head_ptr);
static int  get_name_lst_ind(char *name, name_list_t *lst);
static void pscr_name_list(name_list_t *list);
static void destroy_name_list(name_list_t *list);
static void pscr_proc_label(process_t *p);
static int  pid_sort_by_user(const void *proc1, const void  *proc2);
static int  pid_sort_by_sys(const void *proc1, const void  *proc2);
static int  sort_by_hitcnt(const void *s1, const void *s2);
static void free_region_symbols(process_t *p);
static int  build_profile_by_pc(unsigned long PC, unsigned int hits, 
    process_t *p);

/* --------------------   Begin public functions ---------------------- */

/*
 * void print_done(void)
 *
 * We land here when all sampling is over with
 * and we need to print the report.
 */
void
print_done(void)
{
    unsigned long idx;
    void **P;
    process_t *p;

    mINFORM("--- In print_done().");

#ifdef PERFTEST
    /* Cache performance test, output data */
    getproc_by_pid_cached(0,42);
#endif

    /* Informative output */
    if (gOp.readerror) {
        ferr("read errors detected, skipping report generation\n");
        prospect_exit(1);
    }
    else 
        ferr("sampling done, creating output report\n");

    /* fill in global pid table */
    gConf.sort_base_pids = (process_t **)CALLOC(gProclist.pl_numProcs, 
                                                sizeof(process_t **));
    if (!gConf.sort_base_pids) {
        ferr("disaster, calloc returns nil for global pid array\n");
        prospect_exit(1);
    }
    /* double for inform prints */
    mINFORM("Entire unsorted process table:");
    gConf.num_procs=0;
    for (idx=0L, P  = DTF(gProclist.pl_procs, idx);
                 P != NULL;
                 P  = DTN(gProclist.pl_procs, idx)
        )
    {
        p = (process_t*) *P;

        gConf.sort_base_pids[gConf.num_procs++] = p;
        
        /* calculate run times */
        p->pr_usrRunMs = ((double)p->pr_profTot / gOp.samplehz) *1000.;
        p->pr_sysRunMs = ((double)p->pr_sysProfTot / gOp.samplehz) *1000.;

        mINFORM(
           " Proc(%05d) Total(%06u) Uniq(%06u) SysTot(%05d)"
           " SysUniq(%05d) Name(%s)",
           p->pr_myPid, p->pr_profTot, p->pr_profUniq,
           p->pr_sysProfTot, p->pr_sysProfUniq,
           ((p->pr_myKname) 
              ? p->pr_myKname 
              : (p->pr_path ? p->pr_path : "NoName")
           )
        );
        if (gConf.bug_level>1 && !p->pr_isKthread) {
            region_t *r;
            int i;
            mINFORM("  region list:");
            r = p->pr_vasHead->vh_fwd;
            i = 0;
            do {
                mINFORM("  %3d: start=0x%lx length=%ld offset=0x%lx "
                        "entry=%d path=%s",
                        i, r->rd_start, r->rd_length, r->rd_offset, 
                        p->pr_vasHead->vh_entries, r->rd_path
                       );
                r = r->rd_fwd;
                i++;
            } while (r != (void*) p->pr_vasHead);
        }
    }
    mINFORM("  %d entries in process table (pl_numProcs=%d)", gConf.num_procs, 
            gProclist.pl_numProcs);

    /* now sort the thing */
    if (gConf.flags.sort_by_system) {
        qsort(
             (void*)gConf.sort_base_pids,
             gConf.num_procs,
             sizeof(process_t **),
             pid_sort_by_sys
             );
    }
    else {
        qsort(
             (void*)gConf.sort_base_pids,
             gConf.num_procs,
             sizeof(process_t **),
             pid_sort_by_user
             );
    }
    
    /* Sort kernel modules into ascending address order */
    if (gConf.flags.do_kernel && (gK.k_vas.vh_entries>1)) {
        /* Revisit: reorder on changes */
    }

    /* We output ascii */
    pscr_header();

    /*
     * Output the profiles 
     */
    pscr_stats();
    if (gConf.flags.kernel_only) {
        pscr_kernel_profiles();
    }
    else {
        pscr_proc_table();
        pscr_proc_summaries();
    }

    /* Done, finally */
    prospect_exit(0);

} /* print_done() */

/* 
 * char *kernel_symbol(void *addr)
 *
 * Returns pointer to the kernel symbol string for the passed
 * address.  This code reuses some of the code in ascii_report.c
 * to get at the kernel and module symbols.  We expect that the
 * System.map file has been read already before calls to this
 * function.
 */
char *
kernel_symbol(void *addr)
{
    static process_t p;
    static char symbol[512];
    symb_t sym;
    syms_attrb_t sym_attrb;

    mINFORM("In kernel_symbol(%p)", addr);

    /* init the process container vars as for a regular profile */
    p.pr_syms_array = &sym;
    p.pr_syms_numb = 0;
    p.pr_myPid = 0;
    p.pr_sysProfile = NULL;
    p.pr_syms_uniq = p.pr_sysProfUniq = 1;
    p.pr_syms_vh = &gK.k_vas;
    p.pr_syms_nm_glob = 0;
    p.pr_syms_hits = 0;
    p.pr_syms_region = gK.k_vas.vh_fwd;

    /* get the symbol info for this hit and return if not found */
    if (build_profile_by_pc((unsigned long)addr, 1, &p))
        return "no_region:symbol_not_found";
    
    /* get properties (i.e. name) of symbol and return it's pointer */
    f_sym_attrb(&sym.ss_regptr->rd_symkey, sym.ss_sym_ptr, &sym_attrb); 

    strncpy(symbol, sym.ss_regptr->rd_name, 512-strlen(sym_attrb.sa_name)-2);
    strcat(symbol, ":");
    strncat(symbol, sym_attrb.sa_name, strlen(sym_attrb.sa_name));
    return symbol;
} /* kernel_symbol() */


/* --------------------   Begin static functions ---------------------- */

/*
 * static void pscr_stats(void)
 *
 * Print run statistics.
 */
static void
pscr_stats(void)
{

 pscr_dash();
 pscr("Statistics of Run\n");
 pscr_dash();
 pscr("  Buffers read:           %8lu     Buffer Flushes:          %8lu\n",
        gPstats[PRO_BUFFERS_READ], gPstats[PRO_FLUSHES]);
 pscr("  Notifications received: %8lu     Num sample traces:       %8lu\n",
        gPstats[PRO_NOTIFICATIONS], gPstats[PRO_NUM_SAMPLE_TRACES]);
 pscr("   Num execs:             %8lu      Num samples:            %8lu\n",
        gPstats[PRO_EXECS], gPstats[PRO_SAMPLES]);

 pscr("   Num forks:             %8lu      Ave hits/sample trace:  %8.3f\n",
        gPstats[PRO_FORKS], (float) gPstats[PRO_SAMPLES] /
                                    (gPstats[PRO_NUM_SAMPLE_TRACES])
       );
 pscr("   Num maps:              %8lu      Low transport samples:  %8lu\n",
        gPstats[PRO_MAPS], 1);
 pscr("   Num incomplete maps:   %8lu      High transport samples: %8lu\n", 
        gPstats[PRO_INCOMPLETE_MAPS], 1);
 pscr("   Num drop modules:      %8lu      System hits:            %8lu\n",
         gPstats[PRO_DROP_MODULES], gPstats[PRO_SYS_HITS]);
 pscr("   Num exits:             %8lu      User hits:              %8lu\n",
        gPstats[PRO_EXITS], gPstats[PRO_USR_HITS]);

 pscr("   Num unknowns:          %8lu      Hits of Prospect:       %8lu\n",
        gPstats[PRO_UNKNOWN_NOTES], gPstats[PRO_SELF_HITS]);
 pscr("  Total processes:        %8lu     Unknown hits:           %8lu\n",
        gProclist.pl_numProcs, gPstats[PRO_UNKNOWN_HITS]);

 pscr("  Total kern-module hits: %8lu     Total kthreads:          %8lu\n",
        gPstats[PRO_MOD_HITS], gProclist.pl_numKthreads);
 pscr("  Total kernel hits:      %8lu     Total kthread hits:      %8lu\n",
        gK.k_prof.kp_profTot, gK.k_prof.kp_kthreadProfTot);
 pscr("   Unique of those:       %8lu      Unique of those:        %8lu\n",
        gK.k_prof.kp_profUniq, gK.k_prof.kp_kthreadProfUniq);
 pscr("  Total process 0 hits:   %8lu     Total usr_kernel:        %8lu\n",
        gK.k_prof.kp_intrProfTot, gK.k_prof.kp_usrProfTot);
 pscr("   Unique of those:       %8lu      Unique of those:        %8lu\n",
        gK.k_prof.kp_intrProfUniq, gK.k_prof.kp_usrProfUniq);
#ifdef __ia64__
 pscr("  Hits to gate sec:       %8lu\n",
        gPstats[PRO_GATE_HITS]);
#endif
 pscr("\n");

} /* pscr_stats() */

/*
 * static void pscr_dash(void)
 *
 * Print a 72 dash separator.
 */
static void
pscr_dash(void)
{
    pscr("\
------------------------------------------------------------------------\n");
   return;
} /* pscr_dash() */

/*
 * static void pscr_dash_div(void)
 *
 * Print a 72 dash separator with == header.
 */
static void
pscr_dash_div(void)
{
    pscr("\
==----------------------------------------------------------------------\n");
   return;
} /* pscr_dash_div() */

/*
 * static void PntHeader(void)
 *
 * Print output header info.
 */
static void
pscr_header(void)
{
    pscr("\nProspect: %s\n\n", gConf.prospect_rev);

    pscr("Uname: %s %s %s %s %s\n",
           gConf.my_utsname.sysname,
           gConf.my_utsname.nodename,
           gConf.my_utsname.release,
           gConf.my_utsname.version,
           gConf.my_utsname.machine
    );

    if (gConf.run_time[0]) {
        pscr("This run occured on: %s", gConf.run_time);
    }
    
    pscr("Prospect was built for: ");
    switch (get_built_for_os()) {
        case cCO_UNKNOWN:    pscr("Unknown! Strange.\n"); break;
        case cCO_LINUX_IA32: pscr("Linux IA32\n"); break;
        case cCO_LINUX_IA64: pscr("Linux IA64\n"); break;
    }
    
    /* Check in we're running on the os that we were built for. */
    if (!correct_os())
    {
        pscr(
        "Warning: This prospect is built for a different version of the OS.\n");
        if (mOUTPUT_IS_FILE)  
            ferr("running prospect built for wrong version of OS\n");
    }

    if (mTRACEIN) pscr("Reading trace file generated by prospect.\n");

    /* printout current command line here */
    pscr("Current command line: '%s'\n", gConf.command_line);
    
    /* more info */
    pscr("CPU Speed: %lf MHz\n", gConf.cpumhz);
    pscr("CPU type: ");
    switch (gConf.cputype) {
        case CPU_NO_GOOD: pscr("Bad Detection!"); break;
        case CPU_PPRO: pscr("Pentium Pro"); break;
        case CPU_PII: pscr("Pentium II"); break;
        case CPU_PIII: pscr("Pentium III"); break;
        case CPU_ATHLON: pscr("Athlon"); break;
        case CPU_TIMER_INT: pscr("CPU with Timer Interrupt"); break;
        case CPU_RTC: pscr("RTC Fallback Mode"); break;
        case CPU_P4: pscr("Pentium 4"); break;
        case CPU_IA64: pscr("Generic IA64"); break;
        case CPU_IA64_1: pscr("IA64 Merced"); break;
        case CPU_IA64_2: pscr("IA64 McKinley"); break;
        case CPU_HAMMER: pscr("AMD Hammer Family"); break;
        case CPU_P4_HT2: pscr("Pentium 4 / Xeon with 2 HyperThreads"); break;
        case CPU_AXP_EV4: pscr("Alpha EV4 Family"); break;
        case CPU_AXP_EV5: pscr("Alpha EV5 Family"); break;
        case CPU_AXP_PCA56: pscr("Alpha PCA56 Family"); break;
        case CPU_AXP_EV6: pscr("Alpha EV6 Family"); break;
        case CPU_AXP_EV67: pscr("Alpha EV7 Family"); break;
        case MAX_CPU_TYPE: pscr("Maximum CPU Limit?"); break;
        default: pscr("Unknown!!!");
    }
    pscr("\n");
    pscr("Sampling frequency: %u Hz\n", gOp.samplehz);
    pscr("Prospect was built for oprofile version: " CURRENT_OPROFILE "\n");
    pscr("Profile output type: ");
    if (gConf.flags.follow_forks)
        pscr("follow forks");
    else if (gConf.flags.system_wide)
        pscr("system wide");
    else
        pscr("direct child only");
    pscr("\n");
    if (gConf.flags.no_sort)
        pscr("Not sorting output\n");
    if (gConf.flags.sort_by_system)
        pscr("Sorting by system time, not user\n");
    if (!gConf.flags.do_kernel) 
        pscr("Kernel profiles disabled (probably missing "
             "System.map, use --system-map)\n");
    
    /*
     * Extra messages 
     */
    pscr("Hint: If the CPU speed seems funny, are you running on battery?\n");
    pscr("\n");

} /* pscr_header() */

/*
 * static void pscr_proc_table(void)
 *
 * Print process table cutoff at -m val.
 */
static void
pscr_proc_table(void)
{
    process_t *p;
    unsigned int ii;

    pscr_dash();
    pscr("Extrapolated Summary of %d Processes Analyzed", gConf.num_procs);
    pscr(" - Sorted by ");
    if (gConf.flags.sort_by_system)
        pscr("System ");
    else
        pscr("User ");
    pscr("Time\n");
    pscr_dash();
    pscr("\n");

    pscr("\
Process name        Pid               User Time           System Time");
    pscr("\n\n");
/*
 123456789-123456 12345     12345678 12345.789-   12345678 12345.789-
123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-
*/
    for(ii=0; ii<gConf.num_procs; ii++) {

        p = gConf.sort_base_pids[ii];
        if (gConf.min_value > 0) {
            if (gConf.flags.sort_by_system) {
                if (p->pr_sysRunMs/1000.0 <= gConf.min_value) {
                    pscr(" - Skip remainder, sys time of %0.6f is "
                           "<= '-m %0.5f' Sec\n", 
                           p->pr_sysRunMs/1000.0, gConf.min_value);
                    break;
                }
            }
            else {
                if (p->pr_usrRunMs/1000.0 <= gConf.min_value) {
                    pscr(" - Skip remainder, user time of %0.6f is "
                           "<= '-m %0.5f' Sec\n", 
                           p->pr_usrRunMs/1000.0, gConf.min_value);
                    break;
                }
            }
        }
        /* process name */
        pscr_proc_label(p);
        /* rest of the line */
        pscr("   %8u %10.4f   %8u %10.4f\n",
               p->pr_profTot, p->pr_usrRunMs/1000.0,
               p->pr_sysProfTot, p->pr_sysRunMs/1000.0);
    }
    pscr("\n");

} /* pscr_proc_table() */

/*
 * static void pscr_proc_summaries(void)
 *
 * Print profile for applicable processes.
 */
static void
pscr_proc_summaries(void)
{
    int ii;
    process_t *p;

    for (ii=0; ii<gConf.num_procs; ii++) {

        /* get the proc from the sorted list */
        p = gConf.sort_base_pids[ii];

        /* see if to print this proc */
        if (!gConf.flags.system_wide) {
            if (!gConf.flags.follow_forks) {
               if (p->pr_myPid != gConf.my_child.pid) continue; 
            }
            else {
                /* skip prospect, but print all in sesssion */
                if (p->pr_myPid == gConf.my_pid) continue;
                if (p->pr_mySid != gConf.my_sid) continue;
            }
        }

        /* ok, then print header */
        if (!pscr_proc_header(p)) continue;
        pscr("\n");

        /* next, print out the profile */
        pscr_proc_profiles(p);
        pscr("\n");
    }

    /* if this is a system-wide run, print out the global kernel profile too */
    if (gConf.flags.system_wide) {
        pscr("\n");
        pscr_kernel_profiles();
    }

} /* pscr_proc_summaries() */

/*
 * static void pscr_proc_header(process_t *p)
 *
 * Print out the process id header to screen.
 * Returns 1 on success, 0 on error.
 */
static int
pscr_proc_header(process_t *p)
{
    /* check if time below threshold */
    if (gConf.min_value > 0) {
        if (gConf.flags.sort_by_system) {
            if (p->pr_sysRunMs/1000.0 <= gConf.min_value) {
                pscr_proc_label(p);
                pscr(" - Skip: sys time %0.6f <= '-m %0.5f' Sec\n", 
                       p->pr_sysRunMs/1000.0, gConf.min_value);
                return 0;
            }
        }
        else {
            if (p->pr_usrRunMs/1000.0 <= gConf.min_value) {
                pscr_proc_label(p);
                pscr(" - Skip: user time %0.6f <= '-m %0.5f' Sec\n", 
                       p->pr_usrRunMs/1000.0, gConf.min_value);
                return 0;
            }
        }
    }
    else {
        if (p->pr_profTot==0 && p->pr_sysProfTot==0) {
            pscr_proc_label(p);
            pscr(" - Skip: No user nor kernel hits, thus no profiles.\n"); 
            return 0;
        }
    }


    /* 
     * process separator
     */
    pscr("\n");
    pscr_dash_div();
    pscr_proc_label(p);
    pscr(" ");

    /*
     * pr_birthBy flags -- print how process was born
     */
    switch(p->pr_birthBy) 
    {
        case cBIRTH_not:
            pscr("No Birth!");
            pscr("\n                          ");
            pscr("PPID(%d) GPID(%d) SID(%d)",
                    p->pr_myParentPid, p->pr_myGroupPid, p->pr_mySid);
            pscr("\n                          ");
            pscr("Birth by fork, parent: ");
            pscr_proc_label(p->pr_parent); 
            break;

        case cBIRTH_fork:
            pscr("Birth by fork");
            pscr("\n                          ");
            pscr("PPID(%d) GPID(%d) SID(%d)",
                    p->pr_myParentPid, p->pr_myGroupPid, p->pr_mySid);
            pscr("\n                          ");
            pscr("Parent was:  ");
            pscr_proc_label(p->pr_parent); 
            break;

        case cBIRTH_vfork:
            pscr("Birth by vfork");
            pscr("\n                          ");
            pscr("PPID(%d) GPID(%d) SID(%d)",
                    p->pr_myParentPid, p->pr_myGroupPid, p->pr_mySid);
            /* This should never happen !!!!!  */
            pscr("\n                          ");
            pscr("BUG! Birth by vfork, parent: ");
            pscr_proc_label(p->pr_parent); 
            break;

        case cBIRTH_exec:
            pscr("Birth by exec");
            pscr("\n                          ");
            pscr("PPID(%d) GPID(%d) SID(%d)",
                    p->pr_myParentPid, p->pr_myGroupPid, p->pr_mySid);
            break;
        
        case cBIRTH_spontaneous:
            pscr("Birth by spontaneous");
            pscr("\n                          ");
            pscr("PPID(%d) GPID(%d) SID(%d)",
                    p->pr_myParentPid, p->pr_myGroupPid, p->pr_mySid);
            break;

        case cBIRTH_preExist:
            pscr("Birth by pre-exist");
            pscr("\n                          ");
            pscr("PPID(%d) GPID(%d) SID(%d)",
                    p->pr_myParentPid, p->pr_myGroupPid, p->pr_mySid);
            break;


    } /* End switch */
    pscr("\n                          ");

    /* if prospect was my parent, print that out */
    if (p->pr_myParentPid == gConf.my_pid)
    {
        pscr("Parent was prospect");
        pscr("\n                          ");
        if (gConf.my_child.signalled) {
            if (gConf.my_child.terminated) {
                pscr("Process terminated by signal ");
            }
            else {
                pscr("Process stopped by signal ");
            }
            pscr("%d", gConf.my_child.ts_signal);
            pscr("\n                          ");
        }
    }
    else if (p->pr_mySid == gConf.my_pid && p->pr_myPid != gConf.my_pid)
    {
        pscr("Descendant of prospect");
        pscr("\n                          ");
    }

    /* is this a kernel thread ? */
    if (p->pr_isKthread) 
    {
        pscr("Probable kernel thread");
        pscr("\n                          ");
    }

    /* how many times did we exec ? */
    if (p->pr_exec_times > 1) 
    {
        pscr("Process exec'ed %d times", p->pr_exec_times);
        pscr("\n                          ");
    }


    /* 
     * pr_endBy flags  -- print how died
     */
    switch(p->pr_endBy)
    {
        case cEND_not:
            pscr("Process did not END");
            break;

        case cEND_exit:
            pscr("Process END by exit()");
            break;

        case cEND_exec:
            pscr("Process END by exec()");
            break;
    } /* End switch */
    pscr("\n");

    pscr("\n");
    pscr("Extrapolated user run time:    %12.3f ms   (from %9d hits)\n",
            p->pr_usrRunMs, p->pr_profTot);
    pscr("Extrapolated system run time:  %12.3f ms   (from %9d hits)\n",
            p->pr_sysRunMs, p->pr_sysProfTot);
    pscr("User/Kernel split:             %3d%%  / %3d%%      (from %9d hits)\n",
            (unsigned int)rint((double)p->pr_profTot*100)/
                (p->pr_profTot+p->pr_sysProfTot),
            (unsigned int)rint((double)p->pr_sysProfTot*100)/
                (p->pr_profTot+p->pr_sysProfTot),
            p->pr_profTot+p->pr_sysProfTot);

    /*
     * Print out a list of cpu's used
     */
    if (gConf.numcpus > 1) {
        pscr("\n");
        pscr_cpu_usage("User distribution by CPU:  ", p->pr_ucpu_log,
                       p->pr_myPid);
        pscr_cpu_usage("Kernel distribution by CPU:", p->pr_kcpu_log,
                       p->pr_myPid);
    }

    return 1;
} /* pscr_proc_header() */

/*
 * static void pscr_cpu_usage(char* title, unsigned long *clog, pid_t pid)
 *
 * Print out the sorted cpu usage for this process from the
 * data stored in p->pr_cpu_log array.
 */
static void
pscr_cpu_usage(char *title, unsigned long *clog, pid_t pid)
{
    unsigned int ii, pcent;
    unsigned long total;

    pscr("%s ", title);
    /* find total */
    ii = 0;
    total = 0;
    while (ii < gConf.numcpus) {
        total += clog[ii];
        ii++;
    }
    /* print out log in percentage of hits logged */
    for (ii=0; ii<gConf.numcpus; ii++) {
        if (ii && ii % 4 == 0) pscr("\n%*s",(strlen(title)+1)," ");
        pscr("  ");
        if (ii < 10) pscr(" ");
        if (clog[ii]) {
            pcent = (unsigned long)rint((double)clog[ii]*100/total);
            if (pcent < 10) pscr(" ");
            (pcent == 100) ?
                pscr("[%u:100%]", ii) : pscr("[%u: %u%%]", ii, pcent);
        }
        else {
            pscr("[%u: ---]", ii);
        }
    }

    pscr("\n");
} /* pscr_sorted_cpu_usage() */

/*
 * static void pscr_proc_profiles(process_t *p)
 *
 * Print out the profiles for this proc.
 */
static void
pscr_proc_profiles(process_t *p)
{
    unsigned long Index;
    unsigned long *P;
    name_list_t  *user_reg_lst=NULL, *kernel_reg_lst=NULL;

    pscr_dash();
    pscr("Section 1: Process Intersection with Sampling Mechanism, ");
    if (!p->pr_profTot || !p->pr_sysProfTot) {
        pscr("N");  /* no profiles */
    }
    else {
        if (p->pr_profTot) pscr("U");
        if (p->pr_sysProfTot) pscr("K");
    }
    pscr("\n");
    pscr_dash();

    /*
     * ----------- print out user profile if there's one  ------------
     */
    if (p->pr_profTot) {
        pscr("USER portion of profile:\n");

        /*  Get max possible (uniq PC's) memory for profile */
        p->pr_syms_array =
            (symb_t*)CALLOC(p->pr_profUniq, sizeof(symb_t));
        if (!p->pr_syms_array) {
            perr("Calloc fail spells disaster at line %d, file %s", 
                    __LINE__, __FILE__);
            prospect_exit(1);
        }
        user_reg_lst = CALLOC(1, sizeof(name_list_t));
        if (!user_reg_lst) {
            perr("Calloc fail spells disaster at line %d, file %s", 
                    __LINE__, __FILE__);
            prospect_exit(1);
        }
        p->pr_syms_name_lst = user_reg_lst;

        /* Init total number of symbols to print */
        p->pr_syms_numb = 0;
        /* Init unique number of symbols to print */
        p->pr_syms_uniq = p->pr_profUniq;
        /* Init head pointer of the vas we're processing */
        p->pr_syms_vh = p->pr_vasHead;
        /* Init total number of global symbols to print */
        p->pr_syms_nm_glob = 0;
        /* Init total PC hits in extracted profile */
        p->pr_syms_hits = 0;
        /* Init the starting vas for profile builder */
        p->pr_syms_region = p->pr_vasHead->vh_fwd;

        /* extract profile and build table */
        for (Index=0L, P  = (unsigned long*) DTF(p->pr_profile, Index);
                       P != (unsigned long*) NULL;
                       P  = (unsigned long*) DTN(p->pr_profile, Index)
            )
        {
            if (*P == 0) {
                mBUG("*P=0 on user profile extract");
                continue;
            }
            build_profile_by_pc(Index, (unsigned long) *P, p);
        }
    
        /* sort the table - should be sorting pointers not elements... */
        if (!gConf.flags.no_sort) qsort(
                          (void*) p->pr_syms_array,
                          p->pr_syms_numb,  /* set by build_profile_by_pc */
                          sizeof(symb_t),
                          sort_by_hitcnt
                         );

        /* print it out */
        pscr_pc_profile(p, p->pr_profile);

        /* free memory used for this */
        FREE(p->pr_syms_array);
        /* and free the symbol table for all regions and close files */
        free_region_symbols(p);
        pscr("\n");
    }
    else {
        pscr("USER portion of profile: no hits to report.\n");
    }

    /*
     * ----------- print out kernel profile if there's one ------------
     */
    if (gConf.flags.do_kernel) {
        pscr("\n");
        if (p->pr_sysProfTot) {
            pscr("KERNEL portion of profile:\n");

            /* symbol table is max unique pc's for profile */
            p->pr_syms_array = 
                (symb_t*) CALLOC(p->pr_sysProfUniq, sizeof(symb_t));
            if (!p->pr_syms_array) {
                perr("Calloc fail spells disaster at line %d, file %s", 
                        __LINE__, __FILE__);
                prospect_exit(1);
            }
            kernel_reg_lst = CALLOC(1, sizeof(name_list_t));
            if (!kernel_reg_lst) {
                perr("Calloc fail spells disaster at line %d, file %s", 
                        __LINE__, __FILE__);
                prospect_exit(1);
            }
            p->pr_syms_name_lst = kernel_reg_lst;

            /* Init total number of symbols to print */
            p->pr_syms_numb = 0;
            /* Init unique number of symbols to print */
            p->pr_syms_uniq = p->pr_sysProfUniq;
            /* Init head pointer of the vas we're processing */
            p->pr_syms_vh = &gK.k_vas;
            /* Init total number of global symbols to print */
            p->pr_syms_nm_glob = 0;
            /* Init total PC hits in extracted profile */
            p->pr_syms_hits = 0;
            /* Init the kernel starting vas for profile builder */
            p->pr_syms_region = gK.k_vas.vh_fwd;
            
            /* extract profile and build table */
            for (Index=0L, P  = (unsigned long*) DTF(p->pr_sysProfile, Index);
                           P != (unsigned long*) NULL;
                           P  = (unsigned long*) DTN(p->pr_sysProfile, Index)
                )
            {
                if (*P == 0) {
                    mBUG("*P=0 on system profile extract");
                    continue;
                }
                build_profile_by_pc(Index, (unsigned long) *P, p);
            }

            /* sort the table - should be sorting pointers not elements... */
            if (!gConf.flags.no_sort) qsort(
                              (void*) p->pr_syms_array,
                              p->pr_syms_numb,  /* set by build_proifle_by_pc */
                              sizeof(symb_t),
                              sort_by_hitcnt
                             );

            /* print it out */
            pscr_pc_profile(p, p->pr_sysProfile);

            /* free the symbol table memory  */
            FREE(p->pr_syms_array);
            /* but keep the kernel/modules open since everybody uses them */
        }
        else {
            pscr("KERNEL portion of profile: no hits to report.\n");
        }
    }
    
    /* If no profiles output, don't print out region lists */
    if (p->pr_profTot == 0 && 
            (gConf.flags.do_kernel == 0 ||  p->pr_sysProfTot == 0)) 
        return;
   
    /*
     * ----------- print out region list indices ------------
     */
    pscr("\n");
    pscr_dash();
    pscr("Section 2: Hit File Index for Process\n");
    pscr_dash();

    if (p->pr_profTot) {
        pscr("USER space:\n");
        pscr_name_list(user_reg_lst);
        destroy_name_list(user_reg_lst);
    }
    
    if (gConf.flags.do_kernel && p->pr_sysProfTot) {
        pscr("\n");
        pscr("KERNEL space:\n");
        pscr_name_list(kernel_reg_lst);
        destroy_name_list(kernel_reg_lst);
    }

} /* pscr_proc_profiles() */

/*
 * static void pscr_name_list(name_list_t *list)
 *
 * Prints out name list in index style.
 */
static void
pscr_name_list(name_list_t *list)
{
    name_list_t *cur;
    int ii;

    if (!list) {
        mBUG("pscr_name_list passed nil list");
        return;
    }

    cur=list;
    ii = 0;
    while (cur->next) {
        pscr_fixlen(72, "[%d]  %s\n", ii, cur->name);
        ii++;
        cur = cur->next;
    }
    pscr_fixlen(72, "[%d]  %s\n", ii, cur->name);
} /* pscr_name_list() */

/*
 * static void destroy_name_list(name_list_t *list)
 *
 * Destroy (free) all elements in list.
 */
static void
destroy_name_list(name_list_t *list)
{
    name_list_t *cur, *nxt;

    if (!list)  return;

    cur=list;
    while (cur->next) {
        nxt = cur->next;
        if (cur->name) FREE(cur->name);
        FREE(cur);
        cur=nxt;
    }
    if (cur->name) FREE(cur->name);
    FREE(cur);

} /* destroy_name_list() */

/*
 * static int get_name_lst_ind(char *name, name_list_t *lst);
 *
 * Returns the index of the passed name from the list passed as lst,
 * if the name is not in the list, appends it to tail.
 * Assumes lst has at least one element allocated even if not assigned.
 * Returns -1 on error.
 */
static int
get_name_lst_ind(char *name, name_list_t *lst)
{
    int ii;

    if (!lst) return -1;

    /* first entry special */
    if (!lst->name) {
        lst->name = strdup(name);
        return 0;
    }
    
    ii = 0;
    while (lst->next != NULL) {
        if (!strcmp(name, lst->name)) return ii;
        lst = lst->next;
        ii++;
    }
    if (!strcmp(name, lst->name)) return ii;

    lst->next = MALLOC(sizeof(name_list_t));
    lst = lst->next;
    lst->next = NULL;
    lst->name = strdup(name);

    return ii+1;
} /* get_name_lst_ind() */

/*
 * static void pscr_kernel_profiles(void)
 * Print kernel profiles.
 */
static void
pscr_kernel_profiles(void)
{
    name_list_t  *kernel_name_lst=NULL;
    
    /* create kernel regions name list */
    kernel_name_lst = CALLOC(1, sizeof(name_list_t));
    if (!kernel_name_lst) {
        perr("Calloc fail spells disaster at line %d, file %s", 
                __LINE__, __FILE__);
        prospect_exit(1);
    }

    /* First, print the global kernel profile. */
    pscr_dash_div();
    if (!gConf.flags.do_kernel) {
        pscr("Note: Kernel profiles disabled since there was a problem\n");
        pscr("      with importing the \"%s\" file.\n\n", gConf.system_map);
        return;
    }
    pscr("Global KERNEL Profile:\n\n");
    if (gK.k_prof.kp_profTot == 0) {
        pscr("No kernel hits in any areas, no profiles available.\n\n");
        return;
    }
    if (gConf.numcpus > 1) {
        pscr_cpu_usage("Distribution by CPU:    ", 
                       gK.k_prof.kp_tot_cpu_log, 0);
        pscr("\n");
    }
    pscr("Total number hits: %d     Of those, %d are unique.\n",
           gK.k_prof.kp_profTot, gK.k_prof.kp_profUniq);
    print_kernel_profile(gK.k_prof.kp_profile, 
                         gK.k_prof.kp_profUniq,
                         kernel_name_lst);
    pscr("\n");

    /*
     * If this is a not a kernel profile, then return here.
     */
    if (!gConf.flags.kernel_only) {
        goto print_k_index;
    }

    /* Kernel profile due to kernel threads */
    pscr_dash();
    pscr("Kernel Profile Due to Kernel Threads:\n\n");
    if (gK.k_prof.kp_kthreadProfTot) {
        if (gConf.numcpus > 1) {
            pscr_cpu_usage("Distribution by CPU:    ",
                           gK.k_prof.kp_kthread_cpu_log, 0);
            pscr("\n");
        }
        pscr("Total number hits: %d     Of those, %d are unique.\n",
               gK.k_prof.kp_kthreadProfTot, gK.k_prof.kp_kthreadProfUniq);
        print_kernel_profile(gK.k_prof.kp_kthreadProfile, 
                             gK.k_prof.kp_kthreadProfUniq,
                             kernel_name_lst);
        pscr("\n");
    }
    else {
        pscr("No hits in this area, thus no profile available.\n\n");
    }

    /* Kernel profile due to kernel threads */
    pscr_dash();
    pscr("Kernel Profile Due to User Processes:\n\n");
    if (gK.k_prof.kp_usrProfTot) {
        if (gConf.numcpus > 1) {
            pscr_cpu_usage("Distribution by CPU:    ", 
                           gK.k_prof.kp_usr_cpu_log, 0);
            pscr("\n");
        }
        pscr("Total number hits: %d     Of those, %d are unique.\n",
               gK.k_prof.kp_usrProfTot, gK.k_prof.kp_usrProfUniq);
        print_kernel_profile(gK.k_prof.kp_usrProfile, 
                             gK.k_prof.kp_usrProfUniq,
                             kernel_name_lst);
        pscr("\n");
    }
    else {
        pscr("No hits in this area, thus no profile available.\n\n");
    }

    /* Now, print the process 0 profile - time stolen 
     * from idle by interrupts.
     */
    pscr_dash();
    pscr("Kernel Profile for the Zero Process:\n\n");
    if (gK.k_prof.kp_intrProfTot) {
        if (gConf.numcpus > 1) {
            pscr_cpu_usage("Distribution by CPU:    ", 
                           gK.k_prof.kp_intr_cpu_log, 0);
            pscr("\n");
        }
        pscr("Total number hits: %d     Of those, %d are unique.\n",
               gK.k_prof.kp_intrProfTot, gK.k_prof.kp_intrProfUniq);
        print_kernel_profile(gK.k_prof.kp_intrProfile, 
                             gK.k_prof.kp_intrProfUniq,
                             kernel_name_lst);
        pscr("\n");
    }
    else {
        pscr("No hits in this area, thus no profile available.\n\n");
    }

print_k_index:
    /* Print out kernel regions name list for indexing purposes */
    pscr_dash_div();
    pscr("Hit File Index for Kernel:\n\n");
    pscr_name_list(kernel_name_lst);
    destroy_name_list(kernel_name_lst);
    pscr("\n");
} /* pscr_kernel_profiles() */

/*
 * static void print_kernel_profile(void *, unsigned int, name_list_t*)
 *
 * Print out a kernel profile stored in the dtree pointed to 
 * by *Profile with total Unique hits.  Note that to re-use the code
 * that already does some of this work, we phony up a process struct
 * and fill it in with the kernel profile data.
 */
static void
print_kernel_profile(void *Profile, unsigned int Unique, name_list_t *nlst)
{
    process_t  p;
    unsigned long Index;
    unsigned long *P;

    /* allocate memory for profile */
    p.pr_syms_array = (symb_t*)CALLOC(Unique, sizeof(symb_t));
    if (!p.pr_syms_array) {
        perr("Calloc fail spells disaster at line %d, file %s",
        __LINE__, __FILE__);
        prospect_exit(1);
    }

    /* Init total number of symbols to print */
    p.pr_syms_numb = 0;
    /* Init pid field to zero for kernel profiles */
    p.pr_myPid = 0;
    /* set the process profile pointer to real profile */
    p.pr_sysProfile = Profile;
    /* Init unique number of symbols to print */
    p.pr_syms_uniq = p.pr_sysProfUniq = Unique;
    /* Init head pointer of the vas we're processing */
    p.pr_syms_vh = &gK.k_vas;
    /* Init total number of global symbols to print */
    p.pr_syms_nm_glob = 0;
    /* Init total PC hits in extracted profile */
    p.pr_syms_hits = 0;
    /* Init the kernel starting vas for profile builder */
    p.pr_syms_region = gK.k_vas.vh_fwd;
    /* Set name list to passed arg */
    p.pr_syms_name_lst = nlst;

    /* extract profile and build table */
    for (Index=0L, P  = (unsigned long*) DTF(Profile, Index);
                   P != (unsigned long*) NULL;
                   P  = (unsigned long*) DTN(Profile, Index)
        )
    {
        if (*P == 0) {
            mBUG("*P=0 on kernel profile extract");
            continue;
        }
        build_profile_by_pc(Index, (unsigned long) *P, &p);
    }

    /* sort the table - should be sorting pointers not elements... */
    if (!gConf.flags.no_sort) qsort(
                      (void*) p.pr_syms_array,
                      p.pr_syms_numb,  /* set by build_profile_by_pc */
                      sizeof(symb_t),
                      sort_by_hitcnt
                     );

    /* print it out */
    pscr_pc_profile(&p, p.pr_sysProfile);

    /* and free the symbol table memory  */
    FREE(p.pr_syms_array);
    /* but keep the kernel and module files open */

} /* print_kernel_profile() */

/*
 * static int 
 * build_profile_by_pc(unsigned long PC, unsigned int hits, process_t *p)
 *
 * Generate the symbol table in the process container for the extracted
 * profile passed in here one pc at a time.  Tightly coupled with some vars
 * in the process container.  This routine does not know whether it's 
 * creating a user space or kernel profile.
 */
static int 
build_profile_by_pc(unsigned long PC, unsigned int hits, process_t *p)
{
    char    *Psym_pt;
    unsigned long    Value;
    region_t *r;
    int      Ci, cant_open;
    tNOSYMREASONS noSymReason=cSS_INIT;
    
    /* Walk the vas to find the hits. 
     * Note that we will get hits in increasing order so we only 
     * have to walk the vas once; the process struct saves the current
     * region pointer for the next time around.
     */
    for (r = p->pr_syms_region;
                  ;
         r = p->pr_syms_region = r->rd_fwd)
    {
        if (r == NULL) return -1;
        /* check if were out of address space */
        if (r == (region_t*)p->pr_syms_vh) {
            pscr("Error [PC=0x%lx hits=%d pid=%d]: Beyond address space.\n",
                  PC, hits, p->pr_myPid);
            mINFORM("Error [PC=0x%lx hits=%d pid=%d]: Beyond address space.",
                    PC, hits, p->pr_myPid);
            /* we're done since anything more will be above too */
            return -1;
        }

        /* see if hit is in between regions */
        if (PC < (unsigned long) r->rd_start) {
            pscr("Error [PC=0x%lx hits=%d pid=%d]: In between regions.\n",
                 PC, hits, p->pr_myPid);
            mINFORM("Error [PC=0x%lx hits=%d pid=%d]: In between regions.",
                    PC, hits, p->pr_myPid);
            return -1;
        }

        /* are we in correct region */
        if (PC < (unsigned long) r->rd_end) break;
    }
    
    /* Get path to identified region file and open for symbols. 
     * Note that symbol code will keep this file open for subsequent
     * reads.
     */
    if (!r->rd_path) {
        if (!r->rd_kerntext) {
            pscr("Error [PC=0x%lx hits=%d pid=%d]: Path=nill for region "
                 "0x%lx:0x%lx\n",PC, hits, p->pr_myPid, r->rd_start, r->rd_end);
            return -1;
        }
    }
    else if (strstr(r->rd_path, "<Executable_Memory_Region>")) {
            /* look for generators */
            noSymReason = cSS_SELF_GEN_CODE;
    }
    else {
        /* try to open the file */
        cant_open = FALSE;
        if (f_sym_name_list(&r->rd_symkey, r->rd_path)) {
            /* Check if kernel module hit, this could be in initrd */
            if (PC > gKernelVmOffset) {
                if (try_fixup_module_path(r)) {
                    cant_open = TRUE;
                }
                else {
                    /* try it again - one chance */
                    if (f_sym_name_list(&r->rd_symkey, r->rd_path)) 
                        cant_open = TRUE;
                }
            }
            else
                cant_open = TRUE;
        }
        else {
            /* success in opening file */
            noSymReason = cSS_FOUND_FILE;
        }
        if (cant_open) {
            mINFORM("Open failed of '%s' for region 0%lx:0%lx", 
                    r->rd_path, r->rd_start, r->rd_end);
            perr("Open failed of '%s' for region 0%lx:0%lx", r->rd_path,
                    r->rd_start, r->rd_end);
            r->rd_symkey.sk_sym_count = 0;
            noSymReason = cSS_FILE_NOT_FOUND;
        }
    }

    /* match PC to a symbol in file */
    if (r->rd_symkey.sk_sym_count) {
        syms_attrb_t syms_attrb;
        /* get name and base value of code for PC */
        Psym_pt = f_sym_name_global(&r->rd_symkey, 
                                    (char*) (PC - (unsigned long)r->rd_start));
        f_sym_attrb(&r->rd_symkey, Psym_pt, &syms_attrb);
        Value = syms_attrb.sa_value + (unsigned long)r->rd_start;
        /* if we have symbols, but find no match, it's the jump table */
        if (Psym_pt) {
            noSymReason = cSS_FOUND_SYMS;
            mINFORM("Good symbol '%s', sa_value=0x%lx, Value=0x%lx, "
                    "PC=0x%lx, hits=%d, file='%s'",
                    syms_attrb.sa_name, syms_attrb.sa_value,
                    Value, PC, hits, r->rd_path);
        }
        else {
            noSymReason = cSS_JUMP_TABLE;
            mINFORM("Jump-table '%s', sa_value=0x%lx, Value=0x%lx, "
                    "PC=0x%lx, hits=%d, file='%s'",
                    syms_attrb.sa_name, syms_attrb.sa_value,
                    Value, PC, hits, r->rd_path);
        }
    }
    else {
        mINFORM("No symbols, assume stripped file [PC=0x%lx hits=%d pid=%d "
                "file=%s map=0x%lx:0x%lx]",
                PC, hits, p->pr_myPid, r->rd_path, r->rd_start, r->rd_end);
        
        if (noSymReason == cSS_FOUND_FILE)
            noSymReason = cSS_STRIPPED_FILE;

        /* use the pregion as value (?) */
        Psym_pt = NULL;
        Value = (unsigned long) r->rd_start;
    }

    /* tally up the hits */
    Ci = p->pr_syms_numb-1;  /* current index into sym array */

    if ( (p->pr_syms_hits == 0) || Value != p->pr_syms_array[Ci].ss_value) {
        /* new symbol, init */
        Ci = p->pr_syms_numb;
        /* watch for overflow */
        if (++p->pr_syms_numb > p->pr_syms_uniq) {
            mBUG("Too many hits: table overflow, "
                 "hits=%d, uniq=%d, pid=%d, region='%s'",
                 p->pr_syms_hits, p->pr_syms_uniq, p->pr_myPid, r->rd_path);
        }

        p->pr_syms_array[Ci].ss_cnt         = hits;
        p->pr_syms_array[Ci].ss_regptr      = r;
        p->pr_syms_array[Ci].ss_sym_ptr     = Psym_pt;
        p->pr_syms_array[Ci].ss_noSymReason = noSymReason;
        p->pr_syms_array[Ci].ss_value       = Value;
        p->pr_syms_array[Ci].ss_icount      = 1;
    }
    else {
        /* same symbol, just bump counts */
        p->pr_syms_array[Ci].ss_cnt += hits;
        p->pr_syms_array[Ci].ss_icount++;
    }

    /* bump total hits in table */
    p->pr_syms_hits += hits; 

    return 0;
} /* build_profile_by_pc() */

/*
 * static void pscr_pc_profile(process_t *p, void *prof_head_ptr)
 *
 * Print out profile stored in the proc's table.
 */
static void
pscr_pc_profile(process_t *p, void *prof_head_ptr)
{
    int ii;
    symb_t *Pss;
    region_t *r;
    unsigned int accum_cnt=0;
    unsigned long Index;
    unsigned int *P;
    double F_Spu_P=0.0;

    if (p->pr_syms_hits == 0 ) return;

    pscr("Extrapolated time from %d hits = %0.2lf seconds.\n\n",
           p->pr_syms_hits, ((double)p->pr_syms_hits / gOp.samplehz) );

    /* print profile header line */
    pscr("Pcnt Accum  Hits   Secs  Routine Name\n");

    Pss = p->pr_syms_array;

    for (ii=0; ii<p->pr_syms_numb; ii++, Pss++) {

        /* get region and calc time spent */
        r = Pss->ss_regptr;
        F_Spu_P = (double) Pss->ss_cnt/gOp.samplehz;

        /* if less than min, don't print */
        if (  (!gConf.flags.disassemble)
                  &&
              (gConf.min_value)
                  && 
              (F_Spu_P < gConf.min_value)
          )
        {
           pscr(" - Skip remainder, CPU of %0.6f is below '-m %0.6f' Sec\n", 
                   F_Spu_P, gConf.min_value);
           return;
        }

        /* accumulate hits */
        accum_cnt += Pss->ss_cnt;

        /* if we have a symbol, output its contents */
        if (Pss->ss_sym_ptr) {
            syms_attrb_t syms_attrb, attr_next;

            f_sym_attrb(&r->rd_symkey, Pss->ss_sym_ptr, &syms_attrb);
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  ",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt,
                   (double)Pss->ss_cnt/gOp.samplehz
                  );

            /* For shared libraries, we use the dynamic symbols.  These
             * do not have static symbols anymore.  So we print a range
             * for the symbol name and hope that if the hit is to a 
             * static, then it's between the range.
             */
            if (!r->rd_symkey.sk_isDynamic) {
                pscr("%s", syms_attrb.sa_name);
            }
            else {
                f_sym_attrb_next(&r->rd_symkey, Pss->ss_sym_ptr, &attr_next);
                /* if the next symbol is the same as current, it's last */
                if (attr_next.sa_value == syms_attrb.sa_value)
                    pscr("%s", syms_attrb.sa_name);
                else
                    pscr("%s->%s", syms_attrb.sa_name, attr_next.sa_name);
            }
        }
        else  if (Pss->ss_noSymReason == cSS_SELF_GEN_CODE) {
            /* executing from data pregion = generated code */
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  0x%-8.8lx "
                   "Self-generated-code, %d unique",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt, 
                   (double)Pss->ss_cnt/gOp.samplehz,
                   Pss->ss_value,
                   Pss->ss_icount); 
        }
        else  if (Pss->ss_noSymReason == cSS_JUMP_TABLE) {
            /* a hit below the first symbol in the jump table  */
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  0x%-8.8lx "
                   "Jump-table, %d unique",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt, 
                   (double)Pss->ss_cnt/gOp.samplehz,
                   Pss->ss_value,
                   Pss->ss_icount); 
        }
        else  if (Pss->ss_noSymReason == cSS_STRIPPED_FILE) {
            /* we have a file, but no sysm.  stripped ? */
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  0x%-8.8lx "
                   "Symbols-stripped, %d unique",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt, 
                   (double)Pss->ss_cnt/gOp.samplehz,
                   Pss->ss_value,
                   Pss->ss_icount); 
        }
        else  if (Pss->ss_noSymReason == cSS_NO_SYMBOLS) {
            /* We do not have a symbol, so output the pregion */
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  0x%-8.8lx "
                   "No-symbols, %d unique",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt, 
                   (double)Pss->ss_cnt/gOp.samplehz,
                   Pss->ss_value,
                   Pss->ss_icount); 
        }
        else  if (Pss->ss_noSymReason == cSS_FILE_NOT_FOUND) {
            /* We do not have a symbol, so output the pregion */
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  0x%-8.8lx "
                   "File-open-fail, %d unique",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt, 
                   (double)Pss->ss_cnt/gOp.samplehz,
                   Pss->ss_value,
                   Pss->ss_icount); 
        }
        else {
            /* should never happen, but never say never */
            mINFORM("  Symbol-confusion, Pss->noSymReason=%d", 
                    Pss->ss_noSymReason);
            pscr("%3.0f%% %4.0f%%  %4d  %5.2f  0x%-8.8lx "
                   "Symbol-confusion, %d unique",
                   (double)(Pss->ss_cnt) * 100.0 / p->pr_syms_hits,
                   (double)(accum_cnt) * 100.0 / p->pr_syms_hits,
                   Pss->ss_cnt, 
                   (double)Pss->ss_cnt/gOp.samplehz,
                   Pss->ss_value,
                   Pss->ss_icount); 
        }

        /* Output pregion vnode index if in list, put it it if not */
        if (r) 
        {
            if (r->rd_path)
                pscr(" [%d]", 
                     get_name_lst_ind(r->rd_path, p->pr_syms_name_lst));
            else if (r->rd_name)
                pscr(" [%d]", 
                     get_name_lst_ind(r->rd_name, p->pr_syms_name_lst));
            else
                pscr(" [?]");
        }

        pscr("\n"); 

        /* If extended  profile, and hits per instruction is 2 or
         * greater, output disassembly instruction hits
         */
        if  (   gConf.flags.disassemble 
                        &&
                (
                        (gConf.flags.force_disassemble)
                                ||
                        (
                        (Pss->ss_cnt > 7)  /* Need more than 7 hits */
                                &&
                        (Pss->ss_icount)   /* No div by zero */
                                && 
                        ((Pss->ss_cnt/Pss->ss_icount) > 1)  
                        )
                )
            )
        {
            /* Initialize Cache routines */
            syms_attrb_t   syms_attrb;
            syms_key_t    *Pcurrent_sym_key;
            unsigned long  PregVaddr;

            /* set to current region and memory offset */
            Pcurrent_sym_key = &Pss->ss_regptr->rd_symkey;
            PregVaddr        = (unsigned long)Pss->ss_regptr->rd_start;
            
            /* 
             * Check for bad pc's - this was usually the gateway page.
             * Note that no symbol table (stripped) files will not 
             * continue out of here and will get disassembled.
             */
            if (Pss->ss_sym_ptr == NULL ) 
            {
                if (Pss->ss_noSymReason == cSS_SELF_GEN_CODE)
                {
                    pscr("                         "
                      "Can't disassemble self-generated code, unfortunately.");
                    pscr("\n\n");
                    continue;
                }
                if (Pss->ss_noSymReason == cSS_JUMP_TABLE)
                {
                    pscr("                         "
                      "Can't disassemble jump table areas, unfortunately.");
                    pscr("\n\n");
                    continue;
                }
                if (Pss->ss_noSymReason != cSS_STRIPPED_FILE)
                {
                    pscr("                         "
                      "Can't match these PC's to symbols, unfortunately.");
                    pscr("\n\n");
                    continue;
                }
                /* else, go ahead and try it */
            }

            /* Get the symbol attributes */
            f_sym_attrb(Pcurrent_sym_key, Pss->ss_sym_ptr, &syms_attrb);
            
            /* output address and mapping offset of symbol */
            pscr("                         ");
            pscr("Symbol address: 0x%lx, reg map@: 0x%lx\n",
                 Pss->ss_value, (Pss->ss_value - syms_attrb.sa_value));

            /* Initialize the Instruction Cache routine */
            init_incache
                    (
                    PregVaddr,           /* Absolute Pregion address */
                    syms_attrb.sa_value, /* Symbol offset into Pregion */
                    Pss->ss_cnt,         /* Hits to symbol name */
                    Pss->ss_icount,      /* Unique PC's hit in symbol */
                    Pcurrent_sym_key     /* Symbols file key */
                    );
            /* Now find all values in the array and process */
            for (Index = Pss->ss_value, 
                     P  = (unsigned int*) DTF(prof_head_ptr, Index);
                     P != (unsigned int*) NULL;
                     P  = (unsigned int*) DTN(prof_head_ptr, Index)
                )
            {
                if (*P == 0) mBUG("*P==0 in IPD extraction");
                /* Stuff Cpc, number Hits to instruction. */
                if (stuff_incache(Index, *P)) break;
            }
        }
        else
            if (gConf.flags.disassemble) {
                pscr("                         "
                     "Disassembly criteria not met.");
                pscr("\n\n");
            }
    }
} /* pscr_pc_profile() */

/*
 * static void pscr_proc_label(process_t *p)
 *
 * Printout process id.
 */
static void
pscr_proc_label(process_t *p)
{
    pscr("%-16s  %5d  ", p->pr_myKname, p->pr_myPid);
   
    /* REVISIT
    if (p->pr_is32bit)
        pscr("N");
    else
        pscr("W");
    if (p->pr_is_mt)
        pscr("M ");
    else
        pscr("  ");
    */

} /* pscr_proc_label() */

/*
 * static int pid_sort_by_user((const void *proc1, const void  *proc2)
 *
 * qsort helper: sort by extrapolated user time.
 */
static int
pid_sort_by_user(const void *proc1, const void  *proc2)
{
    unsigned long S1, S2;

    S1 = (*(process_t **)proc1)->pr_profTot;
    S2 = (*(process_t **)proc2)->pr_profTot;

    return((S1 > S2) ? -1 : ((S1 < S2) ? 1 : 0));

} /* pid_sort_by_user() */

/*
 * static int pid_sort_by_sys(const void *proc1, const void  *proc2)
 *
 * qsort helper: sort by extrapolated system time.
 */
static int
pid_sort_by_sys(const void *proc1, const void  *proc2)
{
    unsigned long S1, S2;

    S1 = (*(process_t **)proc1)->pr_sysProfTot;
    S2 = (*(process_t **)proc2)->pr_sysProfTot;

    return((S1 > S2) ? -1 : ((S1 < S2) ? 1 : 0));

} /* pid_sort_by_sys() */

/*
 * int sort_by_hitcnt(
 *
 * qsort helper: sort by symbol hit counts
 */
static int
sort_by_hitcnt(const void *s1, const void *s2)
{
    symb_t *ps1 = (symb_t*) s1;
    symb_t *ps2 = (symb_t*) s2;

    /* if hit count the same, use value */
    if (ps1->ss_cnt == ps2->ss_cnt) {
        if (ps1->ss_value == ps2->ss_value) return 0;
        return ( (ps1->ss_value < ps2->ss_value) ? -1 : 1 );
    }

    return ( (ps1->ss_cnt > ps2->ss_cnt) ? -1 : 1 );

} /* sort_by_hitcnt() */

/*
 * void free_region_symbols(process_t *p)
 *
 * Walk the vas of this proc and tell symbol code to
 * free allocated memory.
 */
static void 
free_region_symbols(process_t *p)
{
    region_t *r;

    if (p->pr_vasHead->vh_fwd != NULL) {
        for (r  = p->pr_vasHead->vh_fwd;
             r != (region_t*) p->pr_vasHead;
             r  = r->rd_fwd
            )
        {
            if (r->rd_syminit) {
                f_sym_name_free(&r->rd_symkey);
                r->rd_symkey.sk_sym_count = 0;
                r->rd_syminit = 0;
            }
        }
    }
} /* free_region_symbols() */
