// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "displaylist.h"
#include "display.h"
#include "exceptions.h"
#include "frame.h"

#include <algorithm>
#include <iterator>
#include <boost/python/extract.hpp>

#include GL_INCLUDE   // just for glGenLists() below

namespace visual {

/************ rView implementation ***********/

rView::rView( const tmatrix& _wct, const lighting& _lights,
	const tmatrix& _fwt, glContext& _cx,
	const vector& mine, const vector& maxe,
	const bool _anaglyph, const bool _coloranaglyph)
	: wct(_wct), lights(_lights), cx(_cx), fwt(_fwt), 
	anaglyph(_anaglyph), coloranaglyph(_coloranaglyph)
{
	ext_set( mine, maxe);
}

rView::~rView()
{
  //for(int i=0; i < sortlists.size(); i++)
  //  glDeleteLists(sortlists[i],1);
}

int 
rView::createSortList()
{
	sortlists.push_back( glGenLists( 1));
	return sortlists.back();
}

void 
rView::absorb_local( rView& local)
{
	ext_set( local.min_extent, local.max_extent);

	std::copy( local.sortlists.begin(), local.sortlists.end(), 
		std::back_inserter(sortlists));
	local.sortlists.clear();
}

void 
rView::ext_set( const vector& mine, const vector& maxe)
{
	min_extent[0] = mine.x; 
	min_extent[1] = mine.y; 
	min_extent[2] = mine.z;
	max_extent[0] = maxe.x; 
	max_extent[1] = maxe.y; 
	max_extent[2] = maxe.z;
}

void 
rView::ext_brect( const tmatrix& mwt, vector bmin, vector bmax)
{
	double b[] = { bmin.x, bmin.y, bmin.z
	             , bmax.x, bmax.y, bmax.z};
	ext_brect( mwt,b);
}

void 
rView::ext_brect( const tmatrix& mft, const double *b)
{
	tmatrix mwt(mft, fwt);
	for(int r=0; r<3; r++) {
		min_extent[r] = max_extent[r] = mwt[r][3];
		for(int c=0; c<3; c++) {
			if (mwt[r][c] < 0) {
				min_extent[r] += mwt[r][c] * b[c];
				max_extent[r] += mwt[r][c] * b[c+3];
			}
			else {
				max_extent[r] += mwt[r][c] * b[c];
				min_extent[r] += mwt[r][c] * b[c+3];
			}
		}
	}
}

void 
rView::ext_point( vector v)
{
	v = fwt * v;
	if (v.x < min_extent[0]) 
		min_extent[0] = v.x;
	if (v.y < min_extent[1])
		 min_extent[1] = v.y;
	if (v.z < min_extent[2]) 
		min_extent[2] = v.z;
	if (v.x > max_extent[0]) 
		max_extent[0] = v.x;
	if (v.y > max_extent[1]) 
		max_extent[1] = v.y;
	if (v.z > max_extent[2]) 
		max_extent[2] = v.z;
}

void 
rView::ext_sphere( vector v, double size)
{
	v = fwt * v;
	if (v.x-size < min_extent[0]) 
		min_extent[0] = v.x-size;
	if (v.y-size < min_extent[1]) 
		min_extent[1] = v.y-size;
	if (v.z-size < min_extent[2]) 
		min_extent[2] = v.z-size;
	if (v.x+size > max_extent[0]) 
		max_extent[0] = v.x+size;
	if (v.y+size > max_extent[1]) 
		max_extent[1] = v.y+size;
	if (v.z+size > max_extent[2]) 
		max_extent[2] = v.z+size;
}

void 
rView::ext_circle( vector p, vector n, double r)
{
	p = fwt * p;
	n = fwt.times_v(n);

	vector d( r*std::sqrt(1 - n.x*n.x)
		, r*std::sqrt(1 - n.y*n.y)
		, r*std::sqrt(1 - n.z*n.z) );
	if (p.x-d.x < min_extent[0]) 
		min_extent[0] = p.x-d.x;
	if (p.y-d.y < min_extent[1]) 
		min_extent[1] = p.y-d.y;
	if (p.z-d.z < min_extent[2]) 
		min_extent[2] = p.z-d.z;
	if (p.x+d.x > max_extent[0]) 
		max_extent[0] = p.x+d.x;
	if (p.y+d.y > max_extent[1]) 
		max_extent[1] = p.y+d.y;
	if (p.z+d.z > max_extent[2]) 
		max_extent[2] = p.z+d.z;
}

/************ DisplayObject implementation **************/

DisplayObject::DisplayObject()
	: visible( true), color(1,1,1)
	// Allow default construction of all of the smart pointer members to zero (NULL).
{
}

DisplayObject::DisplayObject( const DisplayObject& other)
	: Cache( other), color( other.color)
{
	
}
 
DisplayObject::~DisplayObject()
{
}

void 
DisplayObject::glRender(rView&) 
{ 
	return;
};

double 
DisplayObject::rayIntersect( const vector &camera, const vector &ray) 
{ 
	return 0.0;
}

tmatrix 
DisplayObject::getChildTransform()
{ 
	return tmatrix::identity(); 
}

boost::python::object 
DisplayObject::py_get_parent()
{
	if (parent)
		return parent->getObject();
	else
		return boost::python::object();
}

boost::shared_ptr<frame>
DisplayObject::getParent()
{
	return parent;
}

bool
DisplayObject::get_visible() const
{
	return visible;
}


void 
DisplayObject::set_display( boost::shared_ptr<Display> n)
{
	if (!n && display) {
		if (!visible) {
			display = n;
			return;
		}
		// Deleting an existing object from it's display.
		// Called by the destructor (never can happen) and set_visible(false).
		mutex::lock display_objects_lock( display->list_mutex);
		write_lock this_lock( mtx);
		this->remove();
		display = n;
		visible = false;
	}
	else if (n && !display && visible) {
		// Moving onto a display from none.
		mutex::lock display_objects_lock( n->list_mutex);
		write_lock this_lock( mtx);
		display = n;
		this->insert();
	}
	// Now n and display are either both NULL, or both values.
	// If they are both NULL, do nothing.
	// If they are both values and equal, do nothing.
	// If they are values and non-equal, do the following:
	else if (n != display && visible) {
		// Moving an object from one display to another.
		{
			// To prevent a race -> deadlock, we must lock in the same order as
			// glDevice::render().
			mutex::lock old_display_objects_lock( display->list_mutex);
			write_lock this_lock( mtx);
			this->remove();
		}
		display = n;
		{
			mutex::lock new_display_objects_lock( display->list_mutex);
			write_lock this_lock( mtx);
			this->insert();
		}
	}
	// one more case:
	// n is non-null, but we are invisible: set display but do nothing
	else if (n && !visible)
		display = n;
}

boost::shared_ptr<Display>
DisplayObject::get_display() const
{
	return display;
}

boost::python::object 
DisplayObject::py_get_display() const
{
	if (display)
		return display->getObject();
	else
		return boost::python::object();
}

void 
DisplayObject::set_visible( bool v)
{
	if (v && !visible && display) {
		// Becoming visible from invisible, a display is available.
		mutex::lock display_objects_lock( display->list_mutex);
		write_lock this_lock( mtx);
		this->insert();
	}
	else if (!v && visible && display) {
		// Becoming invisible from visible, a display is available.
		mutex::lock display_objects_lock( display->list_mutex);
		write_lock this_lock( mtx);
		this->remove();
	}

	visible=v;
}

// The caller must lock both display->objects and itself prior to calling this function.
void
DisplayObject::insert()
{
	using boost::shared_ptr;
	using boost::python::extract;
	
	shared_ptr<DisplayObject> strong_this;
	if (weak_this.expired()) {
		// Possible if the object is invisible and no shared_ptr owns it in C++.
		strong_this = extract< shared_ptr<DisplayObject> >( self);
		weak_this = strong_this;
	}
	else {
		strong_this = shared_ptr<DisplayObject>( weak_this);
	}
	display->objects.push_back( strong_this);
	display->addObject();
}

// The caller must lock both display->objects and itself prior to calling this function.
void 
DisplayObject::remove()
{
	assert( !weak_this.expired());
	boost::shared_ptr<DisplayObject> strong_this( weak_this);
	display->objects.remove( strong_this);
}

void 
DisplayObject::setParent( boost::shared_ptr<frame> _frame)
{
	write_lock L(mtx);
	if (_frame) {
		if (display != _frame->display) {
			throw std::runtime_error( "Attempt to set parent to object on different display");
		}
	}
	
	parent = _frame;
	if (_frame) {
		boost::shared_ptr<frame> frame_iterator = _frame->parent;
		while (frame_iterator) {
			if (frame_iterator == parent) {
				parent.reset();
				throw std::runtime_error( "Attempt to create a cycle of reference frames");
			}
			frame_iterator = frame_iterator->parent;
		}
	}
}

// We keep two smart pointers to self - a boost::weak_ptr, and a boost::python::object.
// The former is used for comparison against the shared_ptr in the owning Display.
// The second is used to return the most-derived object when reading 
// display.objects and frame.objects from Python.  We must ensure that we don't
// get a cycle of references.  For boost::shared_ptr<>, we simply use 
// boost::weak_ptr<>, and for boost::python::object, we perform an explicit
// decriment on the underlying PyObject*.  Note however, that the object in Python
// is not managed by a shared_ptr, one is created as needed with a deallocator
// that simply decriments the reference count of the owning PyObject*.  So, it
// is possible that the weak_ptr doesn't point to anything if no shared_ptr exists
// in C++ and we must regenerate it as required.
void
DisplayObject::py_complete_init( 
	boost::shared_ptr<DisplayObject> cpp_self, 
	boost::python::object py_self,
	bool _visible,
	boost::shared_ptr<Display> _display,
	boost::shared_ptr<frame> _parent)
{
	weak_this = cpp_self;
	self = py_self;
	Py_DECREF(self.ptr());
	visible = _visible;
	if (visible) {
		// Insert into the required Display.
		set_display( _display);
		setParent( _parent);
	}
	else {
		// Then don't insert into the appropriate display, but make a
		// note of the request for when the object is made visible.  This
		// prevents the spurious activation of an empty display in the
		// case where the user creates an object in the invisible state.
		display = _display;
		parent = _parent;
	}
}



} // !namespace visual
