/*
 *  Copyright (C) 1999
 *  Robert Lissner
 *  Jay MacDonald
 *  Sam Phillips
 *  Keith Wesolowski. 

 *  www.quicklist.org */

#include "includes.h"
#include "globals.h"

#define FROM_LIST_WIDTH   180
#define FROM_LIST_HEIGHT  180
#define TO_LIST_WIDTH  180
#define TO_LIST_HEIGHT 180



/* definitions of all the internal routines */
void adjust_preview (GtkObject *object, gpointer *entry);
void adjust_report ();
void add_ok (GtkObject *object, gpointer *entry);
void add_cancel (GtkObject *object, gpointer *entry);
static void col_from_clicked (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data);
void display_column (gint col); 
void display_all_columns ();

void filter_change (GtkObject *object, gpointer *entry);
static  void from_clicked (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data);
static void new_report_column_width (GtkSheet *sheet, gint col, gint width);
gint pick6rows ();
void pos_check_group ();
void pos_edit (GtkObject *object, gpointer *entry);
void pos_delete (GtkObject *object, gpointer *entry);
void pos_justify (gint jx);
void pos_list_mode (GtkObject *object, gpointer *entry);
void pos_page_heading ();
void pos_print_header ();
void pos_print_footer ();
void pos_newline ();
void print_on_screen ();
void pos_get_dim ();  
void pos_print_total (char level);
void pos_print  (GtkObject *object, gpointer *entry);
gint report_activate (GtkSheet *sheet, gint row, gint col);
void report_add_cancel (GtkWidget *w, gpointer *data);
void report_go_to (gint row0, gint rowi, gint col0, gint coli);
void report_set_col_field (gint col);  
void return_to_list ();
static  void select_ok (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data);
void select_report_range (GtkSheet *sheet, GtkSheetRange *range);
void sort_change (GtkObject *object, gpointer *entry);
GtkWidget* sort_filter_menu  ();
void swap_columns (gint from, gint to);
static void to_clicked (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data);
/* these are for basic_report_mode */

char* newrow [2];

/* these variables are just used to format a report, and then 
   can be reused */
static gint total_width; /* adjust for spaces between columns */
 GtkWidget *ok_button;

/* for checking for group breaks */
static gboolean any_group;
static gboolean any_total;
static gint last_group_row;


static gint rowx;

static gint page_no;
static gint widthx;
static gint line_count;
static GdkFont *font;
static char spaces [84];
static char hash [84];
static char dashes [84];
static gint local_last_column;
static char* pos_text;
static size_t local_len;
static gint rolling_skip_over;

/* not static so it can be called from filepr.c 
   this is part of a worldwide bad hack conspiracy  -sdp */
GtkText *text_box;


/* Here's a bunch of information stored for print_on_screen, for each
   column, to keep the speed up */
/* for group and grand totals */
typedef struct _col_info {
  double group_total;
  double grand_total;
  gint skip_over;
  gboolean do_group;
  gboolean error;
  gboolean do_total;
  gint16 field;
  gint16 list_col;
  gint16 justification;
  gint16 formatting;
  gint16 decimal_places;
  gint16 width; /* when printed, put a space after */
  gint16 type;
} col_info;

static col_info pos_col [MAX_FIELDS + 2]; 


/* _________________________________
   |                                |
   |         add_cancel             |
   |________________________________| 
   OK clicked setting up a list of report columns */
 void add_cancel (GtkObject *object, gpointer *entry) {
  front->report_ct--;
  front->report_ptr = NULL;
  destroy_dialog (); 
}

/* _________________________________
   |                                |
   |         add_ok                 |
   |________________________________| 
   OK clicked setting up a list of report columns */
 void add_ok (GtkObject *object, gpointer *entry) {
  gchar* text;
  text = gtk_entry_get_text (GTK_ENTRY (front->entry1));
  if (check_entry (text))
    return;
  front_is_changed ();
  strcpy (front->report_ptr->name, text);
  adjust_report ();
}


/* _________________________________
   |                                |
   |      adjust_preview            |
   |________________________________| 
   Preview clicked while adjusting report format  */
void adjust_preview (GtkObject *object, gpointer *entry) {
  gchar * text;
  text = gtk_entry_get_text (GTK_ENTRY (front->entry1));
  if (check_entry (text))
    return;
  strcpy (front->report_ptr->name, text);  

  text = gtk_entry_get_text (GTK_ENTRY (front->entry2));
  if (check_entry (text))
    return;
  strcpy (front->report_ptr->header, text);  

  text = gtk_entry_get_text (GTK_ENTRY (front->entry3));
  if (check_entry (text))
    return;
  strcpy (front->report_ptr->footer, text);  

  print_on_screen ();
}

/* |--------------------------------|
   |   adjust_report                |
   |--------------------------------|
   Initialize the report mode window, fill it up and display it */
void adjust_report () {
  char *temp_file_name;
  gint rowx;
  GtkWidget *vbox;
  GtkWidget *menu_bar;
  GtkWidget *hbox1;
  GtkWidget *name_label;

  GtkWidget *hbox2; /* clists and header */
  GtkWidget *header_label; /* for basic */

  GtkWidget *hbox3; /* sort and filter menus */
  
  GtkWidget *scrolled_window;
  
  GtkWidget *hbox5; /* Footer */
  GtkWidget *footer_label;
  
  GtkWidget *hbox6; /* same as action area */
  GtkWidget *preview_button;
  GtkWidget *edit_list_button;
  GtkWidget *delete_button;

  destroy_dialog ();  /* gets into from front-> */
  get_window_size_loc (front->list_win);
  gtk_widget_hide (GTK_WIDGET (front->list_win));
  pick6rows (); /* find six rows to display */
  
  /* now make a window for editing this report format */
  front->report_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_container_set_border_width (GTK_CONTAINER (front->report_win), 5);
  front->display_mode = 'R';
  front->report_mode = 'E';
  temp_file_name = g_strconcat (front->file_name, "  (Edit Report Layout)", NULL);
  gtk_window_set_title (GTK_WINDOW (front->report_win), temp_file_name);
  g_free (temp_file_name);

  gtk_window_set_policy (GTK_WINDOW (front->report_win), TRUE, TRUE, FALSE);
  set_window_size_loc (front->report_win);
 
  vbox = gtk_vbox_new (FALSE, 3);

  menu_bar = build_report_mb ();
  gtk_box_pack_start (GTK_BOX (vbox), menu_bar,
		      FALSE, TRUE, 0);

  /* Show and allow changes to report name */
  hbox1 = gtk_hbox_new (FALSE, 5);
  name_label = gtk_label_new (_("Report Name"));
  gtk_box_pack_start (GTK_BOX (hbox1), name_label, FALSE, FALSE, 0);
  
  front->entry1 = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox1), front->entry1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox1, FALSE, FALSE, 0); 
  gtk_entry_set_text (GTK_ENTRY (front->entry1), front->report_ptr->name);

  /* Show and allow changes to report header*/
  hbox2 = gtk_hbox_new (FALSE, 5);
  header_label = gtk_label_new (_("Print at top of page"));
  gtk_box_pack_start (GTK_BOX (hbox2), header_label, FALSE, FALSE, 0);
  
  front->entry2 = gtk_entry_new_with_max_length (80);
  gtk_box_pack_start (GTK_BOX (hbox2), front->entry2, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox2, FALSE, FALSE, 0); 
  gtk_entry_set_text (GTK_ENTRY (front->entry2), front->report_ptr->header);

  /* build menus for sorts and filters */
  hbox3 = sort_filter_menu ();
  gtk_box_pack_start (GTK_BOX (vbox), hbox3, FALSE, FALSE, 0); 

  /* make sheet and add to scroll_win */
  front->report_sheet = gtk_sheet_new_browser (8, front->report_ptr->
					       last_column + 1,
					       temp_file_name);
  report_set_col_field (0);  /* cursor starts here */
  gtk_sheet_row_button_add_label (GTK_SHEET (front->report_sheet), 
				    0, "Group"); 
  gtk_sheet_row_button_add_label (GTK_SHEET (front->report_sheet), 
				    1, "Total"); 
  for (rowx = 2; rowx < 8; rowx++)
    gtk_sheet_row_button_add_label (GTK_SHEET (front->report_sheet), 
				    rowx, "Sample"); 
  GTK_SHEET_SET_FLAGS (GTK_SHEET (front->report_sheet), 
		       /* GTK_SHEET_ADJUST_TEXT + */
		       GTK_SHEET_AUTO_SCROLL);
  GTK_SHEET_UNSET_FLAGS (GTK_SHEET (front->report_sheet), 
		       GTK_SHEET_AUTORESIZE);
  gtk_sheet_set_row_titles_width (GTK_SHEET (front->report_sheet), 60);
  gtk_container_border_width (GTK_CONTAINER (front->report_sheet), 4);
  
  /* set sensitivity for all column and row buttons */
  gtk_sheet_show_row_titles (GTK_SHEET (front->report_sheet));
  gtk_sheet_show_column_titles (GTK_SHEET (front->report_sheet));
  gtk_sheet_columns_set_sensitivity (GTK_SHEET (front->report_sheet), TRUE);
  gtk_sheet_rows_set_sensitivity  (GTK_SHEET (front->report_sheet), FALSE);
  display_all_columns ();

  /* create a new scrolled window. */
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  
  gtk_container_set_border_width (GTK_CONTAINER (scrolled_window), 
				  3);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW 
				  (scrolled_window),
				  GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS);
	 
  gtk_container_add (GTK_CONTAINER (scrolled_window), front->report_sheet);
  gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, 
		      TRUE, TRUE, 0); 

  /* Show and allow changes to report footer*/
  hbox5 = gtk_hbox_new (FALSE, 5);
  footer_label = gtk_label_new (_("Print at bottom of page"));
  gtk_box_pack_start (GTK_BOX (hbox5), footer_label, FALSE, FALSE, 0);
  
  front->entry3 = gtk_entry_new_with_max_length (80);
  gtk_box_pack_start (GTK_BOX (hbox5), front->entry3, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (vbox), hbox5, FALSE, FALSE, 0); 
  gtk_entry_set_text (GTK_ENTRY (front->entry3), front->report_ptr->footer);

  /* Now do Preview,  OK  and Cancel buttons */
  hbox6 = gtk_hbox_new (TRUE, 5);
  preview_button = gtk_button_new_with_label (_("Preview"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      preview_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(preview_button), "clicked", 
		      GTK_SIGNAL_FUNC (adjust_preview), NULL);

  edit_list_button = gtk_button_new_with_label (_("Edit the list"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      edit_list_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(edit_list_button), "clicked", 
		      GTK_SIGNAL_FUNC (pos_list_mode), NULL);
  
  delete_button = gtk_button_new_with_label (_("Delete"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      delete_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(delete_button), "clicked", 
		      GTK_SIGNAL_FUNC (pos_delete), NULL);

  gtk_box_pack_start (GTK_BOX (vbox), hbox6, FALSE, FALSE, 0); 

  gtk_container_add (GTK_CONTAINER (front->report_win), vbox);
  dim_all_menus ();    
  gtk_signal_connect (GTK_OBJECT (front->report_win), "delete_event",
		      GTK_SIGNAL_FUNC (pos_list_mode), NULL);
  gtk_signal_connect (GTK_OBJECT (front->report_win), "focus-in-event",  
		      GTK_SIGNAL_FUNC (focus_in_event_cb), NULL);
  gtk_signal_connect(GTK_OBJECT( front->report_sheet),
		     "new_column_width",
		     GTK_SIGNAL_FUNC(new_report_column_width), NULL); 
  gtk_signal_connect(GTK_OBJECT( front->report_sheet),
		     "select_range",
		     GTK_SIGNAL_FUNC(select_report_range), NULL);
  gtk_signal_connect(GTK_OBJECT( front->report_sheet),
		     "activate",
		     GTK_SIGNAL_FUNC(report_activate), NULL);
gtk_widget_show_all (front->report_win);
} /* end of build_basic_report_mode */


/* _________________________________
   |                                |
   |      col_from_clicked          |
   |________________________________| 
  An item clicked while adding a column   */
static  void col_from_clicked (GtkWidget *widget, gint row, 
			 gint col, GdkEventButton *event, gpointer data) { 

report_column *this_column;
  gint colx, col_spot, col_last;
  gint fieldx;
  if (event->type != GDK_2BUTTON_PRESS)
    return;
      
  col_spot = front->report_sel_col;
  col_last = front->report_ptr->last_column;
  if (col_last >= MAX_FIELDS - 1) {
    level2_error ("You are limited to 30 columns on a report.",
		  "Go back");
    return;
  }
  
  /* move all existing colums up one */
  for (colx = col_last; colx >= col_spot; colx--)
    front->report_ptr->column [colx+1] = front->report_ptr->column [colx];
  
  /* build the contents of this new column */
  this_column = &front->report_ptr->column [col_spot];
  fieldx = front->col_to_field [row];
  newrow [0] = front->fields [fieldx].name;

  /* now add to report definition */
  this_column->field = fieldx;
  this_column->width = front->fields [fieldx].width;
  this_column->group = 0;
  this_column->total = 0;

  /* now add to report sheet */
  gtk_sheet_insert_columns (GTK_SHEET (front->report_sheet), col_spot, 1);
  display_column (col_spot); /* this thaws it out */
  front->report_ptr->last_column++;
} /* end of col_from_clicked */

/* _________________________________
   |                                |
   |      display_all_columns       |
   |________________________________| */
void display_all_columns () {
  gint colx;
  big_draw_start ();
  for ( colx = 0; colx <= front->report_ptr->last_column; colx++ )
    display_column (colx);
  big_draw_end ();
} /* end of display_all_columns */

/* _________________________________
   |                                |
   |      display_column            |
   |________________________________| */
void display_column (gint colx) {
  gint fieldx;
  gint rowx;
  gint list_mode_col;
  char* text;
  report_column *this_column;

  this_column = &front->report_ptr->column [colx];

  fieldx = this_column->field;
  list_mode_col = front->fields [fieldx].sheet_column;
  gtk_sheet_set_column_width (GTK_SHEET (front->report_sheet), 
			      colx, this_column->width*8);
  gtk_sheet_column_button_add_label( GTK_SHEET (front->report_sheet), colx, 
				       front->fields[fieldx].name);
  gtk_sheet_column_set_justification(GTK_SHEET (front->report_sheet), colx,
				       front->fields [fieldx].justification);

  /* now display the six rows that we are showing as examples */
  if (this_column->group) 
    gtk_sheet_set_cell_text (GTK_SHEET (front->report_sheet), 0, colx,
				"Group");
  else
    gtk_sheet_cell_clear (GTK_SHEET (front->report_sheet), 0, colx);

  if (this_column->total) 
    gtk_sheet_set_cell_text (GTK_SHEET (front->report_sheet), 1, colx,
				"Total");
  else 
    gtk_sheet_cell_clear (GTK_SHEET (front->report_sheet), 1, colx);
   
  /* insert the up to six rows of data */
  for (rowx = 2; rowx < 8; rowx++) {
    gtk_sheet_cell_clear (GTK_SHEET (front->report_sheet), rowx, colx);
    if (front->adjust_rows [rowx] >= 0) {
      text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), 
				    front->adjust_rows [rowx],
				    list_mode_col);
      if (text)
	gtk_sheet_set_cell_text (GTK_SHEET (front->report_sheet), rowx, 
			       colx, text);
    }
  }
}

void filter_change (GtkObject *object, gpointer *entry){
  front->report_ptr->filter = (gint) entry - 1;
  filter_do_apply (front->report_ptr->filter);  
  pick6rows ();
  display_all_columns ();
}


/* _________________________________
   |                                |
   |      from_clicked              |
   |________________________________| 
   From clist clicked.  This adds one column to a new report layout  */
static void from_clicked (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data) {
  report_column *this_column;
  gint colx;
  gint fieldx;  
  if (event->type != GDK_2BUTTON_PRESS)
    return;
      
  if (front->report_ptr->last_column >= MAX_FIELDS - 1)
    level2_error ("You are limited to 30 columns on a report.",
		  "Go back");
  colx = ++front->report_ptr->last_column;
  this_column = &front->report_ptr->column [colx];
  fieldx = front->col_to_field [row];
  newrow [0] = front->fields [fieldx].name;
  gtk_clist_append (GTK_CLIST (front->clist_to), newrow);

  /* now add to report definition */
  this_column->field = fieldx;
  this_column->width = front->fields [fieldx].width; 
  gtk_widget_set_sensitive (ok_button, TRUE);
} /* end of from clicked */


/* |--------------------------------|
   |   new_report_column_width      |
   |--------------------------------|
*/ 
static void new_report_column_width (GtkSheet *sheet, gint col, gint width) {
  front_is_changed ();
  if (width > 640) /* gotta have some sort of limit.  Paper is finite */
    front->report_ptr->column [col].width = 80;
  else
    front->report_ptr->column [col].width = width / 8;
}

/* _________________________________
   |                                |
   |       pick6rows                |
   |________________________________|
   Adjust_report displays six rows of data.  They must match the current
   filter.  returns number of rows found */
gint pick6rows () {
  gint hits = 0;
  gint rowx, torow, fromrow;
  
  /* start out in case all rows are shown */
  for (rowx = 0; rowx < 6; rowx++) {
    if (rowx <= front->last_row)
      front->adjust_rows [rowx + 2] = rowx;
    else front->adjust_rows [rowx + 2] = 0;
  }

  torow = fromrow = 0;
  do {
    if (row_is_visible (fromrow)) {
      hits++;
      front->adjust_rows [torow+2] = fromrow;
      torow++;
    }
   }
  while (++fromrow <= front->last_row
	 && torow < 6);
  for (; torow < 6; torow++)
    front->adjust_rows [torow+2] = -1;
  if (!hits)
    level1_error ("No rows match the current filter", "   OK    ");
  return (hits);
} /* End of pick6 rows */
    
/* _________________________________
   |                                |
   |         pos_edit               |
   |________________________________|
   User is looking at print on screen, now wants to edit it */
void pos_edit (GtkObject *object, gpointer *entry){
  pos_get_dim ();  
  get_window_size_loc (front->report_win); /* save size and location */
  clean_destroy (&front->report_win);
  adjust_report ();
}


/* _________________________________
   |                                |
   |         pos_get_dim            |
   |________________________________|
   Save the dimensions of the print on screen window */
void pos_get_dim () {
  front->report_ptr->width = 
    GTK_WIDGET(front->report_win)->allocation.width;
  front->report_ptr->height = 
    GTK_WIDGET(front->report_win)->allocation.height;
}

/* _________________________________
   |                                |
   |         pos_delete             |
   |________________________________|
   Get rid of this report */
void pos_delete (GtkObject *object, gpointer *entry){
  gint reportx;

  /* move all others down one spot.  There may be none to move */
  for (reportx = front->report_no; reportx < front->report_ct - 1; reportx++)
    memcpy (&front->reports [reportx], &front->reports [reportx + 1],
	    sizeof (report_info));
  front->report_ct--;
  return_to_list ();
}

/* _________________________________
   |                                |
   |         pos_list_mode          |
   |________________________________|
   Clicked  in print_on_screen, go to list mode */
void pos_list_mode (GtkObject *object, gpointer *entry){
  pos_get_dim ();
  return_to_list ();
  dim_all_menus ();
}


/* _________________________________
   |                                |
   |         pos_newline            |
   |________________________________|
   Start a new line in the text buffer */
void pos_newline () {
  gtk_text_insert (text_box, NULL, NULL,
			 NULL, "\n", 1);
} /* end of pos_newline */

/* _________________________________
   |                                |
   |         pos_print              |
   |________________________________|
   Print what's currently on the screen onto lpr instead */
void pos_print (GtkObject *object, gpointer *entry) {
  gchar* big_block;
  FILE *printer;
  
  printer = popen ("lpr", "w");
  if (printer == NULL) {
    level1_error ("    Unable to open the printer.     ",
		  "Go back");
    return;
  }
  big_block = 
    (gchar*) gtk_editable_get_chars (GTK_EDITABLE (text_box), 0, -1);
  fprintf (printer, big_block);
  pclose (printer);
  g_free (big_block);
}


/* _________________________________
   |                                |
   |       pos_check_group          |
   |________________________________|
   Check for and print group totals if that's the thing to do */
void pos_check_group (gint row) {
  gint colx;
  char* old_text;
  char* new_text;
  gboolean got_one;
  if (last_group_row < 0) { /* store the first record if not done */
    last_group_row = row;
    return;
  }
  got_one = FALSE;
  for (colx = 0; colx <= local_last_column; colx++) { 
    if (!pos_col [colx].do_group)
      continue;
    old_text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), 
					last_group_row, 
					pos_col [colx].list_col);
    new_text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), 
					row, pos_col [colx].list_col);
    if (!new_text) { 
      if (!old_text)
	continue;
      else {
	got_one = TRUE;
	break;
      }
    }
    /* therefore, there is new text */
    if (!old_text) {
      got_one = TRUE;
      break;
    }
    /* both have text, compare */
    if (strcmp (old_text, new_text)) {
      got_one = TRUE;
      break;
    }
  }
  /* end of loop to look for break */
  if (!got_one)
    return;
  
  /* OH MAN, GOT A BREAK, HAVE TO PRINT IT NOW */
  pos_print_total ('G');
  last_group_row = row;
} /* end of pos_check_group */


/* _________________________________
   |                                |
   |       pos_print_total          |
   |________________________________|
   Print group or grand totals on screen */
void pos_print_total (char level) {
  gint colx;
  gint widthx;
  gint skipped = 0;
  gint skip_now;
  char linebuf [40];
  for (colx = 0; colx <= local_last_column; colx++) { 
    if (!pos_col [colx].do_total)
      continue;
    /* skip over to right column */
    widthx = pos_col [colx].width;
    skip_now =  pos_col [colx].skip_over - skipped + 1;
    skipped += pos_col [colx].skip_over + widthx;
    if (skip_now > 80)
      skip_now = 80;
    if (skip_now)
      gtk_text_insert (text_box, font, NULL,
		     NULL, spaces, skip_now);
    if (pos_col [colx].error) 
      local_len = d2qls (linebuf, (double) HUGE_VAL, 
			 (gint) FIELD_TYPE_NUMERIC, 0, 0);
    else if (level == 'G') {
      local_len = d2qls (linebuf, pos_col [colx].group_total,
			 (gint) FIELD_TYPE_NUMERIC,
			 pos_col [colx].formatting,
			 pos_col [colx].decimal_places);
      pos_col [colx].group_total = 0;
    }
    else local_len = d2qls (linebuf, pos_col [colx].grand_total,
			    (gint) FIELD_TYPE_NUMERIC,
			    pos_col [colx].formatting,
			    pos_col [colx].decimal_places);
    if (local_len >= widthx) {
      /* text is longer than column width allowed */
      gtk_text_insert (text_box, font, NULL,
		       NULL, hash, widthx - 1);
      continue; 
    }
    /* total fits, right justify it */
    if (widthx - 1 > local_len)
      gtk_text_insert (text_box, font, NULL,
		       NULL, spaces, widthx - local_len - 1);
    gtk_text_insert (text_box, font, NULL,
		       NULL, linebuf, local_len);
  }
  pos_newline ();
  pos_newline (); /* blank line */
  line_count += 2;
} /* end of pos_print_total */


/* _________________________________
   |                                |
   |       print_on_screen          |
   |________________________________|
   Print the selected report on the screen */
void print_on_screen () {

  GtkWidget *vbox;
  gint hits = 0;
  gint colx;
  /* some local variables to speed up the iterative process */
  gint fieldx;
  
  GtkWidget *scrolled_window;
  GtkWidget *hbox6; 
  GtkWidget *print_button;
  GtkWidget *edit_report_button;
  GtkWidget *edit_list_button;
  GtkWidget *delete_button;
  double temp_number;

  last_group_row = -1;
  total_width = -1; /* adjust for spaces between columns */
  any_group = any_total =  FALSE;
  rolling_skip_over = 0; /* spaces to skip */
  page_no = 0;
  line_count = 0;
  local_last_column = front->report_ptr->last_column;

  /* setup some local variables to  save iterative cycles later */
  for (colx = 0; colx <= local_last_column; colx++) { 
    pos_col [colx].group_total = 0;
    pos_col [colx].grand_total = 0;
    pos_col [colx].error = FALSE;
    pos_col [colx].do_group = FALSE;
    pos_col [colx].do_total = FALSE;
    pos_col [colx].field = fieldx = front->report_ptr->column [colx].field;
    pos_col [colx].list_col = front->fields [fieldx].sheet_column;
    pos_col [colx].justification = front->fields [fieldx].justification;
    pos_col [colx].formatting = front->fields [fieldx].formatting;
    pos_col [colx].decimal_places = front->fields [fieldx].decimal_places;
    pos_col [colx].width = front->report_ptr->column [colx].width;
    pos_col [colx].type = front->fields [fieldx].type;
    
    if (front->report_ptr->column [colx].group) {
      pos_col [colx].do_group = TRUE;
      any_group = TRUE;
    }
    if (front->report_ptr->column [colx].total  &&
	pos_col [colx].type == FIELD_TYPE_NUMERIC) {
      pos_col [colx].do_total = TRUE;
      any_total = TRUE;
    }
    if (front->report_ptr->column [colx].width < 2)
      front->report_ptr->column [colx].width = 2;

     /* any less, nothing would display */
    if (front->report_ptr->column [colx].width > 80)
      front->report_ptr->column [colx].width = 80; 
    pos_col [colx].skip_over = rolling_skip_over;
    rolling_skip_over += front->report_ptr->column [colx].width + 1; 

    /* paper has finite limits */    
    total_width += front->report_ptr->column [colx].width + 1;
  }

  /* build strings of 80 characters each */
  spaces [0] = '\0';
  hash [0] = '\0';
  dashes [0] = '\0';
  for (rowx = 0; rowx < 10; rowx++) {
    strcat (spaces, "        "); /* eventually 80 spaces */
    strcat (hash, "########"); /* eventually 80 of them */
    strcat (dashes, "--------"); /* eventually 80 of them */
  }
  
  if (front->report_win) {
    get_window_size_loc (front->report_win); /* save size and location */
    clean_destroy (&front->report_win);
  }
  front->report_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  front->display_mode = 'R';
  front->report_mode = 'P';
  gtk_widget_set_name (front->report_win, "text window");

  set_window_size_loc (front->report_win);
  gtk_window_set_policy (GTK_WINDOW(front->report_win), TRUE, TRUE, FALSE);
  gtk_signal_connect (GTK_OBJECT (front->report_win), "focus-in-event",  
		      GTK_SIGNAL_FUNC (focus_in_event_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (front->report_win), "delete_event",
		      GTK_SIGNAL_FUNC(pos_list_mode), NULL);
  gtk_window_set_title (GTK_WINDOW (front->report_win), 
			front->report_ptr->name);
  gtk_container_set_border_width (GTK_CONTAINER (front->report_win), 0);
  
  vbox = gtk_vbox_new (FALSE, 3);
  gtk_container_add (GTK_CONTAINER (front->report_win), vbox);
  
  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_box_pack_start (GTK_BOX (vbox), scrolled_window, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_ALWAYS,
				  GTK_POLICY_ALWAYS);
  
  front->text_box = gtk_text_new (NULL, NULL);
  text_box = GTK_TEXT (front->text_box);
  gtk_text_set_editable (text_box, FALSE);
  gtk_container_add (GTK_CONTAINER (scrolled_window), front->text_box);
  font = 
    gdk_font_load ("-adobe-courier-medium-r-normal--*-120-*-*-*-*-*-*");
  gtk_text_freeze (text_box);

  /* this is the MAIN LOOP of printing individual rows to the screen */
  /* last row doesn't count */   
  for (rowx = 0; rowx < front->last_row; rowx++) 
    if (row_is_visible (rowx)) {
      hits++;
      if (any_group) /* won't do much if page = 0 */
	 pos_check_group (rowx); /* might print a total and space */

      /* MAIN LOGIC CHANGE.  PRINT REPORT HEADING. */
      if (!page_no || line_count > 54) {
	if (page_no)
	  pos_print_footer ();
	pos_print_header ();
      }
	
      /* MAIN BREAK.  PRINT DATA HERE */
      for (colx = 0; colx <= local_last_column; colx++) { 
	widthx = pos_col [colx].width;
		
	if (colx) /* space betwen columns, except before first */
	  gtk_text_insert (text_box, font, NULL,
		     NULL, spaces, 1);
	pos_text = gtk_sheet_cell_get_text (GTK_SHEET (front->sheet), 
					    rowx, pos_col [colx].list_col);
	if (!pos_text) {
	  gtk_text_insert (text_box, font, NULL,
		     NULL, spaces, widthx);
	  continue;
	}
	/* text is not null, add to totals if appropriate */
	if (pos_col [colx].do_total && !pos_col [colx].error) {
	   temp_number = qls2d (pos_text, (gint) FIELD_TYPE_NUMERIC,
				pos_col [colx].formatting);
	   if (temp_number >= HUGE_VAL)
	     pos_col [colx].error = TRUE;
	   else {
	     pos_col [colx].group_total += temp_number;
	     pos_col [colx].grand_total += temp_number;
	   }
   	}
	/* now let's format it on paper */
	local_len = strlen (pos_text);
	if (local_len > widthx) {
	  /* text is longer than column width allowed */
	  if (pos_col [colx].type == FIELD_TYPE_TEXT) /* text */
	    gtk_text_insert (text_box, font, NULL,
			     NULL, pos_text, widthx);
	  else 
	    gtk_text_insert (text_box, font, NULL,
			     NULL, hash, widthx);
	  continue;
	}

	/* maybe text is same width as allowed */
	if (local_len == widthx) {
	  gtk_text_insert (text_box, font, NULL,
			   NULL, pos_text, widthx);
	  continue;
	}

	/* what's left is that the text is shorter than column, so
	   we have to justify the mess */
	pos_justify (pos_col [colx].justification);
      } /* end of loop to display one column */
   
      pos_newline ();
      line_count++;
    } 
  if (any_group)
    pos_print_total ('G');
  if (any_total)
    pos_print_total ('T');

  gtk_text_thaw (text_box);

  /* Now do Print, edit, cancel and delete */
  hbox6 = gtk_hbox_new (TRUE, 5);
  print_button = gtk_button_new_with_label (_("Print"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      print_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(print_button), "clicked", 
		      GTK_SIGNAL_FUNC (file_print_dlg), NULL);

  edit_report_button = gtk_button_new_with_label (_("Edit the report"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      edit_report_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(edit_report_button), "clicked", 
		      GTK_SIGNAL_FUNC (pos_edit), NULL);

  edit_list_button = gtk_button_new_with_label (_("Edit the list"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      edit_list_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(edit_list_button), "clicked", 
		      GTK_SIGNAL_FUNC (pos_list_mode), NULL);
  
  delete_button = gtk_button_new_with_label (_("Delete the report"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      delete_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(delete_button), "clicked", 
		      GTK_SIGNAL_FUNC (pos_delete), NULL);

  gtk_box_pack_start (GTK_BOX (vbox), hbox6, FALSE, FALSE, 0); 
  gtk_widget_show_all (front->report_win);
  if (!hits)
    level1_error ("No rows were selected",  "    OK    ");
} /* end of print_on_screen */


/* |--------------------------------|
   |     pos_justify                |
   |--------------------------------|
   Just put one cell on the screen, justified properly */ 
void pos_justify (gint jx) {
  gint half;
  if (jx == GTK_JUSTIFY_LEFT) {
      gtk_text_insert (text_box, font, NULL,
		       NULL, pos_text, local_len);
      gtk_text_insert (text_box, font, NULL,
		       NULL, spaces, widthx - local_len);
      return;
    }
    
    if (jx == GTK_JUSTIFY_RIGHT) {
      gtk_text_insert (text_box, font, NULL,
		       NULL, spaces, widthx - local_len);
      gtk_text_insert (text_box, font, NULL,
		       NULL, pos_text, local_len);
      return;
    }
    
    /* oh fun, what's left is center */
    half = (widthx - local_len) / 2;
    gtk_text_insert (text_box, font, NULL,
		     NULL, spaces, half);
    gtk_text_insert (text_box, font, NULL,
		     NULL, pos_text, local_len);	
    gtk_text_insert (text_box, font, NULL,
		     NULL, spaces, widthx - local_len - half);
} /* end of pos_justify */





/* |--------------------------------|
   |     pos_print_header           |
   |--------------------------------|
*/   
void pos_print_header () {
  gint colx;
  gint fieldx;

  page_no++;
  line_count = 0;  
  if (front->report_ptr->header [0]) {
    gtk_text_insert (text_box, font, NULL,
		     NULL, front->report_ptr->header, -1);
    pos_newline ();
  }
  /* column headings */
  for (colx = 0; colx <= local_last_column; colx++) {  
    if (colx) /* space betwen columns, except before first */
      gtk_text_insert (text_box, font, NULL,
		       NULL, spaces, 1);
    fieldx = pos_col [colx].field;
    widthx = pos_col [colx].width;
    pos_text = (char*) &front->fields [fieldx].name;
    local_len = strlen (pos_text);
    if (local_len >= widthx) {
      /* heading is longer than column width allowed */
      gtk_text_insert (text_box, font, NULL,
		       NULL, pos_text, widthx);
      continue;
    }
    
    /* what's left is that the heading is shorter than column, so
       we have to justify it */
    pos_justify (pos_col [colx].justification);
  } /* end of displaying column headings */

  pos_newline ();
  if (total_width < 80)
    gtk_text_insert (text_box, font, NULL,
		     NULL, dashes, total_width);
  else
    gtk_text_insert (text_box, font, NULL,
		     NULL, dashes, 80);
  pos_newline ();
} /* end of pos_print_header */
      
/* |--------------------------------|
   |     pos_print_footer           |
   |--------------------------------|
*/   
void pos_print_footer () {
  if (front->report_ptr->footer [0]) {
    pos_newline ();
    gtk_text_insert (text_box, font, NULL,
		     NULL, front->report_ptr->footer, -1);
    pos_newline ();
  }
  pos_newline (); /* just for appearance on screen */
  pos_newline ();
  pos_newline ();
  gtk_text_insert (text_box, font, NULL,
		   NULL, "\f", 1); 
}



/* |--------------------------------|
   |   report_activate              |
   |--------------------------------|
*/ 
gint report_activate (GtkSheet *sheet, gint row, gint col) {
  if (row < 0 || col < 0)
    return (TRUE); /* row or column buttons */
  report_set_col_field (col);
  dim_all_menus ();
  return (TRUE);   
}

/* _________________________________
   |                                |
   |      report_add_cancel         |
   |________________________________| */
void report_add_cancel (GtkWidget *w, gpointer *data) {
  front->report_ct--;
  destroy_dialog ();
}


/* _________________________________
   |                                |
   |         report_add             |
   |________________________________| 
   This creates a new report format */
void report_add (GtkWidget *w, gpointer *data) {
  gint colx;
  gint fieldx;  

 GtkWidget *top_vbox;
 GtkWidget *hbox1;
 GtkWidget *name_label;

 GtkWidget *hbox2; /* clists and header */

  GtkWidget *scrwin_from;
 GtkWidget *scrwin_to;

 GtkWidget *hbox6; /* same as action area */

 GtkWidget *cancel_button;

 gchar* from_titles [1] = {"Double-click to add to report"};
 gchar* to_titles [1] = {"Double-click to remove"};
 gchar original_name [] = "Please type a report name";
  
 if (check_if_changed ())
    return;
 front->report_no = front->report_ct++;
 front->report_ptr = front->report_ptr = &front->reports [front->report_no];
  
  memset ((char*) front->report_ptr, '\0', sizeof (report_info));
  front->report_ptr->last_column = -1;
  front->report_ptr->sort = -1;
  front->report_ptr->filter = -1;
  
  make_basic_dialog1 ();

  gtk_window_set_title (GTK_WINDOW(dialog1_win), 
			_("New report:  select columns in any order"));
  top_vbox = GTK_DIALOG(dialog1_win) -> vbox;
  gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
  hbox6 = GTK_DIALOG (dialog1_win) -> action_area;

  /* allow for report name */
  hbox1 = gtk_hbox_new (FALSE, 5);
  name_label = gtk_label_new (_("Report Name"));
  gtk_box_pack_start (GTK_BOX (hbox1), name_label, FALSE, FALSE, 0);
  
  front->entry1 = gtk_entry_new ();
  gtk_box_pack_start (GTK_BOX (hbox1), front->entry1, TRUE, TRUE, 0);
  gtk_box_pack_start (GTK_BOX (top_vbox), hbox1, FALSE, FALSE, 0); 
  gtk_entry_set_text (GTK_ENTRY (front->entry1), original_name);
  gtk_entry_select_region(GTK_ENTRY (front->entry1), 0, 
			  strlen (original_name));

  /* now the area containing clists */
  /*  The horizontal box containing the from and to listboxes  */
  hbox2 = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

  /* The from (column names)  clist */
  front->clist_from = gtk_clist_new_with_titles (1, from_titles);     

  gtk_clist_set_column_width (GTK_CLIST (front->clist_from), 0, 
			       FROM_LIST_WIDTH - 30);  
  gtk_widget_set_usize (front->clist_from, FROM_LIST_WIDTH, 
			FROM_LIST_HEIGHT);
  gtk_signal_connect (GTK_OBJECT (front->clist_from), "select_row",
		      (GtkSignalFunc) from_clicked, NULL);
  gtk_clist_column_titles_passive (GTK_CLIST (front->clist_from));

  scrwin_from = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrwin_from), front->clist_from);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_from),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrwin_from), 5);
  gtk_box_pack_start (GTK_BOX (hbox2), scrwin_from, TRUE, TRUE, 0);

  /* The to files clist */
  front->clist_to = gtk_clist_new_with_titles (1, to_titles);     

  gtk_clist_set_column_width(GTK_CLIST (front->clist_to), 0, 
			     TO_LIST_WIDTH -30);
  gtk_widget_set_usize (front->clist_to,  TO_LIST_WIDTH, TO_LIST_HEIGHT);
  gtk_signal_connect (GTK_OBJECT (front->clist_to), "select_row",
		      (GtkSignalFunc) to_clicked, NULL);
  gtk_clist_column_titles_passive (GTK_CLIST (front->clist_to));

  scrwin_to = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrwin_to), front->clist_to);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_to),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrwin_to), 5);
  gtk_box_pack_start (GTK_BOX (hbox2), scrwin_to, TRUE, TRUE, 0);

  /* Now do OK, Done and Cancel buttons */
  ok_button = gtk_button_new_with_label (_("OK"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      ok_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(ok_button), "clicked", 
		      GTK_SIGNAL_FUNC (add_ok), NULL);
  gtk_widget_set_sensitive (ok_button, FALSE);
  
  cancel_button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      cancel_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(cancel_button),"clicked", 
		      GTK_SIGNAL_FUNC (report_add_cancel), NULL);
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) report_add_cancel, NULL);
  /* now populate the clist of column names */
  newrow [1] = NULL;
  for (colx = 0; colx <= front->last_field; colx++) {
    fieldx = front->col_to_field [colx];
    newrow [0] = front->fields [fieldx].name;
    gtk_clist_append (GTK_CLIST (front->clist_from), newrow);
  }
  gtk_widget_show_all (dialog1_win);  
  gtk_widget_grab_focus (GTK_WIDGET (front->entry1));

} /* End of Report_add */  
 

/* |--------------------------------|
   |   report_col_add               |
   |--------------------------------|
  Make a clist of fields names that can be added to report */
void report_col_add (GtkWidget *w, gpointer *
data) {
  gint colx;
  gint fieldx;  
  
  GtkWidget *top_vbox;
  
  GtkWidget *hbox2; /* clists and header */
  GtkWidget *scrwin_from;
  
  GtkWidget *hbox6; /* same as action area */
  GtkWidget *done_button;
  
  gchar* from_titles [1] = {"Double-click to add to report"};
  make_basic_dialog1 ();

  gtk_window_set_title (GTK_WINDOW(dialog1_win), 
			_("Add columns to report"));
  top_vbox = GTK_DIALOG(dialog1_win) -> vbox;
  gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
  hbox6 = GTK_DIALOG (dialog1_win) -> action_area;

  /* now the area containing clist of fields that can be added */
  /*  The horizontal box containing the from and to listboxes  */
  hbox2 = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

  /* The from (column names)  clist */
  front->clist_from = gtk_clist_new_with_titles (1, from_titles);     

  gtk_clist_set_column_width (GTK_CLIST (front->clist_from), 0, 
			       FROM_LIST_WIDTH - 30);  
  gtk_widget_set_usize (front->clist_from, FROM_LIST_WIDTH, 
			FROM_LIST_HEIGHT);
  gtk_signal_connect (GTK_OBJECT (front->clist_from), "select_row",
		      (GtkSignalFunc) col_from_clicked, NULL);
  gtk_clist_column_titles_passive (GTK_CLIST (front->clist_from));

  scrwin_from = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrwin_from), front->clist_from);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_from),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrwin_from), 5);
  gtk_box_pack_start (GTK_BOX (hbox2), scrwin_from, TRUE, TRUE, 0);

  /* Now do OK, Done and Cancel buttons */
  done_button = gtk_button_new_with_label (_("Done"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      done_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(done_button), "clicked", 
		      GTK_SIGNAL_FUNC (cancel_level1), NULL);
  
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) cancel_level1, NULL);
  /* now populate the clist of column names */
  newrow [1] = NULL;
  for (colx = 0; colx <= front->last_field; colx++) {
    fieldx = front->col_to_field [colx];
    newrow [0] = front->fields [fieldx].name;
    gtk_clist_append (GTK_CLIST (front->clist_from), newrow);
  }
  gtk_widget_show_all (dialog1_win);  
} /* End of Report_col_add */ 



/* |--------------------------------|
   |   report_col_left              |
   |--------------------------------|
*/
void report_col_left (GtkWidget *w, gpointer *data) {
  swap_columns (front->report_sel_col, front->report_sel_col -1);
}


/* |--------------------------------|
   |   report_col_right             |
   |--------------------------------|
*/
void report_col_right (GtkWidget *w, gpointer *data) {
  swap_columns (front->report_sel_col, front->report_sel_col + 1);
}

/* |--------------------------------|
   |      report_go_to              |
   |--------------------------------|
   Move the selection to the given point in report sheet */
void report_go_to (gint row0, gint rowi, gint col0, gint coli) {
  GtkSheetRange range;
  range.col0 = col0;
  range.coli = coli;
  range.row0 = row0;
  range.rowi = rowi;
  gtk_sheet_select_range( GTK_SHEET (front->report_sheet), &range);
}

 
/* |--------------------------------|
   |      report_col_del            |
   |--------------------------------|
*/
void report_col_del (GtkWidget *w, gpointer *data) {
  gint colx;
  gint this_col;
  
  this_col = front->report_sel_col;
  /* someday we should have code to ask if the user really wants this */
  gtk_sheet_delete_columns (GTK_SHEET (front->report_sheet), 
			    this_col, 1);

  /* remove entry from report->column by moving down all higher */
  for (colx = this_col; 
       colx < front->report_ptr->last_column; colx++)
    front->report_ptr->column [colx] = front->report_ptr->column [colx + 1];
  front->report_ptr->last_column--;
  if (this_col > front->report_ptr->last_column)
    this_col = front->report_ptr->last_column;
  report_go_to (0,0, this_col, this_col);
} /* end of report_col_del */


/* _________________________________
   |                                |
   |         report_select          |
   |________________________________| */
void report_select (GtkWidget *w, gpointer *data){
  gint reportx;
   GtkWidget *top_vbox;
  
  GtkWidget *hbox2; /* clists and header */
  GtkWidget *scrwin_from;
  
  GtkWidget *hbox6; /* same as action area */
  GtkWidget *cancel_button;
  gchar* from_titles [1] = {"Double-click to select a report"};

  if (check_if_changed ())
    return;
  make_basic_dialog1 ();

  gtk_window_set_title (GTK_WINDOW(dialog1_win), 
			_("Select a report"));
  top_vbox = GTK_DIALOG(dialog1_win) -> vbox;
  gtk_box_set_spacing (GTK_BOX (top_vbox), 10);
  hbox6 = GTK_DIALOG (dialog1_win) -> action_area;

  /* now the area containing clist of fields that can be added */
  /*  The horizontal box containing the from and to listboxes  */
  hbox2 = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX (top_vbox), hbox2, TRUE, TRUE, 0);

  /* The report names clist */
  front->clist_from = gtk_clist_new_with_titles (1, from_titles);     

  gtk_clist_set_column_width (GTK_CLIST (front->clist_from), 0, 
			       FROM_LIST_WIDTH - 30);  
  gtk_widget_set_usize (front->clist_from, FROM_LIST_WIDTH, 
			FROM_LIST_HEIGHT);
  gtk_signal_connect (GTK_OBJECT (front->clist_from), "select_row",
		      (GtkSignalFunc) select_ok, NULL);
  gtk_clist_column_titles_passive (GTK_CLIST (front->clist_from));

  scrwin_from = gtk_scrolled_window_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER (scrwin_from), front->clist_from);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrwin_from),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_container_set_border_width (GTK_CONTAINER (scrwin_from), 5);
  gtk_box_pack_start (GTK_BOX (hbox2), scrwin_from, TRUE, TRUE, 0);

  cancel_button = gtk_button_new_with_label (_("Cancel"));
  gtk_box_pack_start (GTK_BOX (hbox6),
		      cancel_button, TRUE, TRUE, 0);
  gtk_signal_connect (GTK_OBJECT(cancel_button),"clicked", 
		      GTK_SIGNAL_FUNC (cancel_level1), NULL);
  gtk_signal_connect (GTK_OBJECT  (dialog1_win),
		      "delete_event",
		      (GtkSignalFunc) cancel_level1, NULL);

  /* now populate the clist of report names */
  newrow [1] = NULL;
  for (reportx = 0; reportx < front->report_ct; reportx++) {
    newrow [0] = front->reports [reportx].name;
    gtk_clist_append (GTK_CLIST (front->clist_from), newrow);
  }
  gtk_widget_show_all (dialog1_win);  
} /* End of Report_select */ 


/* |--------------------------------|
   |     report_set_col_field       |
   |--------------------------------|
   Sets current column and current field.  Dim thinks this is great */
void report_set_col_field (gint col) {
  front->report_sel_col = col;
  front->report_sel_field = front->report_ptr->column [col].field;
}


/* |--------------------------------|
   |     report_totals_group        |
   |--------------------------------|
*/
void report_totals_group (GtkWidget *w, gpointer *data) {
  front->report_ptr->column [front->report_sel_col].group =
    !front->report_ptr->column [front->report_sel_col].group;
    display_column (front->report_sel_col); 
}

/* |--------------------------------|
   |      report_totals_totals      |
   |--------------------------------|
*/
void report_totals_total (GtkWidget *w, gpointer *data) {  
  front->report_ptr->column [front->report_sel_col].total =
    !front->report_ptr->column [front->report_sel_col].total;
  display_column (front->report_sel_col);
}

/* |--------------------------------|
   |      return_to_list            |
   |--------------------------------|
   Return to list mode and its window */
void return_to_list () {
  get_window_size_loc (front->report_win); /* save size and location */
  clean_destroy (&front->report_win);
  front->report_ptr = NULL;
  front->report_no = 0;
  set_window_size_loc (front->list_win);
  gtk_widget_show (GTK_WIDGET (front->list_win));
  front->display_mode = 'L';
  dim_all_menus ();
}


/* _________________________________
   |                                |
   |      select_ok                 |
   |________________________________| 
  User clicked OK to select a specific report format, go print on screen */
static  void select_ok (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data) {
  if (event->type != GDK_2BUTTON_PRESS)
    return;
  front->report_no = row;
  front->report_ptr = &front->reports [row];
  destroy_dialog ();
  if (front->report_ptr->sort >= 0) /* -1 means none*/
    sort_do_it (front->report_ptr->sort);
  filter_do_apply (front->report_ptr->filter);  
  get_window_size_loc (front->list_win);
  print_on_screen ();
  gtk_widget_hide (GTK_WIDGET (front->list_win));
}


/* |--------------------------------|
   |   select_report_range          |
   |--------------------------------| */
void select_report_range (GtkSheet *sheet, GtkSheetRange *range) {
  GtkSheetRange local_range;
  local_range = *range;
  if (local_range.col0 == local_range.coli) {
    report_set_col_field (local_range.col0);
    dim_all_menus ();
  }
  else report_set_col_field (-1); /* no selection */
}



/* ________________________________
  |                                |
  |          sort_filter_menu      |
  |________________________________|
  Build the sort menu button for the report window */
GtkWidget* sort_filter_menu () {
  GtkWidget *hbox3;
  GtkWidget *menu,
            *menu_item,
            *name_label;
  gint sortx;
  gint filterx; 
  GtkWidget *sort_button;
  GtkWidget *filter_button;
  menu = gtk_menu_new ();
    menu_item = gtk_menu_item_new_with_label ("-Do not sort-");
     gtk_widget_show (menu_item);
     gtk_signal_connect (GTK_OBJECT (menu_item),
			"activate",
			GTK_SIGNAL_FUNC (sort_change), (gpointer) 0);
    gtk_menu_append (GTK_MENU(menu), menu_item);
    /* add the individual sorts to the menu */
    for (sortx = 0; sortx < front->sort_ct; sortx++) {
       menu_item = 
	 gtk_menu_item_new_with_label (front->sorts [sortx].name);
       gtk_widget_show (menu_item);
       gtk_signal_connect (GTK_OBJECT (menu_item),
		      "activate",
		      GTK_SIGNAL_FUNC (sort_change), 
			  (gpointer) (sortx + 1));
      gtk_menu_append (GTK_MENU(menu), menu_item);  
    }
  
   sort_button = gtk_option_menu_new ();
   gtk_option_menu_set_menu (GTK_OPTION_MENU (sort_button), menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (sort_button), 
				front->report_ptr->sort + 1);
   hbox3 = gtk_hbox_new (FALSE, 5);
 
  /* Pack Widgets into boxes */
  name_label = gtk_label_new (_("Sort rule: "));
  gtk_box_pack_start (GTK_BOX (hbox3), name_label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (hbox3), sort_button, FALSE, FALSE, 0);

  menu = gtk_menu_new ();
    menu_item = gtk_menu_item_new_with_label ("-Print all rows-");
    gtk_widget_show (menu_item);
    gtk_signal_connect (GTK_OBJECT (menu_item),
			"activate",
			GTK_SIGNAL_FUNC (filter_change), (gpointer) 0);
    gtk_menu_append (GTK_MENU(menu), menu_item);

    /* add the individual filters to the menu */
    for (filterx = 0; filterx < front->filter_ct; filterx++) {
       menu_item = gtk_menu_item_new_with_label (front->filters 
						 [filterx].name);  
       gtk_widget_show (menu_item);
       gtk_signal_connect (GTK_OBJECT (menu_item),
			   "activate",
			   GTK_SIGNAL_FUNC (filter_change), 
			   (gpointer) (filterx+1));
      gtk_menu_append (GTK_MENU(menu), menu_item);  
    }
  
   filter_button = gtk_option_menu_new ();
   gtk_option_menu_set_menu (GTK_OPTION_MENU (filter_button), menu);
   gtk_option_menu_set_history (GTK_OPTION_MENU (filter_button), 
				front->report_ptr->filter + 1);
  /* Pack Widgets into boxes */
  name_label = gtk_label_new (_("Filter: "));
  gtk_box_pack_end (GTK_BOX (hbox3), filter_button, FALSE, FALSE, 0); 
   gtk_box_pack_end (GTK_BOX (hbox3), name_label, FALSE, FALSE, 0);
   return (hbox3);
} /* end of sort_filter_menu () */

/* _________________________________
   |                                |
   |      sort_change               |
   |________________________________|
*/
void sort_change (GtkObject *object, gpointer *entry){
  front->report_ptr->sort = (gint) entry - 1;
   if (!entry)
     return; /* -1  means none */
   sort_do_it (front->report_ptr->sort);
   pick6rows ();
   display_all_columns ();
}

/* _________________________________
   |                                |
   |       swap_columns             |
   |________________________________|
*/
void swap_columns (gint from, gint to) {
  report_column temp_column;
  
  temp_column = front->report_ptr->column [from];
  front->report_ptr->column [from] = front->report_ptr->column [to];
  front->report_ptr->column [to] = temp_column;

  display_column (from);
  display_column (to);
  report_set_col_field (to);
  gtk_sheet_select_column (GTK_SHEET (front->report_sheet), to);
  dim_all_menus ();
}

/* _________________________________
   |                                |
   |         to_clicked             |
   |________________________________| 
   To clist clicked, remove the clicked upon entry */
static  void to_clicked (GtkWidget *widget, gint row, 
		     gint col, GdkEventButton *event, gpointer data) {
  gint colx;
  if (event->type != GDK_2BUTTON_PRESS)
    return;
  gtk_clist_remove (GTK_CLIST (front->clist_to), row);
  
  /* remove from report definition */
  for (colx = row; colx < front->report_ptr->last_column; colx++)
    front->report_ptr->column [colx] = front->report_ptr->column [colx+1];
  front->report_ptr->last_column--;
  if (front->report_ptr->last_column < 0)
    gtk_widget_set_sensitive (ok_button, FALSE);
  
}














