/***************************************************************************
                          temporaryfile.cpp  -  description
                             -------------------
    begin                : Thu Jun 6 2002
    copyright            : (C) 2002 by Michael Herder
    email                : crapsite@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/* This file is based on ktempfile.cpp, which is part of the KDE libraries.
 * The licence has been changed from LGPL to GPL according to the GPL/LGPL
 *
 *  Copyright (c) 1999 Waldo Bastian <bastian@kde.org>
 *
 **/
#include "temporaryfile.h"
#include "quiteinsane/resource.h"

#include <sys/types.h>
#include <sys/stat.h>

#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>

#ifndef _PATH_TMP
#define _PATH_TMP "/tmp"
#endif

#include <qdatetime.h>
#include <qfile.h>
#include <qdatastream.h>
#include <qtextstream.h>

TemporaryFile::TemporaryFile(QString filePrefix, int mode)
{
  mAutoDelete = false;
  mFd = -1;
  mpStream = 0;
  mpFile = 0;
  mpTextStream = 0;
  mpDataStream = 0;
  mError = 0;
  mOpen = false;
  if (filePrefix.isEmpty())
  {
     filePrefix = "/tmp/";
  }
  (void) create(filePrefix, mode);
}

bool TemporaryFile::create(const QString &filePrefix,int mode)
{
  QString nme = QString(QFile::encodeName(filePrefix)) + "XXXXXX";

  char* ts;
  ts = new char[strlen(nme.latin1())+1];
  strcpy(ts,nme.latin1());

  if((mFd = mkstemp(ts)) < 0)
  {
    qWarning("TemporaryFile: Error trying to create %s: %s", nme.latin1(), strerror(errno));
    mError = errno;
    mTmpName = QString::null;
    delete [] ts;
    return false;
  }
  nme = ts;
  delete [] ts;

  // got a file descriptor. nme contains the name
  mTmpName = QFile::decodeName(nme.latin1());
  mode_t tmp = 0;
  mode_t umsk = umask(tmp);
  umask(umsk);
  chmod(nme, mode&(~umsk));

  // Success!
  mOpen = true;

  // Set uid/gid (neccesary for SUID programs)
  chown(nme, getuid(), getgid());
  return true;
}

TemporaryFile::~TemporaryFile()
{
  close();
  if (mAutoDelete)
    unlink();
}

int TemporaryFile::status() const
{
  return mError;
}

QString TemporaryFile::name() const
{
  return mTmpName;
}

int TemporaryFile::handle() const
{
  return mFd;
}

FILE* TemporaryFile::fstream()
{
  if (mpStream) return mpStream;
  if (mFd < 0) return 0;

  // Create a stream
  mpStream = fdopen(mFd, "r+");
  if (!mpStream)
  {
    qWarning("TemporaryFile: Error trying to open %s: %s", mTmpName.latin1(), strerror(errno));
    mError = errno;
  }
  return mpStream;
}

QFile* TemporaryFile::file()
{
  if (mpFile)
    return mpFile;
  if (!fstream())
    return 0;

  mpFile = new QFile();
  mpFile->setName( name() );
  mpFile->open(IO_ReadWrite, mpStream);
  return mpFile;
}

QTextStream* TemporaryFile::textStream()
{
  if (mpTextStream) return mpTextStream;
  if ( !file() ) return 0; // Initialize mpFile

  mpTextStream = new QTextStream( mpFile );
  return mpTextStream;
}

QDataStream* TemporaryFile::dataStream()
{
  if (mpDataStream) return mpDataStream;
  if ( !file() ) return 0;  // Initialize mpFile

  mpDataStream = new QDataStream( mpFile );
  return mpDataStream;
}

void TemporaryFile::unlink()
{
  if (!mTmpName.isEmpty())
     QFile::remove( mTmpName );
  mTmpName = QString::null;
}

bool TemporaryFile::close()
{
  int result = 0;
  delete mpTextStream; mpTextStream = 0;
  delete mpDataStream; mpDataStream = 0;
  delete mpFile; mpFile = 0;

  if (mpStream)
  {
    result = ferror(mpStream);
    if (result)
      mError = ENOSPC; // Assume disk full.

    result = fclose(mpStream);
    mpStream = 0;
    mFd = -1;
    if (result != 0)
    {
       qWarning("TemporaryFile: Error trying to closing %s: %s", mTmpName.latin1(), strerror(errno));
       mError = errno;
    }
  }

  if (mFd >= 0)
  {
    result = ::close(mFd);
    mFd = -1;
    if (result != 0)
    {
      qWarning("TemporaryFile: Error trying to closing %s: %s", mTmpName.latin1(), strerror(errno));
      mError = errno;
    }
  }

  mOpen = false;
  return (mError == 0);
}

