// 
// $Id: qaregexpwidget.cpp,v 1.5 2000/06/12 10:53:28 amos-cvs Exp $
//
// Implementation of QRegExpWidget class
//
// Jan Borsodi <jb@ez.no>
// Created on: <21-Jul-1999 21:22:46 amos>
//
// Copyright (C) Jan Borsodi.  All rights reserved.

#include "qaregexpwidget.hpp"
#include <qaregexp.hpp>
#include <qaregexperror.hpp>

#include "qasignalredirect.hpp"

#include "qaregexplanguage.hpp"
#include "qaplainlanguage.hpp"
#include "qalisplanguage.hpp"
#include "qaperllanguage.hpp"
#include "qaphplanguage.hpp"
#include "qaqtlanguage.hpp"

#include <qframe.h>
#include <qlayout.h>
#include <qwidget.h>
#include <qstring.h>
#include <qlineedit.h>
#include <qmultilinedit.h>
#include <qtextview.h>
#include <qsplitter.h>
#include <qlabel.h>
#include <qspinbox.h>
#include <qcolor.h>
#include <qstylesheet.h>
#include <qmainwindow.h>
#include <qpopupmenu.h>
#include <qmenubar.h>
#include <qstatusbar.h>
#include <qvgroupbox.h>
#include <qlistbox.h>
#include <qpoint.h>
#include <qrect.h>
#include <qfile.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qcolordialog.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qsize.h>
#include <qevent.h>
#include <qaccel.h>
#include <qglobal.h>
#include <qstringlist.h>
#include <qvaluelist.h>

/*!
  \class QRegExpWidget qregexpwidget.hpp
  \brief A widget with the possibility to see regexps in action.

  Has a text field for regexp, a multiline edit for the text,
  and a output text view. When a regexp matches it will display
  the match in a different color, sub expressions can also be matched.
*/

int FunctionKeys[] = { Qt::Key_F1, Qt::Key_F2, Qt::Key_F3, Qt::Key_F4, Qt::Key_F5, Qt::Key_F6,
                       Qt::Key_F7, Qt::Key_F8, Qt::Key_F9, Qt::Key_F10, Qt::Key_F11, Qt::Key_F12 };

/*!
  Default constructor
*/

QaRegExpWidget::QaRegExpWidget( QWidget *parent, const char *name )
    : QMainWindow( parent, name ), Language( new QaPlainLanguage() ), Reg( "", true, true ),
    Automatic( false ), Loading( false )
{
    setCaption( "RegExplorer - " + version() );
    SubValue = 0;
    MatchCount = 0;
    SubMatch = 0;
    SingleLine = false;
    Update = true;
    Back = QColor( 150, 152, 255 );
    Select = QColor( 255, 255, 0 );
    Matching = QaRegExpWidget::Single;
    Central = new QWidget( this );
    QAccel *acc = new QAccel( this );
    {
        QaSignalRedirect *director;

        for ( unsigned int i = 0; i < 12; ++i )
        {
            director = new QaSignalRedirect( i, this );
            connect( director, SIGNAL( redirect( int ) ),
                     this, SLOT( doFunctionKey( int ) ) );
            acc->connectItem( acc->insertItem( FunctionKeys[i] ), director, SLOT( activate() ) );
        }

//          acc->connectItem( acc->insertItem( Key_F1 ), this, SLOT( doF1() ) );


        for ( unsigned int i = 0; i < 12; ++i )
        {
            director = new QaSignalRedirect( i, this );
            connect( director, SIGNAL( redirect( int ) ),
                     this, SLOT( doShiftedFunctionKey( int ) ) );
            acc->connectItem( acc->insertItem( SHIFT + FunctionKeys[i] ), director, SLOT( activate() ) );
        }

//          acc->connectItem( acc->insertItem( SHIFT + Key_F1 ), this, SLOT( doShiftF1() ) );
    }

    QVBoxLayout *clay = new QVBoxLayout( Central, 1 );
    {
        MainSplit = new QSplitter( Horizontal, Central );
        clay->addWidget( MainSplit, 1 );
        MainSplit->setOpaqueResize( true );

        QWidget *LeftSplit = new QWidget( MainSplit, "left split widget" );
        Lay = new QVBoxLayout( LeftSplit, 0, 3 );
        {
            QHBoxLayout *hlay = new QHBoxLayout();
            Lay->addLayout( hlay, 0 );
            {
                QLabel *patlab = new QLabel( tr( "Pattern:" ), LeftSplit );
                hlay->addWidget( patlab, 0 );
                RegEd = new QLineEdit( LeftSplit );
                hlay->addWidget( RegEd, 0 );
                connect( RegEd, SIGNAL( textChanged( const QString & ) ),
                         this, SLOT( doTextChange( const QString & ) ) );
                connect( RegEd, SIGNAL( returnPressed( ) ),
                         this, SLOT( theTextHasChanged( ) ) );
            }

            RightSplit = new QSplitter( Horizontal, LeftSplit );
            RightSplit->setOpaqueResize( true );
            Lay->addWidget( RightSplit, 1 );
            {
                Edit = new QMultiLineEdit( RightSplit );
//              Edit->setText( "void test::setText( const QString &s )\n"
//                             "{\n"
//                             "}\n" );
                connect( Edit, SIGNAL( textChanged() ),
                         this, SLOT( multiTextChanged() ) );

                TextVu = new QTextBrowser( RightSplit );
                TextVu->setPaper( Back );
//              TextVu->setText( "void test::setText( const QString &s )<br>"
//                               "{<br>"
//                               "}<br>" );
                TextVu->setFocusPolicy( NoFocus );
            }
            QValueList<int> ilst;
            ilst.append( 1 );
            ilst.append( 1 );
            RightSplit->setSizes( ilst );

            SubSplit = new QSplitter( Vertical, MainSplit );
            {
                SubWidget = new QWidget( SubSplit );
                QVBoxLayout *vlay1 = new QVBoxLayout( SubWidget, 1 );
                {
                    SubLabel = new QLabel( "Sub expression", SubWidget );
                    vlay1->addWidget( SubLabel );

                    Sub = new QListBox( SubWidget );
                    vlay1->addWidget( Sub );
                    connect( Sub, SIGNAL( highlighted( int ) ),
                             this, SLOT( valueChange( int ) ) );
                }

                MatchWidget = new QWidget( SubSplit );
                QVBoxLayout *vlay2 = new QVBoxLayout( MatchWidget, 1 );
                {
                    MatchLabel = new QLabel( MatchWidget );
                    vlay2->addWidget( MatchLabel );

                    Matches = new QListBox( MatchWidget );
                    vlay2->addWidget( Matches );
                    connect( Matches, SIGNAL( highlighted( int ) ),
                             this, SLOT( valueChangeMultiple( int ) ) );
                }
                MatchWidget->hide();
            }
            SubSplit->setOpaqueResize( true );

            ErrLabel = new QLabel( " ", statusBar() );
            statusBar()->addWidget( ErrLabel, 1 );
            SpeedLabel = new QLabel( "", statusBar() );
            statusBar()->addWidget( SpeedLabel, 0, true );
        }
        QValueList<int> ilst;
        ilst.append( 4 );
        ilst.append( 1 );
        MainSplit->setSizes( ilst );

        PatDisp = new QMultiLineEdit( Central );
        PatDisp->setText( "sdvdsf" );
        PatDisp->setReadOnly( true );
        clay->addWidget( PatDisp,  0 );
    }
    setCentralWidget( Central );

    FileMenu = new QPopupMenu( 0, "file menu" );
    {
        menuBar()->insertItem( tr( "&File" ), FileMenu );

        FileMenu->setItemEnabled( FileMenu->insertItem( tr( "&Load" ), this, SLOT( doLoad() ), CTRL + Key_L ), false );
        FileLoadTextID = FileMenu->insertItem( tr( "Load &Text File" ), this, SLOT( doLoadText() ), CTRL + SHIFT + Key_L );
        FileMenu->setItemEnabled( FileMenu->insertItem( tr( "&Save" ), this, SLOT( doSave() ), CTRL + Key_S ), false );

        FileMenu->insertSeparator();

	HistoryMenu = new QPopupMenu( 0, "Recent files" );
        FileMenu->insertItem( tr( "&Recent Files" ), HistoryMenu );
        connect( HistoryMenu, SIGNAL( activated( int ) ),
                        this, SLOT( doLoadHistory( int ) ) );

        FileMenu->insertItem( tr( "&Exit" ), this, SLOT( close() ), CTRL + Key_Q );
    }

    EditMenu = new QPopupMenu( 0, "modes menu" );
    {
        menuBar()->insertItem( tr( "&Edit" ), EditMenu );

        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&New Group" ) ), false );
        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&Delete Group" ) ), false );
        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&Add RegExp" ) ), false );
        EditMenu->setItemEnabled( EditMenu->insertItem( tr( "&Remove RegExp" ) ), false );
        EditMenu->insertSeparator();
        EditBackColorID = EditMenu->insertItem( tr( "&Background Color" ), this, SLOT( doBackColor() ) );
        EditSelectColorID = EditMenu->insertItem( tr( "&Selection Color" ), this, SLOT( doSelectColor() ) );
    }

    MatchMenu = new QPopupMenu( 0, "match menu" );
    {
        menuBar()->insertItem( tr( "&Match Type" ), MatchMenu );

        MatchNormalID = MatchMenu->insertItem( tr( "&Normal" ), this, SLOT( doNormal() ), CTRL + Key_N );
        MatchMultipleID = MatchMenu->insertItem( tr( "&Multiple" ), this, SLOT( doMultiple() ), CTRL + Key_M );
        MatchSplitID = MatchMenu->insertItem( tr( "&Split" ), this, SLOT( doSplit() ), CTRL+Key_T );
        MatchMenu->insertSeparator();
        MatchCaseID = MatchMenu->insertItem( tr( "&Case Sensitive" ), this, SLOT( doCase() ), CTRL + Key_I );
        SingleLineID = MatchMenu->insertItem( tr( "Single &Line" ), this, SLOT( doSingleLine() ), CTRL + Key_G );
        NewLineID = MatchMenu->insertItem( tr( "&Differentiate NewLine" ), this, SLOT( doNewLine() ), CTRL + Key_D );
        AutomaticID = MatchMenu->insertItem( tr( "&Automatic Matching" ), this, SLOT( doAutomatic() ), CTRL + Key_A );

        MatchMenu->setCheckable( true );
        MatchMenu->setItemChecked( MatchNormalID, true );
        MatchMenu->setItemChecked( MatchCaseID, true );
        MatchMenu->setItemEnabled( SingleLineID, false );
        MatchMenu->setItemChecked( NewLineID, true );
        MatchMenu->setItemChecked( AutomaticID, true );
    }

    ModesMenu = new QPopupMenu( 0, "modes menu" );
    {
        menuBar()->insertItem( tr( "Em&ulation Mode" ), ModesMenu );

        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&None" ) ), true );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&Lisp" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&Perl" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "P&HP" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "&Qt" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "P&ython" ) ), false );
        ModesMenu->setItemEnabled( ModesMenu->insertItem( tr( "T&cl" ) ), false );

        ModesMenu->setCheckable( true );
        ModesMenu->setItemChecked( ModesMenu->idAt( 0 ), true );
    }

    OutputMenu = new QPopupMenu( 0, "output menu" );
    {
        menuBar()->insertItem( tr( "&Output Type" ), OutputMenu );

        PlainTextID = OutputMenu->insertItem( tr( "Plain &Text" ), this, SLOT( doPlain() ), CTRL + Key_1 );
        OutputMenu->setItemEnabled( PlainTextID, true );
        LispID = OutputMenu->insertItem( tr( "&Lisp" ), this, SLOT( doLisp() ), CTRL + Key_2 );
        OutputMenu->setItemEnabled( LispID, true );
        PerlID = OutputMenu->insertItem( tr( "&Perl" ), this, SLOT( doPerl() ), CTRL + Key_3 );
        OutputMenu->setItemEnabled( PerlID, true );
        PHPID = OutputMenu->insertItem( tr( "P&HP" ), this, SLOT( doPHP() ), CTRL + Key_4 );
        OutputMenu->setItemEnabled( PHPID, true );
        QtID = OutputMenu->insertItem( tr( "&Qt" ), this, SLOT( doQt() ), CTRL + Key_5 );
        OutputMenu->setItemEnabled( QtID, true );
        OutputMenu->setItemEnabled( OutputMenu->insertItem( tr( "P&ython" ) ), false );
        OutputMenu->setItemEnabled( OutputMenu->insertItem( tr( "T&cl" ) ), false );

        OutputMenu->insertSeparator();

        CompleteID = OutputMenu->insertItem( tr( "Code output" ), this, SLOT( doCompleteOutput() ), CTRL + Key_C );

        OutputMenu->setCheckable( true );
        OutputMenu->setItemChecked( OutputMenu->idAt( 0 ), true );
    }

    menuBar()->insertSeparator();

    HelpMenu = new QPopupMenu( 0, "help menu" );
    {
        menuBar()->insertItem( tr( "&Help" ), HelpMenu );

        HelpMenu->insertItem( tr( "&About RegExplorer" ), this, SLOT( doAbout() ), 0 );
        HelpMenu->insertItem( tr( "&About Qt" ), this, SLOT( doAboutQt() ), 0 );
    }

//      QString s;
//      for ( int i = 0; i < Edit->numLines(); i++ )
//      {
//          s += Edit->textLine( i );
//          if ( i < Edit->numLines() )
//              s+= "\n";
//      }
//      EditString = s;
//      textChange( "" );
//      multiTextChanged();
    loadConfig();
    ConversionText = RegEd->text();
//      textChange( RegEd->text() );
}

/*!
  Destroys the object
*/

QaRegExpWidget::~QaRegExpWidget()
{
//    delete Lay;
    saveConfig();
}

void QaRegExpWidget::doNormal()
{
    if ( MatchMenu->isItemChecked( MatchNormalID ) )
        return;
    MatchMenu->setItemChecked( MatchNormalID, true );
    MatchMenu->setItemChecked( MatchMultipleID, false );
    MatchMenu->setItemChecked( MatchSplitID, false );
    MatchMenu->setItemEnabled( SingleLineID, false );
    Sub->setEnabled( true );
    Matching = QaRegExpWidget::Single;

    SubLabel->setText( "Sub expression" );

    Matches->setEnabled( false );
    updateMatch();

    // Fix for listbox repaint bug
    MatchWidget->hide();
//    Matches->show();
    updateConversion();
}

void QaRegExpWidget::doMultiple()
{
    if ( MatchMenu->isItemChecked( MatchMultipleID ) )
        return;
    MatchMenu->setItemChecked( MatchNormalID, false );
    MatchMenu->setItemChecked( MatchMultipleID, true );
    MatchMenu->setItemChecked( MatchSplitID, false );
    MatchMenu->setItemEnabled( SingleLineID, false );
    Sub->setEnabled( true );
    Matching = QaRegExpWidget::Multiple;

    SubLabel->setText( "Sub expression" );
    MatchLabel->setText( "Match number" );

    Matches->setEnabled( true );
    updateMatch();

    // Fix for listbox repaint bug
//    Matches->hide();
    MatchWidget->show();
    updateConversion();
}

void QaRegExpWidget::doSplit()
{
    if ( MatchMenu->isItemChecked( MatchSplitID ) )
        return;
    MatchMenu->setItemChecked( MatchNormalID, false );
    MatchMenu->setItemChecked( MatchMultipleID, false );
    MatchMenu->setItemChecked( MatchSplitID, true );
    MatchMenu->setItemEnabled( SingleLineID, true );
    Sub->setEnabled( MatchMenu->isItemChecked( SingleLineID ) );
    Matching = QaRegExpWidget::Split;

    SubLabel->setText( "Line number" );
    MatchLabel->setText( "Match number" );

    Matches->setEnabled( true );
    updateMatch();

    // Fix for listbox repaint bug
//    Matches->hide();
    MatchWidget->show();
    updateConversion();
}

void QaRegExpWidget::doLoad()
{
}

void QaRegExpWidget::doLoadText()
{
    QString str = QFileDialog::getOpenFileName( QString::null, QString::null, this );
    if ( !str.isEmpty() )
    {
        if ( loadFile( str ) )
            addToHistory( str );
    }
}

bool QaRegExpWidget::loadFile( const QString &name )
{
    // TODO: Use QDataSink and QDataSource for better loading of files, especially big ones.
    QFile f( name );
    if ( f.open( IO_ReadOnly ) )
    {
        char *tmp = new char[f.size()+1];
        f.readBlock( tmp, f.size() );
        tmp[f.size()] = 0;
        QString s( tmp );
        delete []tmp;
        Edit->setText( s );
        return true;
    }
    else
        QMessageBox::warning( this, "File error",
                              QString( "Error reading file \"%1\"" ).arg( name ),
                              0, 1 );
    return false;
}

void QaRegExpWidget::addToHistory( const QString &name, bool infront )
{
    QStringList tmp = OldFiles;
    if ( infront )
        tmp.prepend( name );
    else
        tmp.append( name );
    OldFiles.clear();

    unsigned int i = 0;
    for ( QStringList::Iterator it = tmp.begin(); it != tmp.end() && i < 5; ++it, ++i )
    {
        OldFiles.append( *it );
    }
    rebuildHistory();
}

void QaRegExpWidget::rebuildHistory()
{
    HistoryMenu->clear();
    for ( QStringList::Iterator it = OldFiles.begin(); it != OldFiles.end(); ++it )
    {
        HistoryMenu->insertItem( *it );
    }
}

void QaRegExpWidget::doLoadHistory( int id )
{
    QString name = HistoryMenu->text( id );
    OldFiles.remove( OldFiles.at( id ) );
    if ( loadFile( name ) )
        addToHistory( name );
}

void QaRegExpWidget::doSave()
{
}

void QaRegExpWidget::doCase()
{
    MatchMenu->setItemChecked( MatchCaseID, !MatchMenu->isItemChecked( MatchCaseID ) );
    Reg.setCaseSensitive( MatchMenu->isItemChecked( MatchCaseID ) );
    updateMatch();
}

void QaRegExpWidget::doSingleLine()
{
    MatchMenu->setItemChecked( SingleLineID, !MatchMenu->isItemChecked( SingleLineID ) );
    Sub->setEnabled( MatchMenu->isItemChecked( SingleLineID ) );
    updateMatch();
}

void QaRegExpWidget::doNewLine()
{
    MatchMenu->setItemChecked( NewLineID, !MatchMenu->isItemChecked( NewLineID ) );
    Reg.setHandleNewline( MatchMenu->isItemChecked( NewLineID ) );
    updateMatch();
}

void QaRegExpWidget::updateConversion()
{
    QaRegExpLanguage::MatchType type = QaRegExpLanguage::Normal;
    if ( MatchMenu->isItemChecked( MatchMultipleID ) )
        type = QaRegExpLanguage::Multiple;
    else if ( MatchMenu->isItemChecked( MatchSplitID ) )
        type = QaRegExpLanguage::Split;
    PatDisp->setText( Language->fromPlainText( ConversionText,
                                               OutputMenu->isItemChecked( CompleteID ),
                                               type ) );
//      if ( OutputMenu->isItemChecked( PlainTextID ))
//      {
//          QString s = ConversionText;
//          s = convertText( s );
//          PatDisp->setText( s );
//      }
//      else if ( OutputMenu->isItemChecked( LispID ) )
//      {
//          PatDisp->setText( toLisp( ConversionText ) );
//      }
//      else
//          PatDisp->setText( "" );
}

void QaRegExpWidget::doTextChange( const QString &t )
{
    if ( Automatic && !Loading )
    {
        qDebug( "doTextChange" );
//        if ( !t.isEmpty() )
            textChange( t );
    }
}

void QaRegExpWidget::setExpression( const QString &t )
{
    try
    {
        updateConversion();
        Reg.setExpression( t );
        Reg.setCaseSensitive( MatchMenu->isItemChecked( MatchCaseID ) );
//          if ( Matching == QaRegExpWidget::Multiple || Matching == QaRegExpWidget::Split )
//          {
//              if ( SubMatch >= MatchCount )
//                  SubMatch = MatchCount - 1;
//          }
    }
    catch ( QaRegExpError &e )
    {
        switch ( e.errorCode() )
        {
            case QaRegExpError::UnmatchedBracket:
            case QaRegExpError::InvalidRange:
            case QaRegExpError::UnmatchedParenthesis:
                ErrLabel->setText( tr( "Incomplete input" ) );
                PatDisp->setText( "" );
                break;
            default:
                ErrLabel->setText( e.error() );
                PatDisp->setText( "" );
        }
    }
}

/*!
  Text has changed
*/

void QaRegExpWidget::textChange( const QString &t )
{
    QString regstr = t;
    ConversionText = regstr;
    regstr = PlainLanguage.fromPlainText( regstr,
                                          false,
                                          QaRegExpLanguage::Normal );
    setExpression( regstr );
    if ( Matching == QaRegExpWidget::Split )
    {
        if ( SubValue > NumLines )
            SubValue = NumLines;
    }
    else
    {
        if ( SubValue > Reg.subCount() )
            SubValue = Reg.subCount();
    }
    if ( SubMatch < 0 )
        SubMatch = 0;
    if ( Update )
        updateMatch();
}

//  QString QaRegExpWidget::toLisp( const QString &s )
//  {
//  }

//  /*!
//    Converts backslashified expressions to their correct ascii values
//  */

//  QString QaRegExpWidget::convertText( const QString &s )
//  {
//  }

/*!
  Adds <br> where \n is located.
*/

QString QaRegExpWidget::toHtml( const QString &s )
{
    return QStyleSheet::convertFromPlainText( s );
}

/*!
  Called whenever the sub expression value changes.
*/

void QaRegExpWidget::valueChange( int val )
{
    if ( val < 0 )
        return;
    if ( SubValue == val )
        return;
    SubValue = val;
    if ( Matching == QaRegExpWidget::Multiple || Matching == QaRegExpWidget::Split )
        updateMatch();
    else
        updateOutput();
}

/*!
  Called whenever the sub expression value changes.
*/

void QaRegExpWidget::valueChangeMultiple( int val )
{
    if ( val < 0 )
        return;
    if ( SubMatch == val )
        return;
    SubMatch = val;
    if ( Automatic )
        updateMatch();
    else
        updateOutput();

    if( SubMatch > 0 ) {
      // normal range
      QString anchor;

      anchor.sprintf( "%d", SubMatch-1 );

      // jump to corrent match
      TextVu->scrollToAnchor( anchor );
    }
}

/*!
  Update the output
*/

void QaRegExpWidget::updateOutput()
{
//      if ( Matching == QaRegExpWidget::Split )
//      {
//      }
    if ( Loading )
        return;
//      qDebug( "updateOutput" );
    QString anchor;
    if ( Matching == QaRegExpWidget::Split )
    {
        int offs = 0;
        int olen = EditString.length();
        QString ahead, behind;
        if ( MatchMenu->isItemChecked( SingleLineID ) )
        {
            if ( SubValue == 0 )
            {
                offs = 0;
                olen = Edit->textLine( 0 ).length() + 1;
            }
            for ( int i = 0; i < SubValue; i++ )
            {
                QString str = Edit->textLine( i );
                offs += str.length() + 1;
                olen = Edit->textLine( i+1 ).length() + 1;
            }
            ahead = EditString.mid( 0, offs );
            behind = EditString.mid( offs + olen );
        }
        if ( SubMatch == 0 )
        {
            if ( MatchCount > 0 )
            {
//                QString st = Reg.matchString( SubMatch );
                QString str = EditString.mid( offs, olen );
                QString font = selectColor();
                QString fontback( "</b></font>" );
                QaRegExpRange range = Reg.matches()[0];
                QString str2;
                int index1 = range.start();
                int index2 = range.start() + range.length();

                QString front,back,mid;
                int index3,index4;
                front = str.mid( 0, index1 );
                front = toHtml( front );
                front = front.mid( 3, front.length() - 7 );
                str2 = front;
//                str += font + mid + fontback + back;

                for ( int i = 0; i < MatchCount; i++ )
                {
                    range = Reg.matches()[i];
                    index1 = range.start();
                    index2 = range.start() + range.length();

                    QaRegExpRange range2;
                    if ( i < MatchCount - 1 )
                    {
                        range2 = Reg.matches()[i+1];
                        index3 = range2.start();
                        index4 = range2.start() + range2.length();
                    }
                    else
                    {
                        index3 = str.length();
                        index4 = index3;
                    }

                    mid = str.mid( index1, range.length() );
                    back = str.mid( index2, index3 - index2 );
                    mid = toHtml( mid );
                    mid = mid.mid( 3, mid.length() - 7 );
                    back = toHtml( back );
                    back = back.mid( 3, back.length() - 7 );
                    anchor.sprintf( "<a name=\"%d\"></a>", i );
                    str2 += font + anchor + mid + fontback + back;
                }

                ahead = toHtml( ahead );
                ahead = ahead.mid( 3, ahead.length() - 7 );
                behind = toHtml( behind );
                behind = behind.mid( 3, behind.length() - 7 );

                str2 = ahead + str2 + behind;

                TextVu->setText( str2 );
//                qDebug( str2 );
                ErrLabel->setText( tr( "Match" ) );
            }
            else
            {
                QString str = QStyleSheet::convertFromPlainText( EditString );
                str = str.mid( 3, str.length() - 7 );
                TextVu->setText( str );
                ErrLabel->setText( tr( "No Match" ) );
            }
        }
        else
        {
            if ( Reg.matchBeginning( SubMatch - 1 ) != -1 )
            {
//                QString st = Reg.matchString( SubMatch-1 );
                QString str = EditString.mid( offs, olen );
                QString font = selectColor();
                QString fontback( "</b></font>" );
                QaRegExpRange range = Reg.matches()[SubMatch - 1];
                int index1 = range.start();
                int index2 = range.start() + range.length();

                QString front,back,mid;
                front = str.mid( 0, index1 );
                mid = str.mid( index1, range.length() );
                back = str.mid( index2 );

                front = toHtml( front );
                front = front.mid( 3, front.length() - 7 );
                mid = toHtml( mid );
                mid = mid.mid( 3, mid.length() - 7 );
                back = toHtml( back );
                back = back.mid( 3, back.length() - 7 );
                anchor.sprintf( "<a name=\"%d\"></a>", SubMatch-1 );
                str = front + font + anchor + mid + fontback + back;

                ahead = toHtml( ahead );
                ahead = ahead.mid( 3, ahead.length() - 7 );
                behind = toHtml( behind );
                behind = behind.mid( 3, behind.length() - 7 );

                str = ahead + str + behind;

                TextVu->setText( str );
                ErrLabel->setText( tr( "Match" ) );
            }
            else
            {
                QString str = QStyleSheet::convertFromPlainText( EditString );
                str = str.mid( 3, str.length() - 7 );
                TextVu->setText( str );
                ErrLabel->setText( tr( "No Match" ) );
            }
        }
    }
    else if ( Matching == QaRegExpWidget::Multiple)
    {
        if ( SubMatch == 0 )
        {
            if ( MatchCount > 0 )
            {
//                QString st = Reg.matchString( SubMatch );
                QString str = EditString;
                QString font = selectColor();
                QString fontback( "</b></font>" );
                QaRegExpRange range = Reg.matches()[0];
                QString str2;
                int index1 = range.start();
                int index2 = range.start() + range.length();

                QString front,back,mid;
                int index3,index4;
                front = str.mid( 0, index1 );
                front = toHtml( front );
                front = front.mid( 3, front.length() - 7 );
                str2 = front;
//                str += font + mid + fontback + back;

                for ( int i = 0; i < MatchCount; i++ )
                {
                    range = Reg.matches()[i];
                    index1 = range.start();
                    index2 = range.start() + range.length();

                    QaRegExpRange range2;
                    if ( i < MatchCount - 1 )
                    {
                        range2 = Reg.matches()[i+1];
                        index3 = range2.start();
                        index4 = range2.start() + range2.length();
                    }
                    else
                    {
                        index3 = str.length();
                        index4 = index3;
                    }

                    mid = str.mid( index1, range.length() );
                    back = str.mid( index2, index3 - index2 );
                    mid = toHtml( mid );
                    mid = mid.mid( 3, mid.length() - 7 );
                    back = toHtml( back );
                    back = back.mid( 3, back.length() - 7 );
                    anchor.sprintf( "<a name=\"%d\"></a>", i );
                    str2 += font + anchor + mid + fontback + back;
                }


                TextVu->setText( str2 );
//                qDebug( str2 );
                ErrLabel->setText( tr( "Match" ) );
            }
            else
            {
                QString str = QStyleSheet::convertFromPlainText( EditString );
                str = str.mid( 3, str.length() - 7 );
                TextVu->setText( str );
                ErrLabel->setText( tr( "No Match" ) );
            }
        }
        else
        {
            if ( Reg.matchBeginning( SubMatch - 1 ) != -1 )
            {
//                QString st = Reg.matchString( SubMatch-1 );
                QString str = EditString;
                QString font = selectColor();
                QString fontback( "</b></font>" );
                QaRegExpRange range = Reg.matches()[SubMatch - 1];
                int index1 = range.start();
                int index2 = range.start() + range.length();

                QString front,back,mid;
                front = str.mid( 0, index1 );
                mid = str.mid( index1, range.length() );
                back = str.mid( index2 );

                front = toHtml( front );
                front = front.mid( 3, front.length() - 7 );
                mid = toHtml( mid );
                mid = mid.mid( 3, mid.length() - 7 );
                back = toHtml( back );
                back = back.mid( 3, back.length() - 7 );
                anchor.sprintf( "<a name=\"%d\"></a>", SubMatch-1 );
                str = front + font + anchor + mid + fontback + back;
                TextVu->setText( str );
                ErrLabel->setText( tr( "Match" ) );
            }
            else
            {
                QString str = QStyleSheet::convertFromPlainText( EditString );
                str = str.mid( 3, str.length() - 7 );
                TextVu->setText( str );
                ErrLabel->setText( tr( "No Match" ) );
            }
        }
    }
    else
    {
        if ( Reg.matchBeginning( SubValue ) != -1 )
        {
            QString st = Reg.matchString( SubValue );
            QString str = EditString;
            QString font = selectColor();
            QString fontback( "</b></font>" );
            QaRegExpRange range = Reg.matches()[SubValue];
            int index1 = range.start();
            int index2 = range.start() + range.length();

            QString front,back,mid;
            front = str.mid( 0, index1 );
            mid = str.mid( index1, range.length() );
            back = str.mid( index2 );

            front = toHtml( front );
            front = front.mid( 3, front.length() - 7 );
            mid = toHtml( mid );
            mid = mid.mid( 3, mid.length() - 7 );
            back = toHtml( back );
            back = back.mid( 3, back.length() - 7 );
            anchor.sprintf( "<a name=\"%d\"></a>", SubValue );
            str = front + font + anchor + mid + fontback + back;
            TextVu->setText( str );
            ErrLabel->setText( tr( "Match" ) );
        }
        else
        {
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "No Match" ) );
        }
    }
}

/*!
  Text in the multilinedit widget has changed.
*/

void QaRegExpWidget::multiTextChanged()
{
    QString s;
    for ( int i = 0; i < Edit->numLines(); i++ )
    {
        s += Edit->textLine( i );
        if ( i < Edit->numLines() - 1 )
            s+= "\n";
    }
    NumLines = Edit->numLines();
    EditString = s;
    if ( Update )
        updateMatch();
}

/*!
  Do a rematch of the regexp.
*/

void QaRegExpWidget::updateMatch()
{
    if ( Loading )
        return;
//      qDebug( "updateMatch" );
    QTime start;
    double secs;
    if ( Matching == QaRegExpWidget::Split )
    {
        if ( MatchMenu->isItemChecked( SingleLineID ) )
        {
            start.start();
            if ( ( MatchCount = Reg.split( Edit->textLine( SubValue ) ) ) > 0 )
            {
                secs = (double)start.elapsed();
                if ( SubMatch >= MatchCount+1 )
                    SubMatch = MatchCount + 1 - 1;
                updateOutput();
            }
            else
            {
                secs = (double)start.elapsed();
                QString str = QStyleSheet::convertFromPlainText( EditString );
                str = str.mid( 3, str.length() - 7 );
                TextVu->setText( str );
                ErrLabel->setText( tr( "No split was done" ) );
            }
        }
        else
        {
            start.start();
            if ( ( MatchCount = Reg.split( EditString ) ) > 0 )
            {
                secs = (double)start.elapsed();
                if ( SubMatch >= MatchCount+1 )
                    SubMatch = MatchCount + 1 - 1;
                updateOutput();
            }
            else
            {
                secs = (double)start.elapsed();
                QString str = QStyleSheet::convertFromPlainText( EditString );
                str = str.mid( 3, str.length() - 7 );
                TextVu->setText( str );
                ErrLabel->setText( tr( "No split was done" ) );
            }
        }
    }
    else if ( Matching == QaRegExpWidget::Multiple )
    {
        start.start();
        if ( ( MatchCount = Reg.matchMultiple( EditString, SubValue ) ) > 0 )
        {
            secs = (double)start.elapsed();
            if ( SubMatch >= MatchCount+1 )
                SubMatch = MatchCount + 1 - 1;
            updateOutput();
        }
        else
        {
            secs = (double)start.elapsed();
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "No match found" ) );
        }
    }
    else
    {
        start.start();
        if ( Reg.match( EditString, SubValue ) )
        {
            secs = (double)start.elapsed();
            updateOutput();
        }
        else
        {
            secs = (double)start.elapsed();
            QString str = QStyleSheet::convertFromPlainText( EditString );
            str = str.mid( 3, str.length() - 7 );
            TextVu->setText( str );
            ErrLabel->setText( tr( "No match found" ) );
        }
    }
    SpeedLabel->setText( QString( "Time taken: %1 sec(s)" ).arg( secs/1000.0 ) );

    Sub->clear();
    Matches->clear();
    if ( Matching == QaRegExpWidget::Split )
    {
        if ( MatchCount > 0 )
        {
            Matches->insertItem( "0 - [All matches]", 0 );
        }
        for ( int i = 0; i < MatchCount; i++ )
        {
            QString str;
            str = QString( "%1 - [%2,%3]" ).arg( i + 1 ).arg( Reg.matchBeginning(i) ).arg( Reg.matchEnd(i) );
            Matches->insertItem( str, i + 1 );
        }
        for ( int i = 0; i < NumLines; i++ )
        {
            QString str;
            str = QString( "%1" ).arg( i + 1 );
            Sub->insertItem( str, i );
        }
        Matches->setSelected( SubMatch, true );
        Matches->setCurrentItem( SubMatch );
    }
    else if ( Matching == QaRegExpWidget::Multiple )
    {
        if ( MatchCount > 0 )
        {
            Matches->insertItem( "0 - [All matches]", 0 );
        }
        for ( int i = 0; i < MatchCount; i++ )
        {
            QString str;
            str = QString( "%1 - [%2,%3]" ).arg( i + 1 ).arg( Reg.matchBeginning(i) ).arg( Reg.matchEnd(i) );
            Matches->insertItem( str, i + 1 );
        }
        for ( int i = 0; i < Reg.subCount() + 1; i++ )
        {
            QString str;
            str = QString( "%1" ).arg( i );
            Sub->insertItem( str, i );
        }
        Matches->setSelected( SubMatch, true );
        Matches->setCurrentItem( SubMatch );
    }
    else
    {
        for ( int i = 0; i < Reg.subCount() + 1; i++ )
        {
            QString str;
            if ( Reg.matches()[i].start() != -1 )
                str = QString( "%1 - [%2,%3]" ).arg( i ).arg( Reg.matchBeginning(i) ).arg( Reg.matchEnd(i) );
            else
                str = QString( tr( "(%1) - No match" ) ).arg( i );
            Sub->insertItem( str, i );
        }
    }
    Sub->setSelected( SubValue, true );
    Sub->setCurrentItem( SubValue );
}

void QaRegExpWidget::doPlain()
{
    if ( OutputMenu->isItemChecked( PlainTextID ) )
        return;
    OutputMenu->setItemChecked( PlainTextID, true );
    OutputMenu->setItemChecked( LispID, false );
    OutputMenu->setItemChecked( PerlID, false );
    OutputMenu->setItemChecked( PHPID, false );
    OutputMenu->setItemChecked( QtID, false );
    delete Language;
    Language = new QaPlainLanguage();
    updateConversion();
}

void QaRegExpWidget::doLisp()
{
    if ( OutputMenu->isItemChecked( LispID ) )
        return;
    OutputMenu->setItemChecked( LispID, true );
    OutputMenu->setItemChecked( PlainTextID, false );
    OutputMenu->setItemChecked( PerlID, false );
    OutputMenu->setItemChecked( PHPID, false );
    OutputMenu->setItemChecked( QtID, false );
    delete Language;
    Language = new QaLispLanguage();
    updateConversion();
}

void QaRegExpWidget::doPerl()
{
    if ( OutputMenu->isItemChecked( PerlID ) )
        return;
    OutputMenu->setItemChecked( PerlID, true );
    OutputMenu->setItemChecked( PlainTextID, false );
    OutputMenu->setItemChecked( LispID, false );
    OutputMenu->setItemChecked( PHPID, false );
    OutputMenu->setItemChecked( QtID, false );
    delete Language;
    Language = new QaPerlLanguage();
    updateConversion();
}

void QaRegExpWidget::doPHP()
{
    if ( OutputMenu->isItemChecked( PHPID ) )
        return;
    OutputMenu->setItemChecked( PHPID, true );
    OutputMenu->setItemChecked( PerlID, false );
    OutputMenu->setItemChecked( PlainTextID, false );
    OutputMenu->setItemChecked( LispID, false );
    OutputMenu->setItemChecked( QtID, false );
    delete Language;
    Language = new QaPHPLanguage();
    updateConversion();
}

void QaRegExpWidget::doQt()
{
    if ( OutputMenu->isItemChecked( QtID ) )
        return;
    OutputMenu->setItemChecked( QtID, true );
    OutputMenu->setItemChecked( PHPID, false );
    OutputMenu->setItemChecked( PerlID, false );
    OutputMenu->setItemChecked( PlainTextID, false );
    OutputMenu->setItemChecked( LispID, false );
    delete Language;
    Language = new QaQtLanguage();
    updateConversion();
}

void QaRegExpWidget::doCompleteOutput()
{
    OutputMenu->setItemChecked( CompleteID, !OutputMenu->isItemChecked( CompleteID ) );
    updateConversion();
}

QString QaRegExpWidget::configPath() const
{
    QString str = QDir::homeDirPath();
    str += "/.regexp";
    QDir d;
    if ( !d.exists( str ) )
    {
        d.mkdir( str );
    }
    return str;
}

QString QaRegExpWidget::outputType() const
{
    if ( OutputMenu->isItemChecked( PlainTextID ) )
        return "PlainText";
    else if ( OutputMenu->isItemChecked( LispID ) )
        return "Lisp";
    else if ( OutputMenu->isItemChecked( PerlID ) )
        return "Perl";
    else if ( OutputMenu->isItemChecked( PHPID ) )
        return "PHP";
    else if ( OutputMenu->isItemChecked( QtID ) )
        return "Qt";
    else
        return "Unknown";
}

void QaRegExpWidget::setOutputType( const QString &str )
{
    if ( str == "PlainText" )
        doPlain();
    else if ( str == "Lisp" )
        doLisp();
    else if ( str == "Perl" )
        doPerl();
    else if ( str == "PHP" )
        doPHP();
    else if ( str == "Qt" )
        doQt();
    else
        doPlain();
}

void QaRegExpWidget::saveConfig()
{
    QString file = configPath() + "/config";
    QFile f( file );
    if ( f.open( IO_WriteOnly ) )
    {
        QTextStream ts( &f );
        ts << "# RegExplorer config file\n\n";
        ts << "Splits = ";
        QValueList<int> lst = MainSplit->sizes();
        ts << lst[0] << "," << lst[1];
        lst = RightSplit->sizes();
        ts << ":" << lst[0] << "," << lst[1];
        lst = SubSplit->sizes();
        ts << ":" << lst[0] << "," << lst[1] << "\n";

        ts << "Size = " << width() << "," << height() << "\n";

        ts << "Match = ";
        if ( Matching == QaRegExpWidget::Split )
            ts << "Split\n";
        else if ( Matching == QaRegExpWidget::Multiple )
            ts << "Multiple\n";
        else
            ts << "Single\n";
        ts << "Case = " << ( MatchMenu->isItemChecked( MatchCaseID ) ? "Use\n" : "Ignore\n" );
        ts << "SingleLine = " << ( MatchMenu->isItemChecked( SingleLineID ) ? "True\n" : "False\n" );
        ts << "NewLine = " << ( MatchMenu->isItemChecked( NewLineID ) ? "True\n" : "False\n" );
        ts << "AutomaticMatch = " << ( MatchMenu->isItemChecked( AutomaticID ) ? "True\n" : "False\n" );
        ts << "Output = " << outputType() << "\n";
        ts << "CodeOutput = " << ( OutputMenu->isItemChecked( CompleteID ) ? "True\n" : "False\n" );

        ts << "SubExpr = " << SubValue << "\n";
        ts << "MatchIndex = " << SubMatch << "\n";

        ts << "SelectColor = " << hexColor( Select ) << "\n";
        ts << "BackColor = " << hexColor( Back ) << "\n";
    }
    f.close();

    file = configPath() + "/regexp";
    f.setName( file );
    if ( f.open( IO_WriteOnly ))
    {
        QTextStream ts( &f );
        ts << RegEd->text();
    }
    f.close();

    file = configPath() + "/history";
    f.setName( file );
    if ( f.open( IO_WriteOnly ))
    {
        QTextStream ts( &f );
        for ( QStringList::Iterator it = OldFiles.begin(); it != OldFiles.end(); ++it )
        {
            ts << *it << "\n";
        }
    }
    f.close();

    file = configPath() + "/text";
    f.setName( file );
    if ( f.open( IO_WriteOnly ))
    {
        QTextStream ts( &f );
        ts << Edit->text();
    }
    f.close();
}

void QaRegExpWidget::loadConfig()
{
//      qDebug( "Loading" );
    Loading = true;
    Update = false;
    QString file = configPath() + "/config";
    QFile f( file );
    if ( f.open( IO_ReadOnly ) )
    {
        QTextStream ts( &f );
        QString line = ts.readLine();
        QaRegExp reg( "^([a-zA-Z]+)[ \t]*=[ \t]*([a-zA-Z]+)[ \t]*$" );
        QaRegExp regnumber( "^([a-zA-Z]+)[ \t]*=[ \t]*([0-9]+)[ \t]*$" );
        QaRegExp reghex( "^([a-zA-Z]+)[ \t]*=[ \t]*([0-9a-f]+)[ \t]*$" );
        QaRegExp regsplit( "^([a-zA-Z]+)[ \t]*=[ \t]*([0-9]+),([0-9]+):([0-9]+),([0-9]+):([0-9]+),([0-9]+)[ \t]*$" );
        QaRegExp regsize( "^([a-zA-Z]+)[ \t]*=[ \t]*([0-9]+),([0-9]+)[ \t]*$" );
        WSize = size();
        const char *tmp = line.ascii();
        if ( line == "# RegExplorer config file" )
        {
            while ( !ts.eof() )
            {
                line = ts.readLine();
                tmp = line.ascii();
                if ( reg.match( line ) )
                {
                    QString name = reg.matchString( 1 );
                    QString type = reg.matchString( 2 );
                    if ( name == "Match" )
                    {
                        if ( type == "Split" )
                            doSplit();
                        else if ( type == "Multiple" )
                            doMultiple();
                        else
                            doNormal();
                    }
                    else if ( name == "Case" )
                    {
                        if ( type == "Use" )
                            MatchMenu->setItemChecked( MatchCaseID, true );
                        else
                            MatchMenu->setItemChecked( MatchCaseID, false );
                    }
                    else if ( name == "SingleLine" )
                    {
                        if ( type == "True" )
                            MatchMenu->setItemChecked( SingleLineID, true );
                        else
                            MatchMenu->setItemChecked( SingleLineID, false );
                    }
                    else if ( name == "NewLine" )
                    {
                        if ( type == "True" )
                            MatchMenu->setItemChecked( NewLineID, true );
                        else
                            MatchMenu->setItemChecked( NewLineID, false );
                        Reg.setHandleNewline( MatchMenu->isItemChecked( NewLineID ) );
                    }
                    else if ( name == "CodeOutput" )
                    {
                        if ( type == "True" )
                            OutputMenu->setItemChecked( CompleteID, true );
                        else
                            OutputMenu->setItemChecked( CompleteID, false );
                    }
                    else if ( name == "AutomaticMatch" )
                    {
                        if ( type == "True" )
                            MatchMenu->setItemChecked( AutomaticID, true );
                        else
                            MatchMenu->setItemChecked( AutomaticID, false );
                        Automatic = MatchMenu->isItemChecked( AutomaticID );
                    }
                    else if ( name == "Output" )
                    {
                        setOutputType( type );
                    }
                }
                else if ( regnumber.match( line ) )
                {
                    QString name = regnumber.matchString( 1 );
                    QString type = regnumber.matchString( 2 );
                    if ( name == "SubExpr" )
                    {
                        SubValue = type.toInt();
                    }
                    else if ( name == "MatchIndex" )
                    {
                        SubMatch = type.toInt();
                    }
                }
                else if ( reghex.match( line ) )
                {
                    QString name = reghex.matchString( 1 );
                    QString type = reghex.matchString( 2 );
                    if ( name == "SelectColor" )
                    {
                        QColor tmp = fromHex( type );
                        if ( tmp.isValid() )
                            Select = tmp;
                    }
                    else if ( name == "BackColor" )
                    {
                        QColor tmp = fromHex( type );
                        if ( tmp.isValid() );
                        {
                            Back = tmp;
                            TextVu->setPaper( Back );
                        }
                    }
                }
                else if ( regsplit.match( line ) )
                {
                    QString name = regsplit.matchString( 1 );
                    QString num1 = regsplit.matchString( 2 );
                    QString num2 = regsplit.matchString( 3 );
                    QString num3 = regsplit.matchString( 4 );
                    QString num4 = regsplit.matchString( 5 );
                    QString num5 = regsplit.matchString( 6 );
                    QString num6 = regsplit.matchString( 7 );
                    if ( name == "Splits" )
                    {
                        QValueList<int> lst;
                        lst.append( num1.toInt() );
                        lst.append( num2.toInt() );
                        MainSplit->setSizes( lst );
                        lst.clear();
                        lst.append( num3.toInt() );
                        lst.append( num4.toInt() );
                        RightSplit->setSizes( lst );
                        lst.clear();
                        lst.append( num5.toInt() );
                        lst.append( num6.toInt() );
                        SubSplit->setSizes( lst );
                    }
                }
                else if ( regsize.match( line ) )
                {
                    QString name = regsize.matchString( 1 );
                    QString num1 = regsize.matchString( 2 );
                    QString num2 = regsize.matchString( 3 );
//                    qDebug( "%s, %s", num1.ascii(), num2.ascii() );
                    if ( name == "Size" )
                    {
                        WSize = QSize( num1.toInt(), num2.toInt() );
                    }
                }
            }
        }
    }
    f.close();

    file = configPath() + "/regexp";
    f.setName( file );
    if ( f.open( IO_ReadOnly ) )
    {
        QTextStream ts( &f );
        QString str = ts.readLine();
        RegEd->setText( str );
        ConversionText = str;
        str = PlainLanguage.fromPlainText( str,
                                           false,
                                           QaRegExpLanguage::Normal );
        setExpression( str );
    }
    f.close();

    file = configPath() + "/history";
    f.setName( file );
    if ( f.open( IO_ReadOnly ) )
    {
        QTextStream ts( &f );
        while ( !ts.eof() )
        {
            QString str = ts.readLine();
            if ( !str.isEmpty() )
                addToHistory( str, false );
        }
    }
    f.close();

    file = configPath() + "/text";
    f.setName( file );
    if ( f.open( IO_ReadOnly ) )
    {
        char *tmp = new char[f.size()+1];
        tmp[f.size()] = 0;
        f.readBlock( tmp, f.size() );
        QString str( tmp );
        Edit->setText( str );
        delete []tmp;
    }
    f.close();

    Update = true;
    Loading = false;
    Reg.setCaseSensitive( MatchMenu->isItemChecked( MatchCaseID ) );
    if ( MatchMenu->isItemChecked( MatchSplitID ) )
        Sub->setEnabled( MatchMenu->isItemChecked( SingleLineID ) );
//      multiTextChanged();
//      updateMatch();
//      qDebug( "Loading done" );
}

void QaRegExpWidget::polish()
{
    resize( WSize );
}

void QaRegExpWidget::resizeEvent( QResizeEvent *e )
{
    QMainWindow::resizeEvent( e );
    WSize = size();
}


void QaRegExpWidget::doBackColor()
{
    QColor tmp = QColorDialog::getColor( Back, this );
    if ( tmp.isValid() )
    {
        Back = tmp;
        TextVu->setPaper( Back );
    }
}

void QaRegExpWidget::doSelectColor()
{
    QColor tmp = QColorDialog::getColor( Select, this );
    if ( tmp.isValid() )
    {
        Select = tmp;
        updateMatch();
    }
}

QColor QaRegExpWidget::fromHex( const QString &s ) const
{
    int red, green, blue;
    if ( s.length() != 6 )
        return QColor();
    const char *tmp = s.ascii();
    for ( int i = 0; i < 6; i++ )
    {
        if ( tmp[i] >= '0' )
        {
            if ( tmp[i] > '9' )
            {
                if ( tmp[i] < 'a' )
                    return QColor();
                else if ( tmp[i] > 'f' )
                    return QColor();
            }
        }
        else
            return QColor();
    }

    char c;
    c = ( tmp[0] > '9' ? tmp[0] - 'a' + 0xa : tmp[0] - '0' );
    red = c << 4;
    c = ( tmp[1] > '9' ? tmp[1] - 'a' + 0xa : tmp[1] - '0' );
    red |= c;

    c = ( tmp[2] > '9' ? tmp[2] - 'a' + 0xa : tmp[2] - '0' );
    green = c << 4;
    c = ( tmp[3] > '9' ? tmp[3] - 'a' + 0xa : tmp[3] - '0' );
    green |= c;

    c = ( tmp[4] > '9' ? tmp[4] - 'a' + 0xa : tmp[4] - '0' );
    blue = c << 4;
    c = ( tmp[5] > '9' ? tmp[5] - 'a' + 0xa : tmp[5] - '0' );
    blue |= c;
    return QColor( red, green, blue );
}

QString QaRegExpWidget::hexColor( const QColor &col ) const
{
    char hex[] = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };
    QString num;
    num += hex[(col.red() >> 4) & 0xf];
    num += hex[col.red() & 0xf];
    num += hex[(col.green() >> 4) & 0xf];
    num += hex[col.green() & 0xf];
    num += hex[(col.blue() >> 4) & 0xf];
    num += hex[col.blue() & 0xf];
    return num;
}

QString QaRegExpWidget::selectColor() const
{
    return QString( "<font color=#%1><b>" ).arg( hexColor( Select ) );
}

//  void QaRegExpWidget::doF1()
//  {
//      if ( Sub->count() > 0 )
//          Sub->setCurrentItem( 0 );
//  }

//  void QaRegExpWidget::doShiftF1()
//  {
//      if ( Matches->count() > 0 )
//          Matches->setCurrentItem( 0 );
//  }

void QaRegExpWidget::doFunctionKey( int id )
{
    if ( (int)Sub->count() > id )
        Sub->setCurrentItem( id );
}

void QaRegExpWidget::doShiftedFunctionKey( int id )
{
    if ( (int)Matches->count() > id )
        Matches->setCurrentItem( id );
}

void QaRegExpWidget::doAutomatic()
{
    Automatic = !MatchMenu->isItemChecked( AutomaticID );
    MatchMenu->setItemChecked( AutomaticID, Automatic );
}

void QaRegExpWidget::theTextHasChanged()
{
//      qDebug( "theTextHasChanged" );
    bool update = true;
    if ( RegEd->text().isEmpty() )
    {
        if ( QMessageBox::warning( this, tr( "Empty Regular Expression" ),
                                   tr( "Empty Regular Expression is used,\n"
                                       "this can result in a long parsing time,\n"
                                       "do you wish to continue?" ),
                                   tr( "&Yes" ), tr( "&No" ), QString::null,
                                   0, 1 ) == 1 )
            update = false;
    }
    if ( update )
        textChange( RegEd->text() );
}

void QaRegExpWidget::doAbout()
{
    QMessageBox::about( this, tr( "About RegExplorer" ),
                        tr( "RegExplorer %1\n\n"
                            "Copyright 1999-2000\n"
                            "Jan Borsodi <jb@ez.no>\n\n"
                            "Download new version at:\n"
                            "http://regexplorer.sourceforge.net\n\n"
                            "Thanks to:\n"
                            "Wim Delvaux, Kevin Ying, Justin Mason, Markus Fischer,\n"
                            "Kent Vander Velden, Jan Prokop, Don Undeen\n"
                            "for tips, bugfixes and general comments" ).arg( version() ) );
}

QString QaRegExpWidget::version() const
{
    return QString( "0.1.6" );
}

void QaRegExpWidget::doAboutQt()
{
    QMessageBox::aboutQt( this, tr( "About Qt" ) );
}
