/* renaming.c - Checking renames and resolving rename order.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* C89 */
#include <stdlib.h>
/* POSIX/gnulib */
#include <stdbool.h>
/* Gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* common */
#include "common/memory.h"
#include "common/io-utils.h"
#include "common/string-utils.h"
#include "common/common.h"
#include "common/error.h"
#include "common/llist.h"
#include "common/hmap.h"
/* qmv */
#include "qmv.h"

/**
 * Find renames that have the same destination filename.
 * Mark these as STATUS_DUPLICATE.
 */
static inline void
duplicate_scan(LList *renames, RenamePlan *plan)
{
	HMap *map_new;
	Iterator *it;
	LList *dupes = llist_new();

	map_new = hmap_new();
	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *snew = iterator_next(it);
		FileSpec *sold = hmap_get(map_new, snew->new_name);

		if (sold != NULL) {
			if (sold->status != STATUS_DUPLICATE)
				llist_add(dupes, sold);
			sold->status = STATUS_DUPLICATE;
			snew->status = STATUS_DUPLICATE;
			sold->next_spec = snew;
			snew->next_spec = NULL;
		}
		hmap_put(map_new, snew->new_name, snew);
	}
	iterator_free(it);	
	hmap_free(map_new);

	/* Copy duplicate-renames to done-error-list */
	for (it = llist_iterator(dupes); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		do {
			llist_add(plan->error, spec);
			spec = spec->next_spec;
		} while (spec != NULL);
	}
	iterator_free(it);	
	llist_free(dupes);
}

/**
 * Sort renames topologically. So that their destination name
 * won't exist when they are to be renamed.
 */
static inline void
topologic_sort(LList *renames, RenamePlan *plan)
{
	LList *pending = llist_new();
	Iterator *it;
	HMap *map_available;
	bool changed;

	map_available = hmap_new();

	/* Find all OK renames where destination file does not exist. */
	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->status == STATUS_NONE) {
			char *fullname = cat_files(renames_directory, spec->new_name);
			if (!file_exists(fullname)) {
				spec->status = STATUS_RENAME;
				hmap_put(map_available, spec->old_name, spec);
				llist_add(plan->ok, spec);
			} else {
				llist_add(pending, spec);
			}
			free(fullname);
		}
	}
	iterator_free(it);

	/* Topologic sorting */
	do {
		changed = false;
		
		for (it = llist_iterator(pending); iterator_has_next(it); ) {
			FileSpec *spec = iterator_next(it);
			FileSpec *depend = hmap_get(map_available, spec->new_name);
			if (depend != NULL) {
				iterator_remove(it);
				spec->status = STATUS_RENAME;
				/*spec->dependency = depend;*/
				hmap_put(map_available, spec->old_name, spec);
				hmap_remove(map_available, spec->new_name);
				llist_add(plan->ok, spec);
				changed = true;
			}
		}
		iterator_free(it);
	} while (changed);

	llist_free(pending);
	hmap_free(map_available);
}

/**
 * Find all renames where the old file is missing.
 * There is no guarantee that the old fild will go
 * missing later (say, a microsecond after we
 * did the file_exists check), but it is probably
 * of some guidance to the user.
 *
 * XXX: Technically, there are other checks similar
 * to this one. For example, maybe the new file
 * is on a different filesystem? Or the new file
 * is a subdirectory of the old file (assuming the
 * old file is a directory)?
 */
static inline void
old_missing_scan(LList *renames, RenamePlan *plan)
{
	Iterator *it;

	/* Find all renames where original file no longer exists. */
	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->status == STATUS_NONE) {
			char *fullname = cat_files(renames_directory, spec->old_name);
			if (!file_exists(fullname)) {
				spec->status = STATUS_OLD_MISSING;
				llist_add(plan->error, spec);
			}
			free(fullname);
		}
	}
	iterator_free(it);
}

/**
 * Find all renames that are already completed.
 */
static inline void
completed_scan(LList *renames, RenamePlan *plan)
{
	Iterator *it;

	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->renamed == RENAME_COMPLETE) {
			spec->status = STATUS_COMPLETED;
			llist_add(plan->no_change, spec);
		}
	}
	iterator_free(it);
}

/**
 * Find all renames that are no renames, or already
 * completed.
 */
static inline void
no_change_scan(LList *renames, RenamePlan *plan)
{
	Iterator *it;

	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->status == STATUS_NONE) {
			if (strcmp(spec->old_name, spec->new_name) == 0) {
				spec->status = STATUS_NO_CHANGE;
				llist_add(plan->no_change, spec);
			}
		}
	}
	iterator_free(it);
}

/**
 * Remaining pending files are either cross-referenced,
 * or can't be renamed because the destination file exists.
 */
static inline void
cross_reference_resolve(LList *renames, RenamePlan *plan)
{
	HMap *map_old;
	HMap *map_new;
	Iterator *it;

	map_old = hmap_new();
	map_new = hmap_new();

	/* Put old names for all remaining files into the table. */
	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (spec->status == STATUS_NONE) {
			hmap_put(map_old, spec->old_name, spec);
			hmap_put(map_new, spec->new_name, spec);
		}
	}
	iterator_free(it);

	for (it = llist_iterator(plan->ok); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		hmap_put(map_new, spec->new_name, spec);
	}
	iterator_free(it);

	/* Follow the renamings from new to old. */
	for (it = llist_iterator(renames); iterator_has_next(it); ) {
		FileSpec *s1 = iterator_next(it);
		FileSpec *s2 = s1;
		FileSpec *prev_circular = s1;
		Iterator *it2;
		int tmp_count = 1;
		char *tmp_name;
		char *fullname = NULL;

		/* Rename has already been processed. Ignore it. */
		if (s1->status != STATUS_NONE)
			continue;

		/* Follow the renamings from new to old, until we reach the rename
		 * we started from, or a dead end (i.e. destination exists). We can
		 * be sure this is not going to loop forever, because no two
		 * renames can have the same destination name. */
		while (true) {
			s2 = hmap_get(map_old, s2->new_name);

			/* Destination file already exists. */
			if (s2 == NULL || s2->status != STATUS_NONE) {
				s1->next_spec = NULL;
				break;
			}
			/* We have found a circular rename-set. */
			if (s2 == s1) {
				prev_circular->next_spec = NULL;
				break;
			}
			prev_circular->next_spec = s2;
			prev_circular = s2;
		}

		/* This was not a cross-reference. The destination already
		 * existed. */
		if (s1->next_spec == NULL) {
			s1->status = STATUS_NEW_EXISTS;
			llist_add(plan->error, s1);
			continue;
		}

		/* Add and update status of selected files (reverse order). */
		for (s2 = s1; s2 != NULL; s2 = s2->next_spec) {
			s2->status = STATUS_CIRCULAR;
			llist_add_first(plan->ok, s2);
		}

		/* Find a temporary name for the new rename */
		do {
			if (fullname != NULL)
				free(fullname);
			tmp_name = xasprintf("%s-%d", s1->old_name, tmp_count++);
			fullname = cat_files(renames_directory, tmp_name);
		} while (file_exists(fullname) || hmap_contains_key(map_new, tmp_name));
		free(fullname);

		/* Add the extra required rename. */
		s2 = new_file_spec();
		llist_add_first(plan->ok, s2);
		s2->status = STATUS_CIRCULAR;
		s2->old_name = s1->old_name;
		s2->new_name = xstrdup(tmp_name);
		s1->old_name = xstrdup(tmp_name);

		/* Update the next_spec field of these circular renames */
		it2 = llist_iterator(plan->ok);
		iterator_next(it2);	/* skip first - that's s2 */
		while (iterator_has_next(it)) {
			FileSpec *spec = iterator_next(it2);
			s2->next_spec = spec;
			/*spec->next_spec = (last->next == NULL ? NULL : last->next->data);*/
			/*spec->dependency = (last->prev == NULL ? NULL : last->prev->data);*/
			s2 = spec;
			if (spec == s1)
				break;
		}
		iterator_free(it2);
		s1->next_spec = NULL;
	}
	iterator_free(it);

	hmap_free(map_new);
	hmap_free(map_old);
}

RenamePlan *
make_plan(LList *renames)
{
	RenamePlan *plan;

	plan = xmalloc(sizeof(RenamePlan));
	plan->ok = llist_new();
	plan->error = llist_new();
	plan->no_change = llist_new();

	completed_scan(renames, plan);
	duplicate_scan(renames, plan);
	old_missing_scan(renames, plan);
	no_change_scan(renames, plan);
	topologic_sort(renames, plan);
	cross_reference_resolve(renames, plan);

	return plan;
}

void
plan_command(char **args)
{
	if (plan == NULL) {
		printf(_("no plan - use `list' and `edit'\n"));
		return;
	}
	display_plan(plan, true);
}

static void
display_names(FileSpec *spec, bool show_status)
{
	char *name_old = quote_output_file(spec->old_name);
	char *name_new = quote_output_file(spec->new_name);
	const char *status = "";
	int len = strlen(name_old)+strlen(_(" -> "))+strlen(name_new);

	if (show_status) {
		if (spec->renamed == RENAME_COMPLETE) {
			status = _("[done] ");
		} else if (spec->renamed == RENAME_FAILED) {
			status = _("[failed] ");
		}
		len += strlen(status);
	}

	/* XXX: assume this is width of screen. should query somehow */
	if (len < 80) {
		printf(_("%s%s -> %s\n"), status, name_old, name_new);
	} else {
		printf(_("%s%s\n -> %s\n"), status, name_old, name_new);
	}

	free(name_old);
	free(name_new);
}

static void
print_error_status(FileSpec *spec)
{
	switch (spec->status) {
	case STATUS_OLD_MISSING:
		puts(_("  Source file is missing, can not rename"));
		break;
	case STATUS_DUPLICATE:
		//if (spec->next_spec == NULL)
		puts(_("  Destination file names are the same, can not rename"));
		break;
	case STATUS_NEW_EXISTS:
		puts(_("  Destination file exists, can not rename"));
		break;
	default:
		internal_error(_("invalid status (%d) after order resolution\n"));
		break;
	}
}

static void
print_ok_status(FileSpec *spec)
{
	switch (spec->status) {
	case STATUS_CIRCULAR:
		//if (spec->next_spec == NULL)
		puts(_("  These renames were created due to circular renaming"));
		break;
	case STATUS_RENAME:
		puts(_("  Regular rename"));
		break;
	default:
		internal_error(_("invalid status (%d) after order resolution\n"));
		break;
	}
}

void
display_plan(RenamePlan *plan, bool show_status)
{
	Iterator *it;
	FileSpecStatus last_status = STATUS_NONE;
	FileSpec *last_spec = NULL;

	if (!llist_is_empty(plan->error)) {
		printf(_("Plan contains errors.\n\n"));
	} else if (llist_is_empty(plan->ok)) {
		printf(_("Plan is empty (no changes made).\n"));
	} else {
		printf(_("Plan is valid.\n\n"));
	}

	for (it = llist_iterator(plan->error); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (last_spec != NULL && last_status != spec->status)
			print_error_status(spec);
		display_names(spec, false);
		last_status = spec->status;
		last_spec = spec;
	}
	iterator_free(it);
	if (last_spec != NULL)
		print_error_status(last_spec);

	if (!llist_is_empty(plan->error))
		return;

	last_spec = NULL;
	for (it = llist_iterator(plan->ok); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (last_spec != NULL && last_status != spec->status)
			print_ok_status(spec);
		display_names(spec, show_status);
		last_status = spec->status;
		last_spec = spec;
	}
	iterator_free(it);
	if (last_spec != NULL)
		print_ok_status(last_spec);

	puts("");
}

void
free_plan(RenamePlan *plan)
{
	llist_free(plan->ok);
	llist_free(plan->error);
	llist_free(plan->no_change);
	free(plan);
}
