/* The right hand pane in the configuration window.  This class is responsible for
 * displaying and changing the currently selected module.
 */

#include "modulemodifier.h"
#include "overridemenu.h"

#include <gnome.h>

void applyClicked(GtkButton* button, ModuleModifier* data)
{
        data->apply();
}

void okayClicked(GtkButton* button, ModuleModifier* data)
{
        data->apply();
        data->cancel();
        data->parent.closeModule(data->getModulePath());
}

void cancelClicked(GtkButton* button, ModuleModifier* data)
{
        data->cancel();
        data->parent.closeModule(data->getModulePath());
}

ModuleModifier::ModuleModifier(Configuration& parent_, const Path& mod_)
        : parent(parent_),
          changed(false),
          reloadInProgress(true)
{
        mod = mod_;
        outerBox = gtk_vbox_new(false, 0);
        GtkWidget* scrollWin = gtk_scrolled_window_new(0, 0);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrollWin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_box_pack_start(GTK_BOX(outerBox), scrollWin, true, true, 0);
        gtk_widget_show(scrollWin);
        innerBox = gtk_vbox_new(false, 2);
        gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrollWin), innerBox);

        load();

        reloadInProgress = false;

        gtk_widget_show_all(innerBox);
}

ModuleModifier::~ModuleModifier()
{
        unload();
}

void ModuleModifier::unload()
{
        unsigned int i;
        for (i = 0; i < contents.size(); i++) {
                gtk_container_remove(GTK_CONTAINER(innerBox), contents[i]->getWidget());
                delete contents[i];
        }
        for (i = 0; i < extraWidgets.size(); i++) {
                gtk_container_remove(GTK_CONTAINER(innerBox), extraWidgets[i]);
        }
        for (i = 0; i < overrideMenus.size(); i++) {
                if (overrideMenus[i]->widgetValid())
                        gtk_container_remove(GTK_CONTAINER(innerBox), overrideMenus[i]->getWidget());
                delete overrideMenus[i];
        }
        if (contents.size() != 0)
                contents.clear();
        if (extraWidgets.size() != 0)
                extraWidgets.clear();
        if (overrideMenus.size() != 0)
                overrideMenus.clear();
}

// valueNum is an index into the module's moduleInfo entry
void ModuleModifier::addEntry(const Path& module, int valueNum, bool owner)
{
        try {
                const DataSetMap *m = ModuleTree::instance()->module_info(module);
                assert(m != NULL);
                const DataSetMap& modInf = *m;
                
                const DataSet& supported_var_types = modInf["supported_var_types"];
                const DataSet& supported_vars = modInf["supported_vars"];
                
                string varName = supported_vars.getString(valueNum);
                string varType = supported_var_types.getString(valueNum);
                
                ModifierEntry* modEnt = ModifierEntry::createByTypeName(varType, varName, mod,
                                                                        *this, owner);
                
                contents.push_back(modEnt);
                if (modEnt->isValid()) {
                        gtk_box_pack_start(GTK_BOX(innerBox), modEnt->getWidget(),
                                           false, false, 3);
                        gtk_widget_show_all(modEnt->getWidget());
                }
        } catch (ModuleError ex) {
                cerr << "Module exception: " << ex.description << endl;
        }
}

void ModuleModifier::addOverride(const Path& module, int entryNum)
{
        addEntry(module, entryNum, false);
}

void ModuleModifier::removeEntry(ModifierEntry* modE)
{
        for (unsigned int i = 0; i < contents.size(); i++) {
                if (contents[i] == modE) {
                        for (unsigned int j = 0; j < overrideMenus.size(); j++)
                                overrideMenus[j]->removeSticky(modE->getName());

                        gtk_container_remove(GTK_CONTAINER(innerBox), contents[i]->getWidget());
                        delete contents[i];
                        vector<ModifierEntry*>::iterator t = contents.begin();
                        t += i;
                        contents.erase(t);
                        break;
                }
        }
}

void ModuleModifier::load()
{
        try {
                const DataSetMap *m = ModuleTree::instance()->module_info(mod);
                assert(m);
                const DataSetMap& modInf = *m;

                const DataSet& supported_vars = modInf["supported_vars"];

                for (int i = 0; i < supported_vars.count(); i++)
                        addEntry(mod, i, true);
        } catch (ModuleError ex) {
                cerr << "Module exception: " << ex.description << endl;
        }

        // scan parents
        Path cmod = mod;
        while (cmod.length() > 0) {
                cmod.remove_last();
                
                OverrideMenu* newOM = new OverrideMenu(mod, *this, cmod);
                overrideMenus.push_back(newOM);
                if (newOM->widgetValid())
                        gtk_box_pack_start(GTK_BOX(innerBox),
                                           newOM->getWidget(), false, false, 3);
        }
        
        for (unsigned int i = 0; i < overrideMenus.size(); i++) {
                overrideMenus[i]->loadOverrides();
        }
        
        addButtons();
}

void ModuleModifier::addButtons()
{
        GtkWidget* hsep = gtk_hseparator_new();
        GtkWidget* hbox = gtk_hbox_new(false, 0);
        GtkWidget* okayButton = gtk_button_new_from_stock(GTK_STOCK_OK);
        GtkWidget* applyButton = gtk_button_new_from_stock(GTK_STOCK_APPLY);
        GtkWidget* cancelButton = gtk_button_new_from_stock(GTK_STOCK_CLOSE);

        gtk_signal_connect(GTK_OBJECT(okayButton), "clicked", GTK_SIGNAL_FUNC(okayClicked), this);
        gtk_signal_connect(GTK_OBJECT(applyButton), "clicked", GTK_SIGNAL_FUNC(applyClicked), this);
        gtk_signal_connect(GTK_OBJECT(cancelButton), "clicked", GTK_SIGNAL_FUNC(cancelClicked), this);

        gtk_box_pack_start(GTK_BOX(hbox), hsep, false, false, 3);
        gtk_box_pack_end(GTK_BOX(hbox), cancelButton, false, false, 3);
        gtk_box_pack_end(GTK_BOX(hbox), applyButton, false, false, 3);
        gtk_box_pack_end(GTK_BOX(hbox), okayButton, false, false, 3);
        gtk_box_pack_end(GTK_BOX(innerBox), hbox, false, false, 3);

        gtk_widget_show(okayButton);
        gtk_widget_show(applyButton);
        gtk_widget_show(cancelButton);
        gtk_widget_show(hsep);
        gtk_widget_show(hbox);

        extraWidgets.push_back(hbox);
}

void ModuleModifier::apply()
{
        try {
                DataSet sticky;
                for (unsigned int i = 0; i < contents.size(); i++)
                        contents[i]->apply();
                for (unsigned int i = 0; i < overrideMenus.size(); i++)
                        sticky.addDataSet(overrideMenus[i]->getSticky());
                ModuleTree::instance()->set_value(mod, "sticky", sticky);
                changed = false;
        } catch (ModuleError ex) {
                cerr << "Module exception: " << ex.description << endl;
        }
}

void ModuleModifier::cancel()
{
        // havn't realy done anything, so nothing needs changing
}

void ModuleModifier::reload()
{
        reloadInProgress = true;
        for (unsigned int i = 0; i < contents.size(); i++)
                contents[i]->reload();
        reloadInProgress = false;
}

// signal that changes have been made to this module
void ModuleModifier::setChanged(bool ch)
{
        if (!reloadInProgress) {
                // code to un-grey "okay" and "apply" buttons would go here
                changed = ch;
        }
}
