
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur
//          Krishnan Subramani
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam
//          Radharamanan Radhakrishnan
//          Swaminathan Subramanian
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIR_DesignFile.hh"
#include "IIR_AssociationList.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ConfigurationDeclaration.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_PackageDeclaration.hh"
#include "IIR_ArchitectureStatementList.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_Label.hh"
#include "IIR_DeclarationList.hh"
#include "IIR_AboveAttribute.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_RangeTypeDefinition.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConstantDeclaration.hh"
#include "IIR_Identifier.hh"
#include "published_file.hh"
#include "error_func.hh"
#include "savant.hh"
#include "symbol_table.hh"
#include "library_manager.hh"
#include "include_manager.hh"
#include "set.hh"
#include "resolution_func.hh"
#include "sstream-wrap.hh"
#include <cstring>
#include <clutils/StringUtilities.h>

extern bool parse_error;
extern bool debug_symbol_table;

IIRScram::IIRScram(){
  _my_design_file = NULL;
  _currently_publishing_simultaneous_stmt = FALSE;
  _currently_publishing_simultaneous_if_stmt = FALSE;
}

IIRScram::~IIRScram() {}

// Following are global variables.  We're giving them file static scope so
// as to not be accessed directly.  Eventually these should all disappear.
string &_get_current_publish_name(){
  static string _current_publish_name = "";
  return _current_publish_name;
}
void _set_current_publish_name( string new_name ){
  _get_current_publish_name() = new_name;
}

static IIR *_current_publish_node = NULL;
IIR *_get_current_publish_node(){
  return _current_publish_node;
}

void _set_current_publish_node( IIR *new_publish_node ){
  _current_publish_node = new_publish_node;
}

string &_get_current_configuration_name(){
  static string _current_configuration_name = "";
  return _current_configuration_name;
}

void _set_current_configuration_name( string new_configuration_name ){
  _get_current_configuration_name() = new_configuration_name;
}

string &_get_current_architecture_name(){
  static string _current_architecture_name = "";
  return _current_architecture_name;
}

void _set_current_architecture_name( string new_name ){
  _get_current_architecture_name() = new_name;
}

string &_get_current_entity_name(){
  static string _current_entity_name = "";
  return _current_entity_name;
}
void _set_current_entity_name( string new_name ){
  _get_current_entity_name() = new_name;
}

string &_get_current_elab_name(){
  static string _current_elab_name = "";
  return _current_elab_name;
}
void _set_current_elab_name( string new_name ){
  _get_current_elab_name() = new_name;
}

static IIR_Int32 _number_of_processes = 0;
IIR_Int32 _get_number_of_processes(){
  return _number_of_processes;
}
void _set_number_of_processes( IIR_Int32 new_number ){
  _number_of_processes = new_number;
}

IIR_Int32 &_get_number_of_interface_signals(){
  static IIR_Int32 _number_of_interface_signals = 0;
  return _number_of_interface_signals;
}

dl_list<break_handler> &_get_breakList() {
  static dl_list<break_handler> breakList;
  return breakList;
}

string &_get_current_package_name(){
  static string _current_package_name = "";
  return _current_package_name;
}
void _set_current_package_name( string new_name ){
  _get_current_package_name() = new_name;
}

string &_get_current_package_body_name(){
  static string _current_package_body_name = "";
  return _current_package_body_name;
}
void _set_current_package_body_name( string new_name ){
  _get_current_package_body_name() = new_name;
}

string &_get_current_suffix_name(){
  static string _current_suffix_name = "";
  return _current_suffix_name;
}
void _set_current_suffix_name( string new_name ){
  _get_current_suffix_name() = new_name;
}

void
IIRScram::set_is_currently_publishing_simultaneous_if_stmt(bool newValue) { 
  _currently_publishing_simultaneous_if_stmt = newValue;
}

void 
IIRScram::set_is_currently_publishing_simultaneous_stmt(bool newValue) {
  _currently_publishing_simultaneous_stmt = newValue;
}

IIR_Boolean
IIRScram::get_is_currently_publishing_simultaneous_if_stmt() {
  return _currently_publishing_simultaneous_if_stmt;
}

IIR_Boolean 
IIRScram::get_is_currently_publishing_simultaneous_stmt() { 
  return _currently_publishing_simultaneous_stmt;
}

static IIR_Int32 _index_level = 0;
IIR_Int32 _get_index_level(){  
 return _index_level;
}

IIR_Int32
IIRScram::_get_stmt_qty_index() {
  return -1;
}

void _set_index_level( IIR_Int32 new_level ){
  _index_level = new_level;
}

string &_get_current_another_name(){
  static string _current_another_name = "";
  return _current_another_name;
}

void _set_current_another_name( string new_name ){
  _get_current_another_name() = new_name;
}

static IIR_Boolean _allocate_return_value = FALSE;
IIR_Boolean _get_allocate_return_value(){
  return _allocate_return_value;
}
void _set_allocate_return_value( IIR_Boolean new_value ){
  _allocate_return_value = new_value;
}

string &_get_publish_prefix_string(){
  static string _publish_prefix_string = "";
  return _publish_prefix_string;
}
void _set_publish_prefix_string( string new_string ){
  _get_publish_prefix_string() = new_string;
}

static IIR_TypeDefinition *_aggregate_iterator_subtype = NULL;
IIR_TypeDefinition *_get_aggregate_iterator_subtype(){
  return _aggregate_iterator_subtype;
}
void _set_aggregate_iterator_subtype( IIR_TypeDefinition *new_type ){
  _aggregate_iterator_subtype = new_type;
}

static int _aggregate_iterator_counter = 0;
int _get_aggregate_iterator_counter(){
  return _aggregate_iterator_counter;
}
void _set_aggregate_iterator_counter( int new_counter ){
  _aggregate_iterator_counter = new_counter;
}

static IIRScram::PublishedUnit _currently_publishing_unit = IIRScram::NONE_PARSING;
IIRScram::PublishedUnit _get_currently_publishing_unit(){
  return _currently_publishing_unit;
}

void _set_currently_publishing_unit( IIRScram::PublishedUnit new_unit ){
  _currently_publishing_unit = new_unit;
}

static IIRScram::PublishedUnit _currently_publishing_vhdl_unit = IIRScram::NONE;
IIRScram::PublishedUnit _get_currently_publishing_vhdl_unit(){
  return _currently_publishing_vhdl_unit;
}

void _set_currently_publishing_vhdl_unit( IIRScram::PublishedUnit new_unit ){
  _currently_publishing_vhdl_unit = new_unit;
}

static symbol_table *cgen_sym_tab = NULL;
symbol_table *_get_cgen_sym_tab(){
  return cgen_sym_tab;
}
void _set_cgen_sym_tab( symbol_table *new_tab ){
  cgen_sym_tab = new_tab;
}

static IIR_Boolean _publish_object_without_prefix = FALSE; 
IIR_Boolean _get_publish_object_without_prefix(){
  return _publish_object_without_prefix;
}
void _set_publish_object_without_prefix( IIR_Boolean new_value ){
  _publish_object_without_prefix = new_value;
}

static IIR_ProcessStatement* _current_process_statement = NULL;
IIR_ProcessStatement *_get_current_process_statement(){
  return _current_process_statement;
}
void _set_current_process_statement( IIR_ProcessStatement *new_stmt ){
  _current_process_statement = new_stmt;
}

IIR_TypeDefinition* 
IIRScram::_get_name_type() {
  _report_undefined_scram_fn("_get_name_type()");
  return NULL;
}

IIR_Boolean
IIRScram::_is_attribute(){ return FALSE; }

IIR_Boolean
IIRScram::_is_longest_static_prefix() { 
  return _is_static_expression();
}

IIR_Boolean
IIRScram::_is_numeric_literal() {
  if( _is_integer_literal() == TRUE ||  _is_floating_literal() == TRUE ){
    return TRUE;
  }
  else {
    return FALSE;
  }
}

IIR_Boolean
IIRScram::_is_quantity() {
  return FALSE;
}

IIR_Boolean
IIRScram::_is_terminal() {
  return FALSE;
}

void 
IIRScram::_publish_vhdl_type_decl(ostream &_vhdl_out) {
  _publish_vhdl_decl(_vhdl_out);
}

void
IIRScram::_publish_vhdl_subtype_decl(ostream &_vhdl_out) {
  _publish_vhdl_decl(_vhdl_out);
}

void 
IIRScram::_type_check( IIR_TypeDefinition *input_decl ){
  set<IIR_TypeDefinition> temp_set;
  temp_set.add( input_decl );
  _type_check( &temp_set );
}

IIR *
IIRScram::_semantic_transform( set<IIR_TypeDefinition> * ){
#ifdef SEMANTIC_DEBUG
  cout << "Semantic transform called for node " << get_kind_text() <<
    " - returning \"this\". " << endl;
#endif
  return (IIR *)this;
}

IIR *
IIRScram::_semantic_transform( IIR_TypeDefinition *input_decl ){
  set<IIR_TypeDefinition> temp_set;
  temp_set.add( input_decl );
  return _semantic_transform( &temp_set );
}

IIR *
IIRScram::_rval_to_decl( IIR_TypeDefinition * ){
#ifdef SEMANTIC_DEBUG
  cout << "_rval_to_decl called for node " << get_kind_text() <<
    " - returning \"this\". " << endl;
#endif

  return (IIR *)this;
}

void
IIRScram::_publish_cc_lvalue( published_file & ){
  _report_undefined_scram_fn("_publish_cc_lvalue( published_file &_cc_out )");
}

void
IIRScram::_publish_cc_rvalue( published_file &_cc_out ){
  _publish_cc_lvalue( _cc_out );
}

void 
IIRScram::_publish_cc_prefix_string( published_file &_cc_out ) {
  if( _get_publish_prefix_string() != "" ) {
    _cc_out << _get_publish_prefix_string();
  }
}

void
IIRScram::_publish_cc_scoping_prefix( ostream &, IIR *, IIR * ){
  _report_undefined_scram_fn("_publish_cc_scoping_prefix( ostream&, IIR *, IIR * )");
}

IIR_Boolean
IIRScram::_is_currently_publishing_subprogram() {
  return (_currently_publishing_unit == FUNCTION ||
	  _currently_publishing_unit == PROCEDURE);
}

IIR_Boolean
IIRScram::_is_currently_publishing_generate_for() {
  return (_currently_publishing_unit == GENERATE_FOR);
}

symbol_table *
IIRScram::_get_symbol_table( ){
  ASSERT( _my_design_file != NULL );
  ASSERT( _my_design_file->_get_symbol_table() != NULL );

  return _my_design_file->_get_symbol_table();
}

void 
IIRScram::_report_undefined_scram_fn(char *fn_name) {
  ostringstream os;
  os << fn_name << " not defined for IIR node: " << get_kind_text();
  report_error( this, os.str() );
  abort();
}

IIR_Boolean
IIRScram::_is_readable() {
  _report_undefined_scram_fn("_is_readable()");
  return FALSE;
}

IIR_Boolean
IIRScram::_is_writable() {
  _report_undefined_scram_fn("_is_writable()");
  return FALSE;
}

void
IIRScram::_set_passed_through_out_port(IIR_Boolean) {
  _report_undefined_scram_fn("_set_passed_through_out_port(IIR_Boolean)");
}

IIR_Boolean 
IIRScram::_is_locally_static_primary(){
  _report_undefined_scram_fn("_is_locally_static_primary()");
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_globally_static_primary(){
  // If no overrided version of this got called, we're going to
  // fall back to calling "_is_locally_static_primary".  See 7.4.2 in the
  // '93 LRM for more info.
  return _is_locally_static_primary();
}

IIR_Boolean 
IIRScram::_is_textio() {
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_standard() {
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_relational_operator() {
  return FALSE;
}

IIR_Boolean 
IIRScram::_is_resolved_type() {
  return FALSE;
}

void 
IIRScram::_publish_vhdl( ostream & ){
  _report_undefined_scram_fn("_publish_vhdl(ostream &)");
}

void 
IIRScram::_publish_vhdl_range( ostream & ){
  _report_undefined_scram_fn("_publish_vhdl_range(ostream &)");
}
void 
IIRScram::_publish_vhdl_decl(ostream & ){
  _report_undefined_scram_fn("_publish_vhdl_decl(ostream &)");
}

void 
IIRScram::_publish_cc_condition( published_file & ){
  _report_undefined_scram_fn("_publish_cc_condition( _cc_out )");
}

void 
IIRScram::_publish_cc_name_elaborate( published_file & ){
  _report_undefined_scram_fn("_publish_cc_name_elaborate( _cc_out )");
}

void 
IIRScram::_publish_cc_init( published_file & ){
  _report_undefined_scram_fn("_publish_cc_init( _cc_out )");
}

void
IIRScram::_publish_cc_init_for_ams(published_file &_cc_out) {
  _report_undefined_scram_fn("_publish_cc_init_for_ams( _cc_out )");
}

const string
IIRScram::_get_cc_object_type(){
  _report_undefined_scram_fn("_get_cc_object_type()");
  return "";
}

void 
IIRScram::_publish_cc_object_name( published_file & ){
  _report_undefined_scram_fn("_publish_cc_object_name( _cc_out )");
}

void 
IIRScram::_publish_cc_initialization_value( published_file &_cc_out ) {
  _publish_cc_lvalue( _cc_out );
}

void
IIRScram::_publish_cc_universal_type( published_file & ){
  _report_undefined_scram_fn("_publish_cc_universal_type( _cc_out )");  
}

void
IIRScram::_publish_cc_kernel_type( published_file &pf ) {
  _publish_cc_kernel_type( pf.get_stream() );
}

void
IIRScram::_publish_cc_kernel_type( ostream &os ) {
  os << _get_cc_kernel_type();
}

const string
IIRScram::_get_cc_kernel_type() {
  if( get_subtype() != 0 ){
    return get_subtype()->_get_cc_kernel_type();
  }
  else{
    _report_undefined_scram_fn("_get_cc_kernel_type()");
    return "";
  }
}

void
IIRScram::_publish_cc_first_objectParameter( published_file & ){
  ASSERT ( _is_object() == TRUE );
  _report_undefined_scram_fn("_publish_cc_first_objectParameter( ostream &os )");
}

void 
IIRScram::_publish_cc_wait_data( published_file & ){
  _report_undefined_scram_fn("_publish_cc_wait_data( _cc_out )");
}

void
IIRScram::_publish_cc_data( published_file & ){
  _report_undefined_scram_fn("_publish_cc_data( _cc_out )");
}

void 
IIRScram::_publish_cc_decl( published_file & ){
  _report_undefined_scram_fn("_publish_cc_decl( _cc_out )");
}

void
IIRScram::_publish_cc_bounds( published_file & ) {
  _report_undefined_scram_fn("_publish_cc_bounds( _cc_out )");
}

void
IIRScram::_publish_cc_range( published_file & ){
  _report_undefined_scram_fn("_publish_cc_range( _cc_out )");
}

void
IIRScram::_publish_cc_value( published_file & ){
  _report_undefined_scram_fn("_publish_cc_value( _cc_out )");
}

void
IIRScram::_publish_cc_universal_value( published_file & ){
  _report_undefined_scram_fn("_publish_cc_universal_value( _cc_out )");
}

void
IIRScram::_publish_cc_binding_name( ostream & ){
  _report_undefined_scram_fn("_publish_cc_binding_name( _cc_out )");
}

void
IIRScram::_publish_cc_class_name( ostream& os ){
  // Publishes the name of the C++ class for this object if any...
  _publish_cc_binding_name(os);
  
  if ((get_kind() == IIR_ARCHITECTURE_DECLARATION) ||
      (get_kind() == IIR_ENTITY_DECLARATION) ||
      ( _is_iir_concurrent_statement() == TRUE && get_kind() != IIR_PROCESS_STATEMENT )) {
    os << "_elab";
  }
}

void
IIRScram::_publish_cc_declarator( published_file & ){
  _report_undefined_scram_fn("_publish_cc_declarator( _cc_out )");
}

const string
IIRScram::_get_cc_type_name(){
  _report_undefined_scram_fn("_get_cc_type_name()");
  return "";
}

void
IIRScram::_publish_cc_for_index( published_file & ){
  _report_undefined_scram_fn("_publish_cc_for_index( _cc_out )");
}

IIR_TextLiteral *
IIRScram::_get_declarator() {
  ASSERT(_is_iir_declaration() == TRUE || get_kind() == IIR_SIMPLE_NAME);
  _report_undefined_scram_fn("_get_declarator()");
  return NULL;
}

IIR_GenericList *
IIRScram::_get_generic_list(){
  _report_undefined_scram_fn("_get_generic_list()");
  return NULL;
}

IIR_PortList *
IIRScram::_get_port_list(){
  _report_undefined_scram_fn("_get_port_list()");
  return NULL;
}

IIR *
IIRScram::get_value() {
  return NULL;
}

IIR_List *
IIRScram::_get_statement_list(){
  _report_undefined_scram_fn("_get_statement_list()");

  return NULL;
}

IIR_Mode
IIRScram::_get_mode() {
  _report_undefined_scram_fn("_get_mode()");
  return IIR_UNKNOWN_MODE;
}

IIR_Boolean
IIRScram::_is_published_attribute_in_state(SignalAttribute) {
  _report_undefined_scram_fn("_is_published_attribute_in_state()");
  return FALSE;
}

void 
IIRScram::_add_published_attribute_in_state(SignalAttribute) {
  _report_undefined_scram_fn("_add_published_attribute_in_state()");
}

IIR_Boolean
IIRScram::_is_published_attribute_in_constructor(SignalAttribute) {
  _report_undefined_scram_fn("_is_published_attribute_in_constructor()");
  return FALSE;
}

void 
IIRScram::_add_published_attribute_in_constructor(SignalAttribute) {
  _report_undefined_scram_fn("_add_published_attribute_in_constructor()");
}

IIR_Boolean
IIRScram::_is_published_attribute_in_initstate(SignalAttribute) {
  _report_undefined_scram_fn("_is_published_attribute_in_initstate()");
  return FALSE;
}

void 
IIRScram::_add_published_attribute_in_initstate(SignalAttribute) {
  _report_undefined_scram_fn("_add_published_attribute_in_initstate()");
}

IIR_Boolean
IIRScram::_is_ascending_range() {
  _report_undefined_scram_fn("_is_ascending_range()");
  return false;
}

IIR * 
IIRScram::_transmute() {
  _report_undefined_scram_fn("_transmute()");
  return NULL;
}

ostream&
IIRScram::_print(ostream& os) {
  _report_undefined_scram_fn("ostream& _print(ostream&)");
  return os;
}

const string 
IIRScram::_to_string(){
  string retval = "";

  ostringstream stream;
  _print( stream );
  retval += stream.str();

  return retval;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup(){
  _report_undefined_scram_fn("_symbol_lookup( )");
  return NULL;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup( IIR_Declaration * ){
  _report_undefined_scram_fn("_symbol_lookup( IIR_Declaration * )");
  return NULL;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup( set<IIR_Declaration> * ){
  _report_undefined_scram_fn("_symbol_lookup( set<IIR_Declaration> * )");
  return NULL;
}

set<IIR_Declaration> *
IIRScram::_symbol_lookup( IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_Declaration> *retval = _symbol_lookup();

  if( retval != NULL ){
    retval->reduce_set( constraint_function );
  }

  return retval;
}

IIR_Label *
IIRScram::_lookup_label( IIR_Boolean complain_on_error ){
  IIR_Label *retval = NULL;
  
  ASSERT( complain_on_error == TRUE ||  complain_on_error == FALSE );
  
  set<IIR_Declaration> *decls = _symbol_lookup( &IIR::_is_iir_label );
  ASSERT( decls != NULL );
  if( decls->num_elements() == 0 && complain_on_error == TRUE ){
    report_undefined_symbol( (IIR *)this );
  }

  switch( decls->num_elements() ){
  case 0:{
    ostringstream err;
    err << "No label |" << *this << "| declared in this scope.";
    report_error( this, err.str() );
    break;
  }
  case 1:{
    IIR_Declaration *temp = decls->get_element();
    ASSERT( temp->_is_iir_label() == TRUE );
    retval = (IIR_Label *)temp;
    break;
  }
  default:{
    report_ambiguous_error( (IIR *)this, decls );
  }
  }

  return retval;
}

set<IIR_TypeDefinition> *
IIRScram::_get_rval_set( IIR_Boolean (IIR::*)() ){
  _report_undefined_scram_fn("_get_rval_set(),\nIIR_Boolean (IIR::*constraint_function)()");
  return NULL;
}

set<IIR_TypeDefinition> *
IIRScram::_get_rval_set( set<IIR_TypeDefinition> *,
			 IIR_Boolean (IIR::*)() ){
  _report_undefined_scram_fn("_get_rval_set( set<IIR_TypeDefinition> * ),\nIIR_Boolean (IIR::*constraint_function)()");
  return NULL;
}

set<IIR_TypeDefinition> *
IIRScram::_get_rval_set( set<IIR_Declaration> *, 
			 IIR_Boolean (IIR::*)() ){
  _report_undefined_scram_fn("_get_rval_set( set<IIR_Declaration> * ,\nIIR_Boolean (IIR::*constraint_function)() )");
  return NULL;
}

IIR_Declaration *
IIRScram::_find_in_implicit_list( const string to_find ){
  IIR_Declaration *current_decl;
  
  if( _get_implicit_declarations() != NULL ){
    current_decl = _get_implicit_declarations()->get_element();
    while( current_decl != NULL ){
      if( IIR_TextLiteral::_cmp( current_decl->get_declarator(), to_find ) ==0){
        return current_decl;
      }
      current_decl = _get_implicit_declarations()->get_next_element();
    }
  }

  return NULL;
}

set<IIR_Declaration> *
IIRScram::_get_implicit_declarations( ){
  _report_undefined_scram_fn("_get_implicit_declarations( )");
  return NULL;
}

void 
IIRScram::_set_implicit_declarations( set<IIR_Declaration> * ){
  _report_undefined_scram_fn("_set_implicit_declarations( set<IIR_Declaration> * )"); 
}

IIR_Declaration *
IIRScram::_find_formal_declaration(){
  _report_undefined_scram_fn("_find_formal_declaration( )");
  return NULL;
}

IIR_TypeDefinition *
IIRScram::_determine_rval_in_set( set<IIR_TypeDefinition> *, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_determine_rval_in_set( set<IIR_TypeDefinition> *, IIR_TypeDefinition * )");
  return NULL;  
}

IIR_Declaration *
IIRScram::_determine_decl_in_set( set<IIR_Declaration> *, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_determine_decl_in_set( set<IIR_Declaration> *, IIR_TypeDefinition *)");
  return NULL;  
}

IIR_TypeDefinition *
IIRScram::get_subtype(){
  // In general things that don't override get_subtype should return NULL.
  // Note that get_subtype can be called during error reporting; in that
  // case crashing with an undefined_scram_fn isn't too nice.
  //  _report_undefined_scram_fn("get_subtype()");
  return NULL;
}

IIR_TypeDefinition *
IIRScram::_get_rval_pointed_at(){
  IIR_TypeDefinition *retval = NULL;

  if( get_subtype()->_is_iir_access_type_definition() == TRUE ){
    retval = ((IIR_AccessTypeDefinition *)get_subtype())->get_designated_type();
  }

  return retval;
}

IIR_TypeDefinition *
IIRScram::_get_type_of_element( int ){
  _report_undefined_scram_fn("_get_type_of_element( int )");
  return NULL;
}

IIR_Int32 
IIRScram::_get_num_indexes( ){
  _report_undefined_scram_fn("_get_num_indexes()");
  return -1;
}

IIR_TypeDefinition *
IIRScram::_get_port_type( int ){
  _report_undefined_scram_fn("_get_port_type");
  return NULL;
}

void 
IIRScram::_type_check( set<IIR_TypeDefinition> * ){
  _report_undefined_scram_fn("_type_check( set<IIR_TypeDefinition> * )");
}

IIR *
IIRScram::_rval_to_decl( IIR_TypeDefinition *, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_rval_to_decl( IIR_TypeDefinition *prefix_rval, IIR_TypeDefinition *suffix_rval )");  
  return NULL;
}

IIR *
IIRScram::_rval_to_decl( IIR_Declaration *, IIR_TypeDefinition * ){
  _report_undefined_scram_fn("_rval_to_decl( IIR_Declaration *prefix_decl, IIR_TypeDefinition *suffix_rval )");  
  return NULL;
}

IIR *
IIRScram::_get_component_name( ){
  _report_undefined_scram_fn("_get_component_name()");
  return NULL;
}

void 
IIRScram::_set_component_name( IIR * ){
  _report_undefined_scram_fn("_set_component_name()");
}

IIR_LibraryUnit *
IIRScram::_get_entity_aspect(){
  _report_undefined_scram_fn("_get_entity_aspect()");
  return NULL;
}

IIR_LibraryUnit *
IIRScram::_find_default_binding( IIR *component_name ){
  ASSERT( component_name->get_kind() == IIR_SIMPLE_NAME ||
	  component_name->get_kind() == IIR_COMPONENT_DECLARATION );

  IIR_LibraryUnit *retval = NULL;

  IIR_EntityDeclaration *entity_decl = 
    library_manager::instance()->lookup_entity( FALSE, component_name, _get_work_library() );
  if( entity_decl != NULL ){
    // We need to go get the most recently analyzed architecure.
    retval = library_manager::instance()->lookup_default_architecture( entity_decl );
  }

  return retval;
}

void 
IIRScram::_publish_cc_elaborate( published_file & ){
  _report_undefined_scram_fn("_publish_cc_elaborate( _cc_out )");
}

void 
IIRScram::_publish_cc_subprogram_arg( published_file & ) {
  _report_undefined_scram_fn("_publish_cc_subprogram_arg( _cc_out )");
}

void 
IIRScram::_publish_cc_constructor_args( published_file & ){
  _report_undefined_scram_fn("_publish_cc_constructor_args( _cc_out )");
}

void 
IIRScram::_publish_cc_sigdest( published_file & ){
  _report_undefined_scram_fn("_publish_cc_sigdest( _cc_out )");
}

void
IIRScram::_publish_cc_headers( published_file & ){
  _report_undefined_scram_fn("_publish_cc_headers( _cc_out )");
}

void 
IIRScram::_publish_cc_headerfiles_for_cc_default( published_file &_cc_out ){
  ostringstream file_name_stream;
  _publish_cc_binding_name( file_name_stream );
  if( get_kind() != IIR_PROCESS_STATEMENT ){
    file_name_stream << "_elab";
  }
  file_name_stream << ".hh";
  _publish_cc_include( _cc_out, file_name_stream.str() );
}

void
IIRScram::_publish_cc_direction( published_file & ){
  _report_undefined_scram_fn("_publish_cc_direction( _cc_out )");
}

void
IIRScram::_publish_cc_addChild( published_file & ) {
  _report_undefined_scram_fn("_publish_cc_addChild( _cc_out )");
}  
  
void 
IIRScram::_get_list_of_input_signals( set<IIR> * ){
  _report_undefined_scram_fn("_get_list_of_input_signals()");
}

void 
IIRScram::_get_signal_source_info( set<IIR> * ){
  _report_undefined_scram_fn("_get_signal_source_info()");
}

IIR_AttributeSpecification*
IIRScram::_get_attribute_specification(IIR*) {
  _report_undefined_scram_fn("_get_attribute_specification()");
  return NULL;
}

IIR_SignalKind
IIRScram::_get_signal_kind() {
  _report_undefined_scram_fn("_get_signal_kind()");
  return IIR_NO_SIGNAL_KIND;
}

IIR_Boolean
IIRScram::_is_guard_signal() {
  if( _is_signal() == TRUE && _get_signal_kind() != IIR_NO_SIGNAL_KIND ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

void 
IIRScram::_build_sensitivity_list( IIR_DesignatorList * ){
  if( get_kind() != IIR_PHYSICAL_LITERAL &&
      get_kind() != IIR_ALLOCATOR ){
    _report_undefined_scram_fn("_build_sensitivity_list()");
  }
}

void
IIRScram::_build_above_attribute_set(set<IIR_AboveAttribute> *) {
  // do Nothing.
}

IIR_Declaration* 
IIRScram::_get_prefix_declaration() {
  _report_undefined_scram_fn("_get_prefix_declaration()");
  return NULL;
}

IIR_Declaration* 
IIRScram::_get_package_declaration() {
  return NULL;
}

const string 
IIRScram::_convert_to_library_name( ) {
  _report_undefined_scram_fn("_convert_to_library_name()");
  return "";
}

IIR_TextLiteral *
IIRScram::_get_prefix_string( ) {
  _report_undefined_scram_fn("_get_prefix_string()");
  return NULL;
}

void 
IIRScram::_make_interface_visible( symbol_table * ){
  _report_undefined_scram_fn("_make_interface_visible( symbol_table * )");
}

void 
IIRScram::_make_interface_visible( ){
  _make_interface_visible( _get_symbol_table() );
}

IIR *
IIRScram::_get_current_declarative_region(){
  ASSERT( _get_symbol_table() != NULL );

  return _get_symbol_table()->get_current_declarative_region();
}

#ifdef PROCESS_COMBINATION
void 
IIRScram::_static_elaborate(IIR_ArchitectureDeclaration*, 
			    IIR_DeclarationList*, char*) {
  _report_undefined_scram_fn("_static_elaborate()");
}
#endif

IIR*
IIRScram::_clone() {
  _report_undefined_scram_fn("IIR *_clone()");
  return NULL;
}

void
IIRScram::_clone(IIR* clone) {
  copy_location(this, clone);
}  

void 
IIRScram::_add_decl_into_cgen_symbol_table() {
  if(_is_literal() == FALSE && 
     _is_iir_element_declaration() == FALSE &&
     get_kind() != IIR_PHYSICAL_UNIT && 
     get_kind() != IIR_PHYSICAL_LITERAL &&
     get_kind() != IIR_TYPE_CONVERSION ){
    _report_undefined_scram_fn("_add_decl_into_cgen_symbol_table()");
  }
}

IIR_IntegerLiteral *
IIRScram::_get_integer_static_value(){
  _report_undefined_scram_fn("_get_integer_static_value()");
  return NULL;
}

IIR *
IIRScram::_decl_to_decl( IIR_Declaration * ){
  _report_undefined_scram_fn("_decl_to_decl( IIR_Declaration * )");
  return NULL;
}

void
IIRScram::_publish_vhdl_operator( ostream & ){
  _report_undefined_scram_fn("_publish_vhdl_operator(ostream &)");
}

IIRScram::OperatorPrecedenceLevel
IIRScram::_get_operator_precedence(){
  return IIRScram::OTHERS;
}

bool
IIRScram::_get_mangling_flag() const {
  return no_mangling;
}

IIR_FunctionDeclaration*
IIRScram::_get_resolution_function(){
  return NULL;
}

void 
IIRScram::_type_check_configuration( IIR_AssociationList &port_map_aspect,
				     IIR_AssociationList &generic_map_aspect,
				     int tmp_mode ){
  // Hack to work around #includeing this in the header.
  IIRScram_ConfigurationSpecification::type_check_mode mode = 
    (IIRScram_ConfigurationSpecification::type_check_mode)tmp_mode;

  _get_symbol_table()->open_scope( (IIR *)this );
  set<IIR_Declaration> *component_decls;
  component_decls = _get_component_name()->_symbol_lookup( &IIR::_is_iir_component_declaration );
  if( component_decls == NULL ){
    report_undefined_symbol( _get_component_name() );
    return;
  }

  IIR_ComponentDeclaration *my_component = 0;
  switch( component_decls->num_elements() ){
  case 0:{
    ostringstream err;
    err << "|" << *_get_component_name() 
	<< "| was not declared as a component in this scope.";
    report_error( this, err.str() );
    return;
  }
  case 1:{
    my_component = (IIR_ComponentDeclaration *)component_decls->get_element();
    _set_component_name( _get_component_name()->_decl_to_decl( my_component ) );
    if( debug_symbol_table == true ){
      cerr << "Type checking configuration - about to make |" << 
	*_get_component_name() << "| visible." << endl;
    }
    my_component->_make_interface_visible( _get_symbol_table() );
    break;
  }
  default:{
    report_ambiguous_error( _get_component_name(), component_decls );
    break;
  }
  }

  // This currently gets resolved by the parser...
  IIR *aspect = _get_entity_aspect();
  if( aspect != NULL ){
    ASSERT( aspect->_is_iir_declaration() == TRUE );
    ASSERT( aspect->_is_resolved() == TRUE );
    IIR_Declaration *aspect_decl = (IIR_Declaration *)aspect;
    
    // The formal generics and ports of the entity must be visible here.
    // See section 10.2 of the LRM.  So, now we're deciding what entity
    // we're looking at, and using this info to put those formals back
    // into scope.
    IIR_EntityDeclaration *entity_decl = NULL;
    switch( aspect_decl->get_kind() ){
      
    case IIR_ARCHITECTURE_DECLARATION:{
      IIR_ArchitectureDeclaration *as_arch = (IIR_ArchitectureDeclaration *)aspect_decl;
      entity_decl = as_arch->get_entity();
      break;
    }
    case IIR_ENTITY_DECLARATION:{
      entity_decl =  (IIR_EntityDeclaration *)aspect_decl;
      break;
    }
    case IIR_CONFIGURATION_DECLARATION:{
      entity_decl = ((IIR_ConfigurationDeclaration *)aspect_decl)->get_entity();
      break;
    }
    default:{
      ostringstream err;
      err << "Internal error in IIRScram_ConfigurationSpecification::_type_check"
	  << " - got a " << aspect_decl->get_kind_text() 
	  << ", expecting architecture or entity.";
      report_error( this, err.str() );
      abort();
    }
    }
    
    _resolve_map( entity_decl->_get_port_list(),
		  my_component->_get_port_list(),
		  port_map_aspect,
		  mode );

    _resolve_map( entity_decl->_get_generic_list(),
		  my_component->_get_generic_list(),
		  generic_map_aspect,
		  mode );
  }
  _get_symbol_table()->close_scope( (IIR *)this );
  // else it's OPEN
  //  ASSERT( _is_resolved() == TRUE );
  delete component_decls;
}

void
IIRScram::_resolve_map( IIR_InterfaceList *entity_interface_list,
			IIR_InterfaceList *component_interface_list,
			IIR_AssociationList &map,
			int tmp_mode ){
  // Hack to work around #includeing this in the header.
  IIRScram_ConfigurationSpecification::type_check_mode mode = 
    (IIRScram_ConfigurationSpecification::type_check_mode)tmp_mode;

  if( entity_interface_list && map.num_elements() > 0 ){
    if( mode == IIRScram_ConfigurationSpecification::CONFIG_SPEC ){
      map._resolve_and_order( entity_interface_list,
			      component_interface_list,
			       static_cast<IIR*>(this) );
    }
    else{
      ASSERT( mode == IIRScram_ConfigurationSpecification::COMPONENT_INSTANT );
      map._resolve_and_order( component_interface_list,
			      entity_interface_list,
			       static_cast<IIR*>(this) );
    }
  }
  else{
    if( mode == IIRScram_ConfigurationSpecification::CONFIG_SPEC ){
      build_default_map( *entity_interface_list,
			 *component_interface_list,
			 map );
    }
    else{
      ASSERT( mode == IIRScram_ConfigurationSpecification::COMPONENT_INSTANT );
      build_default_map( *component_interface_list,
			 *entity_interface_list,
			 map );
    }
  }
}

IIR_TypeDefinition *
IIRScram::_type_check_iteration_scheme( IIR_ConstantDeclaration *iteration_scheme ){
  IIR_TypeDefinition *retval = NULL;

  ASSERT( iteration_scheme != NULL );
  IIR_TypeDefinition *range_type = iteration_scheme->get_subtype();
  ASSERT( iteration_scheme->get_subtype() != NULL );

  if( range_type->get_kind() == IIR_RANGE_TYPE_DEFINITION ){
    IIR_RangeTypeDefinition *temp = (IIR_RangeTypeDefinition *)range_type;
    IIR_TypeDefinition *constant_base_type =
      IIR_ScalarTypeDefinition::_determine_discrete_type( temp );
    if( constant_base_type != NULL ){
      retval = constant_base_type->_construct_new_subtype( NULL, temp );
    }
  }
  else{
    // else it should be resolved.
    ASSERT( iteration_scheme->get_subtype()->_is_resolved() == TRUE );
    retval = iteration_scheme->get_subtype();
  }

  return retval;
}

IIR *
IIRScram::_get_enclosing_scope() {
  _report_undefined_scram_fn("_get_enclosing_scope()");
  return NULL;
}

IIR_LibraryDeclaration *
IIRScram::_get_work_library(){
  ASSERT( _my_design_file != 0 );
  return _my_design_file->_get_work_library();
}

void 
IIRScram::copy_location ( const IIRScram *source, IIRScram *dest ){
  ASSERT( source != NULL || parse_error == true  );
  ASSERT( dest != NULL || parse_error == true );
  ASSERT( source->_get_design_file() != NULL || parse_error == true );
  ASSERT( source->get_line_number() != -1 || parse_error == true );

#ifdef EXTRA_LOCATERS
  ASSERT( source->get_column_offset() != -1 || parse_error == true );
  ASSERT( source->get_character_offset() != -1 || parse_error == true );
#endif

#ifdef EXTRA_LOCATERS
  dest->_set_base_info( source->_get_design_file(), source->get_line_number(),
			source->get_column_offset(), source->get_character_offset() );
#else
  dest->_set_base_info( source->_get_design_file(), source->get_line_number(), 0 ,0 );
#endif
}

char *
IIRScram::_strdup( const char * const to_copy, unsigned int size ){
  if( size == 0 ){
    size = strlen( to_copy );
  }
  size++;

  char *retval = new char[size];
  memcpy( retval, to_copy, size - 1 );
  retval[ size ] = '0';

  return retval;
}

void 
IIRScram::_set_base_info( IIR_DesignFile *design_file,
			  int line_number,
			  int column_offset, 
			  int character_offset ){
  ASSERT( design_file != NULL );
  (void)column_offset;
  (void)character_offset;

  _set_design_file( design_file );
  set_line_number( line_number );

#ifdef EXTRA_LOCATERS
  set_column_offset( column_offset );
  set_character_offset( character_offset );
#endif
}

void
IIRScram::_copy_symbols_defined_in_enclosing_scope(symbol_table *current, symbol_table *outer) {
  
  IIR_Declaration *decl;

  decl = current->in_scope_by_type[IIRScram_Declaration::CONSTANT].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::CONSTANT].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }

  decl = current->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::INTERFACE_SIGNAL].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::VARIABLE].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::VARIABLE].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::ALIAS].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::ALIAS].successor(decl);
  }
  
  decl = current->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::ATTRIBUTE].successor(decl);
  }

  decl = current->in_scope_by_type[IIRScram_Declaration::S_FILE].first();
  while (decl != NULL) {
    if ((decl->_get_declarative_region() != this) && (!outer->in_scope(decl))){
      outer->add_declaration(decl);
    }
    decl = current->in_scope_by_type[IIRScram_Declaration::S_FILE].successor(decl);
  }
}

void
IIRScram::_group_component_instantiations(IIR_ArchitectureStatementList* conc_stmt_list, int blockSize) {
  // This method groups (blockSize) number of component instantiations into
  // a single implicit block. This was done to reduce the size of the published
  // individual elaboration class. Just a neat trick to get things going -DJ
  int componentInstantiationCount = conc_stmt_list->_get_number_of_component_instantiations();
  int componentCounter                 = 0;
  IIR_BlockStatement *newBlock         = NULL;
  IIR_ArchitectureStatement* conc_stmt = conc_stmt_list->first();
  IIR_ArchitectureStatement* next_stmt = NULL;
  
  while (componentInstantiationCount > blockSize) {
    componentCounter = 0;
    newBlock         = new IIR_BlockStatement;
    copy_location( this, newBlock );
    newBlock->_set_enclosing_scope( (IIR *) this );
    
    while ((componentCounter < blockSize) && (conc_stmt != NULL))  {
      next_stmt = conc_stmt_list->successor(conc_stmt);
      if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
	conc_stmt_list->remove(conc_stmt);
	
	newBlock->block_statement_part.append(conc_stmt);
	conc_stmt->_set_enclosing_scope(newBlock);
	
	componentCounter++;
	componentInstantiationCount--;
	
	ASSERT ( conc_stmt_list->_get_number_of_component_instantiations() ==
		 componentInstantiationCount );
      }
      
      conc_stmt = next_stmt;
    }

    conc_stmt_list->append(newBlock);
  }
}

void
IIRScram::_publish_cc_class_includes( published_file &_cc_out, 
				      IIR_ArchitectureStatementList* conc_stmt_list) {
  ASSERT(conc_stmt_list != NULL);
  ASSERT( _is_iir_concurrent_statement() == TRUE ||
	  _is_entity_decl() == TRUE ||
	  _is_component_decl() == TRUE ||
	  get_kind() == IIR_ARCHITECTURE_DECLARATION );
	  
  IIR_ArchitectureStatement* conc_stmt = conc_stmt_list->first();
  while( conc_stmt != NULL ){
    _cc_out << "class ";
    if( conc_stmt->get_kind() == IIR_PROCESS_STATEMENT ){
      if( conc_stmt->_get_label() != NULL ){
	_cc_out.get_stream() << *(conc_stmt->_get_label());
      }
      else {
	_cc_out << "ANON_PROCESS" << conc_stmt;
      }
      _cc_out << ";\n";
    }
    else {
      _cc_out << conc_stmt->_get_cc_elaboration_class_name() << ";\n";
    }
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
}

void
IIRScram::_publish_cc_signals( published_file &_cc_out,
			       IIR_DeclarationList* decl_list) {
  ASSERT(decl_list != NULL);
  ASSERT( _is_iir_concurrent_statement() == TRUE ||
	  _is_entity_decl() == TRUE ||
	  _is_component_decl() == TRUE ||
	  get_kind() == IIR_ARCHITECTURE_DECLARATION );
  
  IIR_Declaration* decl = decl_list->first();
  while (decl != NULL) {
    if(decl->_is_signal() == TRUE) {
      _cc_out << decl->get_subtype()->_get_cc_type_name();
      _cc_out << " ";
      decl->_publish_cc_elaborate( _cc_out );
      _cc_out << ";\n";
      
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL &&
	  decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->_is_signal() == TRUE) {
	    imp_decl->_publish_cc_implicit_signal_type( _cc_out );
	    _cc_out << " ";
	    imp_decl->_publish_cc_elaborate( _cc_out );
	    _cc_out << ";\n";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(decl->_is_signal() == TRUE) {
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_definition( _cc_out );
      }
    }
    decl = decl_list->successor(decl);    
  }
}

void
IIRScram::_publish_cc_object_pointers( published_file &_cc_out, 
				       IIR_ArchitectureStatementList* conc_stmt_list ){

  IIR_ArchitectureStatement *conc_stmt = conc_stmt_list->first();
  while (conc_stmt != NULL) {
    if( conc_stmt->get_kind() == IIR_PROCESS_STATEMENT ){
      IIR_ProcessStatement *process_statement = static_cast<IIR_ProcessStatement *>(conc_stmt);
      const string process_name = process_statement->_get_cc_process_class();
      // If the concurrent statement is a concurrent generate for
      // statement, then we need to instantiate multiple process. So we
      // declare a process_class ** array so that process_class[i] contains
      // the process instantiated seperately and not deleted in the
      // destructor. Because the processes are deleted in the main.cc of
      // the generate code.
      _cc_out << process_name;
      if ( get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT ){
	_cc_out << " **";
      }
      else {
	_cc_out << " *";
      }
      _cc_out << process_name << "_elab_obj;" << NL();
    }
    else if( conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT ||
	     conc_stmt->_is_block_statement() ||
	     conc_stmt->_is_concurrent_generate_statement() ){
      _cc_out << conc_stmt->_get_cc_elaboration_class_name() << " *";
      conc_stmt->_get_label()->_publish_cc_elaborate( _cc_out.get_stream() );
      _cc_out << "_elab_obj;" << NL();
    }
    else if( conc_stmt->get_kind() == IIR_SIMPLE_SIMULTANEOUS_STATEMENT ||
	     conc_stmt->get_kind() == IIR_SIMULTANEOUS_IF_STATEMENT ) {
      // No action required
    }
    else {
      ostringstream err;
      err << "ERROR! IIRScram_BlockStatement::_publish_cc_object_pointers( _cc_out ):"
	  << " unknown concurrent statement type |" 
	  << conc_stmt->get_kind_text() << "| in arch";
      report_error( this, err.str() );
    }
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
}

void
IIRScram::_publish_cc_object_pointers_init( published_file &_cc_out,
					    IIR_ArchitectureStatementList* conc_stmt_list,
					    IIR_DeclarationList* decl_list) {

  // Publish pointers from entity statement part
  //  entitydecl->_publish_cc_object_pointers_init( _cc_out );
  // Publish pointers from architecture statement part
  IIR_ArchitectureStatement *conc_stmt = conc_stmt_list->first();
  while(conc_stmt != NULL) {
    conc_stmt->_publish_cc_concurrent_stmt_init( _cc_out, decl_list );
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
}

void
IIRScram::_publish_cc_destructor( published_file &_cc_out, 
				  IIR_ArchitectureStatementList* conc_stmt_list) {
  IIR_ArchitectureStatement* conc_stmt = conc_stmt_list->first();
  _cc_out << _get_cc_elaboration_class_name() << "::~"
	  << _get_cc_elaboration_class_name() << "(){\n";
  while (conc_stmt != NULL) {
    switch(conc_stmt->get_kind()) {
    case IIR_PROCESS_STATEMENT:
      if(get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
	_cc_out << "delete ";
	_cc_out << " [] ";
	
	if (conc_stmt->_get_label() != NULL) {
	  _cc_out.get_stream() << *(conc_stmt->_get_label());
	}
	else {
	  _cc_out << "ANON_PROCESS" << conc_stmt;
	}
	_cc_out << "_elab_obj;\n";
      }
      break;
      
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
    case IIR_BLOCK_STATEMENT:
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      _cc_out << "delete ";
      if( get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT ){
	_cc_out << " [] ";
      }
      conc_stmt->_get_label()->_publish_cc_elaborate( _cc_out.get_stream() );
      _cc_out << "_elab_obj;\n";
      break;
    case IIR_SIMPLE_SIMULTANEOUS_STATEMENT:
    case IIR_SIMULTANEOUS_IF_STATEMENT:
      // No need to do anything for this case
      break;
    default:
      cerr << "ERROR!!!!! _publish_cc_destructor( _cc_out ): unknown conc_statement "
	   << "type |" << conc_stmt->get_kind_text() << "| in arch" << endl;
    }
    
    conc_stmt = conc_stmt_list->successor(conc_stmt);
  }
  _cc_out << "}\n";
}

void
IIRScram::_publish_cc_anonymous_drivers( published_file &_cc_out, IIR_DeclarationList* decl_list ){
  const string old_current_publish_name = _get_current_publish_name();
  _set_current_publish_name( ", SourceBase::ANONYMOUS_PROCESS_ID);\n" );
  
  SCRAM_CC_REF( _cc_out, "IIRScram::_publish_cc_anonymous_drivers" );

  IIR_Declaration *current_decl = decl_list->first();
  while( current_decl != NULL ){
    switch( current_decl->get_kind() ){
    case IIR_SIGNAL_DECLARATION:{
      if ( static_cast<IIR_SignalDeclaration *>(current_decl)->get_value() &&
	   !static_cast<IIR_SignalDeclaration *>(current_decl)->_is_passed_through_out_port() ){
	current_decl->_publish_cc_addChild( _cc_out );
      }
      break;
    }
    case IIR_SIGNAL_INTERFACE_DECLARATION:{
      if ( static_cast<IIR_SignalInterfaceDeclaration *>(current_decl)->get_value() ){
	current_decl->_publish_cc_addChild( _cc_out );
      }
      break;
    }
    default:
      break;
    }
    current_decl = decl_list->successor(current_decl);
  }

  _set_current_publish_name( old_current_publish_name );
}

void
IIRScram::_publish_cc_copy_generics_to_globals( published_file &_cc_out ) {
  ASSERT ( _is_iir_concurrent_statement() == TRUE ||
	   get_kind() == IIR_ENTITY_DECLARATION );
  if (_get_generic_list() != NULL && _get_generic_list()->first() != NULL) {
    _cc_out << "void" << NL()
	    << _get_cc_elaboration_class_name()
	    << OS("::copyGenericsToGlobals(){");
    _get_generic_list()->_publish_cc_copy_generics_to_globals( _cc_out );
    _cc_out << CS("}");
  }
}

IIR_AttributeSpecificationList * 
IIRScram::_get_attribute_specification_list( ){
  return NULL;
}

IIR_Boolean 
IIRScram::_attach_attribute_specification( IIR_AttributeSpecification *to_attach ){
  IIR_AttributeSpecificationList *list = _get_attribute_specification_list();
  if( list == NULL ){
    return FALSE;
  }
  else{
    list->append( to_attach );
    return TRUE;
  }
}

IIR_Boolean 
IIRScram::_attach_disconnection_specification( IIR_AttributeSpecification * ){
  _report_undefined_scram_fn("_attach_disconnection_specification( IIR_AttributeSpecification *)");
  abort();
  return FALSE;
}

IIR_DeclarationList*
IIRScram::_get_declaration_list() {
  _report_undefined_scram_fn("_get_declaration_list()");
  return NULL;
}

void
IIRScram::_publish_cc_reference_info( published_file &_cc_out, char *method_info, int line_number ){
  if (gen_cc_ref == TRUE) {
    string info( method_info + string("(") + intToString( line_number ) + ") on ");
    if( get_file_name() == NULL ){
      info += "NULL";
    }
    else{
      info += get_file_name()->_to_string();
    }
    info += "(" + intToString( get_line_number() ) +")";
    _cc_out.insert_comment( info );
  }
}

void
IIRScram::_publish_make_reference_info( published_file &_cc_out,
					char *method_info, int line_number ){
  if( gen_cc_ref == true ){
    _cc_out << "\n# " << method_info << "(" << line_number << ") on ";
    if( get_file_name() == NULL ){
      _cc_out << "NULL";
    }
    else{
      _cc_out.get_stream() << *get_file_name();
    }
    _cc_out << "(" << get_line_number() << ")\n";
  }
}

void
IIRScram::_publish_cc_dump( published_file &_cc_out ){

  SCRAM_CC_REF( _cc_out, "IIRScram::_publish_cc_dump" );;

  _cc_out.get_stream() << "// Kind = " << get_kind_text() << endl;
}

void 
IIRScram::_publish_cc_include( published_file & ){
  // By default we're going to do nothing...
} 

void 
IIRScram::_publish_cc_include_decls( published_file & ){
  _report_undefined_scram_fn( "_publish_cc_include_decls" );
} 

void 
IIRScram::_publish_cc_include_elab( published_file & ){
  _report_undefined_scram_fn( "_publish_cc_include_elab" );
} 
			       

void 
IIRScram::_publish_cc_include( published_file &os, 
			       const string &file_to_include,
			       bool system_include ){
  //  _publish_cc_include( os, file_to_include, os.get_file_name( os ), system_include );
  os.add_include( file_to_include, system_include );
}

void
IIRScram::_publish_cc_ams_function(published_file &) {
  _report_undefined_scram_fn("_publish_cc_ams_function()");
}

include_manager *
IIRScram::get_include_manager(){
  // lazy initialization
  static include_manager *my_include_manager = new include_manager();

  return my_include_manager;
}

const string
IIRScram::_get_cc_elaboration_class_name(){
  string retval;

  ostringstream binding_name_stream;
  _publish_cc_binding_name( binding_name_stream );
  retval += binding_name_stream.str();
  retval += "_elab";
  
  return retval;
}

IIR_ArchitectureDeclaration *
IIRScram::_get_cc_instantiated_architecture(){
  _report_undefined_scram_fn("_get_cc_instantiated_architecture()");
  return 0;
}

IIR_EntityDeclaration *
IIRScram::_get_cc_instantiated_entity(){
  _report_undefined_scram_fn("_get_cc_instantiated_entity()");
  return 0;
}

IIR_Int32
IIRScram::_get_stmt_signal_index() {
  return -1;
}

void
IIRScram::_set_stmt_node_index(IIR_Int32 *, bool, bool&) {
  // No need to do anything
  return;
}

void 
IIRScram::_set_stmt_qty_index(IIR_Int32 *, set<IIR_Declaration> *) {
  // No neeed to do anything
  return;
}

void
IIRScram::_flush_stmt_index() {
  // No need to do anything
  return;
}

void 
IIRScram::_set_stmt_signal_index(IIR_Int32 *, set<IIR_Declaration> *) {
  // No need to do anything
  return;
}

IIR_Boolean
IIRScram::_is_branchQ() {
  return FALSE;
}

IIR_Boolean
IIRScram::_reference_quantity_found() {
  return FALSE;
}

IIR_Boolean
IIRScram::_contribution_quantity_found() {
  return FALSE;
}

IIR_Boolean
IIRScram::_differential_quantity_found() {
  return FALSE;
}

void
IIRScram::_build_generic_parameter_set(set<IIR_Declaration> *) {
  // do nothing
}

void
IIRScram::_build_contribution_quantity_list(dl_list<IIR_ContributionAttribute> *) {
  // do nothing
}

void
IIRScram::_build_reference_quantity_list(dl_list<IIR_ReferenceAttribute> *) {
  // do nothing
}

void
IIRScram::_build_differential_quantity_list(dl_list<IIR_DotAttribute> *) {
  // do nothing
}

void
IIRScram::_publish_cc_savantnow_process_ptr( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram::_publish_cc_savant_now_process_ptr" );
  _publish_cc_savantnow( _cc_out, "processPtr->" );
}

void
IIRScram::_publish_cc_savantnow_no_process_ptr( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram::_publish_cc_savant_now_no_process_ptr" );
  _publish_cc_savantnow( _cc_out, "" );
}

void
IIRScram::_publish_cc_array_index( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, "IIRScram::_publish_cc_array_index" );
  _publish_cc_lvalue( _cc_out );
}

void
IIRScram::_publish_cc_savantnow( published_file &_cc_out, const string &prefix ){
  _cc_out << "#ifdef savantnow" << NL()
	  << "#undef savantnow" << NL()
	  << "#define savantnow (PhysicalType(ObjectBase::VARIABLE,"
	  << "UniversalLongLongInteger(" << prefix << "getTimeNow().getMajor()), "
	  << "SavanttimeType_info))" << NL()
	  << "#endif" << NL();
}

void
IIRScram::_publish_cc_subprogram_call_name( published_file &_cc_out ){
  _report_undefined_scram_fn("_publish_cc_subprogram_call_name");
}


published_file &
operator<<(published_file &pf, IIRScram &is ){
  is._print( pf.get_stream() );
  return pf;
}

