
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Radharamanan Radhakrishnan
//          Narayanan Thondugulam
//          Swaminathan Subramanian 
//	    Magnus Danielson	cfmd@swipnet.se
//	    Harish Venkataramani venkath@ececs.uc.edu
//          Sivakumar Gowrisankar gowriss@ececs.uc.edu

#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_AssociationElement.hh"
#include "IIR_Attribute.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_BranchQuantityDeclaration.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_ConcurrentGenerateIfStatement.hh"
#include "IIR_ConcurrentGenerateForStatement.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_Designator.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_FileDeclaration.hh"
#include "IIR_FreeQuantityDeclaration.hh"
#include "IIR_TextLiteral.hh"
#include "IIR_Label.hh"
#include "IIR_Name.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_QuantityDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_UseClause.hh"
#include "IIR_WaitStatement.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "published_header_file.hh"
#include "published_cc_file.hh"
#include "sstream-wrap.hh"
#include "language_processing_control.hh"

extern symbol_table *cgen_sym_tab_ptr;
extern symbol_table *cgen_arch_sym_tab_ptr;
extern language_processing_control *lang_proc;

using std::ends;

#ifdef PROCESS_COMBINATION
#include "ProcessCombiner.hh"
extern int num_partitions;
extern char *partition_algorithm;
extern void cone_partition( IIR_ArchitectureDeclaration*, 
			    IIR_ArchitectureStatementList&, ProcessCombiner* );
#include <cstdlib> // for rand()
#endif

#ifdef PROCESS_GRAPH
extern bool signal_graph;
#endif

IIRScram_ArchitectureDeclaration::IIRScram_ArchitectureDeclaration() : cgen_arch_sym_tab( 4093, false ){
  cgen_sym_tab_ptr = NULL;
}

IIRScram_ArchitectureDeclaration::~IIRScram_ArchitectureDeclaration() {}

void 
IIRScram_ArchitectureDeclaration::_publish_vhdl_decl(ostream &_vhdl_out) {
  PublishedUnit oldUnit = _get_currently_publishing_vhdl_unit();

  _set_currently_publishing_vhdl_unit(IIRScram::ARCHITECTURE_DECL);
  
  context_items._publish_vhdl(_vhdl_out);

  _vhdl_out << "architecture ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out  << " of ";
  get_entity()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " is\n";
  architecture_declarative_part._publish_vhdl_decl(_vhdl_out);
  _vhdl_out << "\nbegin\n";
  architecture_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << "end architecture ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ";\n\n";

  _set_currently_publishing_vhdl_unit(oldUnit);
}

void 
IIRScram_ArchitectureDeclaration::_publish_vhdl(ostream &_vhdl_out) {
  get_declarator()->_publish_vhdl(_vhdl_out);
}

void
IIRScram_ArchitectureDeclaration::_publish_vhdl_binding_name(ostream &_vhdl_out){
  _vhdl_out << " entity ";
  get_entity()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "(";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ") ";
}

void
IIRScram_ArchitectureDeclaration::_publish_vhdl_with_library_name(ostream &_vhdl_out){
  _vhdl_out << " entity ";
  ASSERT(_get_declarative_region() != NULL);
  ASSERT(_get_declarative_region()->get_kind() == IIR_LIBRARY_DECLARATION);
  _get_declarative_region()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ".";
  get_entity()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "(";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ")";
}

void 
IIRScram_ArchitectureDeclaration::_type_check( ){
  architecture_declarative_part._type_check_configuration_specifications( architecture_statement_part );
  architecture_declarative_part._type_check_disconnection_specifications(  );
  architecture_declarative_part._type_check_attribute_specifications( architecture_statement_part );
  architecture_statement_part._type_check_instantiate_statements();
}

IIR_PortList *
IIRScram_ArchitectureDeclaration::_get_port_list(){
  ASSERT( get_entity() != NULL );
  return get_entity()->_get_port_list();
}

IIR_GenericList *
IIRScram_ArchitectureDeclaration::_get_generic_list(){
  ASSERT( get_entity() != NULL );
  return get_entity()->_get_generic_list();
}

IIR_List *
IIRScram_ArchitectureDeclaration::_get_statement_list(){
  return &architecture_statement_part;
}

IIR_EntityDeclaration*
IIRScram_ArchitectureDeclaration::_get_entity() {
  return get_entity();
}

IIR_ArchitectureDeclaration *
IIRScram_ArchitectureDeclaration::_get_cc_instantiated_architecture(){
  return static_cast<IIR_ArchitectureDeclaration *>(this);
}
 
IIR_EntityDeclaration *
IIRScram_ArchitectureDeclaration::_get_cc_instantiated_entity(){
  return _get_entity();
}

void 
IIRScram_ArchitectureDeclaration::_publish_cc(){
  IIR* temp = _get_current_publish_node();
  const string old_current_publish_name = _get_current_publish_name();
  symbol_table cgen_sym_tab( 4093, false );

  cgen_sym_tab_ptr = &cgen_sym_tab;
  cgen_arch_sym_tab_ptr = &cgen_arch_sym_tab;

  _set_currently_publishing_unit(ARCHITECTURE_DECL);
  _set_current_publish_node( this );
  _set_current_entity_name( get_entity()->_get_declarator()->_convert_to_string() );
  _set_current_architecture_name( _get_declarator()->_convert_to_string() );

  cerr << "Publishing design unit: " << _get_current_entity_name()
       << "(" << _get_current_architecture_name() << ")\n";

  _set_current_publish_name( "" );

  // Check and group component instantiation statements for large designs
  _group_component_instantiations(&architecture_statement_part, 300);
  
  cgen_sym_tab_ptr = &cgen_sym_tab;
  _publish_cc_declarations();
  
  _set_current_publish_name( old_current_publish_name );
  _set_current_publish_node( temp );
  
  _set_currently_publishing_unit(NONE);

  cgen_sym_tab_ptr = NULL;
}

void 
IIRScram_ArchitectureDeclaration::_publish_cc_declarations(){
  // Publish the header file
  published_header_file header_file( _get_library_directory(), 
				     _get_cc_design_unit_name(),
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_ArchitectureDeclaration::_publish_cc_declarations" );
  _publish_cc_include_decls( header_file );
  context_items._publish_cc( header_file );
  IIRScram::_publish_cc_include( header_file, "tyvis/STDTypes.hh" );

  SCRAM_CC_REF( header_file, "IIRScram_ArchitectureDeclaration::_publish_cc_declarations" );
  architecture_declarative_part._publish_cc( header_file );
  
  architecture_statement_part._publish_cc_ams_function_prototype(header_file);
  // Publish the body
  published_cc_file cc_file( _get_library_directory(), 
			     _get_cc_design_unit_name(),
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_ArchitectureDeclaration::_publish_cc_declarations" );
  _publish_cc_include_decls( cc_file );
  SCRAM_CC_REF( cc_file, "IIRScram_ArchitectureDeclaration::_publish_cc_declarations" );
  _publish_cc_include_elab( cc_file );
  IIRScram::_publish_cc_include( cc_file, "tyvis/VHDLProcess.hh" );
  SCRAM_CC_REF( cc_file, "IIRScram_ArchitectureDeclaration::_publish_cc_declarations" );
  architecture_declarative_part._publish_cc_decl( cc_file );
  SCRAM_CC_REF( cc_file, "IIRScram_ArchitectureDeclaration::_publish_cc_declarations" );
  architecture_statement_part._publish_cc( cc_file );
  architecture_statement_part._publish_cc_ams_function_body(cc_file);
  IIR_ConcurrentStatement *conc_stmt = static_cast <IIR_ConcurrentStatement *>
    (architecture_statement_part.first());
  while(conc_stmt != NULL) {
    if(conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      (dynamic_cast <IIR_ProcessStatement *>(conc_stmt))->
        _publish_cc_ams_tick_above_function_prototype(header_file);
      (dynamic_cast <IIR_ProcessStatement *>(conc_stmt))->
        _publish_cc_ams_breakExpression_function_prototype(header_file);
    }
    conc_stmt =  static_cast <IIR_ConcurrentStatement *>
      (architecture_statement_part.successor(conc_stmt));
  }
  conc_stmt = static_cast <IIR_ConcurrentStatement *>
    (architecture_statement_part.first());
  while(conc_stmt != NULL) {
    if(conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      (dynamic_cast <IIR_ProcessStatement *>(conc_stmt))->
        _publish_cc_ams_break_function_include(cc_file);
      (dynamic_cast <IIR_ProcessStatement*>(conc_stmt))->
        _publish_cc_ams_breakExpression_function_body(cc_file);
      (dynamic_cast <IIR_ProcessStatement *>(conc_stmt))->
        _publish_cc_ams_tick_above_function_body(cc_file);
    }
    conc_stmt =  static_cast <IIR_ConcurrentStatement *>
      (architecture_statement_part.successor(conc_stmt));
  }
}  

IIRScram_Declaration::declaration_type 
IIRScram_ArchitectureDeclaration::_get_type(){
  return ARCHITECTURE;
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_elaborate(){
  IIR* temp_node                     = _get_current_publish_node();
  const string old_current_publish_name = _get_current_publish_name();
  PublishedUnit old_publish_unit     = _get_currently_publishing_unit();

  cgen_arch_sym_tab_ptr = &cgen_arch_sym_tab;
  _set_current_publish_node( this );
  _set_current_publish_name( "" );
  _set_currently_publishing_unit(ARCHITECTURE_DECL);
  _set_current_configuration_name( "" );

  _set_current_entity_name( get_entity()->_get_declarator()->_convert_to_string() );
  _set_current_architecture_name( _get_declarator()->_convert_to_string() );
  
  _publish_cc_headerfile();
  _publish_cc_ccfile();

  cgen_arch_sym_tab_ptr = NULL;
  _set_current_publish_node( temp_node );
  _set_current_publish_name( old_current_publish_name );
  _set_currently_publishing_unit(old_publish_unit);
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_blocks_elaborate( published_file &_cc_out ) {
  IIR_ArchitectureStatement* arch_stmt = NULL;
  const string temp = _get_current_publish_name();
 
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_blocks_elaborate" );

  _set_current_publish_name( "" );
  
  arch_stmt = architecture_statement_part.first();
  while (arch_stmt != NULL) {
    if (arch_stmt->get_kind() == IIR_BLOCK_STATEMENT) {
      arch_stmt->_publish_cc_elaborate( _cc_out );
    }
    if (arch_stmt->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
      arch_stmt->_publish_cc_elaborate( _cc_out );
    }
    if (arch_stmt->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT) {
      arch_stmt->_publish_cc_elaborate( _cc_out );
    }
    
    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }
  _set_current_publish_name( temp );
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_headerfile(){

  // Publish the .hh file
  
  published_header_file header_file( _get_library_directory(), 
				     _get_cc_elaboration_class_name(),
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_ArchitectureDeclaration::_publish_cc_headerfile" );
  header_file << "// Header files needed for this architecture\n";
  IIRScram::_publish_cc_include( header_file, "tyvis/STDTypes.hh" );
  
  // #include all the vhdl-ams related files
  if( lang_proc->processing_vhdl_ams() ){
    _publish_cc_ams_includes( header_file );
  }

  // #include the entity_elab header file
  get_entity()->_publish_cc_include_elab( header_file );

  // Request the Hierarchy.hh for our hierarchy stuff
  IIRScram::_publish_cc_include( header_file, "tyvis/Hierarchy.hh" );

  // Request AMSType.hh if VHDL-AMS constructs are invoked.
  if( lang_proc->processing_vhdl_ams() ) {
    IIRScram::_publish_cc_include( header_file, "tyvis/AMSType.hh" );
  }

  // Just include the decls file which has all the required includes
  // _publish_cc_include_decls( header_file );
  IIRScram::_publish_cc_include( header_file, _get_cc_design_unit_name() + ".hh" );
  
  _publish_cc_class_includes( header_file, &architecture_statement_part );
  IIR_Declaration* decl = architecture_declarative_part.first();
  while(decl != NULL) {
    if( decl->_is_iir_configuration_specification() ){
      if( decl->_get_entity_aspect() != NULL) {
	decl->_get_entity_aspect()->_publish_cc_include_elab( header_file );
      }
    }
    decl =  architecture_declarative_part.successor(decl);
  }
  
  _publish_cc_class( header_file );
}

IIR_DeclarationList*
IIRScram_ArchitectureDeclaration::_get_declaration_list() {
  return &architecture_declarative_part;
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_class( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_class" );

  IIR_EntityDeclaration* entitydecl = get_entity();
    
  _cc_out << "class " << _get_cc_elaboration_class_name() 
	  << " : public "
	  << OS( get_entity()->_get_cc_elaboration_class_name()
		 + "{" )
	  << "public:" << NL();

  // Publish the constructor
  _cc_out << _get_cc_elaboration_class_name() << "(";
  entitydecl->port_clause._publish_cc_unconstrained_ports( _cc_out, TRUE, FALSE, FALSE);
  _cc_out << ");" << NL();
  

  if(entitydecl->generic_clause.num_elements() > 0) {
    _cc_out << _get_cc_elaboration_class_name() << "(" << NL();
    entitydecl->generic_clause._publish_generic_parameter_list( _cc_out );
    entitydecl->port_clause._publish_cc_unconstrained_ports( _cc_out, TRUE, FALSE, TRUE );
    _cc_out << " );" << NL();
  }

  // The destructor.
  _cc_out << "~" << _get_cc_elaboration_class_name() << "();" << NL() << NL()
	  << "void instantiate(Hierarchy *);" << NL()
	  << "void createNetInfo();" << NL()
	  << "void connect(int, int, ...);" << NL()
	  << "void partition() {}" << NL() << NL();

  _publish_cc_ams_objects( _cc_out );
  if( lang_proc->processing_vhdl_ams() ){
    _cc_out << "void formCharacteristicExpressions();\n";
    _cc_out << "void initializeSignals();\n";
    _cc_out << "void connectTerminals(int, ...);\n";
  }
  // Publish the necessary type info structures that have been declared
  // within the scope of this architecture here...
  // Publish any constants declaraed here so that the other declarations
  // can use them without any trouble.
  
  architecture_declarative_part._publish_cc_type_info_and_constant_declarations( _cc_out );

  //Publish the signal info data members
  _publish_cc_signals( _cc_out, &architecture_declarative_part );

  if( lang_proc->processing_vhdl_ams() ){
    _publish_cc_break_signal_info(_cc_out);
  }

  // Since the implicit declarations for the signals declared in the port
  // clause in the entity is NULL. But get_entity()->port_clause in the
  // architecture declaration gives implicit declartions for the signals
  // in the port clause of the entity also. we are going to publish the
  // implicit declarations for those signals here in the architecture
  _publish_cc_implicit_signals( _cc_out, &get_entity()->port_clause );

  // If we're a foreign arch, we'll publish that.  Otherwise we'll publish
  // process pointers.
  if( _is_foreign_architecture() ){
    _publish_cc_foreign_process_pointer( _cc_out );
  }
  else{
    // Publish the pointers to processes and objects used in this architecture
    _publish_cc_object_pointers( _cc_out, &architecture_statement_part );
  }

  // Publish file objects.
  architecture_declarative_part._publish_cc_file_objects( _cc_out );

  _cc_out << "};" << NL();
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ccfile(){
 
  // Publish the .cc file
  published_cc_file cc_file( _get_library_directory(), 
			     _get_cc_elaboration_class_name(),
			     this );
  SCRAM_CC_REF( cc_file, "IIRScram_ArchitectureDeclaration::_publish_cc_ccfile" );
  IIRScram::_publish_cc_include( cc_file, "tyvis/SourceBase.hh" );
  if( lang_proc->processing_vhdl_ams() ) {
    cc_file << "extern list <VHDLType *> globalFreeQuantityList;" <<NL();
  }
  _publish_cc_include_elab( cc_file );

  IIR_ArchitectureStatementList* arch_stmt_list = &architecture_statement_part;
  IIR_ArchitectureStatement *arch_stmt = NULL;
  arch_stmt = arch_stmt_list->first();
  while(arch_stmt != NULL) {
    if (arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT)  {
      IIR_ComponentInstantiationStatement *instantiation_stmt = 
	static_cast<IIR_ComponentInstantiationStatement *>(arch_stmt);
      if( instantiation_stmt->_get_configuration() &&
	  instantiation_stmt->_get_configuration()->get_kind() == IIR_COMPONENT_CONFIGURATION &&
	  instantiation_stmt->_get_configuration()->_get_entity_aspect() ){
	instantiation_stmt->_get_configuration()->_get_entity_aspect()->_publish_cc_include_elab( cc_file );
      }
    }
    arch_stmt = arch_stmt_list->successor(arch_stmt);
  }
  
  _publish_cc_headerfiles_for_cc( cc_file );

#ifdef PROCESS_GRAPH
  if (signal_graph == TRUE) {
    cc_file << "extern ofstream fp;\n";
  }
#endif

  _publish_cc_constructor( cc_file );
  _publish_cc_destructor( cc_file, &architecture_statement_part );
  _publish_cc_instantiate( cc_file );
  _publish_cc_createNetInfo( cc_file );
  _publish_cc_connect( cc_file );
  _publish_cc_blocks_elaborate( cc_file );
  _publish_cc_allocate_module( cc_file );
  if( lang_proc->processing_vhdl_ams() ) {
    _publish_cc_ams_form_characteristic_expressions( cc_file );
    _publish_cc_ams_initialize_signals( cc_file );
    _publish_cc_ams_form_break_set(cc_file);
    _publish_cc_ams_connect_terminals( cc_file );
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_headerfiles_for_cc( published_file &_cc_out ) {
  IIR_EntityDeclaration *entitydecl = get_entity();
 
  // Publish the header files for the binding architectures specified
  // in configuration specification.

  // Publish the forward references for the entity statement part.
  IIR_ArchitectureStatement *arch_stmt = architecture_statement_part.first();
  while (arch_stmt != NULL) {
    // Currently the process name doesn't include its entity and architecture name. 
    switch ( arch_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
      arch_stmt->_publish_cc_include( _cc_out );
      break;
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      arch_stmt->_publish_cc_headerfiles_for_cc_default( _cc_out );
      break;
    case IIR_BLOCK_STATEMENT:
      arch_stmt->_publish_cc_include_elab( _cc_out );
      break;
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      IIR_ComponentInstantiationStatement *as_component_instantiation_stmt;
      as_component_instantiation_stmt = (IIR_ComponentInstantiationStatement *)arch_stmt;
      IIR_AssociationElement *asselem;
      
      asselem = as_component_instantiation_stmt->port_map_aspect.first();
      while ( asselem != NULL ){
	if (asselem->_get_actual() != NULL) {
	  ASSERT( asselem->get_formal() != NULL );
	  // Okay. The actual is not open
	  if ((asselem->get_formal()->get_kind() == IIR_FUNCTION_CALL) ||
	      (asselem->_get_actual()->get_kind() == IIR_FUNCTION_CALL)) {
	    IIRScram::_publish_cc_include( _cc_out, "tyvis/TypeConvert.hh" );
	    // We don't need to publish this multiple times.
	    break;
	  }
	}
	asselem = as_component_instantiation_stmt->port_map_aspect.successor(asselem);
      }
      break;
    case IIR_FILE_DECLARATION:
      IIRScram::_publish_cc_include(_cc_out, "tyvis/SharedFileType.hh");
      break;
    default:
      // Nothing to be done for other statements
      break;
    }
    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }

  // Publish the header files for the processes defined in the entity
  arch_stmt = entitydecl->entity_statement_part.first();
  while (arch_stmt != NULL) {
    if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      // What about processes with no labels?  Also, if there is going to
      // be one more "get_kind" checked, go to a case statment.  Actually,
      // it's not even possible to have a process in the statements of an
      // entity, is it?  Check the LRM.
      arch_stmt->_get_label()->_publish_cc_include( _cc_out );
    }
    else {
      cerr << "IIR_ArchitectureDeclaration_publish_cc_headefiles_for_cc( published_file &_cc_out ) - "
	   << "Un-handled concurrent statement (" << arch_stmt->get_kind()
	   << ") in entity\n";
    }

    arch_stmt = entitydecl->entity_statement_part.successor(arch_stmt);
  }

  // Include the architecture decls header file
  _publish_cc_include_decls( _cc_out );
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_constructor( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_constructor" );
 
  _publish_cc_constructor_with_no_arguments( _cc_out );
  if (get_entity()->generic_clause.num_elements() > 0) {
    _publish_cc_constructor_with_arguments( _cc_out );
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_no_arguments( published_file &_cc_out ) {
  IIR_Boolean firstDeclFlag         = TRUE;
  IIR_EntityDeclaration *entityDecl = get_entity();
  int noOfUnconstrainedPorts        = 0;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_no_arguments" );
 
  _cc_out << _get_cc_elaboration_class_name() << "::"
	  << _get_cc_elaboration_class_name() << OS("( ");
  noOfUnconstrainedPorts = 
    entityDecl->port_clause._publish_cc_unconstrained_ports( _cc_out, TRUE, TRUE, FALSE );
  _cc_out << CS(")");
  
  if (noOfUnconstrainedPorts > 0) {
    // Have to pass the parameters to the entity .ie. parent class...
    _cc_out << ":" << NL();
    _cc_out << OS(entityDecl->_get_cc_elaboration_class_name() + "(");
    entityDecl->port_clause._publish_cc_unconstrained_ports( _cc_out, FALSE, TRUE, FALSE );
    _cc_out << CS(")");
  }
  
  _publish_cc_ams_objects_init(_cc_out, firstDeclFlag);

  // Initialize all the type info structures first here...
  firstDeclFlag = architecture_declarative_part._publish_cc_type_info_and_constant_definitions( _cc_out, firstDeclFlag);
  _publish_cc_signal_objects_init( _cc_out, firstDeclFlag );
  _cc_out << OS("{");
  _publish_cc_ams_form_global_quantity_list( _cc_out );
  _cc_out.insert_comment( "all of the global constants..." );
  architecture_declarative_part._publish_cc_global_constants_assignments( _cc_out );
  _cc_out.insert_comment( "all of the file objects..." );
  architecture_declarative_part._publish_cc_file_objects_init( _cc_out );
  _cc_out.insert_comment( "all of the object pointers..." );
  _publish_cc_object_pointers_init( _cc_out );
  _cc_out << CS("}");
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_arguments(published_file &_cc_out){
  IIR_EntityDeclaration* entitydecl = get_entity();
  int numGenericClause = entitydecl->generic_clause.num_elements();
  IIR_Boolean firstDeclFlag         = TRUE;
  
  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_arguments" );

  _cc_out << _get_cc_elaboration_class_name() << "::" << _get_cc_elaboration_class_name() 
	  << OS("(");
  entitydecl->generic_clause._publish_generic_parameter_list( _cc_out );
  entitydecl->port_clause._publish_cc_unconstrained_ports( _cc_out, TRUE, TRUE, TRUE );
  _cc_out << CS(")");
  
  if(numGenericClause > 0) {
    SCRAM_CC_REF( _cc_out, 
		  "IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_arguments" );
    _cc_out << ":" << NL()
	    << entitydecl->_get_cc_elaboration_class_name() << OS("(");
    entitydecl->generic_clause._publish_generic_parameters_notypes( _cc_out );
    entitydecl->port_clause._publish_cc_unconstrained_ports( _cc_out, FALSE, TRUE, TRUE);
    _cc_out << CS(")");
    firstDeclFlag = FALSE;
  }

  bool tempFlag = true;
  tempFlag = architecture_declarative_part._publish_cc_type_info_and_constant_definitions( _cc_out, firstDeclFlag);
  firstDeclFlag = tempFlag && firstDeclFlag;

  _publish_cc_ams_objects_init(_cc_out, firstDeclFlag);
  _publish_cc_signal_objects_init( _cc_out, firstDeclFlag );
  
  _cc_out << OS("{");
  _publish_cc_ams_form_global_quantity_list( _cc_out );
  architecture_declarative_part._publish_cc_global_constants_assignments( _cc_out );
  architecture_declarative_part._publish_cc_file_objects_init( _cc_out );
  _publish_cc_object_pointers_init( _cc_out );
  _cc_out << CS("}");
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_object_pointers_init( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_object_pointer_init" );
 
  // we are setting the _current_publish_node to the architecture declaration
  // so that if during the publish_cc_object_pointers_init for architecture declaration
  // the node needs to get the archicture declaration. it can use the _current
  // _publish_node variable.
  
  IIR *tempNode = _get_current_publish_node();
  get_entity()->_publish_cc_object_pointers_init( _cc_out );

  _set_current_publish_node( this );
  IIRScram::_publish_cc_object_pointers_init( _cc_out, 
					      &architecture_statement_part, 
					      &architecture_declarative_part);
  _set_current_publish_node( tempNode );
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_instantiate( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_instantiate" );
  _cc_out << "void" << NL()
	  << OS( _get_cc_elaboration_class_name() + "::instantiate( Hierarchy *hier ){");
  // Publish instantiate stmts from entity statement part
  get_entity()->entity_statement_part._publish_cc_instantiate_calls( _cc_out );
  // Publish instantiate stmts from architecture statement part
  architecture_statement_part._publish_cc_instantiate_calls( _cc_out );

  if( _is_foreign_architecture() ){
    _publish_cc_foreign_instantiate( _cc_out );
  }

  // Now we'll need to call "createNetInfo" to connect the nets.
  _cc_out << "createNetInfo();" << NL();
  _cc_out << CS("}");
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_createNetInfo( published_file &_cc_out ) {
  IIR_EntityDeclaration* entitydecl = get_entity();

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_createNetInfo" );

  const string tmp = _get_current_elab_name();
  const string tmp2 = _get_current_publish_name();
  IIR* tmpNode = _get_current_publish_node();

  _cc_out << "void" << NL();
  _cc_out << _get_cc_elaboration_class_name() << OS("::createNetInfo(){");

  // Publish createNetInfo from entity statement part
  entitydecl->_publish_cc_createNetInfo( _cc_out );

  _publish_cc_anonymous_drivers( _cc_out, &get_entity()->port_clause );
  _publish_cc_anonymous_drivers( _cc_out, &architecture_declarative_part );

  architecture_statement_part._publish_cc_createNetInfo( _cc_out );

  if( _is_foreign_architecture() ){
    _publish_cc_foreign_createNetInfo( _cc_out );
  }
  _cc_out << CS("}");
  _set_current_elab_name( tmp );
  _set_current_publish_name( tmp2 );
  _set_current_publish_node( tmpNode );

}

void
IIRScram_ArchitectureDeclaration::_publish_cc_connect( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_connect" );

   _cc_out << "void" << NL()
          << OS( _get_cc_elaboration_class_name() +
                 "::connect( int inputSignals, int outputSignals, ... ){")
	   << "int noOfSignals = inputSignals + outputSignals;" << NL()
	   << "va_list ap;" << NL()
	   << "va_start(ap, outputSignals);" << NL()
	   << OS("for( int i=0; i < noOfSignals; i++ ){")
	   << "addToFanOut( va_arg(ap, VHDLType*) );" << NL()
	   << CS("}")
	   << "va_end(ap);" << NL();

  //Pass on the output connection  inforamtion to its output signals
  IIR_EntityDeclaration* entitydecl = get_entity();
  IIR_InterfaceDeclaration* portelement = NULL;
  IIR_Mode mode;
  int index = 0;

  _cc_out << OS("if( inputSignals > 0 ){");
  portelement = entitydecl->port_clause.first();
  while( portelement != NULL ){
    mode = portelement->get_mode();
    if(mode == IIR_IN_MODE ) {
      _cc_out << OS("setSourceInfo(");
      portelement->_publish_cc_elaborate( _cc_out ); 
      _cc_out << "," << NL() 
	      << "*(fanOutInfo[";
      _cc_out << index << "])" << CS(");");
      index++;
    }
    portelement = entitydecl->port_clause.successor(portelement);
  }
  _cc_out << CS("}");

  _cc_out << OS("if( outputSignals > 0 ){");
  portelement = entitydecl->port_clause.first();
  while( portelement != NULL ){
    if (portelement->get_kind() != IIR_TERMINAL_INTERFACE_DECLARATION) {
      switch( portelement->get_mode() ){
      case IIR_INOUT_MODE:{
	_cc_out << OS("Add(");
	portelement->_publish_cc_elaborate( _cc_out ); 
	_cc_out << "," << NL() << "*(fanOutInfo[";
	_cc_out << index << "])" << CS(");");
	
	_cc_out << OS("setSourceInfo(");
	portelement->_publish_cc_elaborate( _cc_out ); 
	_cc_out << "," << NL() << "*(fanOutInfo[";
	_cc_out << index << "])" << CS(");");
	index++;
	break;
      }
      case IIR_OUT_MODE:{
	_cc_out << OS("Add(");
	portelement->_publish_cc_elaborate( _cc_out ); 
	_cc_out << "," << NL() << "*(fanOutInfo[";
	_cc_out << index << "])" << NL() << CS(");");
	index++;
	break;
      }
      case IIR_IN_MODE:{
	break;
      }
      default:{
	ostringstream err;
	err << "IIRScram_ArchitectureDeclaration::_publish_cc_connect, mode |" 
	    << portelement->get_mode() << "| unsupported.";
	report_error( portelement, err.str() );
	abort();
      }
      }
    }
    portelement = entitydecl->port_clause.successor(portelement);
  }
  _cc_out << CS("}");

  IIR_ArchitectureStatement* arch_stmt = NULL;
  // Publish output  connections for entity statement part
  arch_stmt = entitydecl->entity_statement_part.first();
  while (arch_stmt != NULL) {
    arch_stmt->_publish_cc_connect_call( _cc_out );
    arch_stmt = entitydecl->entity_statement_part.successor(arch_stmt);
  }

  // Publish output connections for  architecture statement part
  arch_stmt = architecture_statement_part.first();
  while (arch_stmt != NULL) {
    arch_stmt->_publish_cc_connect_call( _cc_out );

    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }
  _cc_out << CS("}");
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_binding_name(ostream& outstream){
  outstream << _get_cc_design_unit_name();
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_signal_objects_init( published_file &_cc_out,
								   IIR_Boolean firstDeclFlag) {
  IIR_Declaration* decl = NULL;
  int first = 0;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_signal_objects_init" );
  
  decl = architecture_declarative_part.first();
  while (decl != NULL) {
    if(decl->get_kind() == IIR_SIGNAL_DECLARATION) {
      if( first == 0 && firstDeclFlag == TRUE ){
	_cc_out << ":" << NL();
	first = 1;
      }
      else {
	_cc_out << "," << NL();
      }
      decl->_publish_cc_elaborate( _cc_out );
      _cc_out << OS("(ObjectBase::SIGNAL_NETINFO");

      if ((decl->get_subtype()->_is_scalar_type() == TRUE) &&
	  (decl->get_subtype()->_is_kernel_type() == FALSE) &&
	  (decl->get_subtype()->_is_anonymous()   == FALSE)) {
	decl->get_subtype()->_publish_cc_object_type_info( _cc_out );
      }
      
      if ((decl->get_subtype()->_is_array_type() == TRUE) ||
	  (decl->get_subtype()->_is_record_type() == TRUE)) {
	IIR_TypeDefinition *type = decl->get_subtype();
	
	type->_publish_cc_object_type_info( _cc_out );
	_cc_out << "," << NL();

	type->_publish_cc_resolution_function_id( _cc_out );
      }

      if (decl->get_subtype()->_is_array_type() == TRUE) {
	IIR_TypeDefinition *type = decl->get_subtype();
	if (type->_is_anonymous()  == TRUE) {
	  _cc_out << "," << NL();
	  type->_publish_cc_range( _cc_out );
	} else if(type->_is_unconstrained_array_type()) {
	  ASSERT(decl->get_value() != NULL);
	  if(decl->get_value()->get_kind() != IIR_CONCATENATION_OPERATOR){
	    _cc_out << "," << NL();
	    decl->get_value()->_publish_cc_range( _cc_out );
	  }
	}
      }
      
      _cc_out << CS(")");
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL
	  && decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->_is_signal() == TRUE) {
	    _cc_out << "," << NL();
	    imp_decl->_publish_cc_elaborate( _cc_out );
	    _cc_out << OS("(ObjectBase::SIGNAL_NETINFO");
	    IIR_TypeDefinition* type_def = imp_decl->_get_attribute_name()->_get_explicit_signal_type();
	    if ((type_def->_is_array_type() == TRUE) ||
		(type_def->_is_record_type() == TRUE)) {
	      type_def->_publish_cc_object_type_info( _cc_out );
	      _cc_out << "," << NL();
	      type_def->_publish_cc_resolution_function_id( _cc_out );
	    }
	    
	    if (type_def->_is_array_type() == TRUE)  {
	      if (type_def->_is_anonymous() == TRUE) {
		_cc_out << "," << NL();
		type_def->_publish_cc_range( _cc_out );
	      }
	    }
	    _cc_out << CS(")");
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(((IIR_AliasDeclaration*)decl)->get_name()->_is_signal()) {
	if( first == 0 && firstDeclFlag == TRUE ){
	  _cc_out << ":" << NL();
	  first = 1;
	}
	else {
	  _cc_out << "," << NL();
	}
	static_cast<IIR_AliasDeclaration*>(decl)->_publish_cc_elaborate_alias_init( _cc_out );
      }
    }
    decl = architecture_declarative_part.successor(decl);    
  }
  
  // Publish the initializations for the implicit_declarations of the signals
  // in the port clause of the entity declartion.
  IIR_DeclarationList *port_clause_list = &(get_entity()->port_clause);
  decl = port_clause_list->first();
  while ( decl != NULL ) {
    IIR_Declaration *nextdecl = port_clause_list->successor(decl);
#ifdef PROCESS_COMBINATION
    // This should retrieve the previously built clone for the interface
    // signal.  The attributes have been attached to the clone, not the
    // original port.
    //decl = (IIR_Declaration*)decl->_clone();
#endif
    if( decl->_get_implicit_declarations() != NULL
	&& decl->_get_implicit_declarations()->num_elements() != 0) {
      IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
      while(imp_decl != NULL) {
	if(imp_decl->_is_signal() == TRUE) {
	  if( first == 0 && firstDeclFlag == TRUE ){
	    _cc_out << ":" << NL();
	    first = 1;
	  } else {
	    _cc_out << "," << NL();
	  }
	  
	  imp_decl->_publish_cc_elaborate( _cc_out );
	  _cc_out << OS("(ObjectBase::SIGNAL_NETINFO");
	  IIR_TypeDefinition* type_def = imp_decl->_get_attribute_name()->_get_explicit_signal_type();
	  if (type_def->_is_array_type() &&
	      type_def->_is_anonymous() == TRUE) {
	    _cc_out << "," << NL();
	    type_def->_publish_cc_range( _cc_out );
	  }
	  _cc_out << CS(")");
	}
	imp_decl = decl->_get_implicit_declarations()->get_next_element();
      }
    }
    decl = nextdecl;
  }
  IIR_ConcurrentStatement *conc_stmt = static_cast <IIR_ConcurrentStatement *>
    (architecture_statement_part.first());
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      for (IIR_SequentialStatement* seqstmt =  dynamic_cast <IIR_ProcessStatement *>(conc_stmt)->process_statement_part.first();
          seqstmt != NULL;
          seqstmt = dynamic_cast <IIR_ProcessStatement *>(conc_stmt)->process_statement_part.successor(seqstmt)) {
        ostringstream breakstr;
        if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
          ASSERT(seqstmt->_get_process_stmt_label() != NULL);
          breakstr <<dynamic_cast <IIR_ProcessStatement *>(conc_stmt)
	    ->_get_cc_process_class() << "_breakSignal" ;
          string breakstring = breakstr.str();
          _cc_out << ",\n" << breakstring << "(new EnumerationType "
		  <<"(ObjectBase::SIGNAL,UniversalBoolean(0), "
		  <<"SavantbooleanType_info))" ;
          _cc_out << ",\n" << breakstring 
		  << "_info(ObjectBase::SIGNAL_NETINFO)" ;
        }
      }
    }
    conc_stmt =  static_cast <IIR_ConcurrentStatement *>
      (architecture_statement_part.successor(conc_stmt));
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_scoping_prefix( ostream &, IIR *, IIR * ){
  // Nothing to be done
}

#ifdef PROCESS_GRAPH
void
IIRScram_ArchitectureDeclaration::_publish_cc_driver_info( published_file &_cc_out ){
  IIR_EntityDeclaration* entitydecl = get_entity();
  IIR_InterfaceDeclaration* portelement = NULL;
  IIR_ArchitectureStatement* arch_stmt = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_driver_info" ); 
  
  portelement = entitydecl->port_clause.first();
  for(; portelement != NULL; ) {
    if (portelement->get_subtype()->_is_scalar_type() == TRUE ||
	portelement->get_subtype()->_is_array_type() == TRUE) {
      _cc_out << "fp << \"Signal:";
      portelement->_publish_cc_elaborate( _cc_out );
      _cc_out << "\" << endl;" << endl;
      portelement->_publish_cc_elaborate( _cc_out );
      _cc_out << ".dump_connectivity_info(fp);" << endl;
    }
    portelement = entitydecl->port_clause.successor(portelement);
  }

  //dump_info for concurrent statements in the architecture body
  arch_stmt = architecture_statement_part.first();
  while (arch_stmt!=NULL) {
    if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      arch_stmt->_publish_cc_driver_info( _cc_out );
    }
    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }
}
#endif

set<IIR_Declaration> *
IIRScram_ArchitectureDeclaration::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;
  set<IIR_Declaration> *current_set = architecture_declarative_part._find_declarations( to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  ASSERT( get_entity() != NULL );
  current_set = get_entity()->_find_declarations( to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }
  
  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

void 
IIRScram_ArchitectureDeclaration::_make_interface_visible( symbol_table *sym_tab ){
  // This isn't _my_ interface...
  //   if( get_entity() != NULL ){
  //     get_entity()->_make_interface_visible( sym_tab );
  //   }

  sym_tab->make_visible( &architecture_declarative_part );
  IIR_ArchitectureStatement *current_statement;
  current_statement = architecture_statement_part.first();
  while( current_statement != NULL ){
    if( current_statement->get_label() != NULL ){
      sym_tab->make_visible( current_statement->get_label() );
    }
    current_statement = architecture_statement_part.successor( current_statement );
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_implicit_signals( published_file &_cc_out,
								IIR_DeclarationList *decl_list ){
  ASSERT(decl_list != NULL);
  ASSERT((_is_iir_concurrent_statement() == TRUE) ||
	 (_is_entity_decl() == TRUE) ||
	 (_is_component_decl() == TRUE) ||
	 (get_kind() == IIR_ARCHITECTURE_DECLARATION));
  
  IIR_Declaration* decl = decl_list->first();

  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_implicit_signals" );

  while (decl != NULL) {
    IIR_Declaration *nextdecl = decl_list->successor(decl);
#ifdef PROCESS_COMBINATION
    // This should retrieve the previously built clone for the interface
    // signal.  The attributes have been attached to the clone, not the
    // original port.
    //decl = (IIR_Declaration*)decl->_clone();
#endif
    if(decl->_is_signal() == TRUE) {
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL &&
	  decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->_is_signal() == TRUE) {
	    imp_decl->_publish_cc_implicit_signal_type( _cc_out );
	    _cc_out << " ";
	    imp_decl->_publish_cc_elaborate( _cc_out );
	    _cc_out << ";" << NL();
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(decl->_is_signal() == TRUE) {
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_definition( _cc_out );
      }
    }
    decl = nextdecl;
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_include(published_file &_cc_out) {
  const string include_file_name = _get_cc_elaboration_class_name() + ".hh";
  IIRScram::_publish_cc_include( _cc_out, include_file_name );
}

const string
IIRScram_ArchitectureDeclaration::_get_cc_design_unit_name(){
  return _get_declarator()->_convert_to_string();
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_include_decls_prefix( ostream &os ){
  os << *_get_entity()->_get_declarator();
}

void
IIRScram_ArchitectureDeclaration::_mangle_declarator(){
  string mangled_declarator_tmp;

  if( _get_entity() != NULL ){
    mangled_declarator_tmp = _get_entity()->_get_declarator()->_convert_to_string();
  }
  mangled_declarator_tmp += string("_") + _get_mangling_prefix();
  mangled_declarator_tmp += get_declarator()->_convert_to_string();

  _set_mangled_declarator( mangled_declarator_tmp );
}

#ifdef PROCESS_COMBINATION
void
IIRScram_ArchitectureDeclaration::_combine() {
  IIR_ArchitectureStatementList onewaitlist, uncondwaitlist;
  IIR_ArchitectureStatement *next;
  IIR_ProcessStatement *to_combine;
  ProcessCombiner *cproc;

  cerr << "The elaborated design has "
       << architecture_statement_part.num_elements() << " processes: ";

  // move all 1-wait processes from arch_stmt_part to onewaitlist
  to_combine = (IIR_ProcessStatement*)architecture_statement_part.first();
  while ( to_combine != NULL ) {
    next = architecture_statement_part.successor(to_combine);
    ASSERT( IIR_PROCESS_STATEMENT == to_combine->get_kind() );
    
    to_combine->_build_wait_list();
    if ( to_combine->_wait_stmt_list.num_elements() == 1 ) {
      IIR_WaitStatement *waitstmt = to_combine->_wait_stmt_list.first();
      if (waitstmt->get_timeout_clause() == NULL
	  && waitstmt->sensitivity_list.num_elements() == 0
	  && waitstmt->get_condition_clause() == NULL ) {
	// Put processes with one uncinditional wait in uncondwaitlist
	uncondwaitlist.append( to_combine );
      }
      else {
	onewaitlist.append( to_combine );
      }
      architecture_statement_part.remove( to_combine );
    }
    to_combine = (IIR_ProcessStatement*)next;
  }
  
  cerr << uncondwaitlist.num_elements() 
       << " with a single unconditional wait,\nand "
       << onewaitlist.num_elements()
       << " with a single combinable wait statement.\n";
  cerr << "Partitioning design into " << num_partitions << " partitions";

  // combine processes in unconditional wait list--put into arch_stmt_part
  if ( NULL != uncondwaitlist.first() ) {
    ProcessCombiner uncondwaitcombiner;
    to_combine = (IIR_ProcessStatement*)uncondwaitlist.first();

    while ( NULL != to_combine ) {
      next = uncondwaitlist.successor( to_combine );
      uncondwaitcombiner.combine_uncondwait( to_combine );
      uncondwaitlist.remove( to_combine );
      to_combine = (IIR_ProcessStatement*)next;
    }
    architecture_statement_part.append( uncondwaitcombiner.get_process() );
  }

  // OK, this check probably isn't necessary...
  if ( NULL != onewaitlist.first() ) {
    int partition = 0;
    cproc = new ProcessCombiner[num_partitions];
    to_combine = (IIR_ProcessStatement*)onewaitlist.first();

    while ( NULL != to_combine ) {
      next = onewaitlist.successor( to_combine );
      if ( !to_combine->_convert_to_TWF() ) {
	// If the process can't be converted to TWF, remove it from the
	// onewaitlist and add it back to the architecture
	architecture_statement_part.append( to_combine );
	onewaitlist.remove( to_combine );
      }
      to_combine = (IIR_ProcessStatement*)next;
    }
    
    
    // Partitoning algorithms start here.  Probably should be elsewhere.
    to_combine = (IIR_ProcessStatement*)onewaitlist.first();
    if ( partition_algorithm != NULL 
	 && !strncmp( partition_algorithm, "ra", 2 )) {
      cerr << " with random partitioning.\n";
      while ( NULL != to_combine ) {
	next = onewaitlist.successor( to_combine );
	partition = (int)(((float)num_partitions * (float)rand()) 
			  / (RAND_MAX + 1.0));
	cproc[partition].combine( to_combine );
	onewaitlist.remove( to_combine );
	to_combine = (IIR_ProcessStatement*)next;
      }
    }
    else if ( partition_algorithm != NULL
	      && !strncmp( partition_algorithm, "c", 1 )) {
      cerr << " with output-cone partitioning.\n";
      cone_partition( (IIR_ArchitectureDeclaration*)this, onewaitlist, cproc );
    }
    else { // Round Robin partitioning is the default
      cerr << " with round-robin partitioning.\n";
      
      while ( NULL != to_combine ) {
	next = onewaitlist.successor( to_combine );
	cproc[partition].combine( to_combine );
	partition = (partition + 1)%num_partitions;
	onewaitlist.remove( to_combine );
	to_combine = (IIR_ProcessStatement*)next;
      }
    }

    // add combined processes back to architecture
    for (partition = 0; partition < num_partitions; partition++ ) {
      architecture_statement_part.append( cproc[partition].get_process() );
    }
    delete [] cproc;
  }
}

void
IIRScram_ArchitectureDeclaration::_static_elaborate(IIR_ArchitectureDeclaration *arch, 
						    IIR_DeclarationList *cfglist,
						    char *hier_location) {
  IIR_Declaration *decl, *declclone, *impdecl;
  IIR_Identifier *new_signal_id, *old_id;

  decl = architecture_declarative_part.first();
  while (decl != NULL) {
    if (decl->get_kind() == IIR_COMPONENT_DECLARATION 
	|| decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      declclone = decl;
    }
    else {
      declclone = (IIR_Declaration*)((IIR*)decl)->_clone();
      // mangle names for certain declarations
      if (declclone->get_kind() == IIR_SIGNAL_DECLARATION 
	  || declclone->get_kind() == IIR_FILE_DECLARATION
          || declclone->get_kind() == IIR_ALIAS_DECLARATION) {
	declclone->_static_elaborate_decl(hier_location);
      }
    }

    if( declclone->_get_implicit_declarations() != NULL ){
      // mangle names for all implicit decls
      impdecl = declclone->_get_implicit_declarations()->get_element();
      while (impdecl != NULL) {
	impdecl->_static_elaborate_decl(hier_location);
	impdecl = declclone->_get_implicit_declarations()->get_next_element();
      }
    }

    arch->architecture_declarative_part.append(declclone);
    decl = architecture_declarative_part.successor(decl);
  }

  decl = context_items.first();
  while ( decl != NULL ) {
    ASSERT( IIR_USE_CLAUSE == decl->get_kind() );
    if (arch->context_items.get_position( decl ) == -1 ) {
      arch->context_items.append( decl );
    }
    decl = context_items.successor( decl );
  }

  IIR_ArchitectureStatement *arch_stmt;
  IIR *clone;
  get_entity()->_static_elaborate(arch, cfglist, hier_location);

  arch_stmt = architecture_statement_part.first();
  while (arch_stmt != NULL) {
    if (arch_stmt->get_kind() == IIR_PROCESS_STATEMENT ||
	arch_stmt->get_kind() == IIR_SENSITIZED_PROCESS_STATEMENT ||
	arch_stmt->get_kind() == IIR_CONCURRENT_SELECTED_SIGNAL_ASSIGNMENT ||
	arch_stmt->get_kind() == IIR_CONCURRENT_CONDITIONAL_SIGNAL_ASSIGNMENT) {
      clone = arch_stmt->_clone();
      clone->_static_elaborate(arch, cfglist, hier_location);
      arch->architecture_statement_part.append(clone);
    }
    else {
      arch_stmt->_static_elaborate(arch, cfglist, hier_location);
    }
    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }

  decl = architecture_declarative_part.first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_SIGNAL_DECLARATION:
      static_cast<IIRScram_SignalDeclaration*>(decl)->_my_clone = NULL;
      break;
    case IIR_FILE_DECLARATION:
      static_cast<IIRScram_FileDeclaration*>(decl)->_my_clone = NULL;
      break;
    case IIR_ALIAS_DECLARATION:
      static_cast<IIRScram_AliasDeclaration*>(decl)->_my_clone = NULL;
      break;
    case IIR_COMPONENT_DECLARATION:
    case IIR_CONFIGURATION_SPECIFICATION:
      arch->architecture_declarative_part.remove(decl);
      break;
    default:
      break;
    }
    decl = architecture_declarative_part.successor(decl);
  }
}
#endif

void 
IIRScram_ArchitectureDeclaration::_publish_cc_foreign_process_pointer( published_file &_cc_out ){
  IIRScram::_publish_cc_include( _cc_out, "tyvis/ForeignProcess.hh" );
  _cc_out << "ForeignProcess *foreignProcess;" << NL();
}

bool 
IIRScram_ArchitectureDeclaration::_is_foreign_architecture(){
  return architecture_declarative_part._contains_foreign_attribute();
}

IIR_AttributeSpecification *
IIRScram_ArchitectureDeclaration::_get_foreign_attribute(){
  IIR_AttributeSpecification *retval = 0;
  set<IIR_AttributeSpecification> *foreign_attributes = 
    architecture_declarative_part._find_foreign_attributes();
  if( foreign_attributes != 0 ){
    IIR_AttributeSpecification *last_element = 0;
    IIR_AttributeSpecification *current = foreign_attributes->get_element();
    while( current != 0 ){
      last_element = current;
      current = foreign_attributes->get_next_element();
    }
    retval = last_element;
  }

  return retval;
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_foreign_instantiate( published_file &_cc_out ){
  // We need to call the foreign process creation method with the string from the 
  // attribute spec.  First we'll get that string.
  _cc_out << OS("foreignProcess = ForeignProcess::instantiate(")
	  << _get_foreign_attribute()->_get_cc_value_string() 
	  << "," << NL() 
	  << "\"" << _get_cc_elaboration_class_name() << "\"," << NL()
	  << "this"
	  << CS(");");
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_foreign_createNetInfo( published_file &_cc_out ){
  string old_publish_name = _get_current_publish_name();
  _set_current_publish_name( ", foreignProcess );" );

  // We need to do:
  // Add( signals, foreignProcess )
  IIR_InterfaceDeclaration *currentSignal = get_entity()->port_clause.first();
  while( currentSignal != 0 ){
    // Add the "Add" call...
    _cc_out << OS("Add(");
    currentSignal->_publish_cc_elaborate( _cc_out );
    _cc_out << "," << NL() << "foreignProcess" << CS(");");
    
    // Add the "addChild" call...
    currentSignal->_publish_cc_addChild( _cc_out );

    currentSignal = get_entity()->port_clause.successor( currentSignal );
  }

  _set_current_publish_name( old_publish_name );
}

visitor_return_type *
IIRScram_ArchitectureDeclaration::_accept_visitor( node_visitor *visitor, 
						   visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ArchitectureDeclaration(this, arg);
}

// additions for vhdl-ams

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_includes(published_file &header_file) {

  SCRAM_CC_REF( header_file, "IIRScram_ArchitectureDeclaration::_publish_cc_ams_includes" );

    header_file << "\n// VHDL-AMS include files. \n";
    header_file << "#include \"adouble.h\"\n";
    header_file << "#include \"adutils.h\"\n";
    header_file << "#include \"Quantity.hh\"\n";
    header_file << "#include \"Terminal.hh\"\n";
    header_file << "#include \"RealType.hh\"\n";
    header_file << "#include \"FreeEquation.hh\"\n";
    header_file << "#include \"DifferentialEquation.hh\"\n";
    header_file << "#include \"BranchEquation.hh\"\n";
    header_file << "\n";
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_objects(published_file &_cc_out) {
  IIR_Declaration *decl = NULL;
  decl = architecture_declarative_part.first();
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_ams_objects" );
  while(decl != NULL) {
    switch(decl->get_kind()) {
         case IIR_FREE_QUANTITY_DECLARATION:
           decl->_publish_cc_decl(_cc_out);
           break;
         case IIR_TERMINAL_DECLARATION:
           decl->_publish_cc_decl(_cc_out);
           break;
         case IIR_BRANCH_QUANTITY_DECLARATION:
           decl->_publish_cc_decl(_cc_out);
           break;
	 case IIR_SOURCE_QUANTITY_DECLARATION:
           decl->_publish_cc_decl(_cc_out);
           break;
         default:
           break;
    }
    decl = architecture_declarative_part.successor(decl);
  }
  decl = cgen_arch_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].first();
  while(decl != NULL) {
    if(decl->_is_implicit_declaration() == TRUE) {
      decl->_publish_cc_decl(_cc_out);
    }
    decl = cgen_arch_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].successor(decl);
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_objects_init(published_file &_cc_out, IIR_Boolean &firstDeclFlag) {
  IIR_Declaration* decl = NULL;
  int first = 0;
  decl = architecture_declarative_part.first();
  while(decl != NULL) {
    switch(decl->get_kind()) {
    case IIR_FREE_QUANTITY_DECLARATION: {
      IIR_FreeQuantityDeclaration *free_qty = (IIR_FreeQuantityDeclaration *)decl;
      if ((first == 0) && (firstDeclFlag == TRUE)) {
	_cc_out << ":" << NL();
	first = 1;
	_is_ams_objects_present = TRUE;
	firstDeclFlag = FALSE;
      }
      else {
        _cc_out << "," << NL();
      }
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "(ObjectBase::QUANTITY, ";
      _cc_out << "\"";
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "\", ";
      free_qty->_publish_cc_tolerances(_cc_out);
      _cc_out << "FREE, NULL, NULL";
      if (free_qty->get_value() != NULL) {
        _cc_out << ", ";
        free_qty->get_value()->_publish_cc_ams_function(_cc_out);
      }
      _cc_out << ")";
    }
      break;
    case IIR_TERMINAL_DECLARATION: {
      if ((first == 0) && (firstDeclFlag == TRUE)) {
	_cc_out << ":" << NL();
	first = 1;
	_is_ams_objects_present = TRUE;
	firstDeclFlag = FALSE ;
      }
      else {
	_cc_out << "," << NL();
      }
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "( ObjectBase::TERMINAL ";
      _cc_out << ",\"";
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "\")";
    }
      break;
    case IIR_BRANCH_QUANTITY_DECLARATION: {
      if((first == 0) && (firstDeclFlag == TRUE)) {
	_cc_out << ":" << NL();
	first = 1;
	_is_ams_objects_present = TRUE;
	firstDeclFlag = FALSE;
      }
      else {
	_cc_out << "," << NL();
      }
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "(ObjectBase::QUANTITY, ";
      _cc_out << "\"";
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "\", ";
      if(decl->_is_implicit_declaration()==TRUE) {
	_cc_out << "IMPLICIT,NULL, NULL)";
      }   
      else {
	IIR_BranchQuantityDeclaration *branch_qty = (IIR_BranchQuantityDeclaration*)decl;
	branch_qty->_publish_cc_tolerances(_cc_out);
	if (branch_qty->_is_across_quantity()) {
	  _cc_out << "ACROSS, ";
	} 
	else {
	  _cc_out << "THROUGH,";
	}
	IIR_Declaration *plus_terminal = (IIR_Declaration *)(branch_qty->get_plus_terminal_name());
	IIR_Declaration *minus_terminal = (IIR_Declaration *)(branch_qty->get_minus_terminal_name());
	_cc_out << "&";
	plus_terminal->_publish_cc_lvalue(_cc_out);
	_cc_out << ", ";
	_cc_out << "&";
	minus_terminal->_publish_cc_lvalue(_cc_out);
	if(branch_qty->_is_across_quantity()==TRUE &&
	   branch_qty->get_across_aspect_expression() != NULL ) {
	  _cc_out << ", ";
	  branch_qty->get_across_aspect_expression()->_publish_cc_ams_function(_cc_out);
	}
	else {
	  if (branch_qty->_is_across_quantity() == FALSE &&
              branch_qty->get_through_aspect_expression() != NULL ) {
	    _cc_out << ", ";
	    branch_qty->get_through_aspect_expression()->_publish_cc_ams_function(_cc_out);
	  }
	}
	_cc_out << ")";
      }
    }
      break;
    default:
      break;
    }
    decl = architecture_declarative_part.successor(decl);
  }
  decl = cgen_arch_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].first();
  while (decl != NULL) {
    if (decl->_is_implicit_declaration() == TRUE) {
      if ((first == 0) && (firstDeclFlag == 1)) {
	_cc_out << ":" << NL();
	first = 1;
	_is_ams_objects_present = TRUE;
      }
      else {
	_cc_out << "," << NL();
      }
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "(ObjectBase::QUANTITY, \"";
      decl->_publish_cc_lvalue(_cc_out);
      if (decl->get_kind() == IIR_FREE_QUANTITY_DECLARATION) {
	_cc_out << "\", 0, 0, IMPLICIT, NULL, NULL)";
      }
      else {   
	_cc_out << "\", 0, 0, IMPLICIT";
      }
      if (decl->get_kind() == IIR_BRANCH_QUANTITY_DECLARATION) {
	IIR_BranchQuantityDeclaration *branch_qty = (IIR_BranchQuantityDeclaration*)decl;
	IIR_Declaration *plus_terminal = (IIR_Declaration *)(branch_qty->get_plus_terminal_name());
	IIR_Declaration *minus_terminal = (IIR_Declaration *)(branch_qty->get_minus_terminal_name());
	_cc_out << ", &";
	plus_terminal->_publish_cc_lvalue(_cc_out);
	_cc_out << ", &";
	minus_terminal->_publish_cc_lvalue(_cc_out);
	_cc_out << ")";
      }
    }
    decl = cgen_arch_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::QUANTITY].successor(decl);
  }
  _cc_out << NL();
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_form_global_quantity_list(published_file &_cc_out) {
  IIR_Declaration *decl = architecture_declarative_part.first();
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_ams_form_global_quantity_list" );
  while ( decl != NULL) {
    if (decl->get_kind() == IIR_FREE_QUANTITY_DECLARATION ) {
      _cc_out << "globalFreeQuantityList.push_back(&(";
      decl->_publish_cc_lvalue(_cc_out);
      _cc_out << "));" << NL();
    }
    decl =  architecture_declarative_part.successor(decl);
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_allocate_module( published_file &_cc_out ){
  _cc_out << "extern \"C\"";
  _cc_out.start_block();
  _cc_out << "void *allocate" << _get_cc_design_unit_name() << "()";
  _cc_out.start_block();    
  _cc_out <<"return new "
	  << _get_cc_elaboration_class_name();
  _cc_out.end_statement();
  _cc_out.end_block();
  _cc_out.end_block();
}

void IIRScram_ArchitectureDeclaration::_publish_cc_ams_form_characteristic_expressions(published_file &_cc_out) {
  IIR_ArchitectureStatement* arch_stmt;
  
  SCRAM_CC_REF(_cc_out , "IIRScram_ArchitectureDeclaration::_publish_cc_ams_form_characteristic_expressions");
  
  _cc_out << "void" << NL();
  _publish_cc_binding_name(_cc_out.get_stream());
  _cc_out << "_elab::formCharacteristicExpressions() {" << NL();
  _cc_out << "component *equation = NULL;" << NL();
  _cc_out << "component **parent_equation = NULL;" << NL();
  
  // Publish the characteristic expressions for simultaneous statements
  architecture_statement_part._publish_cc_characteristic_expressions(_cc_out);
  
  // Publishing characteristic expressions in the components instantiated
  // in this architecture, if any...
  arch_stmt = architecture_statement_part.first();
  while ( arch_stmt != NULL ) {
    ostringstream objectname;
    if (arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      objectname << *(arch_stmt->_get_label()) << "_elab_obj" << ends;
      _set_current_elab_name( objectname.str() );
      ((IIR_ComponentInstantiationStatement*)arch_stmt)->_publish_form_characteristic_expressions(_cc_out);
    }   
    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }
  _cc_out << "}" << NL();
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_break_signal_info(published_file &_cc_out) {
  IIR_ConcurrentStatement* conc_stmt;
  
  conc_stmt = static_cast<IIR_ConcurrentStatement *>(architecture_statement_part.first());
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      for (IIR_SequentialStatement* seql_stmt =
	     dynamic_cast <IIR_ProcessStatement *>(conc_stmt)->process_statement_part.first() ;
	   seql_stmt != NULL;
	   seql_stmt = dynamic_cast<IIR_ProcessStatement *>(conc_stmt)->process_statement_part.successor(seql_stmt)) {
        ostringstream breakstr;
        if (seql_stmt->get_kind() == IIR_BREAK_STATEMENT) {
          breakstr << dynamic_cast <IIR_ProcessStatement *>(conc_stmt)->_get_cc_process_class() << "_breakSignal" ;
          string breakstring = breakstr.str();
          _cc_out << "EnumerationType *" << breakstring << ";" << NL();
          _cc_out << "EnumerationType " << breakstring << "_info;" << NL();
        }
      }
    }
    conc_stmt = static_cast<IIR_ConcurrentStatement *>(architecture_statement_part.successor(conc_stmt));
  }
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_form_break_set(published_file &_cc_out) {
  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(BREAK_STATEMENT);
  IIR_ConcurrentStatement *conc_stmt = NULL;
  _cc_out << "\nvoid\n";
  _publish_cc_binding_name(_cc_out.get_stream());
  _cc_out << "_elab::formBreakSet() {\n";
  _cc_out << "  list<breakElement *>breakElementList;"<<NL();
  _cc_out << "  breakElement *current_element;" << NL();
  _cc_out << "  breakSet *current_set;" << NL() ;
  _cc_out << "  signalDS *breakSignal;" << NL() << NL();
  conc_stmt = static_cast<IIR_ConcurrentStatement *>(architecture_statement_part.first());
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      for (IIR_SequentialStatement* seql_stmt =
	     dynamic_cast<IIR_ProcessStatement *>(conc_stmt)->process_statement_part.first() ;
	   seql_stmt != NULL;
	   seql_stmt = dynamic_cast<IIR_ProcessStatement *>(conc_stmt)->process_statement_part.successor(seql_stmt)) {
        if (seql_stmt->get_kind() == IIR_BREAK_STATEMENT) {
          seql_stmt->_publish_cc(_cc_out);
        }
      }
    }
    conc_stmt = static_cast<IIR_ConcurrentStatement *>(architecture_statement_part.successor(conc_stmt));
  }
  _cc_out << "\n}" << NL();
  _set_currently_publishing_unit(_saved_publishing_unit);
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_initialize_signals(published_file &_cc_out) {
  IIR_ConcurrentStatement* conc_stmt = NULL;
  IIR_Declaration *decl;
  set<IIR_Declaration> quantity_set;
  PublishedUnit _temp_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(ABOVE_ATTRIBUTE);
  _cc_out << "void\n";
  _publish_cc_binding_name(_cc_out.get_stream());
  _cc_out << "_elab::initializeSignals() {\n";
  _cc_out << "    aboveAttributeInfo *tickAbove = NULL;\n";
  PublishedUnit _saved_publishing_unit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(SIMULTANEOUS_STATEMENT);
  decl = cgen_arch_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].first();
  while(decl != NULL) {
    decl->_publish_cc_init_for_ams(_cc_out);
    decl = cgen_arch_sym_tab_ptr->in_scope_by_type[IIRScram_Declaration::SIGNAL].successor(decl);
  }
 _set_currently_publishing_unit(_saved_publishing_unit);
  IIR_Declaration *current_quantity_decl = NULL;
  conc_stmt = static_cast <IIR_ConcurrentStatement *>
    (architecture_statement_part.first());
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      for (IIR_SequentialStatement* seqstmt = 
	     dynamic_cast<IIR_ProcessStatement *>(conc_stmt)->process_statement_part.first();
	   seqstmt != NULL;
	   seqstmt = dynamic_cast<IIR_ProcessStatement *>(conc_stmt)->process_statement_part.successor(seqstmt)) {
        ostringstream breakstr;
        const string design_unit_name = _get_cc_design_unit_name();
        if (seqstmt->get_kind() == IIR_BREAK_STATEMENT) {
          ASSERT(seqstmt->_get_process_stmt_label() != NULL);
          breakstr <<dynamic_cast <IIR_ProcessStatement *>(conc_stmt)->
	    _get_cc_process_class() << "_breakSignal" ;
          string breakstring = breakstr.str();
          _cc_out << "{" << NL();
          _cc_out << "\t Signal<UniversalBoolean> *signal = " << NL();
          _cc_out << "\t (Signal<UniversalBoolean>*) ((ScalarType*)this->" ;
          _cc_out << breakstring << ")->object;" << NL();
          _cc_out << "\t SignalNetinfo *signal_info = " << NL();
          _cc_out << "\t (SignalNetinfo*) (*(ScalarType*)&"
                  << breakstring << "_info).object;" << NL();
          _cc_out << "\t signal->fanDest = signal_info->obj_ids;" << NL();
          _cc_out << "\t signal->fanout = signal_info->fanout;" << NL();
          _cc_out << "\t signal->id = signal_info->id;" << NL();
          _cc_out << "\t signal->type = EXPLICIT;" << NL();
          _cc_out << "\t signal->sensitive = true;" << NL();
          _cc_out << "\t signal->numAttributes = 0;" << NL();
          _cc_out << "\t signal->attributeList = NULL;" << NL();
          _cc_out << "\t signal->source = signal_info->source;" << NL();
          _cc_out << "\t signal->name = \"" << breakstring << "\";" << NL();
          _cc_out << "}" << NL();
        }
      }
      IIR_ProcessStatement *current_process_stmt = static_cast <IIR_ProcessStatement *> (conc_stmt);
      current_quantity_decl = current_process_stmt->cgen_sym_tab.in_scope_by_type[IIRScram_Declaration::QUANTITY].first();
      while (current_quantity_decl != NULL) {
        quantity_set.add((IIR_Declaration*)current_quantity_decl);
        current_quantity_decl = current_process_stmt->cgen_sym_tab.
          in_scope_by_type[IIRScram_Declaration::QUANTITY].successor(current_quantity_decl);
      }
      current_process_stmt->_publish_cc_ams_tick_above_info( _cc_out );
    }
    conc_stmt = static_cast <IIR_ConcurrentStatement *>
      (architecture_statement_part.successor(conc_stmt));
  }
  current_quantity_decl = quantity_set.get_element();
  while (current_quantity_decl != NULL) {
    current_quantity_decl->_publish_cc_init(_cc_out);
    current_quantity_decl = quantity_set.get_next_element();
  }
  _cc_out << "}\n";
  _set_currently_publishing_unit(_temp_publishing_unit);
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ams_connect_terminals(published_file &_cc_out) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ArchitectureDeclaration::_publish_cc_ams_connect_terminals" );
  
  IIR_ArchitectureStatement *arch_stmt = NULL;
  IIR_InterfaceDeclaration *portelement = NULL;
  IIR_EntityDeclaration *entitydecl = get_entity();
  
  IIR_Declaration *decl = NULL;
  IIR_BranchQuantityDeclaration *branch_qty = NULL;
  IIR *plus_terminal = NULL;
  IIR *minus_terminal = NULL;
  IIR_Boolean firstDeclFlag = TRUE;
  
  // to assign terminal index ...
  int terminal_index = 0;
  
  _cc_out << "void\n";
  this->_publish_cc_binding_name(_cc_out.get_stream()); 
  _cc_out << "_elab::connectTerminals(int numberOfTerminals, ...) {\n";
  _cc_out << "va_list ap;\n";
  _cc_out << "terminalInfo = (AMSType**)new "
          << "char[numberOfTerminals * sizeof(AMSType*)];\n";
  _cc_out << "va_start(ap, numberOfTerminals);\n";
  _cc_out << "for(int i=0; i <numberOfTerminals; i++) {\n";
  _cc_out << "  terminalInfo[i] = va_arg(ap, AMSType*);\n";
  _cc_out << "}\n";
  
  // adding code for mangling the names of the quantities inside component
  // thus a quantity vr inside a component will be renamed as vr_<label>
  // where this label is obtained from the component instantiation
  // statement from that component instantiation. This will enable us to
  // associate quantities in the output with their components.

  _cc_out << "char *labelInfo = va_arg(ap, char *);\n";
  
  _cc_out << "va_end(ap);\n";
  
  _cc_out << "if (numberOfTerminals > 0 ) { \n";
  
  portelement = entitydecl->port_clause.first();  
  
  for (; portelement != NULL;
      portelement = entitydecl->port_clause.successor(portelement) ) {
  
    switch (portelement->get_kind()) {
    case IIR_TERMINAL_INTERFACE_DECLARATION: {
      _cc_out << "  ";
      _cc_out << "setTerminalInfo(";
      portelement->_publish_cc_lvalue(_cc_out);
      _cc_out << " , *(terminalInfo[" << terminal_index << "] ));\n";
      terminal_index++;
    }
      break;
    default:
      break;
    } 
  }
  
  // now we have to publish such that all the quantities which had
  // interface terminals as their plus/minus terminals, get their
  // elaboration information properly ...
  for (decl = architecture_declarative_part.first();
       decl != NULL;
       decl = architecture_declarative_part.successor(decl)) {
    if (decl->_is_branchQ() == TRUE) {
      branch_qty = ((IIR_BranchQuantityDeclaration*)decl);
      plus_terminal = branch_qty->get_plus_terminal_name();
      minus_terminal = branch_qty->get_minus_terminal_name();
      
      if (plus_terminal->_is_interface() == TRUE) {
        // code for name mangling for internal quantities of components.
        if (firstDeclFlag == TRUE) {
          _cc_out << "  char *temp = new char [strlen(\"";
          branch_qty->_publish_cc_lvalue(_cc_out);
          _cc_out << "_\") + strlen(labelInfo)];\n";
          firstDeclFlag = FALSE;
        }
        else {
          _cc_out << "  temp = new char [strlen(\"";
          branch_qty->_publish_cc_lvalue(_cc_out);
          _cc_out << "_\") + strlen(labelInfo)];\n";
        }
        _cc_out << "  strcpy(temp, \"";
        branch_qty->_publish_cc_lvalue(_cc_out);
        _cc_out << "_\");\n";
        _cc_out << "  strcat(temp, labelInfo);\n";
        _cc_out << "  setPlusTerminal(&(";
        branch_qty->_publish_cc_lvalue(_cc_out);
        _cc_out << "), &";
        plus_terminal->_publish_cc_lvalue(_cc_out);
        _cc_out << ", temp);" << NL();
        _cc_out << "   delete [] temp;\n";
      }
      if (minus_terminal->_is_interface() == TRUE) {
        _cc_out << "  setMinusTerminal(&(";
        branch_qty->_publish_cc_lvalue(_cc_out);
        _cc_out << "), &";
        minus_terminal->_publish_cc_lvalue(_cc_out);
        _cc_out << ");" << NL();
      }
    }
  }
  _cc_out << "}\n";
  
  //Publishing output connections for  architecture statement part
  arch_stmt = architecture_statement_part.first();
  while (arch_stmt != NULL) {
    ostringstream objectname;
    if (arch_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      objectname << *(arch_stmt->_get_label()) << "_elab_obj" << ends;
      ((IIR_ComponentInstantiationStatement*)arch_stmt)->_publish_connect_terminals(_cc_out);
    }
    else {
      //No need to do anything for other type of statements
    }
    arch_stmt = architecture_statement_part.successor(arch_stmt);
  }
  _cc_out << "}\n";
}
