// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//	    Magnus Danielson	cfmd@swipnet.se

//---------------------------------------------------------------------------

#include "IIRScram.hh"
#include "IIRScram_BitStringLiteral.hh"
#include "IIR_ArrayTypeDefinition.hh"
#include "IIR_ScalarTypeDefinition.hh"
#include "published_file.hh"
#include "error_func.hh"
#include "sstream-wrap.hh"


IIRScram_BitStringLiteral::~IIRScram_BitStringLiteral() {}


void
IIRScram_BitStringLiteral::_publish_cc_lvalue( published_file &_cc_out ) {
  IIR_TypeDefinition *type = get_subtype();

  SCRAM_CC_REF( _cc_out, "IIRScram_BitStringLiteral::_publish_cc" );

  //Work Around till this node is resolved properly
  if( type != NULL ){
    ASSERT(type->_is_resolved());
    
    if (type->_is_anonymous() == TRUE) {
      if (type->_get_base_type() != NULL) {
	type = type->_get_base_type();
      }
      else {
	if (type->_get_type_mark() != NULL) {
	  ASSERT ( type->_get_type_mark()->_is_anonymous() == FALSE );
	  type = type->_get_type_mark();
	}
      }
    }
    _cc_out << type->_get_cc_type_name();
  }
  else {
    abort();
  }
  
  _cc_out << "(ObjectBase::VARIABLE, ";

  if (type != NULL) {
    if (type->_is_array_type() == TRUE) {
      type->_publish_cc_object_type_info( _cc_out, FALSE );
      _cc_out << ", ";
    }
    
    type->_publish_cc_resolution_function_id( _cc_out );
    _cc_out << ", ";
    
    if ((type->_is_anonymous() == TRUE) || (type->_is_unconstrained_array_type() == TRUE)) {
      _publish_cc_range( _cc_out );
      _cc_out << ", ";
    }
  }
  
  _publish_cc_initialization_value( _cc_out );
 
  _cc_out << ")";
}


void
IIRScram_BitStringLiteral::_publish_cc_initialization_value( published_file &_cc_out ) {
  IIR_Int32 length = get_text_length();
  IIR_Char base_specifier = (*this)[0];
  string value_string;

  SCRAM_CC_REF( _cc_out, "IIRScram_BitStringLiteral::_publish_cc_initialization_value" );

  switch(base_specifier) {
  case 'b':
  case 'B':
    // We have to new the cstringere since this will be deleted at the end.
    value_string = get_text();
    break;
  case 'o':
  case 'O':
    value_string = _convert_octal_to_binary();
    length = value_string.length();
    break;
  case 'x':
  case 'X':
    value_string = _convert_hex_to_binary();
    length = value_string.length();
    break;
  default:
    ostringstream err;
    err << "IIRScram_BitStringLiteral::_publish_cc_initialization_value( published_file &_cc_out ):"
	<< " unrecognized base specifier `" << base_specifier << "'";
    report_error( this, err.str() );
    abort();
  }

  _cc_out << "\"";
  for( IIR_Int32 i = 0; i < length; i++) { 
    if(value_string[i] == '1' || value_string[i] == '0') {
      _cc_out << value_string[i];
    }
  }
  _cc_out << "\"";
}

void 
IIRScram_BitStringLiteral::_publish_cc_condition( published_file &_cc_out ){
  _publish_cc_lvalue( _cc_out );
}


const string
IIRScram_BitStringLiteral::_convert_octal_to_binary() {
  ostringstream bitStream;
  string bitString;
  IIR_Int32 i;
  IIR_Int32 length = get_text_length();

  // Omit the first character since it is B, X, or O
  for(i = 1; i < length; i++) {
    switch((*this)[i]) {
    case '0':
      bitStream << "000"; break;
    case '1':
      bitStream << "001"; break;
    case '2':
      bitStream << "010"; break;
    case '3':
      bitStream << "011"; break;
    case '4':
      bitStream << "100"; break;
    case '5':
      bitStream << "101"; break;
    case '6':
      bitStream << "110"; break;
    case '7':
      bitStream << "111"; break;
    default:
      // The code-generator assumes that all the error-checking has been
      // done by the front-end, and hence no error is reported here.
      break;
    } // switch
  } // for
  // The return memory is newed here.  It is the caller's responsibility
  // to deallocate this memory.
  bitString = bitStream.str();
  return bitString;
}

      
const string
IIRScram_BitStringLiteral::_convert_hex_to_binary() {
  ostringstream bitStream;
  string bitString;
  IIR_Int32 i;
  IIR_Int32 length = get_text_length();

  // Omit the first character since it is B, X, or O
  for(i = 1; i < length; i++) {
    switch((*this)[i]) {
    case '0':
      bitStream << "0000"; break;
    case '1':
      bitStream << "0001"; break;
    case '2':
      bitStream << "0010"; break;
    case '3':
      bitStream << "0011"; break;
    case '4':
      bitStream << "0100"; break;
    case '5':
      bitStream << "0101"; break;
    case '6':
      bitStream << "0110"; break;
    case '7':
      bitStream << "0111"; break;
    case '8':
      bitStream << "1000"; break;
    case '9':
      bitStream << "1001"; break;
    case 'a':
    case 'A':
      bitStream << "1010"; break;
    case 'b':
    case 'B':
      bitStream << "1011"; break;
    case 'c':
    case 'C':
      bitStream << "1100"; break;
    case 'd':
    case 'D':
      bitStream << "1101"; break;
    case 'e':
    case 'E':
      bitStream << "1110"; break;
    case 'f':
    case 'F':
      bitStream << "1111"; break;
    default:
      // The code-generator assumes that all the error-checking has been
      // done by the front-end, and hence no error is reported here.
      break;
    } // switch
  } // for
  // The return memory is newed here.  It is the caller's responsibility
  // to deallocate this memory.
  bitString = bitStream.str();
  return bitString;
}


IIR_Int32
IIRScram_BitStringLiteral::_get_effective_length() {
  IIR_Int32 length = _get_num_digits();
  IIR_Char base_specifier = (*this)[0];
  
  // Get the length of the string as a bit string.
  switch(base_specifier) {
  case 'b':
  case 'B':
    break;
  case 'o':
  case 'O':
    length *= 3;		// One octal digit = 3 bits.
    break;
  case 'x':
  case 'X':
    length *= 4;		// One hex digit = 4 bits.
    break;
  default:
    ostringstream err;
    err << "IIRScram_BitStringLiteral::_publish_cc_range( published_file &_cc_out ):"
	<< " unrecognized base specifier `" << base_specifier << "'";
    report_error( this, err.str() );
    abort();
  }
  return length;
}

      
void
IIRScram_BitStringLiteral::_publish_cc_range( published_file &_cc_out ) {
  IIR_ScalarTypeDefinition *index_type = NULL;
  IIR_ArrayTypeDefinition *array_type = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_BitStringLiteral::_publish_cc_range" );

  if ( get_subtype()->_is_array_type() == TRUE && get_subtype()->_is_single_dimensional_array_type() == TRUE ){
    array_type = (IIR_ArrayTypeDefinition *)get_subtype();
    index_type = array_type->get_index_subtype();
    if ( index_type != NULL && index_type->get_left() != NULL && index_type->get_right() != NULL ){
      index_type->get_left()->_publish_cc_value( _cc_out );
      if ( index_type->_is_ascending_range() == TRUE){
	_cc_out << ", ArrayInfo::to, ";
      } else {
	_cc_out << ", ArrayInfo::downto, ";
      }
      index_type->get_right()->_publish_cc_value( _cc_out );
    } else {
      IIR_Int32 length = _get_effective_length();
      _cc_out << "0, ArrayInfo::to, " << (length - 1);
    }
  } else {
    IIR_Int32 length = _get_effective_length();
    _cc_out << "0, ArrayInfo::to, " << (length - 1);
  }
}


void
IIRScram_BitStringLiteral::_publish_cc_bounds( published_file &_cc_out ) {
  IIR_ScalarTypeDefinition *index_type = NULL;
  IIR_ArrayTypeDefinition  *array_type = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_BitStringLiteral::_publish_cc_bounds" );
  
  if ( get_subtype()->_is_array_type() == TRUE && 
       get_subtype()->_is_single_dimensional_array_type() == TRUE ){
    array_type = (IIR_ArrayTypeDefinition *)get_subtype();
    index_type = array_type->get_index_subtype();
    if (index_type != NULL ){
      index_type->_publish_cc_array_info( _cc_out );
    } else {
      _cc_out << "nullInfo";
    }
  } else {
    _cc_out << "nullInfo";
  }
}


void
IIRScram_BitStringLiteral::_publish_cc_state_object_init( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_BitStringLiteral::_publish_cc_state_object_init" );

  _cc_out << " ";
  if(get_subtype()->_is_iir_access_type_definition() != TRUE) {
    _cc_out << "*";
  }
  _cc_out << "( new ";
  _publish_cc_lvalue( _cc_out );
  _cc_out << ")";
}


void
IIRScram_BitStringLiteral::_publish_cc_headers( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_BitStringLiteral::_publish_cc_headers" );
  if(get_subtype() != NULL) {
    get_subtype()->_publish_cc_headers( _cc_out );
  }
}


// This method counts the number of digits in the string.  It assumes that
// the string is made of a base specifier, a quote, a sequence of digits
// and ending with a quote.  Further, the string within the quotes is made
// of *valid* digits and the '_' character only.
IIR_Int32
IIRScram_BitStringLiteral::_get_num_digits() {
  IIR_Int32 length = get_text_length();
  IIR_Int32 num_digits = 0;

  // We omit the bit specifier and the two quotes.
  for( int i = 2; i < length-1; i++ ){
    if((*this)[i] != '_') {
      num_digits++;
    }
  }
  return num_digits;
}


set<IIR_TypeDefinition> *
IIRScram_BitStringLiteral::_get_rval_set( IIR_Boolean (IIR::*constraint_function)() ){
  return _get_rval_set_for_string();
}


void
IIRScram_BitStringLiteral::_type_check( set<IIR_TypeDefinition> * ){
}


IIR *
IIRScram_BitStringLiteral::_clone() {
  return this;
}

visitor_return_type *IIRScram_BitStringLiteral::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_BitStringLiteral(this, arg);
};
