// Copyright (c) 1996-2000 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Umesh Kumar V. Rajasekaran urajasek@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu

//---------------------------------------------------------------------------

#include "IIR_SelectedName.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_SimpleName.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_RecordTypeDefinition.hh"
#include "IIR_FileDeclaration.hh"
#include "error_func.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"


extern symbol_table *cgen_sym_tab_ptr;
extern bool debug_symbol_table;

IIRScram_SelectedName::~IIRScram_SelectedName() {}

void 
IIRScram_SelectedName::_publish_vhdl(ostream &_vhdl_out) {
  ASSERT(_is_resolved() == TRUE);
  get_prefix()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ".";
  get_suffix()->_publish_vhdl(_vhdl_out);
}

void
IIRScram_SelectedName::_publish_cc_object_name( published_file &_cc_out ) {
  _publish_cc_lvalue( _cc_out );
}

void 
IIRScram_SelectedName::_publish_cc_lvalue( published_file &_cc_out ) {
  if ((_get_currently_publishing_unit() == PROCESS) ||
      (_get_currently_publishing_unit() == PROCESS_STATE)) {
    if(_get_package_declaration() != NULL) {
      if(!cgen_sym_tab_ptr->in_scope(_get_package_declaration())) {
	cgen_sym_tab_ptr->add_declaration(_get_package_declaration());
      }
    }
  }

  ASSERT( _is_resolved() == TRUE );
  if (get_prefix()->_is_iir_attribute_declaration() == FALSE)  {
    ASSERT( get_subtype() != NULL );

    if ((get_suffix()->get_kind() != IIR_FILE_DECLARATION) ||
	(((IIR_FileDeclaration *) get_suffix())->_is_shared_file() == FALSE)) {
      _cc_out << "((";
      _cc_out << get_subtype()->_get_cc_type_name();
      _cc_out << " &)";
    }
  }
 
  if( _get_prefix_declaration() != NULL && 
      _get_prefix_declaration()->_is_object() == TRUE &&
      get_prefix()->_is_iir_label() == FALSE ){
    
    if ( get_prefix()->get_subtype()->_is_iir_access_type_definition() == TRUE &&
	 _get_suffix()->get_kind() != IIR_DESIGNATOR_BY_ALL ){
      // Okay. For access types we need to publish the .all() after the
      // variable and explicitly type qualify it so that the suceeding
      // expressions can use the value without much ado
      IIR_AccessTypeDefinition *type = (IIR_AccessTypeDefinition *) get_prefix()->get_subtype();
      ASSERT ( type->_is_iir_access_type_definition() == TRUE );
      _cc_out << "((";
      _cc_out << type->get_designated_type()->_get_cc_type_name();
      _cc_out << " &)";
    }
    else if (get_prefix()->get_subtype()->_is_iir_access_type_definition() == TRUE) {
      IIR_AccessTypeDefinition *type = (IIR_AccessTypeDefinition *) get_prefix()->get_subtype();
      _cc_out << "(";
      _cc_out << type->get_designated_type()->_get_cc_type_name();
      _cc_out << " &) ((AccessType &) ";
    }
    
    get_prefix()->_publish_cc_lvalue( _cc_out );
    
    if (get_prefix()->get_subtype()->_is_iir_access_type_definition() == TRUE)  {
      if (_get_suffix()->get_kind() != IIR_DESIGNATOR_BY_ALL) {
	_cc_out << ".all())";
      }
      else {
        _cc_out << ")";
      }
    }
    
    _cc_out << ".";
  }

  if ((get_prefix()->get_subtype() != NULL) &&
      (get_prefix()->get_subtype()->_is_record_type() == TRUE)) {
    // Okay... for record types we don't publish the field name, but rather access
    // the field using the field number...
    
    IIR_RecordTypeDefinition *recType = NULL;

    if (get_prefix()->get_subtype()->_is_iir_access_type_definition() == FALSE)  {
      recType = (IIR_RecordTypeDefinition *) get_prefix()->get_subtype();
    }
    else {
      recType = (IIR_RecordTypeDefinition *) (((IIR_AccessTypeDefinition *) get_prefix()->get_subtype())->get_designated_type());
    }

    ASSERT ( recType != NULL );
    ASSERT ( recType->_is_iir_record_type_definition() ==  TRUE );

    _cc_out << "get_field(";
    _cc_out << recType->_get_field_number(_get_suffix());
    _cc_out << ")";
  }
  else {
    _get_suffix()->_publish_cc_lvalue( _cc_out );
  }

  if (get_prefix()->_is_iir_attribute_declaration() == FALSE)  {
    if ((get_suffix()->get_kind() != IIR_FILE_DECLARATION) ||
	(((IIR_FileDeclaration *) get_suffix())->_is_shared_file() == FALSE)) {
      _cc_out << ")";
    }
  }
}

void 
IIRScram_SelectedName::_publish_cc_declarator( published_file &_cc_out ) {
  if(get_prefix()->_is_signal() == TRUE) {
    get_prefix()->_publish_cc_declarator( _cc_out );
  }
  else {
    get_suffix()->_publish_cc_declarator( _cc_out );
  }
}

void
IIRScram_SelectedName::_publish_cc_elaborate( published_file &_cc_out ) {
  get_prefix()->_publish_cc_elaborate( _cc_out );
  _cc_out << ".";
  get_suffix()->_publish_cc_elaborate( _cc_out );
}

void
IIRScram_SelectedName::_set_passed_through_out_port(IIR_Boolean val) {
  ASSERT ( get_prefix() != NULL );
  get_prefix()->_set_passed_through_out_port(val);
}

void 
IIRScram_SelectedName::_add_decl_into_cgen_symbol_table() {
  get_prefix()->_add_decl_into_cgen_symbol_table();
  get_suffix()->_add_decl_into_cgen_symbol_table();
}
  
IIR_Declaration*
IIRScram_SelectedName::_get_prefix_declaration() {
  // Since we mangle names, we need not have a selected name.  We just
  // need the object declaration, so return the suffix.
  if(get_prefix()->_is_iir_label() == TRUE) {
    ASSERT(get_prefix()->_is_iir_declaration() == TRUE);
    return (IIR_Declaration *) get_suffix();
  } 
  else {
    return get_prefix()->_get_prefix_declaration();
  }
}

IIR_Declaration*
IIRScram_SelectedName::_get_package_declaration() {
  if(_get_suffix()->get_kind() == IIR_PACKAGE_DECLARATION) {
    return (IIR_Declaration*) _get_suffix();
  }
  else {
    return get_prefix()->_get_package_declaration();
  }
}

IIR_LibraryDeclaration *
IIRScram_SelectedName::_get_library_declaration(){
  ASSERT( _is_resolved() == true );
  ASSERT( get_prefix() != NULL );
  if( get_prefix()->get_kind() == IIR_LIBRARY_DECLARATION ){
    return (IIR_LibraryDeclaration *)get_prefix();
  }
  else{
    ASSERT( get_prefix()->get_kind() == IIR_SELECTED_NAME );
    return ((IIR_SelectedName *)get_prefix())->_get_library_declaration();
  }
}

ostream &
IIRScram_SelectedName::_print( ostream &os ){
  os << *get_prefix();
  os << ".";
  os << *get_suffix();
  
  return os;
}

void
IIRScram_SelectedName::_publish_cc_state_object_init( published_file &_cc_out ){
  _publish_cc_lvalue( _cc_out );
}

void
IIRScram_SelectedName::_publish_cc_bounds( published_file &_cc_out )
{
  _cc_out << "defaultInfo";
}

set<IIR_Declaration> *
IIRScram_SelectedName::_symbol_lookup(){
  set<IIR_Declaration> *prefix_decls;
  set<IIR_Declaration> *retval = NULL;

  ASSERT( get_prefix() != NULL );
  ASSERT( get_suffix() != NULL );

  prefix_decls = get_prefix()->_symbol_lookup();
  if( prefix_decls == NULL ){
    return NULL;
  }
  else{
    IIR_Declaration *current_decl = prefix_decls->get_element();
    set<IIR_Declaration> *suffix_decls = NULL;
    while( current_decl != NULL ){
      IIR_Boolean found_in_library = FALSE;
      delete suffix_decls;
      suffix_decls = get_suffix()->_symbol_lookup( prefix_decls );
      if( suffix_decls == NULL ){
	set<IIR_Declaration> *decls = current_decl->_find_declarations( this );
	if( decls != NULL ){
	  suffix_decls = decls;
	  found_in_library = TRUE;
	}
	delete decls;
      }
      if( suffix_decls != NULL ){
	if( retval == NULL ){
	  retval = new set<IIR_Declaration>;
	}
	retval->add( suffix_decls );
	if( found_in_library == TRUE ){
	  current_decl->_add_to_declarative_region( suffix_decls );
	}
      }      
      current_decl = prefix_decls->get_next_element();
    }
    delete suffix_decls;
  }

  delete prefix_decls;
  return retval;
}

IIR_ArchitectureDeclaration *
IIRScram_SelectedName::_get_cc_instantiated_architecture(){
  return get_suffix()->_get_cc_instantiated_architecture();
}

IIR_EntityDeclaration *
IIRScram_SelectedName::_get_cc_instantiated_entity(){
  return get_suffix()->_get_cc_instantiated_entity();
}


IIR *
IIRScram_SelectedName::_get_suffix(){
  ASSERT( get_suffix() != NULL );
  return get_suffix();
}

const string
IIRScram_SelectedName::_convert_to_library_name(){
  string retval;
  string tmp_str = get_prefix()->_convert_to_library_name();
  if(get_suffix()->get_kind() != IIR_DESIGNATOR_BY_ALL){
    if((get_prefix()->get_kind() == IIR_SELECTED_NAME) && 
       (get_suffix()->get_kind() == IIR_SIMPLE_NAME)){
      //this is for handling the case where IIR_DESIGNATOR_BY_ALL is not
      //used in the use clause. ie. some type/subtype/function has been
      //specified.
      return tmp_str;
    }
    else {
      retval = tmp_str + library_manager::get_library_suffix();
    }
  }
  else {
    retval = tmp_str;
  }

  if(get_suffix()->get_kind() != IIR_DESIGNATOR_BY_ALL){
    retval += "/";
  }

  tmp_str = get_suffix()->_convert_to_library_name();

  if( tmp_str != "" ){
    retval += library_manager::get_package_suffix();
  }

  return retval;
}

IIR *
IIRScram_SelectedName::_rval_to_decl( IIR_TypeDefinition *my_type ){
  if( _is_resolved() == FALSE ){
    set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup();
    if( prefix_decls == NULL ){
      set<IIR_TypeDefinition> *prefix_rvals = get_prefix()->_get_rval_set();
      
      // This returns to us the proper type of the prefix...
      IIR_TypeDefinition *prefix_rval = get_suffix()->_determine_rval_in_set( prefix_rvals,my_type );
      // Resolve the prefix...
      set_prefix( get_prefix()->_semantic_transform( prefix_rval ) );
      get_prefix()->_type_check( prefix_rval );
      set_prefix( get_prefix()->_rval_to_decl( prefix_rval ) );
      
      // Finish resolving the suffix...
      set_suffix( get_suffix()->_rval_to_decl( prefix_rval, my_type ) );
      delete prefix_rvals;
    }
    else{
      IIR_Declaration *prefix_decl;
      prefix_decl = get_suffix()->_determine_decl_in_set( prefix_decls, my_type );
      ASSERT( prefix_decl != NULL );
      // Tell the prefix this is it's decl, and continue
      set_prefix( get_prefix()->_decl_to_decl( prefix_decl ) );
      set_suffix( get_suffix()->_rval_to_decl( prefix_decl, my_type ));
    }
    delete prefix_decls;
  }

  return this;
}

void 
IIRScram_SelectedName::_type_check( set<IIR_TypeDefinition> * ){
  // Don't need to do anything here...
}

IIR_TypeDefinition *
IIRScram_SelectedName::get_subtype(){
  ASSERT( _is_resolved() == TRUE );
  
  if( get_suffix()->get_kind() != IIR_DESIGNATOR_BY_ALL ){
    return get_suffix()->get_subtype();
  }
  else{
    return get_prefix()->_get_rval_pointed_at();
  }
}

set<IIR_TypeDefinition> *
IIRScram_SelectedName::_get_rval_set(IIR_Boolean (IIR::*constraint_function)()){
  set<IIR_TypeDefinition> *retval;
  if( _is_resolved() == FALSE ){
    set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup();
    
    if( prefix_decls == NULL ){
      set<IIR_TypeDefinition> *prefix_rvals = get_prefix()->_get_rval_set(constraint_function);
      if( prefix_rvals == NULL ){
	return NULL;
      }
      retval = get_suffix()->_get_rval_set( prefix_rvals, constraint_function );
      delete prefix_rvals;
    }
    else{
      retval = get_suffix()->_get_rval_set( prefix_decls, constraint_function );
    }
    delete prefix_decls;
  }
  else{
    retval = new set<IIR_TypeDefinition>( get_subtype() );
  }

  return retval;
}

IIR_Boolean
IIRScram_SelectedName::_is_resolved( ){
  if( get_prefix()->_is_resolved() == TRUE && get_suffix()->_is_resolved() ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean
IIRScram_SelectedName::_is_signal( ){
  ASSERT( _is_resolved() == TRUE );
  if( get_prefix()->_is_signal() == TRUE || get_suffix()->_is_signal() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean
IIRScram_SelectedName::_is_variable( ){
  ASSERT( _is_resolved() == TRUE );
  if( get_prefix()->_is_variable() == TRUE || get_suffix()->_is_variable() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean
IIRScram_SelectedName::_is_object( ){
  ASSERT( _is_resolved() == TRUE );
  if( get_prefix()->_is_object() == TRUE || get_suffix()->_is_object() == TRUE ){
    return TRUE;
  }
  else{
    return FALSE;
  }
}

IIR_Boolean
IIRScram_SelectedName::_is_readable( ){
  ASSERT( _is_resolved() == TRUE );
  if( get_prefix() && get_prefix()->_is_object() ){
    return get_prefix()->_is_readable();
  }
  else{
    return get_suffix()->_is_readable();
  }
}

IIR_Boolean
IIRScram_SelectedName::_is_writable( ){
  ASSERT( _is_resolved() == TRUE );
  if( get_prefix() && get_prefix()->_is_object() ){
    return get_prefix()->_is_writable();
  }
  else{
    return get_suffix()->_is_writable();
  }
}

IIR_Boolean
IIRScram_SelectedName::_is_type( ){
  ASSERT( _is_resolved() == TRUE );
  return get_suffix()->_is_type();
}

IIR_Boolean
IIRScram_SelectedName::_is_entity_decl() {
  ASSERT ( _is_resolved() == TRUE );
  
  if ((get_suffix()->_is_entity_decl() == TRUE) ||
      (get_prefix()->_is_entity_decl() == TRUE)) {
    return TRUE;
  }
  
  return FALSE;
}

IIR_Boolean
IIRScram_SelectedName::_is_configuration_decl() {
  ASSERT ( _is_resolved() == TRUE );
  
  if ((get_suffix()->_is_configuration_decl() == TRUE) ||
      (get_prefix()->_is_configuration_decl() == TRUE)) {
    return TRUE;
  }
  
  return FALSE;
}

IIR_Boolean 
IIRScram_SelectedName::_is_by_all(){
  return get_suffix()->_is_by_all();
}

void
IIRScram_SelectedName::_set_suffix(IIR *suffix){
  ASSERT( suffix != NULL );

  set_suffix(suffix);
}

IIR *
IIRScram_SelectedName::_decl_to_decl( IIR_Declaration *suffix_decl ){
  set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup();
  if( prefix_decls == NULL ){
    // Then someting probably went wrong - we shouldn't have ended up here,
    // I don't think...
    ostringstream err;
    err << "Internal error in IIRScram_SelectedName::_set_selected_name_declaration -"
	<< " found no prefix declarations.";
    report_error( this, err.str() );
    abort();
  }
  
  if( get_suffix()->get_kind() == IIR_DESIGNATOR_BY_ALL ){
    IIR_Declaration *current_decl = prefix_decls->get_element();
    while( current_decl != NULL ){
      if( current_decl == suffix_decl ){
	set_prefix( get_prefix()->_decl_to_decl( current_decl ) );
	break;
      }
      current_decl = prefix_decls->get_next_element();
    }    
  }
  else{
    set_suffix( suffix_decl );
    
    IIR_Declaration *current_outer_decl = prefix_decls->get_element();
    while( current_outer_decl != NULL ){
      IIR_SimpleName temp_name;
      temp_name.set_prefix( suffix_decl->get_declarator() );
      set<IIR_Declaration> *found = current_outer_decl->_find_declarations( &temp_name );
      if( found != NULL ){
	IIR_Declaration *current_inner_decl = found->get_element();
	while( current_inner_decl != NULL ){
	  if( current_inner_decl == suffix_decl ){
	    set_prefix( get_prefix()->_decl_to_decl( current_outer_decl ) );
	    break;
	  }
	  current_inner_decl = found->get_next_element();
	}
	delete found;
      }
      current_outer_decl = prefix_decls->get_next_element();
    }
  }

  delete prefix_decls;

  ASSERT( _is_resolved() == TRUE );
  return this;
}
  
void 
IIRScram_SelectedName::_build_lib( IIR_LibraryUnit *new_unit ){
  if( get_prefix()->get_kind() == IIR_SIMPLE_NAME ){
    set<IIR_Declaration> *prefix_decls = get_prefix()->_symbol_lookup();
    if( prefix_decls != NULL ){
      IIR_Declaration *current_lib = prefix_decls->get_element();
      while( current_lib != NULL ){
	if( current_lib->get_kind() != IIR_LIBRARY_DECLARATION ){
	  prefix_decls->remove( current_lib );
	}
	current_lib = prefix_decls->get_next_element();
      }
      
      ASSERT( prefix_decls->num_elements() == 1 );
      IIR_Declaration *decl = prefix_decls->get_element();
      ASSERT( decl->get_kind() == IIR_LIBRARY_DECLARATION );
      IIR_LibraryDeclaration *lib_decl = (IIR_LibraryDeclaration *)decl;
      
      lib_decl->primary_units.append( new_unit );
      
    }
    else{
      ostringstream err;
      err << "Can't find library |" << *get_prefix() << "| - perhaps you're"
	  << " missing a library clause?";
      report_error( this, err.str() );
    }
  }
  else{
    ASSERT( get_prefix()->get_kind() == IIR_SELECTED_NAME );
    ((IIR_Name *)get_prefix())->_build_lib( new_unit );
  }
}

void 
IIRScram_SelectedName::_make_interface_visible( symbol_table *sym_tab ){
  ASSERT( _is_resolved() == TRUE );

  if( get_suffix()->get_kind() == IIR_DESIGNATOR_BY_ALL ){
    get_prefix()->_make_interface_visible( sym_tab );
  }
  else{
    // This should actually be "_make_visible", not
    // "_make_interface_visible".  However the there is 
    // no "make_visible" defined yet, and they are essentially
    // equivalent for the moment.
    get_suffix()->_make_interface_visible( sym_tab );
  }
}

void 
IIRScram_SelectedName::_come_into_scope( symbol_table *sym_tab ){
  ASSERT( get_suffix()->_is_resolved() == TRUE );
  get_suffix()->_come_into_scope( sym_tab );
}

void 
IIRScram_SelectedName::_come_out_of_scope( symbol_table *sym_tab ){
  get_suffix()->_come_out_of_scope( sym_tab );
}


IIR_GenericList *
IIRScram_SelectedName::_get_generic_list(){
  ASSERT( _is_resolved() == TRUE );
  ASSERT( get_suffix() != NULL );
  return get_suffix()->_get_generic_list();
}

IIR_PortList *
IIRScram_SelectedName::_get_port_list(){
  ASSERT( _is_resolved() == TRUE );
  ASSERT( get_suffix() != NULL );
  return get_suffix()->_get_port_list();
}


IIR *
IIRScram_SelectedName::_clone(){
  IIR_SelectedName *retval = new IIR_SelectedName();
  IIR_Name::_clone( retval );

  IIR *suffix = get_suffix()->_clone();
  retval->set_suffix( suffix );
  
  return retval;
}

IIR_Boolean 
IIRScram_SelectedName::_is_locally_static(){
  IIR_Boolean retval = TRUE;
  
  if( get_prefix()->_is_locally_static() == FALSE || 
      get_suffix()->_is_locally_static() == FALSE ){
    retval = FALSE;
  }
  
  return retval;
}

IIR_Boolean 
IIRScram_SelectedName::_is_globally_static(){
  IIR_Boolean retval = TRUE;
  
  if( get_prefix()->_is_globally_static() == FALSE || 
      get_suffix()->_is_globally_static() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

IIR_SignalKind
IIRScram_SelectedName::_get_signal_kind(){
  if( get_suffix()->_is_signal() == TRUE ){
    return get_suffix()->_get_signal_kind();
  }
  else if( get_prefix()->_is_signal() == TRUE ){
    return get_suffix()->_get_signal_kind();
  }
  else{
    return IIR_NO_SIGNAL_KIND;
  }
}

IIR_TypeDefinition *
IIRScram_SelectedName::_get_rval_pointed_at(){
  ASSERT( _is_resolved() == TRUE );
  return get_suffix()->_get_rval_pointed_at();
}

visitor_return_type *
IIRScram_SelectedName::_accept_visitor( node_visitor *visitor,
					visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_SelectedName(this, arg);
}
