
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Krishnan Subramani 
//          Narayanan Thondugulam

#include "IIRScram_SequentialStatementList.hh"
#include "IIR_Identifier.hh"
#include "IIR_ProcedureCallStatement.hh"
#include "IIR_AboveAttribute.hh"
#include "savant.hh"
#include "published_file.hh"
#include <clutils/StringUtilities.h>

IIRScram_SequentialStatementList::~IIRScram_SequentialStatementList() {}

void 
IIRScram_SequentialStatementList::_build_wait_list(dl_list<IIRScram_WaitStatement> *list) {
  IIR_SequentialStatement *stmt = first();
  while (stmt != NULL) {
    stmt->_build_wait_list(list);
    stmt = successor(stmt);
  }
}

void 
IIRScram_SequentialStatementList::_build_procedure_call_stmt_list(dl_list<IIR_ProcedureCallStatement> *list) {
  IIR_SequentialStatement *stmt = first();
  while (stmt != NULL) {
    stmt->_build_procedure_call_stmt_list(list);
    stmt = successor(stmt);
  }
}

void 
IIRScram_SequentialStatementList::_publish_vhdl(ostream &_vhdl_out) {
  IIR_SequentialStatement *stmt;
  for (stmt = first(); stmt != NULL; stmt = successor(stmt)) {
    stmt->_publish_vhdl(_vhdl_out);
    _vhdl_out << ";\n";
  }
}

void 
IIRScram_SequentialStatementList::_publish_cc( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_SequentialStatementList::_publish_cc" );
  IIR_SequentialStatement *stmt;
  for (stmt = first(); stmt != NULL; stmt = successor(stmt)) {
    if(stmt->get_line_number() != -1 &&
       stmt->get_file_name() != NULL) {
      _cc_out.insert_comment( string( "line " ) + intToString( stmt->get_line_number() ) +
			      string( " of file " ) + stmt->get_file_name()->_to_string()
			      + "(" + stmt->get_kind_text() + ")" );
    }
    stmt->_publish_cc( _cc_out );
  }
}

// Publishes the declaration for the iterator used in a for loop
// statement.  If not done outside all the loops, C++ will complain that
// it can not jump to a label because it crosses the initialization of
// some variable.
void
IIRScram_SequentialStatementList::_publish_cc_decl_for_loop_iterator( published_file &_cc_out ) {
  IIR_SequentialStatement* stmt = first();
  while(stmt != NULL) {
    stmt->_publish_cc_decl_for_loop_iterator( _cc_out );
    stmt = successor(stmt);
  }
}  

void
IIRScram_SequentialStatementList::_get_list_of_input_signals( set<IIR> *list ){
  IIR_SequentialStatement* stmt = first();
  while( stmt != NULL ) {
    stmt->_get_list_of_input_signals(list);
    stmt = successor(stmt);
  }
}

void
IIRScram_SequentialStatementList::_get_signal_source_info( set<IIR> *siginfo ){
  IIR_SequentialStatement* stmt = first();
  while( stmt != NULL ){
    // XXX SK This has to be handled for all statements.
    if(stmt->_has_statement_list() || 
       stmt->_is_sequential_signal_assignment() ||
       stmt->get_kind() == IIR_PROCEDURE_CALL_STATEMENT) {
      stmt->_get_signal_source_info(siginfo);
    }
    stmt = successor(stmt);
  }
}

IIR_Boolean
IIRScram_SequentialStatementList::_is_above_attribute_found() {
  IIR_Boolean retval = FALSE;

  IIR_SequentialStatement *stmt = first();
  while (stmt != NULL) {
    retval = stmt->_is_above_attribute_found() || retval;
    stmt = successor(stmt);
  }
  return retval;
}

void
IIRScram_SequentialStatementList::_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  IIR_SequentialStatement *stmt = first();
  while (stmt != NULL) {
    stmt->_build_above_attribute_set(to_build);
    stmt = successor(stmt);
  }
}

visitor_return_type *
IIRScram_SequentialStatementList::_accept_visitor( node_visitor *visitor,
						   visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_SequentialStatementList(this, arg);
};
