/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_column_dialog.h"
#include "sg_stock.h"

static GtkWidget *name_entry;
static GtkWidget *apply_check;
static GtkWidget *type_combo;
static GtkWidget *format_combo;
static GtkWidget *internal_combo;
static GtkWidget *precision_spin;

static gint the_column;
static GtkSheet *the_sheet;

static gchar *types[] = {
                           "Number",
                           "Text",
                           "Date",
                           "Time",
                           NULL
};

static gchar *formats[] = {
                           "Decimal (1000)",
                           "Scientific (1.E3)",
                           NULL
};

static gchar *internals[] = {
                           "Integer",
                           "Double",
                           NULL
};

static gboolean
mw_destroy(GtkWidget *widget)
{
  /* This is needed to get out of gtk_main */
  gtk_main_quit ();

  return FALSE;
}

static void
update_worksheet(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gchar *name;
  gint precision;
  SGcolumnformat format;
  SGcolumntype type;
  SGcolumninternal internal;
  gboolean all_columns;
  gint width;
  gint col_width;

  if(active_worksheet->mode != SG_MODE_MATRIX){
    name = gtk_entry_get_text(GTK_ENTRY(name_entry));

    gtk_sheet_set_column_title(the_sheet, the_column, name);
    gtk_sheet_column_button_add_label(the_sheet, the_column, name);

    width = gdk_string_width(GTK_WIDGET(the_sheet)->style->font, name) + 12;
    if(width > the_sheet->column[the_column].width)
      gtk_sheet_set_column_width(the_sheet, the_column, width); 
  
  }

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(type_combo)->list)->selection->data;
  type = (SGcolumntype)gtk_list_child_position(GTK_LIST(GTK_COMBO(type_combo)->list), child);

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(format_combo)->list)->selection->data;
  format = (SGcolumnformat)gtk_list_child_position(GTK_LIST(GTK_COMBO(format_combo)->list), child);

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(internal_combo)->list)->selection->data;
  internal = (SGcolumninternal)gtk_list_child_position(GTK_LIST(GTK_COMBO(internal_combo)->list), child);

  precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(precision_spin));

  if(active_worksheet->mode != SG_MODE_MATRIX)
    all_columns = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(apply_check));
  else
    all_columns = TRUE;

  gtk_sheet_freeze(the_sheet);
  if(active_worksheet->mode != SG_MODE_MATRIX)
    sg_worksheet_column_set_format(active_worksheet, the_column, 
                                   type,
                                   format,
                                   internal,
                                   precision);

  if(all_columns || active_worksheet->mode == SG_MODE_MATRIX){
   gint i;
   for(i = MAX(the_column, 0); i <= the_sheet->maxcol; i++)
     sg_worksheet_column_set_format(active_worksheet, i, 
                                    type,
                                    format,
                                    internal,
                                    precision);
  }

  gtk_sheet_thaw(the_sheet);

  sg_project_refresh_datasets();
  gtk_widget_destroy(widget);
}


static void
type_changed(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gint type;
  gint internal;
 
  child = (GtkWidget*)GTK_LIST(GTK_COMBO(type_combo)->list)->selection->data;
  type = gtk_list_child_position(GTK_LIST(GTK_COMBO(type_combo)->list), child);

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(internal_combo)->list)->selection->data;
  internal = gtk_list_child_position(GTK_LIST(GTK_COMBO(internal_combo)->list), child);

  if(type == SG_TYPE_NUMBER){
      gtk_widget_set_sensitive(internal_combo, TRUE);
      gtk_widget_set_sensitive(format_combo, TRUE);
      if(internal == SG_INTERNAL_INTEGER)
        gtk_widget_set_sensitive(precision_spin, FALSE);
      else
        gtk_widget_set_sensitive(precision_spin, TRUE);
  } else {
      gtk_widget_set_sensitive(precision_spin, FALSE);
      gtk_widget_set_sensitive(format_combo, FALSE);
      gtk_widget_set_sensitive(internal_combo, FALSE);
  }

}


void
sg_column_dialog (SGworksheet *worksheet, gint col)
{
  GtkWidget *window = NULL;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *main_box;
  GtkWidget *ok_button, *cancel_button;
  GtkWidget *action_area;
  GtkWidget *table;
  GtkSheetColumn *column;
  SGcolumn *wcolumn;
  GtkAdjustment *adj;
  gchar text[100];

  the_column = col;
  the_sheet = GTK_SHEET(worksheet->sheet);
  if(worksheet->mode != SG_MODE_MATRIX){
    column = &(GTK_SHEET(worksheet->sheet)->column[col]);
    wcolumn = &worksheet->column[the_column];
  } else {
    wcolumn = &worksheet->matrix;
  }

  window=gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);

  if(worksheet->mode != SG_MODE_MATRIX)
    gtk_window_set_title (GTK_WINDOW(window),"Column properties");
  else
    gtk_window_set_title (GTK_WINDOW(window),"Matrix properties");

  gtk_window_set_modal (GTK_WINDOW(window),TRUE);

  /* Create widgets */
  main_box = gtk_vbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 10);
  gtk_container_add (GTK_CONTAINER (window), main_box);

  frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (main_box), frame, FALSE, FALSE, 0);

  table = gtk_table_new(7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  if(worksheet->mode != SG_MODE_MATRIX){
    name_entry = gtk_entry_new();   

    label = gtk_label_new("Column name"),
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), 
                              label,
                              0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table), name_entry, 1, 2, 0, 1);


    gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 2, 1, 2);
  }

  type_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(type_combo), types);

  label = gtk_label_new("Data type"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), type_combo, 1, 2, 2, 3);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(type_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(type_combo)->list), 
                       wcolumn->type); 
  if(worksheet->mode == SG_MODE_MATRIX)
    gtk_widget_set_sensitive(type_combo, FALSE);

  format_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(format_combo), formats);

  label = gtk_label_new("Format"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), format_combo, 1, 2, 3, 4);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(format_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(format_combo)->list), 
                       wcolumn->format); 


  label = gtk_label_new("Precision"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 4, 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 16., 1., 1., 0.);
  precision_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(precision_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(precision_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), precision_spin, 1, 2, 4, 5);


  internal_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(internal_combo), internals);

  label = gtk_label_new("Representation"),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 5, 6);
  gtk_table_attach_defaults(GTK_TABLE(table), internal_combo, 1, 2, 5, 6);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(internal_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(internal_combo)->list), 
                       wcolumn->internal); 

  if(worksheet->mode != SG_MODE_MATRIX){
    apply_check = gtk_check_item_new_with_label("Apply to all columns to the right");
    gtk_table_attach_defaults(GTK_TABLE(table), apply_check, 0, 2, 6, 7);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(apply_check), FALSE);
  }

  /* Action Area */
  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_end (GTK_BOX (main_box), action_area, FALSE, FALSE, 0);
  gtk_widget_show (action_area);

  ok_button = sg_stock_button(GNOME_STOCK_BUTTON_OK);
  GTK_WIDGET_SET_FLAGS (ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (ok_button);
  gtk_widget_show (ok_button);

  cancel_button = sg_stock_button(GNOME_STOCK_BUTTON_CANCEL);
  GTK_WIDGET_SET_FLAGS (cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (cancel_button);

  /* connect signals */
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (gtk_widget_destroy),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
                             GTK_SIGNAL_FUNC (update_worksheet),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (GTK_COMBO(type_combo)->entry), 
                             "changed",
                             GTK_SIGNAL_FUNC (type_changed),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (GTK_COMBO(internal_combo)->entry), 
                             "changed",
                             GTK_SIGNAL_FUNC (type_changed),
                             GTK_OBJECT (window));

  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (mw_destroy),NULL);

  /* Show widgets */

  if(worksheet->mode != SG_MODE_MATRIX){
    if(column->name)
      sprintf(text,"%s",column->name);
    else
      sprintf(text,"%d",col);

    gtk_entry_set_text(GTK_ENTRY(name_entry), text);
  }

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(precision_spin), 
                            active_worksheet->column[the_column].precision);

  type_changed(NULL, NULL);
  gtk_widget_show_all (window);

  /* wait until dialog get destroyed */
  gtk_main();

  return;
}
