/*  Screem:  imageWizard.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <libgnomeui/gnome-file-entry.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-util.h>

#include <glib/gfileutils.h>

#include <gtk/gtk.h>

#include <glade/glade.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "screem-page.h"
#include "screem-dtd.h"

#include "fileops.h"

static const gchar *align[] = {
	"bottom",
	"middle",
	"top",
	"left",
	"right",
	NULL
};

/* per wizard struct */
typedef struct {
	GladeXML *xml;
	ScreemWindow *window;
	ScreemEditor *editor;

	GtkWidget *dialog;

	GSList *popup_attrs;
	guint popup_start;
	guint popup_len;
} ImageWizard;

/* keep track of how many wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 4;

#define NAME "Image Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG "img"

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void popup( ScreemWindow *window,
			    guint start, guint len, GSList *attrs );
void image_wizard_display( GtkAction *action, gpointer user_data );
void image_wizard_clicked( GtkWidget *widget, gint button, gpointer data );
gboolean image_wizard_preview( GtkWidget *widget );

static void image_wizard_align_init( GladeXML *xml );

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	ImageWizard *image_wizard;
	const gchar *ui = "\
<ui>\
<menubar>\
<menu action=\"Insert\">\
<menu action=\"Wizards\">\
<menuitem action=\"ImageWizard\"/>\
</menu>\
</menu>\
</menubar>\
<toolbar name=\"Wizards Toolbar\">\
<toolitem action=\"ImageWizard\" />\
</toolbar>\
</ui>";
	GtkAction *action;
	
	gchar *label;
	gchar *tip;
	GError *error;

	image_wizard = g_new0( ImageWizard, 1 );
	image_wizard->window = SCREEM_WINDOW( window );
	image_wizard->editor = SCREEM_EDITOR( editor );

	label = g_strdup( _( "Image Wizard" ) );
	tip = g_strdup( _( "Insert an Image" ) );
	action = gtk_action_new( "ImageWizard", label, tip,
			"Screem_Image" );
	g_signal_connect( G_OBJECT( action ), "activate",
			G_CALLBACK( image_wizard_display ), image_wizard );
	gtk_action_group_add_action( GTK_ACTION_GROUP( image_wizard->window->action_group ), action );
	g_free( label );
	g_free( tip );

	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( image_wizard->window->merge ),
						ui, strlen( ui ), &error ) ) {
		g_message( "%s ui error = %s", "image wizard",
				error->message );
		g_error_free( error );
	}

	/* NOTE: make sure that the window being obtained isn't visible */
	image_wizard->xml = glade_xml_new( GLADE_PATH"/imageWizard.glade",
					   "imagewizard_dialog", NULL );

	image_wizard_align_init( image_wizard->xml );

	image_wizard->dialog = glade_xml_get_widget( image_wizard->xml, 
						     "imagewizard_dialog" );

	g_object_set_data( G_OBJECT( image_wizard->dialog ),
			   "image_wizard", image_wizard );

	glade_xml_signal_autoconnect( image_wizard->xml );

	wizards = g_list_append( wizards, image_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	ImageWizard *image_wizard;

	for( list = wizards; list; list = list->next ) {
		image_wizard = (ImageWizard*)list->data;
		if( image_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* link_wizard is the one to erase */
	wizards = g_list_remove( wizards, image_wizard );
	g_free( image_wizard );
}

/* for popup tag editing in the wizard */
G_MODULE_EXPORT void popup( ScreemWindow *window,
			    guint start, guint len, GSList *attrs )
{
	GList *list;
	GSList *attr;
	ImageWizard *image_wizard;
	GtkWidget *widget;
	gboolean layout;
	ScreemPage *page;
	const gchar *pagepath;

	for( list = wizards; list; list = list->next ) {
		image_wizard = (ImageWizard*)list->data;
		if( image_wizard->window == window ) {
			/* got it */
			break;
		}
	}
	g_return_if_fail( list != NULL );

	screem_editor_select_region( image_wizard->editor, start, len );

	page = screem_window_get_document( window );
	pagepath = screem_page_get_pathname( page );

	/* fill in attr values */
	layout = FALSE;
	for( attr = attrs; attr; attr = attr->next ) {
		const gchar *name;
		const gchar *value;
		
		value = (const gchar*)attr->data;
		attr = attr->next;
		name = (const gchar*)attr->data;
		
		if( ! g_strcasecmp( "src", name ) ) {
			gchar *dir;
			gchar *full;
			widget = glade_xml_get_widget( image_wizard->xml,
						       "imagepath" );

			/* convert value to a full path */
			if( pagepath ) {
				dir = g_dirname( pagepath );
				full = relative_to_full( value, dir );
				g_free( dir );
			} else {
				full = g_strdup( value );
			}
			if( ! strncmp( "file://", full, strlen( "file://" ) ) ) {
				gtk_entry_set_text( GTK_ENTRY( widget ), 
						    full + strlen( "file://" ) );
			} else {
				gtk_entry_set_text( GTK_ENTRY( widget ),
						    value );
			}
			image_wizard_preview( widget );
			g_free( full );
		} else if( ! g_strcasecmp( "alt", name ) ) {
			widget = glade_xml_get_widget( image_wizard->xml,
						       "imagealt" );
			gtk_entry_set_text( GTK_ENTRY( widget ), value );
		} else if( ! g_strcasecmp( "width", name ) ) {
			gint val;
			
			if( value[ strlen( value ) - 1 ] == '%' ) {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "widthper" );
			} else {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "widthpx" );
			}
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						      TRUE );
			widget = glade_xml_get_widget( image_wizard->xml, 
						       "width" );
			val = atoi( value );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   (gdouble)val );
		} else if( ! g_strcasecmp( "height", name ) ) {
			gint val;
			if( value[ strlen( value ) - 1 ] == '%' ) {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "heightper" );
			} else {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "heightpx" );
			}
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						      TRUE );
			widget = glade_xml_get_widget( image_wizard->xml, 
						       "height" );
			val = atoi( value );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   (gdouble)val );
		} else if( ! g_strcasecmp( "align", name ) ) {
			gint i;
			if( ! layout ) {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "applylayout" );
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
							      TRUE );
				layout = TRUE;
			}
			widget = glade_xml_get_widget( image_wizard->xml,
						       "image_align" );
			for( i = 0; align[ i ]; ++ i ) {
				if( ! g_strcasecmp( align[ i ], value ) ) {
					break;
				}
			}
			if( ! align[ i ] ) {
				i = 0;
			}
			gtk_option_menu_set_history( GTK_OPTION_MENU( widget ), i );
		} else if( ! g_strcasecmp( "border", name ) ) {
			gint val;

			if( ! layout ) {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "applylayout" );
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
							      TRUE );
				layout = TRUE;
			}
			widget = glade_xml_get_widget( image_wizard->xml, 
						       "border" );
			val = atoi( value );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   (gdouble)val );
		} else if( ! g_strcasecmp( "hspace", name ) ) {
			gint val;

			if( ! layout ) {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "applylayout" );
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
							      TRUE );
				layout = TRUE;
			}
			widget = glade_xml_get_widget( image_wizard->xml, 
						       "hspace" );
			val = atoi( value );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   (gdouble)val );
		} else if( ! g_strcasecmp( "vspace", name ) ) {
			gint val;

			if( ! layout ) {
				widget = glade_xml_get_widget( image_wizard->xml,
							       "applylayout" );
				gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
							      TRUE );
				layout = TRUE;
			}
			widget = glade_xml_get_widget( image_wizard->xml, 
						       "vspace" );
			val = atoi( value );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ),
						   (gdouble)val );
		}
	}

	image_wizard->popup_attrs = attrs;
	image_wizard->popup_start = start;
	image_wizard->popup_len = len;

	gtk_dialog_run( GTK_DIALOG( image_wizard->dialog ) );

	image_wizard->popup_attrs = NULL;
}


/* End of required section */


void image_wizard_display( GtkAction *action, gpointer user_data )
{
	ImageWizard *wizard;
	ScreemPage *page;
	ScreemApplication *app;
	ScreemSession *session;
		
	wizard = (ImageWizard*)user_data;
	app = SCREEM_APPLICATION( wizard->window->application );
	session = screem_application_get_session( app );

	page = screem_window_get_document( wizard->window );

	if( ! page )
		return;
       
	if( ! GTK_WIDGET_VISIBLE( wizard->dialog ) ) {
		screem_session_restore_dialog( session,
				wizard->dialog );
	}
	
	gtk_widget_show_all( wizard->dialog );
	gdk_window_raise( wizard->dialog->window );
}

gboolean image_wizard_preview( GtkWidget *widget )
{
	GladeXML *xml;
	const gchar *filename;

	xml = glade_get_widget_tree( GTK_WIDGET( widget ) );
	widget = glade_xml_get_widget( xml, "imagepath" );

	filename = gtk_entry_get_text( GTK_ENTRY( widget ) );

	if( *filename ) {
		struct stat s;
		GdkPixbuf *pixbuf;
		GtkWidget *widget;

		if( stat( filename, &s ) < 0 ) {
			return FALSE;
		}
		if( g_file_test( filename, G_FILE_TEST_IS_DIR ) ) {
			return FALSE;
		}
		pixbuf = gdk_pixbuf_new_from_file( filename, NULL );
		if( pixbuf ) {
			GdkPixbuf *temp;
			gchar *w;
			gchar *h;
			gchar *size;

			widget = glade_xml_get_widget( xml, "previewimage" );
			temp = gdk_pixbuf_scale_simple( pixbuf,
							160, 120,
							GDK_INTERP_BILINEAR );
			gtk_image_set_from_pixbuf( GTK_IMAGE( widget ), temp );

			w = g_strdup_printf( "%i", 
					     gdk_pixbuf_get_width( pixbuf ) );
			h = g_strdup_printf( "%i", 
					     gdk_pixbuf_get_height( pixbuf ) );
			size = g_strdup_printf( "%i bytes", (gint)s.st_size );

			gdk_pixbuf_unref( temp );

			widget = glade_xml_get_widget( xml, "imagewidth" );
			gtk_label_set_text( GTK_LABEL( widget ), w );
			widget = glade_xml_get_widget( xml, "imageheight" );
			gtk_label_set_text( GTK_LABEL( widget ), h );
			widget = glade_xml_get_widget( xml, "imagesize" );
			gtk_label_set_text( GTK_LABEL( widget ), size );

			g_free( w );
			g_free( h );
			g_free( size );

			widget = glade_xml_get_widget( xml, "width" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), 
						   (gfloat)gdk_pixbuf_get_width( pixbuf ) );
			widget = glade_xml_get_widget( xml, "height" );
			gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), 
						   (gfloat)gdk_pixbuf_get_height( pixbuf ) );

			gdk_pixbuf_unref( pixbuf );
		}
	}

	return FALSE;
}

void image_wizard_clicked( GtkWidget *widget, gint button, gpointer data )
{
	ImageWizard *wizard;
	GladeXML *xml;
	GList *list;
	ScreemPage *page;
	ScreemApplication *app;
	ScreemSession *session;

	xml = glade_get_widget_tree( widget );
	for( wizard = NULL, list = wizards; list; list = list->next ) {
		wizard = (ImageWizard*)list->data;
		if( wizard->xml == xml )
			break;
		wizard = NULL;
	}
	g_assert( wizard );

	page = screem_window_get_document( wizard->window );
	app = wizard->window->application;
	session = screem_application_get_session( app );
	
	if( ! page ) {
		return;
	}

	if( button == GTK_RESPONSE_OK || button == GTK_RESPONSE_APPLY ) {
		const gchar *pathname;
		GtkWidget *widget;
		const gchar *filename;
		const gchar *alt;
		const gchar *alignment;
		GString *tag;
		gint w;
		gint h;
		gint i;
		gchar *temp;
		gboolean copyNeeded;
		gboolean makeThumb;
		gchar *uri;
		gint pos;

		gint filenamepos;
		gchar *rel;
		gchar *dir;

		ScreemDTD *dtd;
		gchar *doctype;

		pathname = screem_page_get_pathname( page );
		
		widget = glade_xml_get_widget( xml, "imagepath" );
		filename = gtk_entry_get_text( GTK_ENTRY( widget ) );

		widget = glade_xml_get_widget( xml, "imagealt" );
		alt = gtk_entry_get_text( GTK_ENTRY( widget ) );

		tag = g_string_new( "<img src=\"" );
		filenamepos = tag->len;
		g_string_append( tag, "\" alt=\"" );
		g_string_append( tag, alt );
		g_string_append( tag, "\" width=\"" );

		widget = glade_xml_get_widget( xml, "width" );
		w = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
		g_string_append_printf( tag, "%i", w );
		
		widget = glade_xml_get_widget( xml, "widthpx" );
		if(! gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
			g_string_append_c( tag, '%' );
		}

		g_string_append( tag, "\" height=\"" );
		widget = glade_xml_get_widget( xml, "height" );
		h = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(widget));
		g_string_append_printf( tag, "%i", h );

		widget = glade_xml_get_widget( xml, "heightpx" );
		if(! gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget))) {
			g_string_append_c( tag, '%' );
		}
		
		g_string_append_c( tag, '"' );

		widget = glade_xml_get_widget( xml, "applylayout" );
		if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget)) ) {
			widget = glade_xml_get_widget( xml, "image_align" );
			widget = GTK_OPTION_MENU( widget )->menu_item;
			
			alignment = (const gchar*)g_object_get_data(G_OBJECT(widget),
								"align" );
			g_string_append_printf( tag, " align=\"%s\"",
						alignment );

			widget = glade_xml_get_widget( xml, "border" );
			i = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
			g_string_append_printf( tag, " border=\"%i\"", i );

			widget = glade_xml_get_widget( xml, "hspace" );
			i = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
			g_string_append_printf( tag, " hspace=\"%i\"", i );

			widget = glade_xml_get_widget( xml, "vspace" );
			i = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );
			g_string_append_printf( tag, " vspace=\"%i\"", i );
		}
		
		/* if we are from the popup we need to fill in all
		   the attributes we haven't handled in the wizard */
		if( wizard->popup_attrs ) {
			GSList *attr;
			for( attr = wizard->popup_attrs; attr;
			     attr = attr->next ) {
				const gchar *name;
				const gchar *value;

				value = (const gchar*)attr->data;
				attr = attr->next;
				name = (const gchar*)attr->data;

				if( g_strcasecmp( "src", name ) &&
				    g_strcasecmp( "alt", name ) &&
				    g_strcasecmp( "width", name ) &&
				    g_strcasecmp( "height", name ) &&
				    g_strcasecmp( "align", name ) &&
				    g_strcasecmp( "border", name ) &&
				    g_strcasecmp( "hspace", name ) &&
				    g_strcasecmp( "vspace", name ) &&
				    g_strcasecmp( "/", name ) ) {
					g_string_append_c( tag, ' ' );
					g_string_append( tag, name );
					if( value ) {
						g_string_append( tag, "=\"" );
						g_string_append( tag, value );
						g_string_append_c( tag, '"' );
					}
				}
			}
		}

		/* bit of a hack */
		dtd = screem_page_get_dtd( page );
		doctype = NULL;
		g_object_get( G_OBJECT( dtd ), "public", &doctype, NULL );
		if( ! doctype ) {
			/* get default doctype string */
			doctype = gconf_client_get_string( wizard->window->application->client,
							   "/apps/screem/editor/default_dtd",
							   NULL );
		}
		if( doctype && strstr( doctype, " XHTML " ) != NULL ) {
			g_string_append( tag, " />" );
		} else {
			g_string_append( tag, ">" );
		}
		if( doctype ) {
			g_free( doctype );
		}
	
		widget  = glade_xml_get_widget( xml, "copy" );
		copyNeeded = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

		widget  = glade_xml_get_widget( xml, "thumb" );
		makeThumb = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );

		if( pathname ) {
			dir = g_dirname( pathname );
		} else {
			dir = g_strdup( "" );
		}

		/* copy the file if we need to */
		if( copyNeeded ) {
			uri = g_strdup_printf( "%s/%s", dir, 
					       g_basename( filename ) );
			copy_file( filename, uri, 
				screem_application_file_op, app );
		} else {
			uri = g_strdup( filename );
		}
	
		rel = relative_path( uri, dir );
		
		/* if its a thumbnail we need to wrap this up in a href tag */
		if( makeThumb ) {
			const gchar *ext;
			gchar *thumb;
			GdkPixbuf *pixbuf;
			GdkPixbuf *scale;

			temp = g_strdup_printf( "<a href=\"%s\">", rel );
			filenamepos += strlen( temp );
			g_string_prepend( tag, temp );
			g_free( temp );
			g_string_append( tag, "</a>" );

			g_free( rel );

			ext = g_extension_pointer( filename );
			thumb = g_strndup( filename, ext - filename - 1 );
			rel = g_strconcat( thumb, "-thumb.", ext, NULL );
			g_free( thumb );

			pixbuf = gdk_pixbuf_new_from_file( filename, NULL );
			scale = NULL;
			if( pixbuf ) {
				scale = gdk_pixbuf_scale_simple( pixbuf,
								 w, h,
								 GDK_INTERP_BILINEAR );
			}
			if( scale ) {
				if( gdk_pixbuf_save( scale, rel, ext, NULL, NULL ) ) {
					screem_application_file_op( GNOME_VFS_MONITOR_EVENT_CREATED, rel, app );
				}
				g_object_unref( scale );
				g_object_unref( pixbuf );
			}
			thumb = rel;
			rel = relative_path( thumb, dir );
			g_free( thumb );
		}
		g_string_insert( tag, filenamepos, rel );
		g_free( dir );
		g_free( rel );
		g_free( uri );

		if( wizard->popup_attrs ) {
			pos = wizard->popup_start;
			screem_editor_delete_forward( wizard->editor,
						      pos,
						      wizard->popup_len );
		} else {
			pos = screem_editor_get_pos( wizard->editor );
		}
		
		/* now we insert it at position pos */
		screem_editor_insert( wizard->editor, pos, tag->str );

		g_string_free( tag, TRUE );
	}

	screem_session_store_dialog( session, widget );
	
	if( button != GTK_RESPONSE_APPLY ) {
		gtk_widget_hide( GTK_WIDGET( widget ) );
	}
}

static void image_wizard_align_init( GladeXML *xml )
{
	gint i;
	GtkWidget *widget;
	GtkWidget *menu;


	menu = gtk_menu_new();
	for( i = 0; align[ i ]; ++ i ) {
		widget = gtk_menu_item_new_with_label( align[ i ] );
		gtk_widget_show( widget );
		gtk_menu_shell_append( GTK_MENU_SHELL( menu ),
				       widget );
		g_object_set_data( G_OBJECT( widget ), "align", 
				   (gpointer)align[ i ] );
	}

	widget = glade_xml_get_widget( xml, "image_align" );
	gtk_option_menu_set_menu( GTK_OPTION_MENU( widget ), menu );
}

void image_wizard_px( GtkWidget *widget )
{
	gtk_spin_button_set_range( GTK_SPIN_BUTTON( widget ),
				   1, 2048 );
}

void image_wizard_percent( GtkWidget *widget )
{
	gint value;

	value = gtk_spin_button_get_value_as_int( GTK_SPIN_BUTTON( widget ) );

	gtk_spin_button_set_range( GTK_SPIN_BUTTON( widget ),
				   1, 100 );

	if( value > 100 ) {
		gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), 100.0 );
	}
}
