/*  Screem: fileops.c,
 *  This file provides file/directory copying/moving/deletion
 * 
 *  Copyright (C) 1999, 2000 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>
#include <dirent.h>
#include <errno.h>

#include <stdio.h>
#include <string.h>

#include <glib/gfileutils.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkprogressbar.h>

#include <glib/gi18n.h>
#include <libgnomeui/gnome-authentication-manager.h>

#include <gconf/gconf-client.h>
#include <glade/glade.h>

#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-directory.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libgnomevfs/gnome-vfs-module-callback.h>
#include <libgnomevfs/gnome-vfs-standard-callbacks.h>

#include "fileops.h"

#include "support.h"

#include "screem-errors.h"

/* not nice including these, we need access to the ScreemSession
 * so we can keep track of the file selector window size */
#include "screem-application.h"
#include "screem-window.h"

/* yuk, needed for our hack on GnomeFileEntry as we can't
 * provide the ScreemWindow in glade dialogs to get access to the app */
extern ScreemApplication *app;

gboolean copy_file( const gchar *source, const gchar *dest,
		ScreemFileOp cb, gpointer data )
{
	GnomeVFSURI *suri;
	GnomeVFSURI *duri;

	GnomeVFSHandle *shandle;
	GnomeVFSHandle *dhandle;

	GnomeVFSOpenMode mode;
	GnomeVFSResult result;

	gboolean ret = FALSE;
	gboolean wrote = FALSE;

	gchar buffer[ BUFSIZ + 1 ];

	if( ! overwrite( dest ) ) {
		return FALSE;
	}

	suri = gnome_vfs_uri_new( source );
	duri = gnome_vfs_uri_new( dest );

	mode = GNOME_VFS_OPEN_READ;
	result = gnome_vfs_open_uri( &shandle, suri, mode );
	if( result == GNOME_VFS_OK ) {
		GnomeVFSFileSize wsize = 0;
		mode = GNOME_VFS_OPEN_WRITE;

		result = gnome_vfs_open_uri( &dhandle, duri, mode );
		if( result == GNOME_VFS_ERROR_NOT_FOUND ) {
			GnomeVFSFileInfo *info;
			GnomeVFSFileInfoOptions options;

			/* get permissions for source */
			options = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
			info = gnome_vfs_file_info_new();
			result = gnome_vfs_get_file_info_uri( suri, info, options );
			if( result == GNOME_VFS_OK ) {
				result = gnome_vfs_create_uri( &dhandle, duri,
						       mode, TRUE, 
						       info->permissions );
			}
			gnome_vfs_file_info_unref( info );
		}
		while( result == GNOME_VFS_OK ) {
			GnomeVFSFileSize rsize;

			wrote = TRUE;
			
			result = gnome_vfs_read( shandle, 
						 buffer, BUFSIZ, &rsize );
			buffer[ rsize ] = '\0';
			
			if( result == GNOME_VFS_OK ) {
				result = gnome_vfs_write( dhandle, buffer, rsize,
							  &rsize );
				wsize += rsize;
			} 
		}
		if( result == GNOME_VFS_ERROR_EOF ) {
			ret = TRUE;
		}

		if( wrote ) {
			gnome_vfs_truncate_handle( dhandle, wsize );
			gnome_vfs_close( dhandle );
			if( cb ) {
				cb( GNOME_VFS_MONITOR_EVENT_CREATED,
						dest, data );
			}
		}
		gnome_vfs_close( shandle );
	}

	gnome_vfs_uri_unref( duri );
	gnome_vfs_uri_unref( suri );

	return ret;
}

gboolean copy_dir( const gchar *source, const gchar *dest, gboolean move,
		ScreemFileOp cb, gpointer data )
{
	gchar *orig;
	gchar *new_path;

	gboolean made;

	GnomeVFSFileInfo *vfsinfo;
	GnomeVFSFileInfoOptions options;
	GnomeVFSResult result;
	GnomeVFSDirectoryHandle *handle;

	GnomeVFSFilePermissions perms;

	made = FALSE;

	vfsinfo = gnome_vfs_file_info_new();
	options = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	result = gnome_vfs_get_file_info( source, vfsinfo, options );
	if( result != GNOME_VFS_OK ) {
		gnome_vfs_file_info_unref( vfsinfo );
		return FALSE;
	}
	perms = vfsinfo->permissions;

	result = gnome_vfs_get_file_info( dest, vfsinfo, options );
	
	if( result != GNOME_VFS_OK ) {
		if( ! mkdir_recursive( dest, perms, cb, data ) ) {
			gnome_vfs_file_info_unref( vfsinfo );
			return FALSE;
		}
		made = TRUE;
		if( cb ) {
			cb( GNOME_VFS_MONITOR_EVENT_CREATED, dest, data );
		}
	}

	if( ! made && vfsinfo->type != GNOME_VFS_FILE_TYPE_DIRECTORY ) {
		gnome_vfs_file_info_unref( vfsinfo );
		return FALSE;
	}

	gnome_vfs_file_info_unref( vfsinfo );

	handle = NULL;
	result = gnome_vfs_directory_open( &handle, source, options );

	if( result != GNOME_VFS_OK ) {
		return FALSE;
	}

	vfsinfo = gnome_vfs_file_info_new();

	while( result == GNOME_VFS_OK ) {
		result = gnome_vfs_directory_read_next( handle, vfsinfo );
		if( ( result == GNOME_VFS_OK ) &&
		    strcmp( "..", vfsinfo->name ) &&
		    strcmp( ".", vfsinfo->name ) ) {
			orig = g_strconcat( source, "/", vfsinfo->name, NULL );
			new_path = g_strconcat( dest, "/", vfsinfo->name,
						NULL );
			if( vfsinfo->type != GNOME_VFS_FILE_TYPE_DIRECTORY ) {
				if( move ) {
					move_file( orig, new_path, cb, data );
				} else {
					copy_file( orig, new_path, cb, data );
				}
			} else {
				copy_dir( orig, new_path, move, cb, data );
			}
			g_free( orig );
			g_free( new_path );
		}
	}

	if( handle ) {
		gnome_vfs_directory_close( handle );
	}

	if( move ) {
		delete_dir( source, cb, data );
	}

	return TRUE;
}

gboolean move_file( const gchar *source, const gchar *dest,
		ScreemFileOp cb, gpointer data )
{
	gboolean ret;
	
	if( ( ! overwrite( dest ) ) ||
		( gnome_vfs_move( source, dest, TRUE ) != GNOME_VFS_OK ) ) {
		ret = FALSE;
	} else {
		ret = TRUE;
		cb( GNOME_VFS_MONITOR_EVENT_CREATED, dest, data );
		cb( GNOME_VFS_MONITOR_EVENT_DELETED, source, data );
	}

	return ret;
}

gboolean delete_file( const gchar *file, ScreemFileOp cb, gpointer data )
{
	GnomeVFSURI *uri;
	gboolean ret;

	uri = gnome_vfs_uri_new( file );

	ret = (gnome_vfs_unlink_from_uri( uri ) == GNOME_VFS_OK);

	gnome_vfs_uri_unref( uri );

	if( ret && cb ) {
		cb( GNOME_VFS_MONITOR_EVENT_DELETED, file, data );
	}
	
	return ret;
}

gboolean delete_dir( const gchar *path, ScreemFileOp cb, gpointer data )
{
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfoOptions options;
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;
	gboolean ret;
	
	options = GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK;

	result = gnome_vfs_directory_open( &handle, path, options );
	if( result != GNOME_VFS_OK) {
		return delete_file( path, cb, data );
	}
       
	info = gnome_vfs_file_info_new();

	while( result == GNOME_VFS_OK ) {
		result = gnome_vfs_directory_read_next( handle, info );
		if( ( result == GNOME_VFS_OK ) &&
		    strcmp( ".", info->name ) && strcmp( "..", info->name ) ) {
			gchar *new_path = g_strdup_printf( "%s%c%s", path,
							   G_DIR_SEPARATOR, 
							   info->name );
			delete_dir( new_path, cb, data );
			g_free( new_path );
		}
	}

	gnome_vfs_file_info_unref( info );

	gnome_vfs_directory_close( handle );

	ret = ( gnome_vfs_remove_directory( path ) == GNOME_VFS_OK );
	if( ret ) {
		if( cb ) {
			cb( GNOME_VFS_MONITOR_EVENT_DELETED, path, data );
		}
	}

	return ret;
}

/* recusivley makes directories (if needed), starting at current path */
gboolean mkdir_recursive( const gchar *path, GnomeVFSFilePermissions perms,
		ScreemFileOp cb, gpointer data )
{
	GnomeVFSURI *uri;
	gboolean ret = FALSE;
	gboolean exists;

	uri = gnome_vfs_uri_new( path );
	if( uri ) {
		ret = TRUE;
		
		exists = gnome_vfs_uri_exists( uri );

		if( ( ! exists ) && gnome_vfs_uri_has_parent( uri ) ) {
			GnomeVFSURI *parent;
			gchar *parentname;

			parent = gnome_vfs_uri_get_parent( uri );

			parentname = gnome_vfs_uri_to_string( parent,
							      GNOME_VFS_URI_HIDE_NONE );
			gnome_vfs_uri_unref( parent );

			ret = mkdir_recursive( parentname, perms, cb, data );

			g_free( parentname );
		}
		if( ret && ! exists ) {
			GnomeVFSResult result;

			result = gnome_vfs_make_directory_for_uri( uri, 
								   perms );
			if( result == GNOME_VFS_ERROR_FILE_EXISTS ) {
				/* ensure it is a directory */
				GnomeVFSFileInfo *info;
				GnomeVFSFileInfoOptions options;
				
				info = gnome_vfs_file_info_new();
				options = GNOME_VFS_FILE_INFO_DEFAULT;
				if( gnome_vfs_get_file_info_uri( uri, 
							info,
							options ) ==
							GNOME_VFS_OK &&
				    info->type ==GNOME_VFS_FILE_TYPE_DIRECTORY) {
					result = GNOME_VFS_OK;
				}
				gnome_vfs_file_info_unref( info );
			}
			
			ret = ( result == GNOME_VFS_OK );
		}
		gnome_vfs_uri_unref( uri );
	}

	if( ret ) {
		if( cb ) {
			cb( GNOME_VFS_MONITOR_EVENT_CREATED, path, data );
		}
	}
	
	return ret;
}

/* converts the given path into one relative to the passed root */
gchar *relative_path( const gchar *text, const gchar *root )
{
	GnomeVFSURI *uri;
	GnomeVFSURI *base;
	gchar *ret;

	GSList *uri_parents = NULL;
	GSList *base_parents = NULL;
	GSList *up;
	GSList *bp;

	GnomeVFSURI *parent;
	GString *temp;

	g_return_val_if_fail( text != NULL, NULL );
	
	if( ! root || *root == '\0' ||
	    ! g_strncasecmp( "mailto:", text, strlen( "mailto:" ) ) ||
	    ! g_strncasecmp( "javascript:", text, strlen( "javascript:" ) ) ) {
		return g_strdup( text );
	}

	uri = gnome_vfs_uri_new( text );
	base = gnome_vfs_uri_new( root );

	if( strcmp( gnome_vfs_uri_get_scheme( uri ),
		    gnome_vfs_uri_get_scheme( base ) ) ) {
		gnome_vfs_uri_unref( uri );
		gnome_vfs_uri_unref( base );
		return g_strdup( text );
	}

	/* get list of uri parents */
	uri_parents = g_slist_prepend( NULL, uri );
	for( parent = uri; gnome_vfs_uri_has_parent( parent ); ) {
		parent = gnome_vfs_uri_get_parent( parent );

		uri_parents = g_slist_prepend( uri_parents, parent );
	}
	/* get list of base parents */
	base_parents = g_slist_prepend( NULL, base );
	for( parent = base; gnome_vfs_uri_has_parent( parent ); ) {
		parent = gnome_vfs_uri_get_parent( parent );

		base_parents = g_slist_prepend( base_parents, parent );
	}

	up = uri_parents;
	bp = base_parents;

       	while( up && bp ) {
		gchar *uri1;
		gchar *uri2;
		gboolean match;

		uri1 = gnome_vfs_uri_extract_short_name( (GnomeVFSURI*)up->data );
		uri2 = gnome_vfs_uri_extract_short_name( (GnomeVFSURI*)bp->data );
		match = ( ! strcmp( uri1, uri2 ) );

		g_free( uri1 );
		g_free( uri2 );
		
		if( match ) {
			up = up->next;
			bp = bp->next;
		} else {
			break;
		}
	}
	
	/* up and bp should now be at the point where they differ */
	temp = g_string_new( NULL );
	if( bp ) {
		gint parents;
		for( parents = g_slist_length( bp ); parents > 0; -- parents ) {
			g_string_append( temp, "../" );
		}
	} else {
		g_string_append( temp, "./" );
	}

	for( ; up; up = up->next ) {
		gchar *tmp = gnome_vfs_uri_extract_short_name( (GnomeVFSURI*)up->data );
		g_string_append( temp, tmp );
		if( up->next ) {
			g_string_append_c( temp, '/' );
		}
		g_free( tmp );
	}

	ret = temp->str;

	g_string_free( temp, FALSE );

	for( up = uri_parents; up; up = up->next ) {
		gnome_vfs_uri_unref( (GnomeVFSURI*) up->data );
	}
	for( bp = base_parents; bp; bp = bp->next ) {
		gnome_vfs_uri_unref( (GnomeVFSURI*) bp->data );
	}
	if( uri_parents ) {
		g_slist_free( uri_parents );
	}
	if( base_parents ) {
		g_slist_free( base_parents );
	}

	return ret;
}

/* converts the given relative path to a full pathname,
   we should be in the path that it is relative from before calling the
   function */
gchar *relative_to_full( const gchar *relPath, const gchar *base_uri )
{
	gchar *base;
	gchar *path = NULL;
	gchar *cwd;
	
	/* if no base uri get current local dir */
	cwd = NULL;
	if( ! base_uri ) {
		cwd = g_get_current_dir();
		if( ! cwd ) {
			cwd = g_strdup( "" );
		}
		base_uri = cwd;
	}

	/* ensure base uri ends in / */
	if( base_uri[ strlen( base_uri ) -1 ] != G_DIR_SEPARATOR ) {
		base = g_strconcat( base_uri, "/", NULL );
	} else {
		base = g_strdup( base_uri );
	}

	path = gnome_vfs_uri_make_full_from_relative( base, relPath );

	g_free( base );
	
	if( cwd ) {
		g_free( cwd );
	}

	return path;
}

gboolean uri_exists( const gchar *filename, GnomeVFSResult *result )
{
	GnomeVFSFileInfo *info;
	GnomeVFSURI *uri;
	gboolean ret;
	
	uri = gnome_vfs_uri_new( filename );

	if( ! result ) {
		ret = gnome_vfs_uri_exists( uri );
	} else {
		info = gnome_vfs_file_info_new();
		*result = gnome_vfs_get_file_info_uri( uri, info,
				       GNOME_VFS_FILE_INFO_FOLLOW_LINKS );

		/* does dest already exist? */
		ret = ( *result == GNOME_VFS_OK );

		gnome_vfs_file_info_unref( info );
	}
	gnome_vfs_uri_unref( uri );

	return ret;
}

gboolean uri_accessible( const gchar *filename, 
		GnomeVFSFilePermissions perms )
{
	gboolean ret;
	GnomeVFSFileInfo *info;
	GnomeVFSResult res;
	
	ret = FALSE;
	info = gnome_vfs_file_info_new();
	res = gnome_vfs_get_file_info( filename, info,
				GNOME_VFS_FILE_INFO_GET_ACCESS_RIGHTS |
			       GNOME_VFS_FILE_INFO_FOLLOW_LINKS );
	if( res == GNOME_VFS_OK ) {
		ret = info->permissions & perms;
	}
	gnome_vfs_file_info_unref( info );

	return ret;
}

gboolean overwrite( const gchar *filename )
{
	GtkWidget *prompt;
	GnomeVFSResult result;

	gboolean ret;

	ret = uri_exists( filename, &result );

	if( ! ret ) {
		ret = TRUE;
	} else if( result == GNOME_VFS_OK ) {
		gchar *glade_path;
		GladeXML *xml;
		GString *str;
		const gchar *ltxt;
		GtkWidget *label;
		const gchar *temp;
		gchar *tmp;
		guint pos;

		glade_path = screem_get_glade_path();
		xml = glade_xml_new( glade_path, "overwrite", NULL );
		g_free( glade_path );
	
		label = glade_xml_get_widget( xml, "overwrite_label" );
		ltxt = gtk_label_get_text( GTK_LABEL( label ) );
		temp = strstr( ltxt, "%s" );
		g_assert( temp );
		pos = (guint)(temp - ltxt);
		str = g_string_new( "<span weight=\"bold\" size=\"larger\">" );
		pos += str->len;
		g_string_append( str, ltxt );
	
		g_string_erase( str, pos, 2 );
		temp = gnome_vfs_unescape_string_for_display( filename );
		tmp = g_markup_escape_text( temp, -1 );
		g_string_insert( str, pos, tmp );
		g_free( (gchar*)temp );
		g_free( tmp );
	
		temp = strstr( str->str, "\n\n" );
		g_assert( temp );
		pos = (guint)(temp - str->str);
		g_string_insert( str, pos, "</span>" );

		gtk_label_set_markup( GTK_LABEL( label ), str->str );
		g_string_free( str, TRUE );
	
		prompt = glade_xml_get_widget( xml, "overwrite" );
		gtk_window_set_wmclass( GTK_WINDOW( prompt ),
					"Screem",
					"overwrite_prompt" );
		gtk_window_set_type_hint( GTK_WINDOW( prompt ), 
					  GDK_WINDOW_TYPE_HINT_DIALOG );
		ret = ( gtk_dialog_run( GTK_DIALOG( prompt ) ) == 
				GTK_RESPONSE_OK );
		gtk_widget_destroy( prompt );
		g_object_unref( G_OBJECT( xml ) );
	} else if( result != GNOME_VFS_ERROR_NOT_FOUND ) {
		/* FIXME: report error reason */
		ret = FALSE;
	}

	return ret;
}

/* does path1 match path2, or is it 
   a subdir going from base_uri,
   path2 must be absolute */
gchar* paths_match( const gchar *base_uri,
		    const gchar *path1, const gchar *path2 )
{
	gchar *abs_path1;

	abs_path1 = relative_to_full( path1, base_uri );

	if( strncmp( abs_path1, path2, strlen( path2 ) ) ) {
		g_free( abs_path1 );
		abs_path1 = NULL;
	}

	return abs_path1;
}

void load_file_internal_cancel( GtkWidget *widget )
{
	g_object_set_data( G_OBJECT( widget ), "cancel",
			GUINT_TO_POINTER( 1 ) );
}

static GString *load_file_internal( GnomeVFSHandle *handle, 
				   const gchar *path,
				   GnomeVFSFileSize size,
				   gboolean interactive,
				   GnomeVFSResult *res )
{
	GString *ret;
	GnomeVFSResult result;
	GnomeVFSFileSize bytes_read;
	GnomeVFSFileSize total_read;
	
	GladeXML *xml;
	gchar *glade_path;
	const gchar *ltxt;
	const gchar *temp;
	gchar *tmp;
	guint pos;
	GString *str;
	GtkWidget *widget;

#define READ_CHUNK_SIZE 8192
	gchar buffer[ READ_CHUNK_SIZE ];
	gdouble fraction;
	gboolean pulse;
	
	/* shut compiler up, they won't be used uninitialised */
	xml = NULL;
	widget = NULL;
	
	ret = g_string_new( NULL );

	pulse = ( size == 0 );
	
	if( interactive ) {
		glade_path = screem_get_glade_path();
		xml = glade_xml_new( glade_path, "loading", NULL );
		g_free( glade_path );

		widget = glade_xml_get_widget( xml, "text" );
		ltxt = gtk_label_get_text( GTK_LABEL( widget ) );
		temp = strstr( ltxt, "%s" );
		g_assert( temp );
		pos = (guint)(temp - ltxt);
		str = g_string_new( "<span weight=\"bold\" size=\"larger\">" );
		pos += str->len;
		g_string_append( str, ltxt );
	
		g_string_erase( str, pos, 2 );
		temp = gnome_vfs_unescape_string_for_display( path );
		tmp = g_markup_escape_text( temp, -1 );
		g_string_insert( str, pos, tmp );
		g_free( (gchar*)temp );
		g_free( tmp );
	
		temp = strstr( str->str, "\n\n" );
		g_assert( temp );
		pos = (guint)(temp - str->str);
		g_string_insert( str, pos, "</span>" );
	
		gtk_label_set_markup( GTK_LABEL( widget ), str->str );
		g_string_free( str, TRUE );
	
		widget = glade_xml_get_widget( xml, "loading" );
		gtk_window_set_wmclass( GTK_WINDOW( widget ),
					"Screem",
					"loading_dialog" );
		gtk_window_set_type_hint( GTK_WINDOW( widget ), 
					  GDK_WINDOW_TYPE_HINT_DIALOG );

		gtk_widget_show( widget );
		glade_xml_signal_autoconnect( xml );

		widget = glade_xml_get_widget( xml, "progress" );
	}
	
	total_read = bytes_read = 0;
	fraction = 0.0;
	do {
		result = gnome_vfs_read( handle, buffer,
				READ_CHUNK_SIZE, &bytes_read );
		
		if( result == GNOME_VFS_OK || 
		    result == GNOME_VFS_ERROR_EOF ) {
			/* gzip method never gives EOF, assume
			 * 0 bytes is EOF,
			 *
			 * see http://bugzilla.gnome.org/show_bug.cgi?id=41173
			 *
			 * */
			if( bytes_read == 0 ) {
				result = GNOME_VFS_ERROR_EOF;
			}
			if( interactive ) {
				total_read += bytes_read;

				if( ! pulse ) {
					fraction = total_read / ( size * 1.0 );
					gtk_progress_bar_set_fraction( GTK_PROGRESS_BAR( widget ), fraction );
				} else {
					gtk_progress_bar_pulse( GTK_PROGRESS_BAR( widget ) );
				}
				gdk_threads_leave();
				while( g_main_iteration( FALSE ) ) {}
				
				gdk_threads_enter();

				if( g_object_get_data( G_OBJECT( widget ),
							"cancel" ) ) {
					result = GNOME_VFS_ERROR_CANCELLED;
				}
			}
			ret = g_string_append_len( ret, buffer, 
						   bytes_read );
		}
	} while( result == GNOME_VFS_OK );

	if( result != GNOME_VFS_ERROR_EOF ) {
		g_string_free( ret, TRUE );
		ret = NULL;
	} else {
		result = GNOME_VFS_OK;
	}
	
	if( interactive ) {
		widget = glade_xml_get_widget( xml, "loading" );
		gtk_widget_destroy( widget );
		g_object_unref( G_OBJECT( xml ) );
	}
	
	if( res ) {
		*res = result;
	}
	
	return ret;
}

GString *load_file( const gchar *path, gboolean *compress,
		gchar **charset, GError **error )
{
	GnomeVFSResult result;
	GnomeVFSFileInfo *info;
	GnomeVFSHandle *handle;
	gboolean interactive;

	gchar *gzpath;
	GnomeVFSFileSize size;
	gboolean local;
	gboolean compressed;
	gchar *temp;
	
#define FILE_SIZE_FOR_DIALOG ( 8 * 1024000 )
	
	GString *ret;

	if( error ) {
		*error = NULL;
	}
	if( ! compress ) {
		compress = &compressed;
	}
	
	ret = NULL;
	
	info = gnome_vfs_file_info_new();
	
	gzpath = g_strconcat( path, "#ugzip:", NULL );
	
	handle = NULL;
	result = gnome_vfs_open( &handle, gzpath, GNOME_VFS_OPEN_READ );
	*compress = TRUE;
	if( result != GNOME_VFS_OK ) {
		result = gnome_vfs_open( &handle, path, GNOME_VFS_OPEN_READ );
		*compress = FALSE;
	}
	if( result == GNOME_VFS_OK ) {
		result = gnome_vfs_get_file_info_from_handle( handle, 
				info,
				GNOME_VFS_FILE_INFO_DEFAULT );
		size = 0;
		local = FALSE;
		if( result == GNOME_VFS_OK ) {
			local = GNOME_VFS_FILE_INFO_LOCAL( info );
			if( info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE )  {
				size = info->size;
			}
		}
		gnome_vfs_file_info_unref( info );
	
		interactive = ( ( size > FILE_SIZE_FOR_DIALOG ) ||
				! local );
		ret = load_file_internal( handle, path, size,
					interactive, &result );
	}

	if( handle ) {
		gnome_vfs_close( handle );
	}
	
	/* if we have a result convert it to UTF-8 if possible */
	if( ret ) {
		temp = screem_support_charset_convert( ret->str, 
				ret->len, charset );
		g_string_assign( ret, temp );
	} else if( error ) {
		*error = g_error_new_literal( g_quark_from_string( SCREEM_LOAD_FILE_ERR ),
				(gint)result,
				gnome_vfs_result_to_string( result ) );
	}
	
	g_free( gzpath );
	
	return ret;
}

gboolean save_file( const gchar *path, const gchar *data,
		    GnomeVFSFilePermissions perms,
		    gboolean compress,
		    GError **error )
{
	GnomeVFSURI *uri;
	GnomeVFSResult result;
	GnomeVFSHandle *handle;
	GnomeVFSFileSize wsize;
	gint len;
	gchar *gzpath;
	
	gboolean ret;

	if( ! data ) {
		data = "";
	}

	if( error ) {
		*error = NULL;
	}
	
	if( ! compress ) {
		uri = gnome_vfs_uri_new( path );
	} else {
		gzpath = g_strconcat( path, "#gzip:", NULL );
		uri = gnome_vfs_uri_new( gzpath );
		g_free( gzpath );
	}

	/* FIXME: make a backup of the original first
	 * to avoid any data loss if the save fails? */
	
	if( uri ) {
		len = strlen( data );
		wsize = 0;
		
		result = gnome_vfs_create_uri( &handle, uri,
					       GNOME_VFS_OPEN_WRITE,
					       FALSE, perms );
	}
	
	if( ! uri ) {
		ret = FALSE;
		result = GNOME_VFS_ERROR_INVALID_URI;
	} else if( result != GNOME_VFS_OK ) {
		ret = FALSE;
	} else {
		while( len > 0 && result == GNOME_VFS_OK ) {
			GnomeVFSFileSize vfssize;
			
			result = gnome_vfs_write( handle, data, len,
						  &vfssize );
			len -= vfssize;
			data += vfssize;
			wsize += vfssize;
		}
		gnome_vfs_truncate_handle( handle, wsize );
		gnome_vfs_close( handle );

		gnome_vfs_uri_unref( uri );

		ret = TRUE;
	}

	if( error && ! ret ) {
		*error = g_error_new_literal( g_quark_from_string( SCREEM_SAVE_FILE_ERR ),
				(gint)result,
				gnome_vfs_result_to_string( result ) );
	}
	
	return ret;
}

gboolean screem_uri_is_dir( const gchar *uri )
{
	GnomeVFSFileInfo *info;
	gboolean ret;

	ret = FALSE;

	info = gnome_vfs_file_info_new();

	if( gnome_vfs_get_file_info( uri, info, 
				     GNOME_VFS_FILE_INFO_DEFAULT ) == 
	    GNOME_VFS_OK ) {
		ret =  ( info->type == GNOME_VFS_FILE_TYPE_DIRECTORY );
	}

	gnome_vfs_file_info_unref( info );

	return ret;
}

/* pinched from Epiphany ephy-file-helpers.c + modified */
gchar *screem_create_tmp_file( const gchar *base, const gchar *ext, gboolean tmpdir )
{
	gint fd;
	gchar *name;

	if( tmpdir ) {
		const gchar *tmp;

		tmp = g_getenv( "TMPDIR" );
		if( ! tmp ) {
#ifdef P_tmpdir
			tmp = P_tmpdir;
#else
			tmp = "/tmp";
#endif
		}
		name = g_build_filename( tmp, base, NULL );
	} else {
		name = g_strdup( base );
	}

	fd = g_mkstemp( name );

	if( fd != -1 ) {
		unlink( name );
		close( fd );

		if( ext ) {
			gchar *tmp;

			tmp = g_strconcat( name, ".", ext, NULL );
			g_free( name );
			name = tmp;
		}
	} else {
		g_free( name );
		name = NULL;
	}

	return name;
}

gchar *screem_get_dot_dir( void )
{
	const gchar *home;
	gchar *ret;
	
	home = g_get_home_dir();

	ret = g_build_filename( home, ".screem", NULL );

	mkdir_recursive( ret, GNOME_VFS_PERM_USER_ALL, NULL, NULL );

	return ret;
}

static gint dummy_filter( const GnomeVFSFileInfo *info )
{
	return 0;
}

GSList *screem_vfs_scandir( const gchar *dir, 
		     gint (*filter)( const GnomeVFSFileInfo *info ),
		     GCompareFunc compare,
		     gboolean recurse )
{
	GSList *ret;
	GSList *rec;
	gchar *full;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSFileInfoOptions options;
	GnomeVFSFileInfo *info;
	GnomeVFSResult result;
	gboolean ignore;
	gchar *utf;
	
	ret = NULL;

	if( ! filter ) {
		filter = dummy_filter;
	}
	
	options = GNOME_VFS_FILE_INFO_GET_MIME_TYPE |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE |
		  GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
	handle = NULL;
	result = gnome_vfs_directory_open( &handle, dir, options );
		
	info = gnome_vfs_file_info_new();
	while( result == GNOME_VFS_OK ) {
		result = gnome_vfs_directory_read_next( handle, info );
		if( result == GNOME_VFS_OK ) {
		
			ignore = ( info->name[ 0 ] == '.' &&
				   ( info->name[ 1 ] == '\0' ||
				     ( info->name[ 1 ] == '.' &&
				       info->name[ 2 ] == '\0' )
				   )
				 );
	
			if( ! g_utf8_validate( info->name, -1, NULL ) ) {
				utf = g_filename_to_utf8( info->name, -1,
						NULL, NULL, NULL );
			} else {
				utf = g_strdup( info->name );
			}
			
			if( ! filter( info ) ) {
				full = g_build_path( G_DIR_SEPARATOR_S,
						     dir, utf,
						     NULL );
				ret = g_slist_prepend( ret, full );
			} else if( recurse && ( ! ignore ) &&
				info->mime_type &&
				( ! strcmp( "x-directory/normal",
					info->mime_type ) ||
				  ! strcmp( "inode/directory",
					  info->mime_type ) ) ) {
				full = g_build_path( G_DIR_SEPARATOR_S,
						     dir, utf,
						     NULL );
				rec = screem_vfs_scandir( full,
						filter, compare,
						recurse );
				g_free( full );
				ret = g_slist_concat( rec, ret );
			}
			g_free( utf );
		}
		gnome_vfs_file_info_clear( info );
	}
	gnome_vfs_file_info_unref( info );
	
	if( handle ) {
		gnome_vfs_directory_close( handle );
	}
	
	if( ret && compare ) {
		ret = g_slist_sort( ret, compare );
		ret = g_slist_reverse( ret );
	}
	
	return ret;
}


/* vfs callbacks */
static void screem_vfs_full_auth_cb( gconstpointer in,
				gsize in_size,
				gpointer out,
				gsize out_size,
				gpointer data )
{
	const GnomeVFSModuleCallbackFullAuthenticationIn *full_in;
	GnomeVFSModuleCallbackFullAuthenticationOut *full_out;

	full_in = (GnomeVFSModuleCallbackFullAuthenticationIn*) in;
	full_out = (GnomeVFSModuleCallbackFullAuthenticationOut*) out;
	
}

static void screem_vfs_fill_auth_cb( gconstpointer in,
				gsize in_size,
				gpointer out,
				gsize out_size,
				gpointer data )
{
	const GnomeVFSModuleCallbackFillAuthenticationIn *fill_in;
	GnomeVFSModuleCallbackFillAuthenticationOut *fill_out;

	fill_in = (GnomeVFSModuleCallbackFillAuthenticationIn*) in;
	fill_out = (GnomeVFSModuleCallbackFillAuthenticationOut*) out;

}

static void screem_vfs_save_auth_cb( gconstpointer in,
				gsize in_size,
				gpointer out,
				gsize out_size,
				gpointer data )
{
	const GnomeVFSModuleCallbackSaveAuthenticationIn *save_in;
	GnomeVFSModuleCallbackSaveAuthenticationOut *save_out;

	save_in = (GnomeVFSModuleCallbackSaveAuthenticationIn*) in;
	save_out = (GnomeVFSModuleCallbackSaveAuthenticationOut*) out;

}
#if 0
static void screem_vfs_auth_cb( gconstpointer in,
				gsize in_size,
				gpointer out,
				gsize out_size,
				gpointer data )
{

}

static void screem_vfs_proxy_auth_cb( gconstpointer in,
				gsize in_size,
				gpointer out,
				gsize out_size,
				gpointer data )
{

}
#endif
void screem_init_vfs_callbacks( void )
{
	static const struct {
		const gchar *type;
		GnomeVFSModuleCallback cb;
	} callbacks[] = {
		{
			GNOME_VFS_MODULE_CALLBACK_FULL_AUTHENTICATION,
			screem_vfs_full_auth_cb
		},
		{
			GNOME_VFS_MODULE_CALLBACK_FILL_AUTHENTICATION,
			screem_vfs_fill_auth_cb
		},
		{
			GNOME_VFS_MODULE_CALLBACK_SAVE_AUTHENTICATION,
			screem_vfs_save_auth_cb
		},
/*		{
			GNOME_VFS_MODULE_CALLBACK_AUTHENTICATION,
			screem_vfs_auth_cb
		},
		{
			GNOME_VFS_MODULE_CALLBACK_HTTP_PROXY_AUTHENTICATION,
			screem_vfs_proxy_auth_cb
		}*/
	};
	static const guint n_cbs = G_N_ELEMENTS( callbacks );
	guint i;
	
	
	for( i = 0; FALSE && i < n_cbs; ++ i ) {
		gnome_vfs_module_callback_set_default( callbacks[ i ].type,
				callbacks[ i ].cb,
				NULL, /* data */
				NULL );
	}

	gnome_authentication_manager_init();
}

gchar *screem_fileselect( const gchar *title, GtkWindow *parent,
			GtkFileChooserAction action,
			const gchar *dir,
			const GSList *filters )
{
	ScreemSession *session;
	
	GtkWidget *dialog;
	gchar *ret;
	const gchar *button;
	gchar *temp;
	GtkFileFilter *all;

	gint x;
	gint y;
	gint w;
	gint h;

	const gchar *role = "screem_file_selector";
	
	session = screem_application_get_session( app );
	ret = NULL;

	button = GTK_STOCK_OPEN;
	if( action == GTK_FILE_CHOOSER_ACTION_SAVE ||
	    action == GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER ) {
		button = GTK_STOCK_SAVE;
	}
		
	dialog = gtk_file_chooser_dialog_new( title, parent,
				action, 
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				button, GTK_RESPONSE_ACCEPT,
				NULL );
	gtk_window_set_role( GTK_WINDOW( dialog ), role );
	gtk_file_chooser_set_local_only( GTK_FILE_CHOOSER( dialog ), 
					FALSE );

	while( filters ) {
		gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( dialog ),
						filters->data );
		filters = filters->next;
	}
	all = screem_get_file_filter( _( "All Files" ) );
	gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( dialog ), all );
	if( dir ) {
		gtk_file_chooser_set_current_folder_uri( GTK_FILE_CHOOSER( dialog ), dir );
	}
	
	screem_session_restore_dialog( session, dialog );

	if( gtk_dialog_run( GTK_DIALOG( dialog ) ) == GTK_RESPONSE_ACCEPT ) {
		ret = gtk_file_chooser_get_uri( GTK_FILE_CHOOSER( dialog ) );
		if( ret ) {
			if( ! g_utf8_validate( ret, -1, NULL ) ) {
				temp = g_filename_to_utf8( ret, -1,
						NULL, NULL, NULL );
				g_free( ret );
				ret = temp;
			}
		}
	}

	/* don't use screem_session_store_dialog(), we don't
	 * want to store the width/height for save dialogs */
	if( session ) {
		if( action == GTK_FILE_CHOOSER_ACTION_SAVE ||
		    action == GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER ) {
			w = h = -1;
		} else {
			gtk_window_get_size( GTK_WINDOW( dialog ),
				&w, &h );
		}
		gtk_window_get_position( GTK_WINDOW( dialog ),
				&x, &y );
		screem_session_set_dialog( session, 
				gtk_window_get_role( GTK_WINDOW( dialog ) ),
				x, y, w, h );
	}
	
	gtk_widget_destroy( dialog );
	
	return ret;
}

GSList *screem_fileselect_multi( const gchar *title, GtkWindow *parent,
			GtkFileChooserAction action,
			const gchar *dir,
			const GSList *filters )
{
	ScreemSession *session;

	GtkWidget *dialog;
	GSList *ret;
	gchar *temp;
	GSList *tmp;
	const gchar *button;
	GtkFileFilter *all;

	gint x;
	gint y;
	gint w;
	gint h;
	
	const gchar *role = "screem_file_selector";
	
	session = screem_application_get_session( app );
	ret = NULL;

	button = GTK_STOCK_OPEN;
	if( action == GTK_FILE_CHOOSER_ACTION_SAVE ||
	    action == GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER ) {
		button = GTK_STOCK_SAVE;
	}
		
	dialog = gtk_file_chooser_dialog_new( title, parent,
				action, 
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				button, GTK_RESPONSE_ACCEPT,
				NULL );
	gtk_window_set_role( GTK_WINDOW( dialog ), role );
	gtk_file_chooser_set_local_only( GTK_FILE_CHOOSER( dialog ), 
					FALSE );
	gtk_file_chooser_set_select_multiple( GTK_FILE_CHOOSER( dialog),
						TRUE );
	while( filters ) {
		gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( dialog ),
						filters->data );
		filters = filters->next;
	}
	all = screem_get_file_filter( _( "All Files" ) );
	gtk_file_chooser_add_filter( GTK_FILE_CHOOSER( dialog ), all );
	if( dir ) {
		gtk_file_chooser_set_current_folder_uri( GTK_FILE_CHOOSER( dialog ), dir );
	}
	
	screem_session_restore_dialog( session, dialog );

	if( gtk_dialog_run( GTK_DIALOG( dialog ) ) == GTK_RESPONSE_ACCEPT ) {
		ret = gtk_file_chooser_get_uris( GTK_FILE_CHOOSER( dialog ) );
		for( tmp = ret; tmp; tmp = tmp->next ) {
			temp = tmp->data;
			if( ! g_utf8_validate( temp, -1, NULL ) ) {
				tmp->data = g_filename_to_utf8( temp, 
						-1, NULL, NULL, NULL );
				g_free( temp );
			}
		}	
	}

	/* don't use screem_session_store_dialog(), we don't
	 * want to store the width/height for save dialogs */
	if( session ) {
		if( action == GTK_FILE_CHOOSER_ACTION_SAVE ||
		    action == GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER ) {
			w = h = -1;
		} else {
			gtk_window_get_size( GTK_WINDOW( dialog ),
				&w, &h );
		}
		gtk_window_get_position( GTK_WINDOW( dialog ),
				&x, &y );
		screem_session_set_dialog( session, 
				gtk_window_get_role( GTK_WINDOW( dialog ) ),
				x, y, w, h );
	}
	
	gtk_widget_destroy( dialog );
	
	return ret;
}

gint screem_compare_file_age( const gchar *a, const gchar *b )
{
	GnomeVFSFileInfo *info;
	time_t atime;
	time_t btime;
	GnomeVFSFileInfoOptions options;
	
	info = gnome_vfs_file_info_new();
	atime = btime = 0;
	options = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;
	if( gnome_vfs_get_file_info( a, info, options ) == GNOME_VFS_OK ) {
		atime = info->mtime;
	}
	if( gnome_vfs_get_file_info( b, info, options ) == GNOME_VFS_OK ) {
		btime = info->mtime;
	}
	gnome_vfs_file_info_unref( info );

	return ( atime - btime );
}

