/*  Screem:  screem-site-ui.c,
 *  this file contains GUI code that deals with the site
 * 
 *  Copyright (C) 1999 - 2001  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <glib/gfileutils.h>

#include <gtk/gtktogglebutton.h>

#include <libgnomeui/gnome-messagebox.h>
#include <libgnomeui/gnome-stock-icons.h>
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-file-entry.h>
#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include <string.h>

#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <time.h>

#include <gtk/gtkinvisible.h>

#include <glade/glade.h>

#include <libxml/tree.h>

#include "fileops.h"
#include "pageUI.h"
#include "support.h"

#include "screem-application.h"
#include "screem-site.h"
#include "screem-site-ui.h"
#include "screem-page.h"
#include "screem-editor.h"
#include "screem-preview.h"

#include "screem-site-view.h"

#include "screem-file-browser.h"

void screem_site_settings_dialog_cb( GtkWidget *widget, gint button );
void screem_site_settings_change_url( GtkWidget *widget, gpointer data );

static void screem_site_gather_stats( ScreemSite *site, GladeXML *xml );
static void screem_site_setup_lists( ScreemSite *site, GladeXML *xml );

static void screem_site_create_template_finish( GtkWidget *widget,
						gpointer data );

static void screem_site_create_template_select( GtkTreeSelection *selection,
						gpointer data );

static void screem_site_create_template_added( ScreemFileBrowser *browser,
					       const gchar *uri,
					       const gchar *mime_type,
					       GtkTreeIter *it,
					       gpointer data );
static void screem_site_create_template_removed( ScreemFileBrowser *browser,
						 const gchar *uri,
						 GtkTreeIter *it,
						 gpointer data );
static xmlNodePtr screem_site_template_locate_uri( xmlNodePtr node, const gchar *uri );

static void screem_site_template_file_op( GnomeVFSMonitorEventType type,
		const gchar *uri, gpointer data );

void screem_site_close_site( ScreemSite *site )
{
       	g_return_if_fail( site != NULL );

	/* if this is the fake site then it can't be closed */
	if( ! screem_site_get_fake_flag( site ) ) {
		screem_site_save( site );
	}
}

gboolean screem_site_open_with_filename( ScreemWindow *window,
					 ScreemApplication *application,
					 const gchar *filename )
{
	ScreemSite *site;
	gchar *path = NULL;
	GnomeVFSURI *uri;
	GnomeVFSFileInfo *info;
	gchar *msg;

	g_return_val_if_fail( filename != NULL, FALSE );

	screem_application_set_cursor( application, GDK_WATCH );

	uri = gnome_vfs_uri_new( filename );
	
	/* is the filename local? */
	if( gnome_vfs_uri_is_local( uri ) ) {
		/* yes */
		/* HACK */
		if( *filename == G_DIR_SEPARATOR ) {
			path = g_strconcat( "file://", filename, NULL );
		} else {
			path = relative_to_full( filename, NULL );
		}
	} else {
		path = g_strdup( filename );
	}
	gnome_vfs_uri_unref( uri );

	/* ensure we have a dir */
	info = gnome_vfs_file_info_new();

	if( gnome_vfs_get_file_info( path, info, 
				 GNOME_VFS_FILE_INFO_FOLLOW_LINKS ) != 
			GNOME_VFS_OK ) {
		g_free( path );
		screem_application_set_cursor( application, 
				GDK_LAST_CURSOR );
		return FALSE;
	}

	if( info->type != GNOME_VFS_FILE_TYPE_DIRECTORY )  {
		gchar *temp = g_path_get_dirname( path );
		g_free( path );
		path = temp;
	}

	gnome_vfs_file_info_unref( info );

	/* if we already have the site loaded, then don't do so again */
	site = screem_application_get_site( application, path );
	if( site ) {
		g_free( path );
		if( window ) {
			screem_window_set_current( window, site );
		}
		screem_application_set_cursor( application, 
				GDK_LAST_CURSOR );
		return TRUE;
	}

	site = screem_site_new( G_OBJECT( application ) );
	screem_site_set_pathname( site, path );
	screem_site_set_name( site, _( "Imported Site" ) );

	if( window ) {
		msg = g_strdup_printf( _( "Loading site: %s" ), path );
		screem_window_show_message( window, msg, FALSE );
		g_free( msg );
		gdk_threads_leave(); 
		while( g_main_context_iteration(NULL,FALSE) ) { }
		gdk_threads_enter();
	}
	
	if( ! screem_site_load( site ) ) {
		/* we failed to load the site */
		g_object_unref( G_OBJECT( site ) );
		if( window ) {
			msg = g_strdup_printf( _( "Failed to load site: %s" ), path );
			screem_window_show_message( window, msg, FALSE );
			g_free( msg );
			g_free( path );
		}
		screem_application_set_cursor( application, 
				GDK_LAST_CURSOR );
		return FALSE;
	}
		
	g_free( path );
		
	/* add this site to the list of loaded sites */
	screem_application_add_site( application, site );
	screem_application_set_cursor( application, GDK_LAST_CURSOR );

	if( window ) {
		screem_window_set_current( window, site );
		
		msg = g_strdup_printf( _( "Loaded site: %s" ), screem_site_get_name( site ) );
		screem_window_show_message( window, msg, FALSE );
		g_free( msg );
		gdk_threads_leave(); 
		while( g_main_context_iteration(NULL,FALSE) ) { }
		gdk_threads_enter();
		
		/* if we are importing a site open the site settings dialog */
		if( screem_site_get_is_import( site ) ) {
			screem_site_settings_dialog( site, window );
			screem_site_set_is_import( site, FALSE );
		}
	}

	/* site loaded, check for cvs update,
	   only do this if the site auto update flag is false,
	   and the user hasn't disabled displaying of the prompting dialog */

	if( screem_site_get_cvs_root( site ) && 
	    screem_site_get_auto_update( site ) ) {
		GladeXML *xml;
		GtkWidget *widget;
		gint button;
		GtkWidget *toggle;
		gboolean ask;
		gchar *gladepath;
	
		gladepath = screem_get_glade_path();
		xml = glade_xml_new( gladepath,
				     "auto_cvs_update", NULL );
		g_free( gladepath );
		widget = glade_xml_get_widget( xml, "auto_cvs_update" );
		toggle = glade_xml_get_widget( xml, "dont_ask" );

		gtk_window_set_wmclass( GTK_WINDOW( widget ), 
					"Screem", "auto_cvs_update" );
		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		if( screem_site_get_auto_update_ask( site ) ) {
			button = gtk_dialog_run( GTK_DIALOG( widget ) );
			ask = !gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(toggle));
			screem_site_set_auto_update_ask( site, ask );
		} else {
			ask = TRUE;
			button = 0;
		}

		gtk_widget_destroy( widget );

		/* FIXME can't update from session load as window == NULL */
		if( button == 0 && window ) {
			/* do a cvs update */
			GtkAction *action;
			
			action = gtk_action_group_get_action( GTK_ACTION_GROUP( window->action_group ),
								"UpdateSite" );
			gtk_action_activate( action );
		} else if( ! ask ) {
			/* don't ask was selected and the dialog was
			   just closed */
			screem_site_set_auto_update( site, FALSE );
		}
	}
	
	return TRUE;
}

/**
 * screem_site_settings_dialog:
 * opens the site settings dialog
 *
 * return values: none
 */
void screem_site_settings_dialog( ScreemSite *site, 
		ScreemWindow *window )
{
	ScreemApplication *app;
	ScreemSession *session;
	GtkWidget *dialog1;
     	GtkWidget *entry;
	GtkWidget *toggle;
	ScreemSitePerms permissions;
	ScreemSiteSymlinks symlinks;
	const gchar *text;
	GladeXML *xml;
	gchar *gladepath;

	gint button;
		
	gchar *spath;

	g_return_if_fail( site != NULL );
	g_return_if_fail( ! screem_site_get_fake_flag( site ) );

	app = SCREEM_APPLICATION( window->application );
	session = screem_application_get_session( app );
	
	gladepath = screem_get_glade_path();	
	xml = glade_xml_new( gladepath, "site_settings", NULL );
	g_free( gladepath );

	glade_xml_signal_autoconnect( xml );
		
	screem_site_gather_stats( site, xml );

	screem_site_setup_lists( site, xml );
	
	dialog1 = glade_xml_get_widget( xml, "site_settings" );
	
	/* fill in the details */
	/* name */
	entry = glade_xml_get_widget( xml, "site_name_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	text = screem_site_get_name( site );
	if( ! text ) {
		text = _( "Imported Site ");
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* path */
	entry = glade_xml_get_widget( xml, "site_path_entry" );
	text = screem_site_get_pathname( site );
	if( ! text ) {
		text = "";
	}
	gtk_label_set_text( GTK_LABEL( entry ), text );
	
	/* cvs root */
	entry = glade_xml_get_widget( xml, "cvs_file_entry" );
	if( ( text = screem_site_get_cvs_root( site ) ) ) {
		toggle = glade_xml_get_widget( xml, "cvs_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );
		entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY(entry ) );
		gtk_entry_set_text( GTK_ENTRY( entry ), text );
	}
	
	/* template path */
	entry = glade_xml_get_widget( xml, "template_file_entry" );
	spath = screem_get_local_site_path( site );
	if( spath ) {
		gnome_file_entry_set_default_path( GNOME_FILE_ENTRY( entry ), spath );	
	}
	g_free( spath );
	if( ( text = screem_site_get_template_path( site ) ) ) {
		toggle = glade_xml_get_widget( xml, "template_check" );
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
					     TRUE );

		entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY(entry ) );
		gtk_entry_set_text(GTK_ENTRY( entry ), text );	
	}
	
	/* auto cvs update */
	toggle = glade_xml_get_widget( xml, "auto_cvs_update" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_auto_update( site ) );

	/* ctags */
	toggle = glade_xml_get_widget( xml, "ctags" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_use_ctags( site ) );
	
	/* internal preview settings */
	toggle = glade_xml_get_widget( xml, "preview_static" );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ),
				      screem_site_get_preview_static( site ) );

	toggle = glade_xml_get_widget( xml, "preview_dynamic" );
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON( toggle ),
				     screem_site_get_preview_dynamic( site ) );

	/* remote url */
	entry = glade_xml_get_widget( xml, "remote_url_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	text = screem_site_get_remote_url( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* http url */
	entry = glade_xml_get_widget( xml, "http_url_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	text = screem_site_get_http_url( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote method */
	entry = glade_xml_get_widget( xml, "remote_method" );
	gtk_combo_box_set_active( GTK_COMBO_BOX( entry ), 
			screem_site_get_remote_method( site ) );
	
	/* passive ftp */
	if( screem_site_get_passive_ftp( site ) ) {
		toggle = glade_xml_get_widget( xml, "ftpmode" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* no file delete */
	if( screem_site_get_no_delete( site ) ) {
		toggle = glade_xml_get_widget( xml, "nodelete" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* check for moved files */
	if( screem_site_get_check_moved( site ) ) {
		toggle = glade_xml_get_widget( xml, "checkmoved" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* no file overwrite */
	if( screem_site_get_no_overwrite( site ) ) {
		toggle = glade_xml_get_widget( xml, "nooverwrite" );
		gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	}
	
	/* permissions handling */
	permissions = screem_site_get_permissions( site );
	toggle = glade_xml_get_widget( xml, permission_strings[permissions] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	
	/* symbolic link handling */
	symlinks = screem_site_get_symlinks( site );
	toggle = glade_xml_get_widget( xml, symlink_strings[ symlinks ] );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( toggle ), TRUE );
	
	/* remote path */
	entry = glade_xml_get_widget( xml, "remote_path_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	text = screem_site_get_remote_path( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote user */
	entry = glade_xml_get_widget( xml, "remote_user_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	text = screem_site_get_remote_user( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* remote password */
	entry = glade_xml_get_widget( xml, "remote_password_entry" );
	text = screem_site_get_remote_pass( site );
	if( ! text ) {
		text = "";
	}
	gtk_entry_set_text( GTK_ENTRY( entry ), text );
	
	/* attach the site to the dialog */
	g_object_set_data( G_OBJECT( dialog1 ), "site", site );
	
	gtk_window_set_transient_for( GTK_WINDOW( dialog1 ),
				      GTK_WINDOW( window ) );

	screem_session_restore_dialog( session, dialog1 );
	
	button = gtk_dialog_run( GTK_DIALOG( dialog1 ) );
	
	if( button == GTK_RESPONSE_CLOSE ) {
		screem_site_settings_dialog_cb( dialog1, button );
	}
	
	screem_session_store_dialog( session, dialog1 );
	
	gtk_widget_destroy( dialog1 );
	g_object_unref( xml );
}

void screem_site_settings_change_url( GtkWidget *widget, gpointer data )
{

}

void screem_site_settings_dialog_cb( GtkWidget *widget, gint button ) 
{
	GtkWidget *entry;
	GtkWidget *toggle;
	ScreemSite *site;
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );

	site = SCREEM_SITE( g_object_get_data( G_OBJECT( widget ), "site") );
	
	/* name */
	entry = glade_xml_get_widget( xml, "site_name_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	screem_site_set_name( site, 
			      gtk_entry_get_text( GTK_ENTRY(entry) ) );
	/* cvs root */
	toggle = glade_xml_get_widget( xml, "cvs_check" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
		entry = glade_xml_get_widget(xml,"cvs_file_entry");
		entry = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( entry ) );
		screem_site_set_cvs_root( site, 
					  gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	} else {
		screem_site_set_cvs_root( site, NULL );
	}
	
	/* template */
	toggle = glade_xml_get_widget( xml, "template_check" );
	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) ) {
		gchar *path;

		entry = glade_xml_get_widget( xml, 
					      "template_file_entry" );
		path = gnome_file_entry_get_full_path( GNOME_FILE_ENTRY( entry ), FALSE );
		
		screem_site_set_template_path( site, path ); 
		g_free( path );
	} else {
		screem_site_set_template_path( site, NULL );
	}

	/* cvs update */
	toggle = glade_xml_get_widget( xml, "auto_cvs_update" );
	screem_site_set_auto_update( site,
				     gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );

	/* ctags */
	toggle = glade_xml_get_widget( xml, "ctags" );
	screem_site_set_use_ctags( site,
		     gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );

	/* preview */
	toggle = glade_xml_get_widget( xml, "preview_static" );
	screem_site_set_preview_static( site,
					gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );

	toggle = glade_xml_get_widget( xml, "preview_dynamic" );
	screem_site_set_preview_dynamic( site,
					 gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( toggle ) ) );
	
	/* remote url */
	entry = glade_xml_get_widget( xml, "remote_url_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	screem_site_set_remote_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	/* http url */
	entry = glade_xml_get_widget( xml, "http_url_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	screem_site_set_http_url( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	
	/* remote method */
	entry = glade_xml_get_widget( xml, "remote_method" );
	screem_site_set_remote_method( site,
			gtk_combo_box_get_active( GTK_COMBO_BOX( entry ) ) );
	
	/* passive ftp */
	toggle = glade_xml_get_widget( xml, "ftpmode" );
	screem_site_set_passive_ftp( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* no file delete */
	toggle = glade_xml_get_widget( xml, "nodelete" );
	screem_site_set_no_delete( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* check for moved files */
	toggle = glade_xml_get_widget( xml, "checkmoved" );
	screem_site_set_check_moved( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* no file overwrite */
	toggle = glade_xml_get_widget( xml, "nooverwrite" );
	screem_site_set_no_overwrite( site, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) ));
	
	/* permissions handling */
	toggle = glade_xml_get_widget( xml, "p_ignore" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_permissions( site, PERMS_IGNORE );
	}
	toggle = glade_xml_get_widget( xml, "p_exec" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_permissions( site, PERMS_EXEC );
	}
	toggle = glade_xml_get_widget( xml, "p_all" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_permissions( site, PERMS_ALL );
	}
	
	/* symbolic link handling */
	toggle = glade_xml_get_widget( xml, "s_ignore" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_symlinks( site, SYM_IGNORE );
	}
	toggle = glade_xml_get_widget( xml, "s_follow" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_symlinks( site, SYM_FOLLOW );
	}
	toggle = glade_xml_get_widget( xml, "s_maintain" );
	if(gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON(toggle) )) {
		screem_site_set_symlinks( site, SYM_MAINTAIN );
	}
	
	/* remote path */
	entry = glade_xml_get_widget( xml, "remote_path_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	screem_site_set_remote_path( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	
	/* remote user */
	entry = glade_xml_get_widget( xml, "remote_user_entry" );
	entry = gnome_entry_gtk_entry( GNOME_ENTRY( entry ) );
	screem_site_set_remote_user( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );
	
	/* remote password */
	entry = glade_xml_get_widget( xml, 
				      "remote_password_entry");
	screem_site_set_remote_pass( site, gtk_entry_get_text( GTK_ENTRY( entry ) ) );

	screem_site_save( site );
}

static void screem_site_gather_stats( ScreemSite *site, GladeXML *xml )
{
	GtkWidget *last_upload;
	GtkWidget *files_list;
	GtkWidget *total_size;

	GtkCellRenderer *prenderer;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;

	GtkTreeModel *model;

	gchar *num;

	gint total;
	gchar *sitecopy_file;
	const gchar *name;

	struct stat st;

	last_upload = glade_xml_get_widget( xml, "last_upload" );
	files_list = glade_xml_get_widget( xml, "files_list" );
	total_size = glade_xml_get_widget( xml, "total_size" );

	name = screem_site_get_name( site );

	sitecopy_file = g_strconcat( g_get_home_dir(), G_DIR_SEPARATOR_S,
				     ".sitecopy", G_DIR_SEPARATOR_S, name,
				     NULL );
	if( ! stat( sitecopy_file, &st ) ) {
		gchar *tmp;

		tmp = ctime( &st.st_mtime );
		tmp[ strlen( tmp ) - 1 ] = '\0';
		gtk_label_set_text( GTK_LABEL( last_upload ), tmp );
	} else {
		gtk_label_set_text( GTK_LABEL( last_upload ), _("Unknown") );
	}
	g_free( sitecopy_file );

	model = screem_site_get_statistics( site );
	total = GPOINTER_TO_INT( g_object_get_data( G_OBJECT( model ), 
						    "total" ) );
	num = g_strdup_printf( _("%i bytes"), total );
	gtk_label_set_text( GTK_LABEL( total_size ), num );
	g_free( num );

	prenderer = gtk_cell_renderer_pixbuf_new();
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();

	gtk_tree_view_column_set_title( column, _( "Mime Type" ) );
	gtk_tree_view_column_pack_start( column, prenderer, FALSE );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_column_set_resizable( column, TRUE );
	
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 0,
					     NULL );
	gtk_tree_view_column_set_attributes( column, prenderer,
					     "pixbuf", 1,
					     NULL );

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, _( "Number" ) );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 2, NULL );

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, _( "Total Size" ) );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 3, NULL );

	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	
	gtk_tree_view_column_set_title( column, _( "Average Size" ) );
	gtk_tree_view_column_pack_start( column, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( files_list ), column );
	gtk_tree_view_column_set_attributes( column, renderer,
					     "text", 4, NULL );

	gtk_tree_view_set_model( GTK_TREE_VIEW( files_list ), model );
	g_object_unref( model );
}

static void screem_site_lists_toggle_remove( GtkTreeSelection *selection,
					     gpointer data )
{
	gint count;

	count = gtk_tree_selection_count_selected_rows( selection );
	gtk_widget_set_sensitive( GTK_WIDGET( data ), ( count > 0 ) );
}

static gboolean screem_site_manage_pattern_add_idle( GtkWidget *view )
{
	GtkTreeViewColumn *col;
	GtkTreePath *path;

gdk_threads_enter();
	
	path = g_object_get_data( G_OBJECT( view ), "path" );
	g_object_set_data( G_OBJECT( view ), "path", NULL );	
	
	col = gtk_tree_view_get_column( GTK_TREE_VIEW( view ), 0 );
	
	gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( view ),
				      path, col, FALSE, 0, 0 );
	
	gtk_tree_view_set_cursor( GTK_TREE_VIEW( view ), path, col, TRUE );

	gtk_tree_path_free( path );
	
	gdk_threads_leave();
	
	return FALSE;
}

void screem_site_manage_pattern_add( GtkWidget *view, gpointer button )
{
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreePath *path;
	
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );

	gtk_list_store_append( GTK_LIST_STORE( model ), &it );
	gtk_list_store_set( GTK_LIST_STORE( model ), &it,
			    0, _( "Enter Pattern" ),
			    1, TRUE,
			    -1 );

	path = gtk_tree_model_get_path( model, &it );
	g_object_set_data( G_OBJECT( view ), "path", path );	

	g_idle_add_full( G_PRIORITY_LOW, 
			(GSourceFunc)screem_site_manage_pattern_add_idle,
			view, NULL );
}

void screem_site_manage_pattern_remove( GtkWidget *view, gpointer button )
{
	GList *rows;
	GList *tmp;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreePath *path;
	GtkTreeIter it;
	GtkTreeRowReference *ref;
	ScreemSite *site;

	site = g_object_get_data( G_OBJECT( view ), "site" );
	g_assert( SCREEM_IS_SITE( site ) );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( view ) );
	
	model = NULL;
	
	rows = gtk_tree_selection_get_selected_rows( selection, &model );
	g_assert( model );
	for( tmp = rows; tmp; tmp = tmp->next ) {
		path = (GtkTreePath*)tmp->data;
		tmp->data = gtk_tree_row_reference_new( model, path );
		gtk_tree_path_free( path );
	}
	for( tmp = rows; tmp; tmp = tmp->next ) {
		ref = (GtkTreeRowReference*)( tmp->data );
		path = gtk_tree_row_reference_get_path( ref );
		
		if( gtk_tree_model_get_iter( model, &it, path ) ) {
			gtk_list_store_remove( GTK_LIST_STORE( model ), &it );
			screem_site_view_update( site );
			screem_site_save( site );
		}	
		gtk_tree_path_free( path );
		gtk_tree_row_reference_free( ref );
	}
	g_list_free( rows );
}

static void screem_site_lists_edited( GtkCellRendererText *renderer,
					const gchar *path,
					const gchar *new_text,
					ScreemSite *site )
{
	GtkListStore *model;
	GtkTreePath *tpath;
	GtkTreeIter it;
	
	model = GTK_LIST_STORE( g_object_get_data( G_OBJECT( renderer ),
							"model" ) );
	
	tpath = gtk_tree_path_new_from_string( path );
	
	if( gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &it, tpath ) ) {
		gtk_list_store_set( model, &it, 0, new_text, -1 );
		screem_site_view_update( site );
		screem_site_save( site );
	}

	gtk_tree_path_free( tpath );
		
}

static void screem_site_setup_lists( ScreemSite *site, GladeXML *xml )
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	GtkWidget *widget;
	GtkTreeSelection *selection;


	/* setup our exclude/ignore/ascii lists */
	widget = glade_xml_get_widget( xml, "exclude_list" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( site->private->excludes ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Pattern" ) );
	gtk_tree_view_column_pack_start( col, renderer,
					TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ),
				     col );
	gtk_tree_view_column_set_attributes( col, renderer,
					    "text", 0,
					    "editable", 1,
					    NULL );
	g_object_set_data( G_OBJECT( widget ), "site", site );
	g_signal_connect( G_OBJECT( renderer ), "edited",
			  G_CALLBACK( screem_site_lists_edited ),
			  site );
	g_object_set_data( G_OBJECT( renderer ), "model", 
			   site->private->excludes );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );
	widget = glade_xml_get_widget( xml, "exclude_remove" );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_site_lists_toggle_remove ),
			  widget );
	
	
	widget = glade_xml_get_widget( xml, "ignore_list" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( site->private->ignores ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Pattern" ) );
	gtk_tree_view_column_pack_start( col, renderer,
					TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ),
				     col );
	gtk_tree_view_column_set_attributes( col, renderer,
					    "text", 0,
					    "editable", 1,
					    NULL );
	g_object_set_data( G_OBJECT( widget ), "site", site );
	g_signal_connect( G_OBJECT( renderer ), "edited",
			  G_CALLBACK( screem_site_lists_edited ),
			  site );
	g_object_set_data( G_OBJECT( renderer ), "model", 
			   site->private->ignores );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );
	widget = glade_xml_get_widget( xml, "ignore_remove" );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_site_lists_toggle_remove ),
			  widget );
	
	widget = glade_xml_get_widget( xml, "ascii_list" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( site->private->asciis ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, _( "Pattern" ) );
	gtk_tree_view_column_pack_start( col, renderer,
					TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ),
				     col );
	gtk_tree_view_column_set_attributes( col, renderer,
					    "text", 0,
					    "editable", 1,
					    NULL );
	g_object_set_data( G_OBJECT( widget ), "site", site );
	g_signal_connect( G_OBJECT( renderer ), "edited",
			  G_CALLBACK( screem_site_lists_edited ),
			  site );
	g_object_set_data( G_OBJECT( renderer ), "model", 
			   site->private->asciis );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	gtk_tree_selection_set_mode( selection, GTK_SELECTION_MULTIPLE );
	widget = glade_xml_get_widget( xml, "ascii_remove" );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_site_lists_toggle_remove ),
			  widget );
}

void screem_site_create_template( ScreemWindow *window )
{
	ScreemApplication *app;
	gchar *uri;

	app = window->application;
	uri = screem_create_tmp_file( "screem-XXXXXX", NULL, TRUE );
	
	if( uri ) {
		if( ! mkdir_recursive( uri,
				       GNOME_VFS_PERM_USER_ALL,
				       screem_application_file_op, app ) ) {
			g_free( uri );
			uri = NULL;
		}
	}
	if( uri ) {
		gchar *glade_path;
		GladeXML *xml;
		ScreemFileBrowser *browser;
		GtkWidget *widget;
		GtkTreeModel *model;
		GtkCellRenderer *prenderer;
		GtkCellRenderer *renderer;
		GtkTreeViewColumn *column;
		GtkTreeSelection *selection;
		xmlDocPtr doc;

		doc = xmlNewDoc( XML_DEFAULT_VERSION );
		xmlDocSetRootElement( doc,
				      xmlNewDocNode( doc, NULL,
						     "screem_template",
						     NULL ) );

		glade_path = screem_get_glade_path();
		xml = glade_xml_new( glade_path, "site_template_window",
				     NULL );

		widget = glade_xml_get_widget( xml, "view" );
		
		browser = screem_file_browser_new();
		
		screem_file_browser_scan_directory( browser, uri,
						    FILE_BROWSE_ROOT |
						    FILE_BROWSE_RECURSE );

		model = screem_file_browser_get_model( browser );

		selection = gtk_tree_view_get_selection( GTK_TREE_VIEW(widget) );
		gtk_tree_selection_set_mode( GTK_TREE_SELECTION( selection ),
					     GTK_SELECTION_SINGLE );
		
		prenderer = gtk_cell_renderer_pixbuf_new();
		renderer = gtk_cell_renderer_text_new();
		column = gtk_tree_view_column_new();
		gtk_tree_view_column_set_title( column, "Template" );
		gtk_tree_view_column_pack_start( column, prenderer, FALSE );
		gtk_tree_view_column_pack_start( column, renderer, TRUE );
		gtk_tree_view_column_set_resizable( column, TRUE );
		gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), column );
		gtk_tree_view_column_set_attributes( column, renderer,
						     "text", 
						     FILE_BROWSER_NAME_COL,
						     NULL );
		gtk_tree_view_column_set_attributes( column, prenderer,
						     "pixbuf",
						     FILE_BROWSER_ICON_COL,
						     NULL );
		
		gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), model );
		g_object_set_data( G_OBJECT( browser ), "view", widget );

		widget = glade_xml_get_widget( xml, "site_template_window" );

		/* connect to selection change signal */
		g_signal_connect( G_OBJECT( selection ), "changed",
				  G_CALLBACK(screem_site_create_template_select),
				  widget );

		g_object_set_data( G_OBJECT( widget ), "doc", doc );
		g_object_set_data( G_OBJECT( browser ), "uri", uri );
		g_object_set_data( G_OBJECT( widget ), "browser", browser );
		g_object_set_data( G_OBJECT( widget ), "window", window );
		g_signal_connect( G_OBJECT( widget ),
				  "destroy",
				  G_CALLBACK(screem_site_create_template_finish),
				  browser );
		g_signal_connect( G_OBJECT( browser ), "added",
				  G_CALLBACK(screem_site_create_template_added),
				  widget );
		g_signal_connect(G_OBJECT( browser ), "removed",
				 G_CALLBACK(screem_site_create_template_removed),
				 widget );

		gtk_window_set_transient_for( GTK_WINDOW( widget ),
					      GTK_WINDOW( window ) );
		gtk_widget_show_all( widget );
		glade_xml_signal_autoconnect( xml );

		g_free( glade_path );
	} else {
		GtkWidget *widget;
	
		widget = gtk_message_dialog_new( GTK_WINDOW( window ),
						GTK_DIALOG_MODAL,
						GTK_MESSAGE_WARNING,
						GTK_BUTTONS_OK,
						_( "Failed to create temporary directory for the template" ) );
		gtk_dialog_run( GTK_DIALOG( widget ) );
	}
}

void screem_site_create_template_response( GtkWidget *dialog,
					   gint response )
{
	ScreemWindow *window;
	ScreemApplication *app;
	xmlDocPtr doc;
	gchar *filename;
	const gchar *uri;
	ScreemFileBrowser *browser;

	doc = (xmlDocPtr)g_object_get_data( G_OBJECT( dialog ), "doc" );
	browser = SCREEM_FILE_BROWSER( g_object_get_data( G_OBJECT( dialog ),
							  "browser" ) );
	uri = (const gchar*)g_object_get_data( G_OBJECT( browser ), "uri" );
	window = SCREEM_WINDOW( g_object_get_data( G_OBJECT( dialog ), 
						   "window" ) );
	g_object_get( G_OBJECT( window ), "app", &app, NULL );
	
	switch( response ) {
	case 0:
		/* load */
		filename = screem_fileselect( _( "Select Site Template" ), 
						GTK_WINDOW( window ),
						GTK_FILE_CHOOSER_ACTION_OPEN,
						NULL, NULL );
		if( filename ) {
			xmlDocPtr newdoc;
			ScreemSite *site;
			gchar *projectfile;
			gchar *url;

			url = screem_create_tmp_file( "screem-XXXXXX", ".xml", TRUE );
			if( url ) {
				if(!mkdir_recursive( url,
						     GNOME_VFS_PERM_USER_ALL,
						     screem_site_template_file_op,
						     browser ) ) {
					g_free( url );
					url = NULL;
				}
			}
			if( url ) {
				g_object_set_data( G_OBJECT( browser ), "uri",
						   url );

				newdoc = xmlParseFile( filename );
				g_object_set_data( G_OBJECT( dialog ), "doc",
						   NULL );
				
				site = screem_site_new( G_OBJECT( app ) );
				screem_site_set_site_template( site, filename );
				screem_site_set_pathname( site, url );
				screem_site_create( site );
				g_object_unref( site );
				projectfile = g_strconcat( url, 
							   G_DIR_SEPARATOR_S,
							   ".project.screem",
							   NULL );
				delete_file( projectfile, 
					screem_site_template_file_op, browser );
				g_free( projectfile );
				
				g_object_set_data( G_OBJECT( dialog ), "doc",
						   newdoc );
				
				xmlFreeDoc( doc );
				
				g_free( filename );

				screem_file_browser_scan_directory( browser,
								    url,
								    FILE_BROWSE_ROOT | FILE_BROWSE_RECURSE );
				delete_dir( uri,
					screem_site_template_file_op, browser );
				g_free( (gchar*)uri );
			}
		}
		break;
	case 1:
		filename = screem_fileselect( _("Select directory to save site as"), 
						GTK_WINDOW( window ),
						GTK_FILE_CHOOSER_ACTION_CREATE_FOLDER, NULL, NULL );
			
		if( filename ) {
			xmlSaveFile( filename, doc );
			g_free( filename );
		}
		break;
	default:
		gtk_widget_destroy( dialog );
		break;
	}
	g_object_unref( app );
}

static void screem_site_create_template_finish( GtkWidget *widget,
						gpointer data )
{
	ScreemFileBrowser *browser;
	xmlDocPtr doc;
	gchar *uri;

	browser = SCREEM_FILE_BROWSER( data );
	uri = (gchar*)g_object_get_data( G_OBJECT( browser ), "uri" );

	doc = (xmlDocPtr)g_object_get_data( G_OBJECT( widget ), "doc" );
	if( doc ) {
		xmlFreeDoc( doc );
	}
	delete_dir( uri, screem_site_template_file_op, browser );
	g_free( uri );
	g_object_unref( browser );
}

static void screem_site_create_template_select( GtkTreeSelection *selection,
						gpointer data )
{
	GtkWidget *widget;
	GladeXML *xml;
	GtkTreeModel *model;
	GtkTreeIter it;
	GtkTreeIter sit;

	xmlDocPtr doc;
	ScreemFileBrowser *browser;
	const gchar *base;

	doc = (xmlDocPtr)g_object_get_data( G_OBJECT( data ), "doc" );
	browser = SCREEM_FILE_BROWSER( g_object_get_data( G_OBJECT( data ),
							  "browser" ) );
	base = (const gchar*)g_object_get_data( G_OBJECT( browser ), "uri" );


	widget = GTK_WIDGET( gtk_tree_selection_get_tree_view( selection ) );
	xml = glade_get_widget_tree( widget );

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );
	gtk_tree_model_get_iter_first( model, &it );

	widget = glade_xml_get_widget( xml, "maintable" );

	if( gtk_tree_selection_get_selected( selection, &model,
					     &sit ) ) {
		gtk_widget_set_sensitive( widget, TRUE );

		if( ! memcmp( &it, &sit, sizeof( GtkTreeIter ) ) ) {
			/* clear ui */
			widget = glade_xml_get_widget( xml, "name" );
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
			widget = glade_xml_get_widget( xml, "template" );
			gtk_entry_set_text( GTK_ENTRY( widget ), "" );
		} else {
			/* fill in ui */
			GList *list;
			GList *tmp;
			const gchar *url;
			const gchar *uri;
			xmlNodePtr node;
			gchar *name;
			gchar *template;
			
			list = g_list_append( NULL, &sit );
			tmp = screem_file_browser_get_pathnames( browser,
								 list, FALSE );
			g_list_free( list );
			uri = url = NULL;
			if( tmp ) {
				uri = url = (const gchar*)tmp->data;
				g_list_free( tmp );
			} 
			if( uri ) {
				if( ! strncmp( "file://", url, strlen( "file://" ) ) ) {
					url += strlen( "file://" );
				}
				url += strlen( base );
			}
			node = xmlDocGetRootElement( doc );
			node = screem_site_template_locate_uri( node, url );
			if( node ) {
				name = xmlGetProp( node, "name" );
				template = xmlGetProp( node, "template" );
			} else {
				name = NULL;
				template = NULL;
			}

			if( ! name ) {
 				name = g_strdup( "" );
			}
			if( ! template ) {
				template = g_strdup( "" );
			}

			widget = glade_xml_get_widget( xml, "name" );
			gtk_entry_set_text( GTK_ENTRY( widget ), name );
			widget = glade_xml_get_widget( xml, "template" );
			gtk_entry_set_text( GTK_ENTRY( widget ), template );

			g_free( name );
			g_free( template );

			if( screem_uri_is_dir( uri ) ) {
				widget = glade_xml_get_widget( xml, "directory");
			} else {
				widget = glade_xml_get_widget( xml, "file" );
			}
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( widget ),
						      TRUE );
		}
	} else {
		gtk_tree_selection_select_iter( selection, &it );
	}
}

static void screem_site_create_template_added( ScreemFileBrowser *browser,
					       const gchar *uri,
					       const gchar *mime_type,
					       GtkTreeIter *it,
					       gpointer data )
{
	GtkTreeModel *model;
	GtkTreeIter parent;
	GtkTreePath *path;
	GtkWidget *widget;
	GladeXML *xml;
	xmlDocPtr doc;
	const gchar *base;
	const gchar *url;

	gchar **names;
	gint i;
	xmlNodePtr node;
	gboolean isdir;

	if( it ) {
		model = screem_file_browser_get_model( browser );
		model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );
		gtk_tree_model_iter_parent( model, &parent, it );
		widget = g_object_get_data( G_OBJECT( browser ), "view" );
		path = gtk_tree_model_get_path( model, &parent );
		gtk_tree_view_expand_row( GTK_TREE_VIEW( widget ), path, FALSE );
		gtk_tree_path_free( path );
	}
	
	widget = GTK_WIDGET( data );
	xml = glade_get_widget_tree( widget );
	doc = (xmlDocPtr)g_object_get_data( G_OBJECT( widget ), "doc" );
	base = (const gchar*)g_object_get_data( G_OBJECT( browser ), "uri" );

	url = uri;
	if( ! strncmp( "file://", url, strlen( "file://" ) ) &&
	    strncmp( "file://", base, strlen( "file://" ) ) ) {
		uri += strlen( "file://" );
	}
	uri += strlen( base );

	/* traverse doc creating nodes as necessary to locate
	   a node with name == uri, if we can't locate it, create it */

	if( doc ) {
		node = xmlDocGetRootElement( doc );
	} else {
		node = NULL;
	}
      
	names = g_strsplit( uri, G_DIR_SEPARATOR_S, -1 );
	for( i = 0; node && names[ i ]; ++ i ) {
		xmlNodePtr child;

		if( *names[ i ] == '\0' ) {
			continue;
		}

		child = node->xmlChildrenNode;
		while( child ) {
			gchar *name;
			gboolean match;

			name = xmlGetProp( child, "name" );
			match = ( name && ! strcmp( names[ i ], name ) );
			g_free( name );
			if( match ) {
				break;
			} else {
				child = child->next;
			}
		}
		if( ! child ) {
			/* add node */
			if( names[ i + 1 ] ) {
				isdir = TRUE;
			} else {
				isdir = screem_uri_is_dir( url );
			}
			if( isdir ) {
				child = xmlNewChild( node, NULL, "directory",
						     NULL );
			} else {
				child = xmlNewChild( node, NULL, "file",
						     NULL );
			}
			xmlSetProp( child, "name", names[ i ] );
		}
		node = child;
	}

	g_strfreev( names );
}

static void screem_site_create_template_removed( ScreemFileBrowser *browser,
						 const gchar *uri,
						 GtkTreeIter *it,
						 gpointer data )
{
	GtkWidget *widget;
	GladeXML *xml;
	xmlDocPtr doc;
	const gchar *base;
	const gchar *url;

	xmlNodePtr node;

	widget = GTK_WIDGET( data );
	xml = glade_get_widget_tree( widget );
	doc = (xmlDocPtr)g_object_get_data( G_OBJECT( widget ), "doc" );
	base = (const gchar*)g_object_get_data( G_OBJECT( browser ), "uri" );

	url = uri;
	uri += strlen( "file://" ) + strlen( base );

	/* traverse doc and delete relevant node */

	if( doc ) {
		node = xmlDocGetRootElement( doc );
	} else {
		node = NULL;
	}
      
	node = screem_site_template_locate_uri( node, uri );
	if( node ) {
		xmlUnlinkNode( node );
		xmlFreeNode( node );
	}
}

static xmlNodePtr screem_site_template_locate_uri( xmlNodePtr node, const gchar *uri )
{
	gint i;
	gchar **names;

	if( ! uri ) {
		return NULL;
	}
	
	names = g_strsplit( uri, G_DIR_SEPARATOR_S, -1 );
	for( i = 0; node && names[ i ]; ++ i ) {
		xmlNodePtr child;

		if( *names[ i ] == '\0' ) {
			continue;
		}

		child = node->xmlChildrenNode;
		while( child ) {
			gchar *name;
			gboolean match;

			name = xmlGetProp( child, "name" );
			match = ( name && ! strcmp( names[ i ], name ) );
			g_free( name );
			if( match ) {
				break;
			} else {
				child = child->next;
			}
		}
		node = child;

		if( node && ! names[ i + 1 ] ) {
			break;
		}
			
	}

	g_strfreev( names );

	return node;
}

static void screem_site_template_file_op( GnomeVFSMonitorEventType type,
		const gchar *uri, gpointer data )
{
	ScreemFileBrowser *browser;
	gchar *dirname;

	browser = SCREEM_FILE_BROWSER( data );
	dirname = g_path_get_dirname( uri );
	screem_file_browser_file_mod( browser, dirname, uri, type );
	g_free( dirname );
}


void screem_site_create_template_add( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemFileBrowser *browser;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	xmlDocPtr doc;
	const gchar *base;
	
	xml = glade_get_widget_tree( widget );

	widget = glade_xml_get_widget( xml, "site_template_window" );
	browser = SCREEM_FILE_BROWSER( g_object_get_data( G_OBJECT( widget ),
							  "browser" ) );
	doc = (xmlDocPtr)g_object_get_data( G_OBJECT( widget ), "doc" );

	widget = glade_xml_get_widget( xml, "view" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );

	base = (const gchar*)g_object_get_data( G_OBJECT( browser ), "uri" );
	
	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		GList *list;
		GList *tmp;
		const gchar *url;
		gchar *uri;
		gchar *temp;
		const gchar *name;
		const gchar *template;
		gboolean isdir;
		gboolean created;

		widget = glade_xml_get_widget( xml, "name" );
		name = gtk_entry_get_text( GTK_ENTRY( widget ) );
		widget = glade_xml_get_widget( xml, "directory");
		isdir = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
		widget = glade_xml_get_widget( xml, "template" );
		template = gtk_entry_get_text( GTK_ENTRY( widget ) );
	
		list = g_list_append( NULL, &it );
		tmp = screem_file_browser_get_pathnames( browser,
							 list, FALSE );
		g_list_free( list );
		url = (const gchar*)tmp->data;
		g_list_free( tmp );

		/* if url isn't a directory, we want its parent */
		if( ! screem_uri_is_dir( url ) ) {
			temp = g_path_get_dirname( url );
		} else {
			temp = g_strdup( url );
		}
		uri = g_strconcat( temp, G_DIR_SEPARATOR_S, name, NULL );
		url = uri;
		url += strlen( "file://" ) + strlen( base );

		g_free( temp );

		if( isdir ) {
			created = mkdir_recursive( uri,  
						GNOME_VFS_PERM_USER_ALL,
						screem_site_template_file_op,
						browser );
		} else {
			created = save_file( uri, "", 
					     GNOME_VFS_PERM_USER_READ |
					     GNOME_VFS_PERM_USER_WRITE,
					     FALSE, NULL );
		}
		if( created ) {
			xmlNodePtr node;

			widget = glade_xml_get_widget( xml,
						       "site_template_window" );
			
			screem_site_create_template_added( browser,
							   uri, 
							   NULL, NULL, 
							   widget );

			node = xmlDocGetRootElement( doc );
			node = screem_site_template_locate_uri( node,
								url );
			if( *template && node ) {
				xmlSetProp( node, "template", template );
			}
		}
		g_free( uri );
	}
}


void screem_site_create_template_remove( GtkWidget *widget )
{
	GladeXML *xml;
	ScreemFileBrowser *browser;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	
	xml = glade_get_widget_tree( widget );

	widget = glade_xml_get_widget( xml, "site_template_window" );
	browser = SCREEM_FILE_BROWSER( g_object_get_data( G_OBJECT( widget ),
							  "browser" ) );

	widget = glade_xml_get_widget( xml, "view" );
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );
	model = gtk_tree_view_get_model( GTK_TREE_VIEW( widget ) );

	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		GList *list;
		GList *tmp;
		const gchar *url;

		list = g_list_append( NULL, &it );
		tmp = screem_file_browser_get_pathnames( browser,
							 list, FALSE );
		g_list_free( list );
		url = (const gchar*)tmp->data;
		g_list_free( tmp );
		delete_dir( url, screem_site_template_file_op, browser );
	}
}

gboolean screem_site_save_confirmation( ScreemSite *site )
{
	gint button;
	GList *list;
	GList *tmp;
	gboolean ret;
	
	ret = TRUE;
	button = GTK_RESPONSE_YES;
	for( tmp = list = screem_site_get_pages( site ); 
		list && button != GTK_RESPONSE_CANCEL ;
		list = list->next ) {
		button = screem_page_save_confirmation( SCREEM_PAGE( list->data ), TRUE );
			
		if( button == GTK_RESPONSE_CANCEL ||
		    button == GTK_RESPONSE_NONE ||
		    button == GTK_RESPONSE_DELETE_EVENT ) {
			ret = FALSE;
		}
	}
	g_list_free( tmp );

	return ret;
}


