/*  Screem:  screem-spell.c
 *
 *  Copyright (C) 2002 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <ctype.h>

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <glade/glade.h>

#include <gtk/gtkliststore.h>
#include <gtk/gtktextbuffer.h>
#include <gtk/gtkdialog.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtkcellrenderer.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkentry.h>

#include "screem-spell.h"
#include "screem-page.h"

#include "support.h"

#include "screemmarshal.h"

static void screem_spell_class_init( ScreemSpellClass *klass );
static void screem_spell_init( ScreemSpell *spell );
static void screem_spell_finalize( GObject *object );

typedef struct ScreemSpellWord {
	gchar *word;
	gint pos;
} ScreemSpellWord;


struct ScreemSpellPrivate {
	GList *words;

	GtkListStore *store;

	/* pid of ispell process */
	gint pid;

	gint inpipe[ 2 ];
	gint outpipe[ 2 ];
};

static const gchar *ispell_command = "ispell";
static gchar *const ispell_command_args[ 16 ] = {
	"screem-spellchecker",
	"-a",
	NULL
};

enum { ACCEPT = 0, REPLACE, INSERT, INSERTCASE, SKIP };


enum {
	HIGHLIGHT,
	LAST_SIGNAL
};
static guint screem_spell_signals[LAST_SIGNAL] = { 0 };


static ScreemSpellWord* screem_spell_word_new( const gchar *text, gint pos );
static void screem_spell_word_destroy( ScreemSpellWord *word );

static void screem_spell_split_text( ScreemSpell *spell, const gchar *text );
static gboolean screem_spell_check( ScreemSpell *spell,
				    const gchar *word,
				    gchar **buffer );


ScreemSpell* screem_spell_new()
{
	ScreemSpell *spell;
	GType type;
	
	type = screem_spell_get_type();

	spell = SCREEM_SPELL( g_object_new( type, NULL ) );

	return spell;
}


gboolean screem_spell_check_word( ScreemSpell *spell, const gchar *word )
{
	return screem_spell_check( spell, word, NULL );
}

void screem_spell_check_interactive( ScreemSpell *spell )
{
	GladeXML *xml;
	gchar *gladepath;
	GtkWidget *widget;
	gpointer data;
	GList *list;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	ScreemPage *page;
	gchar *text;
	GtkTextBuffer *tbuffer;
	GtkTextIter it;
	GtkTextIter eit;
	GtkWidget *dialog;
	gint offset;

	data = g_object_get_data( G_OBJECT( spell ), "page" );
	
	if( ! data )
		return;
	
	page = SCREEM_PAGE( data );
	tbuffer = GTK_TEXT_BUFFER( page );
	text = screem_page_get_data( page );
	screem_spell_split_text( spell, text );
	g_free( text );

	gladepath = screem_get_glade_path();
	xml = glade_xml_new( gladepath, "spellcheck_dialog", NULL );
	g_free( gladepath );

	widget = glade_xml_get_widget( xml, "spellaltlist" );
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ),
				 GTK_TREE_MODEL( spell->private->store ) );
	renderer = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	gtk_tree_view_column_set_title( col, "Alternatives" );
	gtk_tree_view_column_pack_start( col, renderer, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	gtk_tree_view_column_set_attributes( col, renderer, "text", 0, NULL );


	dialog = glade_xml_get_widget( xml, "spellcheck_dialog" );

	glade_xml_signal_autoconnect( xml );

	offset = 0;
	list = spell->private->words;
	while( dialog ) {
		gchar *buffer;
		GList *alts = NULL;
		GList *alist;
		gchar c;
		gchar *curword;
		gboolean replace = FALSE;
		ScreemSpellWord *word;

		if( list ) {
			word = (ScreemSpellWord*)list->data;
			list = list->next;
		} else {
			GtkWidget *notebook;
			notebook = glade_xml_get_widget( xml, "notebook3" );
			gtk_widget_set_sensitive( notebook, FALSE );
			word = NULL;
			gtk_dialog_run( GTK_DIALOG( dialog ) );
			gtk_widget_destroy( dialog );
			dialog = NULL;
		}
		
		if( word && 
		    ! screem_spell_check( spell, word->word, &buffer ) ) {
			/* buffer format:
			   <symbol> <word> <number> <word number>: <word>, ...
			*/
			if( buffer ) {
				gchar tmp[ BUFSIZ ];
				gchar *temp;
				gint num;
				
				sscanf( buffer, "%c %s %d", &c, tmp, &num );
				temp = strchr( buffer, ':' );
				temp += 2;
				while( num ) {
					curword = temp;
					temp = strchr( curword, ',' );
					if( temp ) {
						*temp = '\0';
						temp += 2;
					} else {
						temp = strpbrk( curword, "\r\n" );
						if( temp )
							*temp = '\0';
					}
					alts = g_list_append( alts, curword );
					num --;
				}
			}
			/* fill in entries in dialog */
			widget = glade_xml_get_widget( xml, "spellword" );
			gtk_entry_set_text( GTK_ENTRY( widget ), word->word );
			gtk_list_store_clear( spell->private->store );
			for( alist = alts; alist; alist = alist->next ) {
				GtkTreeIter it;
				gtk_list_store_append( spell->private->store,
						       &it );
				gtk_list_store_set( spell->private->store,
						    &it,
						    0, alist->data,
						    1, spell,
						    -1 );
			}
			g_list_free( alts );
			/* highlight the word */
			g_signal_emit( G_OBJECT( spell ),
				       screem_spell_signals[ HIGHLIGHT ], 0,
				       word->pos + offset, 
				       word->pos + offset + 
				       strlen( word->word ) );
			/* run dialog */
			c = '\0';
			switch( gtk_dialog_run( GTK_DIALOG( dialog ) ) ) {
			case ACCEPT:
				c = '@';
				break;
			case REPLACE:
				replace = TRUE;
				break;
			case INSERT:
				c = '+';
				replace = TRUE;
				break;
			case INSERTCASE:
				c = '&';
				replace = TRUE;
				break;
			case SKIP:
				break;
			default:
				gtk_widget_destroy( dialog );
				dialog = NULL;
				break;
			}
			if( ! dialog )
				break;
			curword = (gchar*)
				gtk_entry_get_text( GTK_ENTRY( widget ) );
			if( c != '\0' ) {
				/* FIXME: remove command characters from
				   curword */
				gchar *temp = g_strdup_printf( "%c%s\n",
							       c, curword );
				write( spell->private->outpipe[ 1 ],
				       temp, strlen( temp ) );
				g_free( temp );
			}
			if( replace ) {
				gint len = strlen( word->word );

				gtk_text_buffer_get_iter_at_offset( tbuffer,
								    &it,
								    word->pos +
								    offset );
				gtk_text_buffer_get_iter_at_offset( tbuffer,
								    &eit,
								    word->pos +
								    offset +
								    len );

				gtk_text_buffer_delete( tbuffer, &it, &eit );
				gtk_text_buffer_insert( tbuffer, &it,
							curword, 
							strlen( curword ) );
				offset += strlen( curword ) - len;
			}
		}
	}

	g_object_unref( xml );
}

void screem_spell_alt_selected( GtkTreeView *view, GtkTreePath *path,
				GtkTreeViewColumn *column, gpointer data )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkTreeIter it;
	GtkTreeModel *model;
	GValue value = {0};
	const gchar *word;

	model = gtk_tree_view_get_model( GTK_TREE_VIEW( view ) );
	if( gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &it, path ) ) {
		gtk_tree_model_get_value( GTK_TREE_MODEL( model ), &it, 0,
					  &value );
		word = g_value_get_string( &value );
		
		xml = glade_get_widget_tree( GTK_WIDGET( view ) );
		widget = glade_xml_get_widget( xml, "spellword" );
		gtk_entry_set_text( GTK_ENTRY( widget ), word );
		
		g_value_unset( &value );
	}
}

void screem_spell_accept( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), ACCEPT );
}
void screem_spell_replace( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), REPLACE );
}
void screem_spell_insert( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), INSERT );
}
void screem_spell_insertcase( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), INSERTCASE );
}
void screem_spell_skip( GtkWidget *widget )
{
	GladeXML *xml;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "spellcheck_dialog" );
	gtk_dialog_response( GTK_DIALOG( widget ), SKIP );
}

/* static stuff */
static ScreemSpellWord* screem_spell_word_new( const gchar *text, gint pos )
{
	ScreemSpellWord *word;

	word = g_new0( ScreemSpellWord, 1 );
	word->word = g_strdup( text );
	word->pos = pos;

	return word;
}

static void screem_spell_word_destroy( ScreemSpellWord *word )
{
	g_free( word->word );
	g_free( word );
}


/* splits text up into its individual words, ignoring html tags,
   FIXME: should really handle any text within tag attributes such as alt
   in an <img> tag */
static void screem_spell_split_text( ScreemSpell *spell, const gchar *text )
{
	GList *list;
	gint len;
	const gchar *start;
	gchar c;
	gint i;
	ScreemSpellWord *w;

	g_list_foreach( spell->private->words,
			(GFunc)screem_spell_word_destroy,
			NULL );
	g_list_free( spell->private->words );
	spell->private->words = NULL;

	list = NULL;
	len = strlen( text );

	for( i = 0, start = text; i < len; ++ i ) {
		c = *( text + i );
		if( isspace( c ) || ( ispunct( c ) && c != '\'' ) ) {
			if( start != text + i ) {
				/* we have come to the end of a word */
				gchar *word;
				gint wordlen;

				wordlen = text + i - start;
				word = g_strndup( start, wordlen );
				w = screem_spell_word_new( word, 
							   start - text );
				g_free( word );
				list = g_list_append( list, w );
			}
			start = text + i + 1;
		}

		/* stopped at the start of a tag, advance forward until we
		   hit the end of the tag */
		if( c == '<' ) {
			while( ( c != '>' ) && ( i < len ) ) {
				i ++;
				c = *( text + i );
			}
			start = text + i + 1;
		} else if( c == '&' ) {
			/* we stopped at a char encoding element */
			while( ( c != ';' ) && ( i < len ) ) {
				i ++;
				c = *( text + i );
			}
			start = text + i + 1;
		}

	}
	spell->private->words = list;
}

static gboolean screem_spell_check( ScreemSpell *spell,
				    const gchar *word,
				    gchar **buffer )
{
	gchar *temp;
	gint size;
	gchar buf[ BUFSIZ ];
	gboolean success = FALSE;

	if( buffer )
		*buffer = NULL;

	temp = g_strconcat( "^", word, "\n", NULL );

	if( write( spell->private->outpipe[ 1 ], temp, strlen( temp ) ) < 0 ) {
		g_free( temp );
		return success;
	}

	g_free( temp );

	if( ( size = read( spell->private->inpipe[ 0 ], buf, BUFSIZ ) ) < 0 ) {
		return success;
	}

	buf[ size ] = '\0';

	switch( buf[ 0 ] ) {
	case '*':
		/* word is ok */
		success = TRUE;
		break;
	case '+':
		/* word found */
		success = TRUE;
		break;
	case '-':
		/* found as compound */
		success = TRUE;
		break;
	case '\n':
		/* ignore the word */
		success = TRUE;
		break;
	case '#':
		/* not found, no idea what it is */
		break;
	case '?':
		/* not found, but we have guesses */
	case '&':
		/* not found, but here is some near misses */
		if( buffer )
			*buffer = g_strdup( buf );
		break;
	default:
		/* probably a number */
		success = TRUE;
		break;
	}

	return success;
}

/* G Object stuff */

#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_spell_class_init( ScreemSpellClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	object_class->finalize = screem_spell_finalize;


	screem_spell_signals[HIGHLIGHT] = 
		g_signal_new( "highlight",
			      G_OBJECT_CLASS_TYPE( object_class ),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET( ScreemSpellClass, highlight ),
			      NULL, NULL,
			      screem_marshal_VOID__INT_INT,
			      G_TYPE_NONE, 2,
			      G_TYPE_INT,
			      G_TYPE_INT );
}

static void screem_spell_init( ScreemSpell *spell )
{
	spell->private = g_new0( ScreemSpellPrivate, 1 );

	spell->private->store = gtk_list_store_new( 2, 
						    G_TYPE_STRING, 
						    G_TYPE_POINTER, 
						    NULL );

	/* spawn an ispell process */
	pipe( spell->private->inpipe );
	pipe( spell->private->outpipe );

	spell->private->pid = fork();

	switch( spell->private->pid ) {
	case -1:
		/* error */
		break;
	case 0:
		/* child process */
		close( 0 );
		dup( spell->private->outpipe[ 0 ] );
		close( spell->private->outpipe[ 1 ] );
		close( 1 );
		dup( spell->private->inpipe[ 1 ] );
		close( spell->private->inpipe[ 0 ] );
		execvp( ispell_command, ispell_command_args );
		_exit( 1 );
		break;
	default:
		/* parent process */
		close( spell->private->outpipe[ 0 ] );
		close( spell->private->inpipe[ 1 ] );

		/* read in the ispell banner */
		{
			gchar buffer[ BUFSIZ + 1 ];
			read( spell->private->inpipe[ 0 ], buffer, BUFSIZ );
		}
		break;
	}
}

static void screem_spell_finalize( GObject *object )
{
	ScreemSpell *spell;
	gint status;

	spell = SCREEM_SPELL( object );

	kill( spell->private->pid, SIGTERM );
	waitpid( spell->private->pid, &status, 0 );
	close( spell->private->inpipe[ 0 ] );
	close( spell->private->outpipe[ 1 ] );

	if( spell->private->words ) {
		g_list_foreach( spell->private->words,
				(GFunc)screem_spell_word_destroy,
				NULL );
		g_list_free( spell->private->words );
	}

	g_object_unref( spell->private->store );

	g_free( spell->private );

	G_OBJECT_CLASS( parent_class )->finalize( object );
}

GType screem_spell_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemSpellClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_spell_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemSpell ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_spell_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemSpell",
					       &info, 0 );
	}

	return type;
}
