#include "packet.hpp"
#include "tcpip.hpp"
#include <new>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

extern "C" {
	#include <errno.h>
	#include <netinet/in.h>
	#include <netinet/ip.h>
	#include <netinet/tcp.h>
	#include <netinet/udp.h>
	#include <arpa/inet.h>
	#include <unistd.h>
	#include "log.h"

	#ifdef WITH_IPQ
	#include <linux/netfilter.h>
	#include <libipq.h>
	#endif //WITH_IPQ

	#ifdef WITH_DIVERT
	#include <sys/types.h>
	#include <sys/socket.h>
	#endif //WITH_DIVERT
}

using namespace std;

static void log_tcp_ip(void *payload, char *buf, int n);

#ifdef WITH_DIVERT
divert_packet::divert_packet()
{
	sock = -1;
	payload = 0;
	payload_size = 0;
	inp_if = out_if = 0;
}

divert_packet::~divert_packet()
{
}

int divert_packet::send_back()
{
	int result;

	result = sendto(sock, buf, payload_size, 
		0, (const struct sockaddr *)(&from), sizeof(from));
	if( result==-1 ) {
		log_info(LL_ERR, "sendto() error (%s)", strerror(errno));
		return -1;
	}
	return 0;
}

void divert_packet::log_packet(char *buf, int n)
{
	char buf2[n];

	log_tcp_ip(payload, buf2, n);
	snprintf(buf, n-1, "type=divert %s", buf2);
}

generic_packet *get_divert_packet(void *arg)
{
	int n, result, sock;
	divert_packet *packet;

	sock = (int)(arg);

	packet = new(nothrow) divert_packet;
	if( packet==NULL ) {
		log_info(LL_ALERT, "no memory for new divert_packet {%s:%d}",
			__FILE__, __LINE__);
		return 0;
	}

	socklen_t fromlen = sizeof(struct sockaddr_in);
	result = n = recvfrom(sock, packet->buf, packet->buf_size, 0, 
		(struct sockaddr *)(&packet->from), &fromlen);
	if( result==-1 ) {
		log_info(LL_ALERT, "recvfrom() error (%s) {%s:%d}", 
			strerror(errno), __FILE__, __LINE__);
		goto __err_recvfrom;
	}

	packet->sock = sock;
	packet->payload = packet->buf;
	packet->payload_size = n;

	return packet;
	__err_recvfrom:
		delete packet;
		return 0;
}
#endif //WITH_DIVERT

#ifdef WITH_IPQ
ipq_packet::ipq_packet()
{
	handle = 0;
	payload = meta = 0;
	payload_size = 0;
	inp_if = out_if = 0;
}

ipq_packet::~ipq_packet()
{
	if( handle ) {
		int result;
		ipq_packet_msg_t *ptr = ipq_get_packet(buf);
		result = ipq_set_verdict(handle, ptr->packet_id, NF_DROP, 0, 0);
		if( result==-1 ) {
			log_info(LL_WARN, "can't NF_DROP packet: "
				"ipq_set_verdict() error (%s) {%s:%d}",
				ipq_errstr(), __FILE__, __LINE__);
		}
		handle = 0;
	}
}

int ipq_packet::send_back()
{
	int result;

	result = ipq_set_verdict(handle, meta->packet_id, NF_ACCEPT, 0, 0);
	if( result<0 ) {
		log_info(LL_WARN, "can't NF_ACCEPT packet: "
			"ipq_set_verdict() error (%s) {%s:%d}",
			ipq_errstr(), __FILE__, __LINE__);
		return -1;
	}
	handle = 0;
	return 0;
}

int ipq_packet::check_packet()
{
	struct iphdr *ip_hdr;
	unsigned int hw_proto, ip_hdr_total_len;

	meta = ipq_get_packet(buf);
	ip_hdr = (struct iphdr *)(meta->payload);

	hw_proto = ntohs(meta->hw_protocol);
	if( hw_proto!=0x0800 ) {
		// current kernels (as of 2.4.18) set the skb->protocol
		// field *after* the outgoing packet has gone into the first
		// output hook (NF_IP_LOCAL_OUT)
		//
		log_info(LL_WARN, "invalid packet (hw_protocol=0x%x) "
			"please apply http://webs.sinectis.com/lesanti/"
			"shaperd/fw-2.4.17.patch to fix this problem {%s:%d}", 
			hw_proto, __FILE__, __LINE__);
		send_back();
		return -1;
	}

	if( meta->data_len<20 ) {
		log_info(LL_ERR, "datagram truncated (data_len=%d) {%s:%d}",
			meta->data_len, __FILE__, __LINE__);
		return -1;
	}

	if( ip_hdr->version!=4 ) {
		log_info(LL_ERR, "not an ipv4 packet (version=%d) {%s:%d}",
			ip_hdr->version, __FILE__, __LINE__);
		return -1;
	}

	ip_hdr_total_len = ntohs(ip_hdr->tot_len);
//	if( ip_hdr_total_len!=meta->data_len ) {
//		log_info(LL_ERR, "error: ip_hdr->tot_len disagrees with "
//			"netfilter! {%s:%d}", __FILE__, __LINE__);
//		return -1;
//	}

	payload = meta->payload;
	payload_size = ip_hdr_total_len;

	return 0;
}


void ipq_packet::log_packet(char *buf, int n)
{
	char buf2[n];

	log_tcp_ip(payload, buf2, n);
	snprintf(buf, n-1, "type=ipq inp_if=\"%s\" out_if=\"%s\" "
		"%s mark=0x%lx",
		inp_if, out_if, buf2, (unsigned long int) (meta->mark));
}

generic_packet *get_ipq_packet(void *arg)
{
	int result;
	ipq_packet *packet;
	struct ipq_handle *handle = (struct ipq_handle*)(arg);

	packet = new(nothrow) ipq_packet;
	if( packet==NULL ) {
		log_info(LL_ALERT, "no memory for new ipq_packet {%s:%d}",
			__FILE__, __LINE__);
		return 0;
	}

	result = ipq_read(handle, packet->buf, packet->buf_size, 0);
	if( result<0 ) {
		log_info(LL_WARN, "ipq_read() error (%s) {%s:%d}",
			ipq_errstr(), __FILE__, __LINE__);
		goto __err_ipq_read;
	}

	result = ipq_message_type(packet->buf);
	if( result==NLMSG_ERROR ) {
		// you are reading this because you've  no idea of 
		// what the logs are sayin'. me neither :)
		// naah, as the docs says, if netlink can't do its
		// job (transport information between the kernel
		// and userland) it'll return with an error code.
		// for example, this can happen if you bring down
		// an interface while shaperd is sending stuff 
		// towards it (it happened to me :)
		// 
		int errnum = ipq_get_msgerr(packet->buf);
		log_info(LL_WARN, "recv'd netlink error %d (%s) {%s:%d}",
			errnum, strerror(errnum), __FILE__, __LINE__);
		goto __err_ipq_msgtype;
	} else if( result!=IPQM_PACKET ) {
		// ...
		log_info(LL_WARN, "??? aborting program {%s:%d}",
			__FILE__, __LINE__);
		exit(-1);
	}

	// ipq_packet::check_packet() could drop it if malformed,
	// so this is really necessary here
	packet->handle = handle;

	result = packet->check_packet();
	if( result!=0 ) {
		goto __err_check_packet;
	}

	packet->inp_if = packet->meta->indev_name;
	packet->out_if = packet->meta->outdev_name;

	return packet;
	__err_check_packet:
	__err_ipq_msgtype:
	__err_ipq_read:
		delete packet;
		return 0;
}
#endif //WITH_IPQ

static void log_tcp_ip(void *payload, char *buf, int n)
{
	struct iphdr *ip_hdr = (struct iphdr *)(payload);
	struct in_addr addr;
	char str_saddr[64], str_daddr[64], *str_proto;
	int len = -1, src_port = -1, dst_port = -1, tcpudp = 0;

	addr.s_addr = ip_hdr->saddr; 
	snprintf(str_saddr, 63, "%s", inet_ntoa(addr));

	addr.s_addr = ip_hdr->daddr; 
	snprintf(str_daddr, 63, "%s", inet_ntoa(addr));

	len = ntohs(ip_hdr->tot_len);

	switch(int(ip_hdr->protocol))
	{
		case __IP_HDR_PROTO_TCP__: {
			void *ip_payload = ((char*)(payload)) + (4*ip_hdr->ihl);
			struct tcphdr *tcp_hdr = (struct tcphdr*)ip_payload;
			str_proto = "tcp";
			src_port = int(ntohs(tcp_hdr->source));
			dst_port = int(ntohs(tcp_hdr->dest));
			tcpudp = 1;
			break;
		}
		case __IP_HDR_PROTO_UDP__: {
			void *ip_payload = ((char*)(payload)) + (4*ip_hdr->ihl);
			struct udphdr *udp_hdr = (struct udphdr*)ip_payload;
			str_proto = "udp";
			src_port = int(ntohs(udp_hdr->source));
			dst_port = int(ntohs(udp_hdr->dest));
			tcpudp = 1;
			break;
		}
		case __IP_HDR_PROTO_ICMP__: {
			str_proto = "icmp";
			break;
		}
		default: {
			str_proto = "?";
			break;
		}
	}

	if( tcpudp ) {
		snprintf(buf, n-1, "len=%d proto=%s src=%s %d dst=%s %d",
			len, str_proto, str_saddr, src_port, str_daddr, 
			dst_port);
	} else {
		snprintf(buf, n-1, "len=%d proto=%s src=%s dst=%s",
			len, str_proto, str_saddr, str_daddr);
	}
}

