#include "time.hpp"
#include "sched.hpp"
#include "packet.hpp"

extern "C" {
	#include "log.h"
	#include <unistd.h>
	#include <sys/time.h>
	#include <sys/types.h>
}

using namespace std;

#define SCHED_SLEEPING 1
#define SCHED_BURST    2
#define SCHED_IDLE     3

scheduler::scheduler()
{
	state = SCHED_IDLE;
	sleep_threshold = (1LL<<32)/10;

	scheduled_classid = -1;
	bw_adm = 0;

	total_bytes = total_packets = 0;
}

scheduler::~scheduler()
{
}

int scheduler::send(generic_packet *packet)
{
	int result;

	result = fsm();
	if( result==-1 ) {
		return -1;
	}

	result = __send(packet);
	if( result==-1 ) {
		return -1;
	}

	return 0;	
}

int scheduler::can_send()
{
	int result;

	result = fsm();
	if( result==-1 ) {
		return -1;
	}

	if( state!=SCHED_IDLE && state!=SCHED_BURST ) {
		return 0;
	}
	return 1;
}

int scheduler::is_sleeping()
{
	int result;

	result = fsm();
	if( result==-1 ) {
		return -1;
	}

	if( state!=SCHED_SLEEPING ) {
		return 0;
	}
	return usleep;
}

int scheduler::fsm_burst()
{
	u_int64_t now = ck();
	u_int64_t dt  = burst_end - now;

	if( burst_end <= now) {
		state = SCHED_IDLE;
		return 0;
	} else if( dt > sleep_threshold ) {
		state = SCHED_SLEEPING;
		usleep = to_usec(dt);

		log_info(LL_ERR,
			"scheduler::fsm_burst() warning: "
			"inconsistent fsm state, usleep=%d {%s:%d}",
			usleep, __FILE__, __LINE__);

		return 0;
	} else {
		return 0;
	}
}

int scheduler::fsm_sleeping()
{
	u_int64_t now = ck();
	u_int64_t dt  = burst_end - now;

	if( burst_end <= now ) {
		state = SCHED_IDLE;
		return 0;
	} else if( dt > sleep_threshold ) {
		state = SCHED_SLEEPING;
		usleep = to_usec(dt);
		return 0;
	} else {
		state = SCHED_BURST;
		burst_start = now;
		return 0;
	}
}

int scheduler::fsm()
{
	int result;

	switch(state) {
		case SCHED_IDLE:
			result = 0;
			break;
		case SCHED_BURST:
			result = fsm_burst();
			break;
		case SCHED_SLEEPING:
			result = fsm_sleeping();
			break;
		default:
			log_info(LL_ALERT, "invalid fsm state {%s:%d}",
				__FILE__, __LINE__);
			result = -1;
			break;			
	}
	return result;
}

// handles low-level stuff
int scheduler::__send(generic_packet *packet)
{
	int result;
	u_int64_t dt, now;

	if( state!=SCHED_IDLE && state!=SCHED_BURST ) {
		log_info(LL_ERR, "scheduler::__send() can't transmit in this "
			"state (%d) {%s:%d}", state, __FILE__, __LINE__);
		return -1;
	}

	result = bw_adm->getbw(scheduled_classid, packet->payload_size, &dt);
	if( result==1 ) {
		log_info(LL_DEBUG1, "dropping packet (class' bw=0) {%s:%d}",
			__FILE__, __LINE__);
		return 0;
	}

	now = ck();

	if( state==SCHED_IDLE ) {
		state = SCHED_BURST;
		burst_start = now;
		burst_end   = now + dt;
	} else {
		burst_end += dt;
	}

	if( burst_end > now ) {
		u_int64_t burst_left = burst_end - now;

		if( burst_left > sleep_threshold ) {
			state = SCHED_SLEEPING;
			usleep = to_usec(burst_left);
		}
	}

	result = packet->send_back();
	if( result==-1 ) {
		return -1;
	}

	total_packets++;
	total_bytes += packet->payload_size;

	return 0;
}

void scheduler::set_classid(int id)
{
	scheduled_classid = id;
}

void scheduler::set_bwadm(bwadm *ptr)
{
	bw_adm = ptr;
}
