/*
 * The Sleuth Kit
 *
 * $Date: 2005/03/15 21:17:18 $
 * 
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2005 Brian Carrier.  All rights reserved 
 *
 * This file is part of disktools
 *
 * disktools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * imgtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "tsk_os.h"
#include "tsk_types.h"
#include "disk_ide.h"
#include "aux_lib.h"
#include "error.h"

void
usage()
{
    fprintf(stderr, "usage: disk_stat [-V] DEVICE\n");
    fprintf(stderr, "\t-V: Print version\n");
    return;
}

#if defined(LINUX2)

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


int
main(int argc, char **argv)
{
    int fd;
    char *device = NULL;	/* e.g. "/dev/hda" */
    struct stat devstat;
    int ch;

    DISK_INFO *di;
    uint64_t nat_max = 0;

    while ((ch = getopt(argc, argv, "V")) > 0) {
	switch (ch) {
	case 'V':
	    print_version(stdout);
	    return 0;
	default:
	    usage();
	    return 0;
	}
    }

    if (optind < argc)
	device = argv[optind];

    if (!device) {
	fprintf(stderr, "no device specified\n");
	usage();
	exit(1);
    }

    if (0 != stat(device, &devstat)) {
	fprintf(stderr, "Error opening %s\n", device);
	exit(1);
    }

    if ((S_ISCHR(devstat.st_mode) == 0) && (S_ISBLK(devstat.st_mode) == 0)) {
	fprintf(stderr, "The file name must correspond to a device\n");
	exit(1);
    }

    fd = open(device, O_RDONLY);
    if (fd == -1) {
	error("error opening device %s (%m)", device);
    }

    di = identify_device(fd);

    /* Check if the disk supports HPA */
    if ((di->flags & DISK_HPA) == 0) {
	nat_max = di->capacity - 1;
    }
    else {
	nat_max = get_native_max(fd);
    }

    close(fd);

    printf("Maximum Disk Sector: %" PRIu64 "\n", nat_max);
    printf("Maximum User Sector: %" PRIu64 "\n", di->capacity - 1);

    if (di->capacity - 1 != nat_max) {
	printf("\n** HPA Detected (Sectors %" PRIu64 " - %" PRIu64
	       ") **\n\n", di->capacity, nat_max);
    }

    if ((di->flags & DISK_HPA) == 0)
	printf("(Disk does not support HPA)\n");

    exit(0);
}

#else

int
main(int argc, char **argv)
{
    int ch;

    while ((ch = getopt(argc, argv, "V")) > 0) {
	switch (ch) {
	case 'V':
	    print_version(stdout);
	    return 0;
	default:
	    usage();
	    return 0;
	}
    }

    fprintf(stderr, "This tool works only on Linux systems\n");
    return 0;
}

#endif
