/*
 * The Sleuth Kit
 *
 * $Date: 2005/01/17 22:40:17 $
 *
 * Brian Carrier [carrier@sleuthkit.org]
 * Copyright (c) 2003-2005 Brian Carrier.  All rights reserved
 *
 * bsd.c: BSD Disk labels
 *
 * This file is part of mmtools
 *
 * mmtools is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * mmtools is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with mactime; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, LOSS OF USE, DATA, OR PROFITS OR
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "mm_tools.h"
#include "bsd.h"


/*
 * Return a buffer with a description of the partition type
 */
static char *
bsd_get_desc(uint8_t fstype)
{
    char *str = mymalloc(64);
    switch (fstype) {

    case 0:
	strncpy(str, "Unused (0x00)", 64);
	break;
    case 1:
	strncpy(str, "Swap (0x01)", 64);
	break;
    case 2:
	strncpy(str, "Version 6 (0x02)", 64);
	break;
    case 3:
	strncpy(str, "Version 7 (0x03)", 64);
	break;
    case 4:
	strncpy(str, "System V (0x04)", 64);
	break;
    case 5:
	strncpy(str, "4.1BSD (0x05)", 64);
	break;
    case 6:
	strncpy(str, "Eighth Edition (0x06)", 64);
	break;
    case 7:
	strncpy(str, "4.2BSD (0x07)", 64);
	break;
    case 8:
	strncpy(str, "MSDOS (0x08)", 64);
	break;
    case 9:
	strncpy(str, "4.4LFS (0x09)", 64);
	break;
    case 10:
	strncpy(str, "Unknown (0x0A)", 64);
	break;
    case 11:
	strncpy(str, "HPFS (0x0B)", 64);
	break;
    case 12:
	strncpy(str, "ISO9660 (0x0C)", 64);
	break;
    case 13:
	strncpy(str, "Boot (0x0D)", 64);
	break;
    case 14:
	strncpy(str, "Vinum (0x0E)", 64);
	break;
    default:
	snprintf(str, 64, "Unknown Type (0x%.2x)", fstype);
	break;
    }

    return str;
}

/* 
 * Process the partition table at the sector address 
 *
 */
static uint8_t
bsd_load_table(MM_INFO * mm, uint8_t test)
{

    bsd_disklabel dlabel;
    uint32_t idx = 0;

    if (verbose)
	fprintf(logfp, "bsd_load_table: Table Sector: %" PRIuDADDR "\n",
		mm->sect_offset);

    /* seek to the table */
    if (mm_read_block_nobuf
	(mm, (char *) &dlabel, sizeof(dlabel),
	 mm->sect_offset) != sizeof(dlabel)) {
	if (test) {
	    fprintf(stderr,
		    "Error reading BSD Disk Label in Sector: %" PRIuDADDR
		    "\n", mm->sect_offset);
	    return 1;
	}
	else {
	    error("Error reading BSD Disk Label in Sector: %" PRIuDADDR
		  "\n", mm->sect_offset);
	}
    }

    /* Check the magic  */
    if (guessu32(mm, dlabel.magic, BSD_MAGIC)) {

	/* Check if the next sector has the correct magic value because
	 * the user may have given the first partition sector, but the
	 * dlabel is in the second sector
	 */
	mm->sect_offset++;
	if (mm_read_block_nobuf
	    (mm, (char *) &dlabel, sizeof(dlabel),
	     mm->sect_offset) != sizeof(dlabel)) {

	    if (test) {
		fprintf(stderr,
			"Error reading BSD Disk Label (second try) in Sector: %"
			PRIuDADDR "\n", mm->sect_offset);
		return 1;
	    }
	    else {
		error
		    ("Error reading BSD Disk Label (second try)in Sector: %"
		     PRIuDADDR "\n", mm->sect_offset);
	    }
	}


	/* If it isn't there, then give an error */
	if (guessu32(mm, dlabel.magic, BSD_MAGIC)) {
	    if (test)
		return 1;
	    else
		error
		    ("Invalid BSD partition table (magic #1) (Sector: %"
		     PRIuDADDR ") %" PRIx32 "", mm->sect_offset - 1,
		     getu32(mm, dlabel.magic));
	}
    }

    /* Check the second magic value */
    if (getu32(mm, dlabel.magic2) != BSD_MAGIC) {
	if (test)
	    return 1;
	else
	    error("Invalid BSD disk label (magic #2) (Sector: %"
		  PRIuDADDR ")  %" PRIx32 "\n", mm->sect_offset, getu32(mm,
									dlabel.
									magic2));
    }

    /* Cycle through the partitions, there are either 8 or 16 */
    for (idx = 0; idx < getu16(mm, dlabel.num_parts); idx++) {

	uint32_t part_start;
	uint32_t part_size;

	part_start = getu32(mm, dlabel.part[idx].start_sec);
	part_size = getu32(mm, dlabel.part[idx].size_sec);

	if (verbose)
	    fprintf(logfp,
		    "load_table: %d  Starting Sector: %" PRIu32
		    "  Size: %" PRIu32 "  Type: %d\n", idx, part_start,
		    part_size, dlabel.part[idx].fstype);

	if (part_size == 0)
	    continue;

	/* Add the partition to the internal sorted list */
	mm_part_add(mm, (DADDR_T) part_start, (DADDR_T) part_size,
		    MM_TYPE_VOL,
		    bsd_get_desc(dlabel.part[idx].fstype), -1, idx);
    }

    return 0;
}


/* 
 * Walk the partitions that have already been loaded during _open
 *
 */
void
bsd_part_walk(MM_INFO * mm, PNUM_T start, PNUM_T last, int flags,
	      MM_PART_WALK_FN action, char *ptr)
{
    MM_PART *part;
    PNUM_T cnt = 0;

    if (start < mm->first_part || start > mm->last_part)
	error("bsd_part_walk: Invalid start partition: %" PRIuPNUM "",
	      start);

    if (last < mm->first_part || last > mm->last_part)
	error("bsd_part_walk: Invalid end partition: %" PRIuPNUM "", last);

    part = mm->part_list;
    while ((part != NULL) && (cnt <= last)) {

	if (cnt >= start)
	    action(mm, cnt, part, 0, ptr);

	part = part->next;
	cnt++;
    }

    return;
}


void
bsd_close(MM_INFO * mm)
{
    mm_part_free(mm);
    free(mm);
}

MM_INFO *
bsd_open(IMG_INFO * img_info, DADDR_T offset, uint8_t test)
{
    MM_INFO *mm = (MM_INFO *) mymalloc(sizeof(*mm));

    mm->img_info = img_info;
    mm->mmtype = MM_BSD;
    mm->str_type = "BSD Disk Label";


    /* use the offset provided */
    mm->sect_offset = offset + BSD_PART_OFFSET;

    /* inititialize settings */
    mm->part_list = NULL;
    mm->first_part = mm->last_part = 0;
    mm->endian = 0;
    mm->dev_bsize = 512;
    mm->block_size = 512;

    /* Assign functions */
    mm->part_walk = bsd_part_walk;
    mm->close = bsd_close;

    /* Load the partitions into the sorted list */
    if (bsd_load_table(mm, test)) {
	bsd_close(mm);
	return NULL;
    }

    /* fill in the sorted list with the 'unknown' values */
    mm_part_unused(mm);

    return mm;
}
