/***************************************************************************
                          smb4kcore  -  description
                             -------------------
    begin                : Do Apr 8 2004
    copyright            : (C) 2004 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KCORE_H
#define SMB4KCORE_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qmap.h>
#include <qtimer.h>

// KDE includes
#include <kconfig.h>

// application specific includes
#include "smb4kscanner.h"
#include "smb4kmounter.h"
#include "smb4kpasswdreader.h"
#include "smb4kfileio.h"
#include "smb4kipfinder.h"
#include "smb4kbookmarkhandler.h"
#include "smb4kprint.h"
#include "smb4kshellio.h"


/**
 * This is the top-level core class. It inherits all other core classes 
 * except the containers.
 */

class Smb4KCore : public QObject
{
  Q_OBJECT
  
  public:
    /**
     * The constructor.
     */
    Smb4KCore( QObject *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KCore();
    /**
     * Returns the state the scanner is in.
     */
    int scannerState() { return m_scanner_state; }
    /**
     * Returns TRUE, if the scanner is running, otherwise FALSE.
     */
    bool scannerIsRunning() { return m_scanner_running; }
    /**
     * Returns the state the mounter is in.
     */
    int mounterState() { return m_mounter_state; }
    /**
     * Returns TRUE, if the mounter is running, otherwise FALSE.
     */
    bool mounterIsRunning() { return m_mounter_running; }
    /**
     * Returns the state the IP finder is in.
     */
    int ipFinderState() { return m_ipfinder_state; }
    /**
     * Returns TRUE, if the IP finder is running, otherwise FALSE.
     */
    bool ipFinderIsRunning() { return m_ipfinder_running; }
    /**
     * Returns the state the print object is in.
     */
    int printState() { return m_print_state; }
    /**
     * Returns TRUE, if the print object is running, otherwise FALSE.
     */
    bool printIsRunning() { return m_print_running; }
    /**
     * Returns TRUE if one of the core classes is doing something.
     */
    bool isRunning();
    /**
     * Returns the current state the core class is in.
     */
    int currentState() { return m_current_state; }
    /**
     * Returns a pointer to the scanner object.
     */
    Smb4KScanner *scanner() { return m_scanner; }
    /**
     * Returns a pointer to the mounter object.
     */
    Smb4KMounter *mounter() { return m_mounter; }
    /**
     * Returns a pointer to the bookmark handler object.
     */
    Smb4KBookmarkHandler *bookmarkHandler() { return m_bookmarkHandler; }
    /**
     * Returns a pointer to the file IO object.
     */
    Smb4KFileIO *fileIO() { return m_fileIO; }
    /**
     * Returns a pointer to the IP finder object.
     */
    Smb4KIPFinder *ipFinder() { return m_finder; }
    /**
     * Returns a pointer to the password reader object.
     */
    Smb4KPasswdReader *passwdReader() { return m_passwdReader; }
    /**
     * Returns a pointer to the printer handler object.
     */
    Smb4KPrint *print() { return m_print; }
    /**
     * Returns the pointer ti the shell IO object.
     */
    Smb4KShellIO *shellIO() { return m_shellIO; }
    /**
     * Aborts any action of the core.
     */
    void abort();
    /**
     * Opens the mount point of a share. This function is used to run the file manager.
     *
     * @param share The share to be opened.
     */
    void open( Smb4KShare *share );
    /**
     * This function returns the UMASK as integer. It is provided 
     * for convenience.
     */
    int getUMASK();
    /**
     * This function returns the Kernel version string. It is provided 
     * for convenience.
     */
    const QString getKernelVersion();
    /**
     * This function returns the system name. It is provided 
     * for convenience.
     */
    const QString getSystemName();
    /**
     * This function will return Samba's version string.
     */
    const QString getSambaVersion() { return m_samba_version; }
    /**
     * This function returns the global options of the smb.conf file. You 
     * can reach the value of each option by providing the lowercase option 
     * name as key, if it was defined in the global section of the smb.conf.
     */
    const QMap<QString, QString> getGlobalSambaOptions() { return m_fileIO->readSMBOptions(); }
    /**
     * This function returns the WINS server the system is using.
     */
    const QString getWINSServer();
    /**
     * Sets the share for which the usage should be returned. You may also pass
     * the NULL pointer. In this case, the usage() signal will return empty values.
     * @param share The share object.
     */
    void setShareForUsageInfo( Smb4KShare *share );
    
  signals:
    /**
     * This signal is emitted, if one of the core objects
     * starts or stops running.
     */
    void runStateChanged();
    /**
     * This signal emits the usage info for a share.
     *
     * @param mountpoint    The mountpoint/path that usage should be determined.
     *
     * @param total         The total space that's available on the filesystem (in kB).
     *
     * @param used          The space that's used (in kB).
     *
     * @param free          The space that's free (in kB).
     *
     * @param percent       The percentage the used space eats up (in kB).
     * 
     * @param dirs          The number of directories present in the mount point. The items
     *                      '.' and '..' are ignored.
     * 
     * @param files         The number of files present in the mount point.
     */
    void usage( const QString &mountpoint, double total, double used, double free, double percent, int dirs, int files );
    
  protected slots:
    /**
     * This slot is called, if the scanner is running. 
     */
    void slotScannerIsRunning( int state, bool on );
    /**
     * This slot is called, if the mounter is running.
     */
    void slotMounterIsRunning( int state, bool on );
    /**
     * This slot is called, if the mounter is running.
     */
    void slotIPFinderIsRunning( int state, bool on );
    /**
     * This slot is called, if the printer handler is running.
     */
    void slotPrinterHandlerIsRunning( int state, bool on );
    /**
     * This slot is called if an error in the core classes
     * occurred.
     */
    void slotShowErrorMessage( int error_code, const QString &error_message = QString::null );
    /**
     * This slot is called by the KApplication::shutDown() signal. Here all actions are
     * taken to exit the core savely.
     */
    void slotShutdown();
    /**
     * This slot receives the sambaVersion() signal by Smb4KShellIO and writes 
     * Samba's version to the config file.
     * @param version Samba's version string.
     */
    void slotReceiveSambaVersion( const QString &version );
    /**
     * This slot will invoke Smb4KShellIO::getSambaVersion(). It is
     * reimplemented for convenience. Connect to Smb4KCore::sambaVersion() to
     * get the result.
     */
    void slotGetSambaVersion();
    /**
     * This slot is connected to the timer. It is used to determine the usage
     * of a share and emits this data via the usage() signal.
     */
    void slotGetUsage();
      
  private:
    /**
     * The scanner object.
     */
    Smb4KScanner *m_scanner;
    /**
     * The mounter object.
     */
    Smb4KMounter *m_mounter;
    /**
     * The bookmark handler object.
     */
    Smb4KBookmarkHandler *m_bookmarkHandler;
    /**
     * The FileIO object.
     */
    Smb4KFileIO *m_fileIO;
    /**
     * The IP finder object.
     */
    Smb4KIPFinder *m_finder;
    /**
     * The password reader object.
     */
    Smb4KPasswdReader *m_passwdReader;
    /**
     * The printer handler object.
     */
    Smb4KPrint *m_print;
    /**
     * The shell IO object.
     */
    Smb4KShellIO *m_shellIO;
    /**
     * The state the sanner is in.
     */
    int m_scanner_state;
    /**
     * Is TRUE when the scanner is running, otherwise it is FALSE.
     */
    bool m_scanner_running;
    /**
     * The state the mounter is in.
     */
    int m_mounter_state;
    /**
     * Is TRUE when the mounter is running, otherwise it is FALSE.
     */
    bool m_mounter_running;
    /**
     * The state the IP finder is in.
     */
    int m_ipfinder_state;
    /**
     * Is TRUE when the IP finder is running, otherwise it is FALSE.
     */
    bool m_ipfinder_running;
    /**
     * Holds the current state.
     */
    int m_current_state;
    /**
     * The state the printer handler is in.
     */
    int m_print_state;
    /**
     * Is TRUE when the printer handler is running, otherwise it is FALSE.
     */
    bool m_print_running;
    /**
     * The advanced options for smbclient.
     */
    QString m_advancedSMBOptions;
    /**
     * The KConfig object.
     */
    KConfig *m_config;
    /**
     * Stores Samba's version string.
     */
    QString m_samba_version;
    /**
     * This function is used to set the current state of the core.
     * @param state One of the states defined in smb4kdefs.h
     */
    void setCurrentState( int state );
    /**
     * Searches for the needed programs and emits an error
     * if mandatory ones are missing.
     */
    void searchPrograms();
    /**
     * The path for which the disk usage should be determined.
     */
    QString m_path;
    /**
     * The timer for retrieving the usage periodically.
     */
    QTimer *m_timer;
};
 
#endif
