/*
 * image2raw.cc -- Converts images to Raw DV
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <iostream>
#include <string>
using std::cerr;
using std::endl;
using std::string;

#include <libdv/dv.h>
#include <unistd.h>
#include <extensions.h>
#include <Diagnostics.h>
#include <PixbufUtils.h>
#include <DVEncoder.h>
#include <PPMPump.h>

class Image2Raw : virtual public DVEncoder, virtual public PixbufUtils
{
	private:
		int repeat;
		uint8_t *image;
		Frame frame;
		bool output_ppm;
		bool output_ppma;
		int ppm_width;
		int ppm_height;

	public:
		bool aborted;
		int count;
	
		Image2Raw( ) : 
			repeat( 1 ), 
			image( NULL ), 
			output_ppm( false ), 
			output_ppma( false ), 
			ppm_width( 0 ),
			ppm_height( 0 ),
			aborted( false ), 
			count( 1 )
		{
			image = (uint8_t *)malloc( 720 * 576 * 3 );
		}

		~Image2Raw( )
		{
			delete image;
		}

		void SetRepeat( int _repeat )
		{
			repeat = _repeat;
		}

		void SetOutputPPM( bool _output_ppm )
		{
			output_ppm = _output_ppm;
		}

		void SetOutputPPMA( bool _output_ppm )
		{
			output_ppma = _output_ppm;
		}

		void SetPPMScaler( char *value )
		{
			sscanf( value, "%dx%d", &ppm_width, &ppm_height );
		}

		bool CreateFrame( string filename )
		{
			int width = GetWidth( );
			int height = GetHeight( );

			if ( ( output_ppm || output_ppma ) && ppm_width != 0 && ppm_height != 0 )
			{
				width = ppm_width;
				height = ppm_height;
			}

			bool ret = false;

			if ( output_ppma )
			{
				PPMFrame pframe;
				ret = pframe.Load( filename );
				if ( ret )
				{
					pframe.Scale( width, height );
					for ( int i = 0; !aborted && i < repeat; i ++ )
						aborted = !pframe.WriteImage( true );
				}
			}
			else
			{
				ret = ReadImageFile( filename, image, width, height );
				if ( ret )
				{
					count ++;
					if ( !output_ppm )
					{
						EncodeRGB( frame, image );
						for ( int i = 0; !aborted && i < repeat; i ++ )
						{
							EncodeAudio( frame );
							if ( fwrite( frame.data, frame.GetFrameSize( ), 1, stdout ) != 1 )
								aborted = true;
							fflush( stdout );
						}
					}
					else
					{
						for ( int i = 0; !aborted && i < repeat; i ++ )
						{
							fprintf( stdout, "P6\n%d %d\n255\n", width, height );
							if ( fwrite( image, width * height * 3, 1, stdout ) != 1 )
								aborted = true;
							fflush( stdout );
						}
					}
				}
			}

			return ret;
		}
};

static void Usage( )
{
	cerr << "Usage: image2raw [ [ options ] [ file ... ] ]" << endl;
	cerr << "Where: file is any image file (jpeg/png/gif et al)" << endl;
	cerr << "       -n        - NTSC usage (MUST come first)" << endl;
	cerr << "       -ppm      - output scaled ppms instead of Raw DV" << endl;
	cerr << "       -s WxH    - Scaling for PPM output (obeys -a, -f and -k)" << endl;
	cerr << "       -a        - Maintain aspect ration (default)" << endl;
	cerr << "       -f        - Scale to full size" << endl;
	cerr << "       -k        - Keep image size as is (crop/centred)" << endl;
	cerr << "       -w        - Wide screen output (default: off)" << endl;
	cerr << "       -r number - repeat frames n times (default: 1)" << endl;
	cerr << "       -b colour - background colour (default: 000000 [black])" << endl;
	cerr << "       -A wav    - wav file to dub the output with" << endl;
	cerr << "       -p number - encoder passes (default: 3)" << endl;
	cerr << "       -q qno    - encoder qno (default: 0)" << endl;
	cerr << "       -2        - 2 pass encoding (default: off)" << endl;
	exit( 0 );
}

int main( int argc, char **argv )
{
	Image2Raw raw;
	bool loop = true;

	Diagnostics::SetApp( "image2raw" );

	if ( isatty( fileno( stdout ) ) )
	{
		cerr << "Output be must redirected to a pipe or a file." << endl;
		Usage( );
	}

	while( loop && ! raw.aborted )
	{
		loop = false;
		
		for ( int index = 1; index < argc; index ++ )
		{
			if ( !strcmp( argv[ index ], "--help" ) )
				Usage( );
			else if ( !strcmp( argv[ index ], "-v" ) )
				Diagnostics::SetLevel( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-A" ) )
				raw.SetAudioFile( string( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-n" ) )
				raw.SetPAL( false );
			else if ( !strcmp( argv[ index ], "-a" ) )
				raw.SetScale( SCALE_ASPECT_RATIO );
			else if ( !strcmp( argv[ index ], "-f" ) )
				raw.SetScale( SCALE_FULL );
			else if ( !strcmp( argv[ index ], "-k" ) )
				raw.SetScale( SCALE_NONE );
			else if ( !strcmp( argv[ index ], "-w" ) )
				raw.SetWide( true );
			else if ( !strcmp( argv[ index ], "-p" ) )
				raw.SetEncodePasses( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-q" ) )
				raw.SetStaticQno( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-r" ) )
				raw.SetRepeat( atoi( argv[ ++ index ] ) );
			else if ( !strcmp( argv[ index ], "-b" ) )
				raw.SetBackgroundColour( argv[ ++ index ] );
			else if ( !strcmp( argv[ index ], "-2" ) )
				raw.SetTwoPassEncoding( true );
			else if ( !strcmp( argv[ index ], "-l" ) )
				loop = true;
			else if ( !strcmp( argv[ index ], "-ppma" ) )
				raw.SetOutputPPMA( true );
			else if ( !strcmp( argv[ index ], "-ppm" ) )
				raw.SetOutputPPM( true );
			else if ( !strcmp( argv[ index ], "-s" ) )
				raw.SetPPMScaler( argv[ ++ index ] );
			else if ( !raw.CreateFrame( string( argv[ index ] ) ) && !raw.aborted )
				cerr << "Unable to open " << argv[ index ] << " -- ignoring." << endl;
			else if ( raw.aborted )
				break;
		}
		
		if ( raw.count == 0 )
			loop = false;
		else
			raw.count = 0;
	}
	
	exit( raw.aborted );
}
