///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// $Id: FrequencyDBImpl_cache.cc,v 1.14 2004/01/17 01:13:17 bburton Exp $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//

#include <unistd.h>
#include <fcntl.h>
#include "WordData.h"
#include "FrequencyDBImpl_cache.h"

static const string DATABASE_TYPE("-cached");

FrequencyDBImpl_cache::FrequencyDBImpl_cache(FrequencyDBImpl *db)
  : m_db(db)
{
  assert(db);
}

FrequencyDBImpl_cache::~FrequencyDBImpl_cache()
{
  close();
}

bool FrequencyDBImpl_cache::open(const string &filename,
                                 bool read_only,
                                 int create_mode)
{
  close();
  return m_db->open(filename, read_only, create_mode);
}

void FrequencyDBImpl_cache::close()
{
  m_db->close();
}

void FrequencyDBImpl_cache::flush()
{
  m_db->beginTransaction();
  for (map_iter_t i = m_cache.begin(); i != m_cache.end(); ++i) {
    if (i->second.is_dirty) {
      m_db->writeWord(i->first, *i->second.counts);
    }
    delete i->second.counts;
  }
  m_db->endTransaction(true);
  m_cache.clear();
  m_db->flush();
}

void FrequencyDBImpl_cache::writeWord(const string &word,
                                      const WordData &counts)
{
  if (!m_db->canCacheTerm(word)) {
    if (is_debug) {
      cerr << "UNCACHED TERM " << word << endl;
    }
    m_db->writeWord(word, counts);
  } else {
    map_iter_t i = m_cache.find(word);
    if (i == m_cache.end()) {
      m_cache.insert(make_pair(word, CacheEntry(true, false, new WordData(counts))));
      if (is_debug) {
        cerr << "CACHED TERM INSERTED " << word << endl;
      }
    } else if ((*i->second.counts).equals(counts)) {
      if (is_debug) {
        cerr << "CACHED TERM UNCHANGED " << word << endl;
      }
    } else {
      if (i->second.is_shared && i->second.counts->hasSameCounts(counts)) {
	// Do nothing because we don't want terms migrating from shared to 
	// private database if only the timestamp changed.
        if (is_debug) {
          cerr << "CACHED TERM FROM SHARED " << word << endl;
        }
      } else {
	i->second.is_dirty = true;
	i->second.is_shared = false;
	*i->second.counts = counts;
        if (is_debug) {
          cerr << "CACHED TERM UPDATED " << word << endl;
        }
      }
    }
  }
}

bool FrequencyDBImpl_cache::readWord(const string &word,
                                     WordData &counts)
{
  map_const_iter_t i = m_cache.find(word);
  if (i != m_cache.end()) {
    counts = *i->second.counts;
    return true;
  }

  bool is_shared = false;
  if (m_db->readWord(word, counts, is_shared)) {
    if (m_db->canCacheTerm(word)) {
      m_cache.insert(make_pair(word, CacheEntry(false, is_shared, new WordData(counts))));
    }
    return true;
  }

  return false;
}

bool FrequencyDBImpl_cache::firstWord(string &word,
                                      WordData &counts)
{
  flush();
  return m_db->firstWord(word, counts);
}

bool FrequencyDBImpl_cache::nextWord(string &word,
                                     WordData &counts)
{
  return m_db->nextWord(word, counts);
}

string FrequencyDBImpl_cache::getDatabaseType() const
{
  return m_db->getDatabaseType() + DATABASE_TYPE;
}

void FrequencyDBImpl_cache::sweepOutOldTerms(int junk_count,
					     int max_age)
{
  flush();
  m_db->sweepOutOldTerms(junk_count, max_age);
}
