/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: DocLpc.pike,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $
 */

constant cvs_version="$Id: DocLpc.pike,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $";

/* this object really represents a factory if executed !
 */
inherit "/classes/Document";

import Attributes;

#include <classes.h>
#include <access.h>
#include <database.h>
#include <attributes.h>
#include <macros.h>
#include <types.h>
#include <classes.h>
#include <events.h>
#include <exception.h>

static mapping       mRegAttributes; // registered attributes for this factory
static array(object)    aoInstances; // Instances of this class

/**
 * Initialize the document.
 *  
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 */
static void
init_document()
{
    mRegAttributes = ([ ]);
    aoInstances     = ({ });
    add_data_storage(STORE_DOCLPC, retrieve_doclpc, restore_doclpc);
}

/**
 * Get the object class - CLASS_DOCLPC in this case.
 *  
 * @author Thomas Bopp (astra@upb.de) 
 */
int
get_object_class()
{
    return ::get_object_class() | CLASS_DOCLPC;
}

/**
 * Destructor of this object.
 *  
 */
static void
delete_object()
{
    aoInstances -= ({ 0 });
    
    foreach(aoInstances, object obj)
	if ( objectp(obj) )
	    obj->delete(); // delete all instances
    ::delete_object();
}

/**
 * Execute the DocLPC which functions as a factory class.
 * The parameters must include a name 'name' and might include
 * a 'moveto' variable to move the object.
 *  
 * @param mapping variables - execution parameters.
 * @return the newly created object.
 */
mixed execute(mapping variables)
{
    if ( objectp(_CODER) && sizeof(_CODER->get_members()) > 0 ) {
	// check if User code is allowed, creator needs to be coder
	// and no other user should have write access on this script
	object creator = get_creator();
	if ( !_CODER->is_member(creator) && !_ADMIN->is_member(creator) )
	    THROW("Unauthorized Script", E_ACCESS);
	mapping sanc = get_sanction();
	foreach(indices(sanc), object grp) {
	    if ( (sanc[grp] & SANCTION_WRITE ) && !_ADMIN->is_member(grp) &&
		 !_CODER->is_member(grp) && grp != _ADMIN && grp != _CODER )
		THROW("Write access for non coder group enabled - aborting !",
		      E_ACCESS);
	}
    }

    try_event(EVENT_EXECUTE, CALLER, 0);
    clean_instances();
    if ( stringp(variables->_action) && sizeof(aoInstances) > 0 ) {
      //just call the script
      object script = aoInstances[0];
      return script->execute(variables);
    }

    //object obj = new("DB:#"+get_object_id()+".pike", variables["name"]);
    
    object obj;
    master()->clear_compilation_failures();
    obj = ((program)("/DB:#"+get_object_id()+".pike"))(variables->name);

    install_attributes(obj->this());
    
    object mv = find_object((int)variables["moveto"]);
    if ( objectp(mv) )
	obj->move(mv);

    if ( !stringp(variables["name"]) )
	variables->name = "";
    obj->set_attribute(OBJ_NAME, variables["name"]);
    obj->set_attribute(OBJ_CREATION_TIME, time());
    obj->set_attribute(OBJ_SCRIPT, this());
    obj->set_acquire(obj->get_environment);
    obj->set_acquire_attribute(OBJ_ICON, _Server->get_module("icons"));
    obj->created();
    aoInstances += ({ obj->this() });
    aoInstances -= ({ 0 });
    require_save(STORE_DOCLPC);
    run_event(EVENT_EXECUTE, CALLER, obj);
    return obj;
}

object provide_instance()
{
    object o;
    array instances = aoInstances;
    if ( arrayp(instances) )
        clean_instances();
	//instances -= ({ 0 });
    o = get_instance();
    if ( objectp(o) )
	return o;

    object e = master()->ErrorContainer();
    master()->set_inhibit_compile_errors(e);
    mixed err = catch {
	o = execute((["name":"temp", ]));
	o->set_acquire(this());
    };
    master()->set_inhibit_compile_errors(0);
    if ( err != 0 ) {
	FATAL("While providing instance of %s\n%s, %s\n%O", 
	      get_identifier(), err[0], e->get(), err[1]);
	throw(err);
    }
    return o->this();
}

/**
 * Call this script - use first instance or create one if none.
 *  
 * @param mapping vars - normal variable mapping
 * @return execution result
 */
mixed call_script(mapping vars)
{
    object script = provide_instance();
    return script->execute(vars);
}

/**
 * register all attributes for an object
 *  
 * @param obj - the object to register attributes
 * @author Thomas Bopp (astra@upb.de) 
 * @see register_class_attribute
 */
private static void
install_attributes(object obj)
{
    object factory = _Server->get_factory(obj->get_object_class());
    if ( !objectp(factory) )
	factory = _Server->get_factory(CLASS_OBJECT);
    
    mapping mClassAttr = factory->get_attributes() + mRegAttributes;
    foreach ( indices(mClassAttr), mixed key ) 
	install_attribute(mClassAttr[key], obj);
}

bool install_attribute(Attribute attr, object obj)
{
    mixed err = catch {
	mixed key = attr->get_key();
	mixed def = attr->get_default_value();
	if ( !zero_type(def) )
	    obj->set_attribute(key, def);
	string|object acq = attr->get_acquire();
	if ( stringp(acq) )
	    obj->set_acquire_attribute(key, obj->find_function(acq));
	else 
	    obj->set_acquire_attribute(key, acq);
	return true;
    };
    FATAL("Error registering attribute: %O", err);
}


bool check_attribute(mixed key, mixed data) 
{
    Attribute a = mRegAttributes[key];
    if ( objectp(a) )
	return a->check_attribute(data);
}

/**
 * register attributes for the class(es) this factory creates.
 * each newly created object will have the attributes registered here.
 *  
 * @param Attribute attr - the new attribute to register.
 * @param void|function conversion - conversion function for all objects
 *
 * @see classes/Object.set_attribute 
 * @see libraries/Attributes.pmod.Attribute
 */
void 
register_attribute(Attribute attr, void|function conversion)
{
    try_event(EVENT_REGISTER_ATTRIBUTE, CALLER, attr);
    register_class_attribute(attr, conversion);

    // register on all dependent factories too
    array(object) factories = values(_Server->get_classes());
    foreach ( factories, object factory ) {
	factory = factory->get_object();
	if ( factory->get_object_id() == get_object_id() )
	    continue;
	if ( search(Program.all_inherits(object_program(factory)),
		    object_program(this_object())) >= 0 )
	    factory->register_attribute(copy_value(attr), conversion);
    }
    run_event(EVENT_REGISTER_ATTRIBUTE, CALLER, attr);
}


/**
 * Register_class_attribute is called by register_attribute,
 * this function is local and does no security checks. All instances
 * of this class are set to the default value and acquiring settings.
 *  
 * @param Attribute attr - the Attribute to register for this factories class
 * @param void|function conversion - conversion function.
 * @see register_attribute
 */
static void register_class_attribute(Attribute attr, void|function conversion)
{
    string|int key = attr->get_key();
    MESSAGE("register_class_attribute(%O)",key);
    Attribute pattr = mRegAttributes[key];
    if ( pattr == attr ) {
	MESSAGE("re-registering class attribute with same value.");
	return;
    }
    foreach(aoInstances, object inst)
	install_attribute(attr, inst);

    mRegAttributes[key] = attr;
    require_save(STORE_DOCLPC);
}

/**
 * get the registration information for one attribute of this class
 *  
 * @param mixed key - the attribute to describe.
 * @return array of registered attribute data.
 */
Attribute describe_attribute(mixed key)
{
    return copy_value(mRegAttributes[key]);
}

/**
 * Get the source code of the doclpc, used by master().
 *  
 * @return the content of the document.
 */
string get_source_code()
{
    return ::_get_content();
}

/**
 * Get the compiled program of this objects content.
 *  
 * @return the pike program.
 */
final program get_program() 
{ 
  program p = master()->lookup_program("/DB:#"+get_object_id()+".pike");
  return p;
}

/**
 * Get an Array of Error String description.
 *  
 * @return array list of errors from last upgrade.
 */
array(string) get_errors()
{
    return master()->get_error("/#"+get_object_id()+".pike");
}

/**
 * Upgrade this script and all instances.
 *  
 */
void upgrade()
{
  program p = get_program();
  mixed res = master()->upgrade(p);
  if ( stringp(res) )
    steam_error(res);
}

static void content_finished()
{
    ::content_finished();
    //    if ( !functionp(CALLER->get_object_id) )
    // return;
    mixed err = catch(upgrade());
    if ( err != 0 ) 
	werror("Error upgrading program...."+err[0]+"\n");
}

/**
 * Retrieve the DocLPC data for storage in the database.
 *  
 * @return the saved data mapping.
 * @author <a href="mailto:astra@upb.de">Thomas Bopp</a>) 
 */
final mapping
retrieve_doclpc()
{
    if ( CALLER != _Database )
	THROW("Invalid call to retrieve_data()", E_ACCESS);
    
    return ([ 
	"RegAttributes":map(mRegAttributes, save_attribute),
	"Instances": aoInstances,
	]);
}

static mapping save_attribute(Attribute attr)
{
    return attr->save();
}

/**
 * Restore the data of the LPC document.
 *  
 * @param mixed data - the saved data.
 */
final void 
restore_doclpc(mixed data)
{
    if ( CALLER != _Database )
	THROW("Invalid call to restore_data()", E_ACCESS);

    aoInstances = data["Instances"];
    if ( !arrayp(aoInstances) )
	aoInstances = ({ });
    foreach(indices(data->RegAttributes), mixed key) { 
	mixed v = data->RegAttributes[key];
	if ( arrayp(v) ) {
	    mixed acq = v[4];
	    if ( intp(acq) && acq == 1 )
		acq = REG_ACQ_ENVIRONMENT;
	    Attribute a = Attribute(key, v[1],v[0],v[6],acq,v[5],v[2],v[3]);
	    mRegAttributes[key] = a;
	}
	else {
	    Attribute a = Attribute(v->key,v->desc,v->type,v->def,v->acquire,
				    v->control, v->event_read, v->event_write);
	    mRegAttributes[key] = a;
	}
    }
}

/**
 * Get the existing instances of this pike program.
 *  
 * @return array of existing objects.
 */
array(object) get_instances()
{
  array instances = ({ });

  for ( int i = 0; i < sizeof(aoInstances); i++ ) {
    if ( objectp(aoInstances[i]) && 
	 aoInstances[i]->status() != PSTAT_FAIL_DELETED && 
	 aoInstances[i]->status() != PSTAT_FAIL_COMPILE)
      instances += ({ aoInstances[i] });
  }
  return instances;
}

void clean_instances()
{
    aoInstances-= ({ 0 });
    
    foreach(aoInstances, object instance) {
      if( objectp(instance) && instance->status() == PSTAT_FAIL_COMPILE)
      {
	  aoInstances-= ({ instance });
	  instance->delete();
      }
    }
}

object get_instance() 
{
    clean_instances();
    foreach ( aoInstances, object instance ) {
	if ( objectp(instance) && instance->status() != PSTAT_FAIL_DELETED 
	     && instance->status() != PSTAT_FAIL_COMPILE)
	    return instance;
    }
    return 0;
}

string describe()
{
    return sprintf("%s+(#%d,%s,%d,%s,%d Instances, ({ %{%O,%} }))", 
                   get_identifier(),
		   get_object_id(),
	           master()->describe_program(object_program(this_object())),
	           get_object_class(),
		   do_query_attribute(DOC_MIME_TYPE),
	           sizeof(aoInstances),
		   aoInstances);
//    return get_identifier()+"(#"+get_object_id()+","+
//	master()->describe_program(object_program(this_object()))+","+
//	get_object_class()+","+do_query_attribute(DOC_MIME_TYPE)+","+
//	sizeof(aoInstances) +" Instances)";
}
