# -*- Mode: ruby; indent-tabs-mode: nil -*-
#
#  $Id: tc_template.rb,v 1.10 2003/09/23 01:20:41 hisa Exp $
#
#  Copyright (c) 2003 FUJIMOTO Hisakuni <hisa@fobj.com>
#

require 'test/unit'
require 'tempura/template'
require 'tempura/charconv_euc'
require 'model'

class TC_Template < Test::Unit::TestCase

  def setup
    @path_en = "view.en.html"
    @path_ja = "view.ja.html"
    @src_en = File.open(@path_en) { |f| f.read }
    @src_ja = File.open(@path_ja) { |f| f.read }
    @model_en = Model.new( [[ 'ebiten', 32 ], [ 'nasu', 12 ], [ 'kakiage', 8 ]] )
    @model_ja = Model.new( [[ 'Ϸŷ', 32 ], [ 'ػ', 12 ], [ 'Ȥ', 8 ]] )
  end

  def test_s_new
    tmpl = nil
    assert_nothing_raised { tmpl = Tempura::Template.new }
    assert_kind_of( Tempura::Template, tmpl )
    assert_nothing_raised { tmpl = Tempura::Template.new(@path_en) }
    assert_kind_of( Tempura::Template, tmpl )
    assert_nothing_raised { tmpl = Tempura::Template.new(@path_ja, Tempura::CharConvEUC) }
    assert_kind_of( Tempura::Template, tmpl )
  end

  def test_s_new_with_string
    assert_raises( ArgumentError) { Tempura::Template.new_with_string }
    tmpl = nil
    assert_nothing_raised { tmpl = Tempura::Template.new_with_string(@src_en) }
    assert_kind_of( Tempura::Template, tmpl )
    assert_nothing_raised { tmpl = Tempura::Template.new_with_string(@src_ja, Tempura::CharConvEUC) }
    assert_kind_of( Tempura::Template, tmpl )
  end

  def test_expander
    tmpl = Tempura::Template.new
    ret = nil
    assert_nothing_raised { ret = tmpl.expander }
    assert_kind_of( Tempura::Expander, ret )
  end

  def test_sources
    tmpl = Tempura::Template.new
    ret = nil
    assert_nothing_raised { ret = tmpl.sources }
    assert_kind_of( Tempura::SourceContainer, ret )
  end

  def test_default_action
    tmpl = Tempura::Template.new
    assert_nothing_raised { tmpl.default_action }
    assert_equal( "", tmpl.default_action )
  end

  def test_set_default_action
    tmpl = Tempura::Template.new
    assert_nothing_raised { tmpl.default_action = "myapp.cgi" }
    assert_equal( "myapp.cgi", tmpl.default_action )
    tmpl.set_with_string '<a href="" _event_="show//msg==>%(hello)">hello</a>'
    doc = tmpl.expand_to_document( nil )
    href = doc.root.attributes['href']
    assert_match( /^myapp.cgi?/, href )
    assert_match( /\?.*\bevent=show\b/, href )
    assert_match( /\?.*\bmsg=hello\b/, href )
  end

  def test_default_event_key
    tmpl = Tempura::Template.new
    assert_nothing_raised { tmpl.default_event_key }
    assert_equal( "event", tmpl.default_event_key )
  end

  def test_set_default_event_key
    tmpl = Tempura::Template.new
    assert_nothing_raised { tmpl.default_event_key = "command" }
    assert_equal( "command", tmpl.default_event_key )
    tmpl.set_with_string '<a href="" _event_="show//msg==>%(hello)">hello</a>'
    doc = tmpl.expand_to_document( nil )
    href = doc.root.attributes['href']
    assert_match( /\?.*\bcommand=show\b/, href )
    assert_match( /\?.*\bmsg=hello\b/, href )
  end

  def test_set
    tmpl = Tempura::Template.new(@path_en)
    assert_nothing_raised { tmpl.set( @path_en ) }
    assert_nothing_raised { tmpl.set( @path_ja, :ja ) }
    assert_nothing_raised { tmpl.set( @path_ja, "ja" ) }
    assert_nothing_raised { tmpl.set( @path_ja, "ja", Tempura::CharConvEUC ) }
  end

  def test_set_with_string
    tmpl = Tempura::Template.new(@path_en)
    assert_nothing_raised { tmpl.set_with_string( @src_en ) }
    assert_nothing_raised { tmpl.set_with_string( @src_ja, :ja ) }
    assert_nothing_raised { tmpl.set_with_string( @src_ja, "ja" ) }
    assert_nothing_raised { tmpl.set_with_string( @src_ja, "ja", Tempura::CharConvEUC ) }
  end

  def test_expand
    tmpl = Tempura::Template.new
    assert_raises(Tempura::Template::SourceUnspecifiedError) { tmpl.expand(@model_en) }

    tmpl = Tempura::Template.new_with_string("<en _child_='items.map{|i|i.name}.join'/>")
    tmpl.set_with_string("<ja _child_='items.map{|i|i.name}.join'/>", :ja)
    ret = nil
    assert_nothing_raised { ret = tmpl.expand(@model_en) }
    assert_equal( "<en>#{@model_en.items.map{|i|i.name}.join}</en>", ret )
    ret = nil
    assert_nothing_raised { ret = tmpl.expand(@model_ja, :ja) }
    assert_equal( "<ja>#{@model_ja.items.map{|i|i.name}.join}</ja>", ret )
  end

  def test_expand_to_document
    tmpl = Tempura::Template.new
    assert_raises(Tempura::Template::SourceUnspecifiedError) { tmpl.expand_to_document(@model_en) }

    tmpl = Tempura::Template.new_with_string("<en _child_='items.map{|i|i.name}.join'/>")
    tmpl.set_with_string("<ja _child_='items.map{|i|i.name}.join'/>", :ja)
    ret = nil
    assert_nothing_raised { ret = tmpl.expand_to_document(@model_en) }
    assert_kind_of( REXML::Document, ret )
    assert_equal( "<en>#{@model_en.items.map{|i|i.name}.join}</en>", ret.to_s )
    ret = nil
    assert_nothing_raised { ret = tmpl.expand_to_document(@model_ja, :ja) }
    assert_kind_of( REXML::Document, ret )
    assert_equal( "<ja>#{@model_ja.items.map{|i|i.name}.join}</ja>", Tempura::CharConvEUC.from_u8(ret.to_s) )
  end

end
