/* -*- mode: c; c-file-style: "gnu" -*-
 * misc.c -- Miscellaneous routines
 * Copyright (C) 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy-Auth.
 *
 * Thy-Auth is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Thy-Auth is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file misc.c
 * Miscellaneous routines.
 *
 * This module contains various unsorted helper functions.
 */

#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include "compat/compat.h"
#include "misc.h"

/** Read command arguments.
 * Read the arguments passed to a command, and store them in a char *
 * array, which must be freed by the caller.
 *
 * @param argc is the number of arguments to read.
 *
 * @returns The arguments stuffed into an array.
 * @note The returned array must be freed by the caller!
 */
char **
thy_auth_args_read (int argc)
{
  char **args;
  int i;
  size_t l;

  args = (char **)bhc_calloc (argc + 2, sizeof (char *));
  for (i = 0; i < argc; i++)
    {
      if (getline (&args[i], &l, stdin) == -1)
	exit (1);
      l = strlen (args[i]);
      args[i][l - 1] = '\0';
    }
  args[i] = NULL;

  return args;
}

/** @internal Respond to a VERSION request.
 * Respond to a VERSION request, by returning one if we talk the same
 * protocol version, 0 otherwise.
 *
 * @note Takes on PROTOCOL_VERSION argument.
 */
static void
_thy_auth_do_one_version (int argc, char **argv)
{
  if (argc < 1)
    thy_auth_respond ("0\n", 2);
  else
    thy_auth_respond ((atoi (argv[0]) == _THY_AUTH_PROTOCOL_VERSION) ?
		      "1\n" : "0\n", 2);
}

/** Handle one command.
 * This is the core of any thy-auth service: receives the commands and
 * their arguments, then dispatches control to the appropriate
 * function, some of which is common to all (like VERSION and QUIT),
 * some need to be implemented by the caller (AUTH and CHECK).
 */
void
thy_auth_do_one (void)
{
  char *cmd = NULL, **args, *argc_s = NULL;
  size_t l;
  int i = 0, argc;

  if (getline (&cmd, &l, stdin) == -1)
    exit (1);
  if (getline (&argc_s, &l, stdin) == -1)
    exit (1);

  argc = atoi (argc_s);
  args = thy_auth_args_read (argc);

  if (!strcmp (cmd, "AUTH\n"))
    _thy_auth_do_one_auth (argc, args);
  else if (!strcmp (cmd, "VERSION\n"))
    _thy_auth_do_one_version (argc, args);
  else if (!strcmp (cmd, "CHECK\n"))
    _thy_auth_do_one_check (argc, args);
  else if (!strcmp (cmd, "QUIT\n"))
    exit (0);
  else
    thy_auth_respond ("0\n", 2);

  while (args[i] != NULL)
    {
      free (args[i]);
      i++;
    }
  free (args);
  free (argc_s);
  free (cmd);
}

/** Print a response.
 * Tries to send the response back to our client. If it does not
 * succeed, Thy-auth will exit immediately.
 *
 * @param msg is the message.
 * @param length is the message length.
 */
void
thy_auth_respond (const char *msg, size_t length)
{
  if (write (STDOUT_FILENO, msg, length) != (ssize_t)length)
    exit (1);
}
