// kl10_cca.c - CCA (Cache System)
//
// Written by
//  Timothy Stark <sword7@speakeasy.org>
//
// This file is part of the TS10 Emulator.
// See ReadMe for copyright notice.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// Device Name: CCA
// Device Code: 014

#include "pdp10/defs.h"
#include "pdp10/kl10.h"
#include "pdp10/dev/iodefs.h"
#include "pdp10/proto.h"

#define NOPXCT 0

extern int KL10apr_srFlags;

// 70010 WRFIL (BLKO APR,)
// Write Refill Table
void KL10_ioOpcode_WRFIL(void *uptr)
{
}

// 70144 SWPIA (DATAI CCA,)
// Sweep Cache, Invalidate All Pages
void KL10_ioOpcode_SWPIA(void *uptr)
{
	// Set sweep busy bit in APR register.
	KL10apr_srFlags |= APRSR_SWPBUSY;

	// Clear all cache entries.

	// Clear sweep busy bit and set sweep done bit in APR register.
	// Also, request an interrupt.
	KL10apr_srFlags = (KL10apr_srFlags & ~APRSR_SWPBUSY) | APRSR_SWPDONE;
}

// 70150 SWPVA (BLKO CCA,)
// Sweep Cache, Validate All Pages
void KL10_ioOpcode_SWPVA(void *uptr)
{
	// Set sweep busy bit in APR register.
	KL10apr_srFlags |= APRSR_SWPBUSY;

	// Write into storage all cached words whose written bits are set.
	// Clear all written but but do not change the validity of any
	// entries.

	// Clear sweep busy bit and set sweep done bit in APR register.
	// Also, request an interrupt.
	KL10apr_srFlags = (KL10apr_srFlags & ~APRSR_SWPBUSY) | APRSR_SWPDONE;
}

// 70154 SWPUA (DATAO CCA,)
// Sweep Cache, Unload All Pages
void KL10_ioOpcode_SWPUA(void *uptr)
{
	// Set sweep busy bit in APR register.
	KL10apr_srFlags |= APRSR_SWPBUSY;

	// Write into storage all cached words whose written bits are set.
	// Invalidate the entire cache (clear all valid and written bits).

	// Clear sweep busy bit and set sweep done bit in APR register.
	// Also, request an interrupt.
	KL10apr_srFlags = (KL10apr_srFlags & ~APRSR_SWPBUSY) | APRSR_SWPDONE;
}

// 70164 SWPIO (CONI CCA,)
// Sweep Cache, Invalidate One Page
void KL10_ioOpcode_SWPIO(void *uptr)
{
	// Set sweep busy bit in APR register.
	KL10apr_srFlags |= APRSR_SWPBUSY;

	// Clear all cache entries.

	// Clear sweep busy bit and set sweep done bit in APR register.
	// Also, request an interrupt.
	KL10apr_srFlags = (KL10apr_srFlags & ~APRSR_SWPBUSY) | APRSR_SWPDONE;
}

// 70170 SWPVO (CONSZ CCA,)
// Sweep Cache, Validate One Page
void KL10_ioOpcode_SWPVO(void *uptr)
{
	// Set sweep busy bit in APR register.
	KL10apr_srFlags |= APRSR_SWPBUSY;

	// Write into storage all cached words whose written bits are set.
	// Clear all written but but do not change the validity of any
	// entries.

	// Clear sweep busy bit and set sweep done bit in APR register.
	// Also, request an interrupt.
	KL10apr_srFlags = (KL10apr_srFlags & ~APRSR_SWPBUSY) | APRSR_SWPDONE;
}

// 70174 SWPUO (CONSO CCA,)
// Sweep Cache, Unload One Page
void KL10_ioOpcode_SWPUO(void *uptr)
{
	// Set sweep busy bit in APR register.
	KL10apr_srFlags |= APRSR_SWPBUSY;

	// Write into storage all cached words whose written bits are set.
	// Invalidate the entire cache (clear all valid and written bits).

	// Clear sweep busy bit and set sweep done bit in APR register.
	// Also, request an interrupt.
	KL10apr_srFlags = (KL10apr_srFlags & ~APRSR_SWPBUSY) | APRSR_SWPDONE;
}

void kl10_InitCCA(KL10_DEVICE *kl10)
{
	P10_IOMAP *io;

	if (io = (P10_IOMAP *)calloc(1, sizeof(P10_IOMAP))) {
		io->devName    = "CCA";
		io->keyName    = "CCA";
		io->emuName    = "KL10: Cache";
		io->emuVersion = "(Internal)";
		io->idDevice   = KL10_CCA;
		io->ResetIO    = NULL;

		// Set APR instructions for KL10 processor
		io->Function[IOF_BLKI]  = NULL;
		io->Function[IOF_DATAI] = KL10_ioOpcode_SWPIA;
		io->Function[IOF_BLKO]  = KL10_ioOpcode_SWPVA;
		io->Function[IOF_DATAO] = KL10_ioOpcode_SWPUA;
		io->Function[IOF_CONO]  = NULL;
		io->Function[IOF_CONI]  = KL10_ioOpcode_SWPIO;
		io->Function[IOF_CONSZ] = KL10_ioOpcode_SWPVO;
		io->Function[IOF_CONSO] = KL10_ioOpcode_SWPUO;
		
		// Assign APR device to I/O mapping
		kx10_SetMap(io);
	}
}
