/* ==================================================== ======== ======= *
 *
 *  unatdisp.hpp  [Native Layer: platform dependent implementation]
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _unatdisp_hpp_
#define	_unatdisp_hpp_
//pragma ident	"@(#)unatdisp.hpp	ubit:03.06.04"
#include <ubit/config.h>
#include <ubit/unatwin.hpp>
#include <ubit/uappli.hpp>


class UNatDisp {
public:
  UNatDisp(UDisp&);
  virtual ~UNatDisp();

  int getStatus() const;
  /**< returns connection status; 
   * > 0 if connected to an X server
   * == OpenGL if connected in OpenGL mode 
   */

  UDisp& getDisp() const {return disp;}
  ///< returns the corresponding abstract Display counterpart.

  int getID() const  {return id;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // X Window

  struct {
    UX_Atom PRIMARY_SELECTION, SECONDARY_SELECTION, 
      WM_PROTOCOLS, WM_DELETE_WINDOW, WM_TAKE_FOCUS,
      UBIT_SELECTION, UBIT_MESSAGE, UBIT_WINDOW;
  } atoms;

  void realize();
  ///< creates X data.

  bool isRealized() const;
  ///< is X data created ?.

  UX_Display    getXDisplay()    const {return xdisplay;}
  UX_Screen     getXScreen()     const {return xscreen;}
  UX_Visual     getXVisual()     const {return xvisual;}
  UX_Colormap   getXColormap()   const {return xcmap;}
  UX_Window     getXRootWindow() const;
  UX_Window     getXWindow()     const {return xwin;}

  int getScreenWidth()  const;
  ///< width of the Screen.

  int getScreenHeight() const;
  ///< height of the Screen.

  int getDepth() const {return depth;}
  ///< depth of this UNatDisp.

  int getScreenDefaultDepth() const;
  ///< default depth of the Screen.

  int getScreenNumber() const;
  ///< number of the Screen of this UNatDisp.

  int getScreenCount() const;
  ///< total number of Screens available on this X Server.

  unsigned long getRedMask()   const {return red_mask;}
  unsigned long getGreenMask() const {return green_mask;}
  unsigned long getBlueMask()  const {return blue_mask;}
  int getRedShift()            const {return red_shift;}
  int getGreenShift()          const {return green_shift;}
  int getBlueShift()           const {return blue_shift;}
  int getRedBits()             const {return red_bits;}
  int getGreenBits()           const {return green_bits;}
  int getBlueBits()            const {return blue_bits;}
  ///< attention: les masks varient suivant l'architecture (big endian ou little endian).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // OpenGL

#ifdef WITH_GL
  /// initializes GL
  bool initGL();

  /// returns the GL Visual Info
  UX_VisualInfo getGLVisual() const {return glvisual;}

  void resizeGLViewport(u_dim w, u_dim h);
  
  /// sets a GL color
  void setGLColor(UX_Color pixel);
  void setGLColor(UX_Color pixel, GLubyte alpha);
  void setGLColor(GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha);

  /// converts an X Pixel to a GL color
  void toGLColor(UX_Color, GLubyte& red, GLubyte& green, GLubyte& blue);

  /// same as getFont but returns a GL display list id
  GLuint getGLFont(const class UFontDesc&); 
#endif

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Colors

  UX_Color getColor(const UColor& c) {return getColor(c.getImpl());}
  UX_Color getColor(class UColorImpl&);
  /**< returns the native color corresponding to this UColor.
   * Note: the color is implicitely realized if necessary */

  bool realizeColor(const UColor& c) {return realizeColor(c.getImpl());}
  bool realizeColor(class UColorImpl&);
  /**< allocates physical resources for this UColor on this Display.
   * Returns true if the color could be allocated.
   * returns false and sets a default color otherwise */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Mouse Cursors

  UX_Cursor getCursor(const UCursor&);
  /**< returns the native cursor corresponding to this UCursor.
   * Note: the cursor is implicitely realized if necessary */

  bool realizeCursor(const UCursor&);
  /**< Allocates physical resources for this UCursor on this UDisp.
   * returns true if the cursor was found and false otherwise */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Fonts

  UX_Font getFont(const class UFontDesc&);
  /**< returns the native font corresponding to this UFont.
   * Note: the font is implicitely realized if necessary */
  
  bool realizeFont(const class UFont&);
  bool realizeFont(const class UFontDesc&);
  bool realizeFontFamily(const class UFontFamily&);
  UX_Font loadNatFont(const class UFontFamily&, int styles, int lsize);
  /**< Allocates physical resources for this UFont/UFontDesc on this UDisp.
   * returns true if the font was found and false otherwise */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Visuals

  
  //bool setScreen(int screen_no);
  /**<
   * changes screen according to number (default is screen 0).
   * returns false if screen number is invalid.
   * Note: must be called before realize() or eventLoop() */

  int setVisual(const XVisualInfo&, UX_Colormap = None);
  /**< sets the visual of this UDisp according to the VisualInfo argument.
   * Notes:
   * - the Colormap is automatically created if this argument is 'None'
   * - this function must be called before the X windows are realized
   */

  int setVisual(int visual_class, int depth_hint, bool linear_gamma,
		UX_Colormap = None);
  /**<
   * sets the visual of this UDisp according to the 'visual_class'
   * and the 'depth_hint' args.
   * Arguments:
   * - 'visual_class' is the X visual class (typically: TrueColor
   *   or PseudoColor)
   * - the 'depth' argument is a hint: the visual with the closest depth 
   *   available is used
   * - tries to find a gamma-corrected visual if 'linear_gamma' is true
   *   (this arg is a hint, it only applies on Solaris)
   * - the Colormap is automatically created if this argument is 'None'
   * - returns the depth of the visual that was found or 0 if no visual
   *   with the specified class could be found on this screen.
   * Note:
   * - this function must be called before the X windows are realized
   */

  int chooseVisual(XVisualInfo&, int visual_class, int depth_hint, 
		   bool linear_gamma);
  /**<
   * searches a visual on this UDisp.
   * Arguments:
   * - 'XVisualInfo' is initialized if the value returned by tis fct. is > 0
   * - 'visual_class' is the X visual class (typically: TrueColor
   *   or PseudoColor)
   * - the 'depth' argument is a hint: the visual with the closest depth 
   *   available is returned
   * - tries to find a gamma-corrected visual if 'linear_gamma' is true
   *   (this arg is a hint, it only applies on Solaris)
   * - returns the depth of the visual that was found or 0 if no visual
   *   with the specified class could be found on this screen.
   */

  bool findExactVisual(XVisualInfo&, int visual_class, int depth, 
		       bool linear_gamma);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // X selection

  bool askSelectionOwnership(class UEvent&);
  /**< [impl] the display becomes the owner of the selection of the X Server.
   * the UEvent must be a button event
   */

  void changeSelectionContentRequest(UX_Event, class UTextsel&);
  // Cas ou une autre application demande a obtenir la valeur de la 
  // selection courante (detenue par 'this' appli)
  // l'event doit etre un XSelectionRequestEvent

  void askSelectionContent(class UEvent&, 
			   class UStr* put_to_this_str, int put_to_this_pos);
  /**< [impl] asks the selection of the X Server
   * the UEvent must be a button event. 
   * The server will send a SelectionNotify event to the UNatDisp that will
   * put the result in this UStr at this position
   */

  void retrieveSelectionContentRequest(UX_Event);
  // the XEvent must be a XSelectionEvent event. 
  // recupere le contenu de la selection lorsque le NatDisp recoit un
  // SelectionNotify event apres avoir demande le contenu de la selection 
  // en appellant askSelectionContent()
  // the XEvent must be a XSelectionEvent event. 

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Internal data

  class UNatGraph* getSharedGraph() const {return sharedGraph;}
  ///< used by UWinGraph objects for standard toolkit drawing

  class UNatGraph* getClientGraph() const {return clientGraph;}
  ///< used by UGraph objects for client drawing

  UX_Pixmap getDefaultPixmap() const {return defaultPixmap;}
  ///< default Pixmap with same depth as UNatDisp (for XCreateWindow).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // Event management

  void dispatchEvents(UX_Event);
  ///< dispatches X Events to Windows (function called by the event loop).

  void showNotify(UX_Window, bool shows);
  ///< notifies the UMS [Ubit Mouse Server] when a window is opened/closed.

  class UFlow* retrieveFlow(unsigned int ev_state, unsigned long flow_id);
  ///< returns an existing flow or create a new one (if flow_id not used).

  void on_selection(UWin*, UView*, UX_Event);
  void on_raw_event(UWin*, UView*, UX_Event);
  void on_expose(UWin*, UView*, UX_Event);
  void on_configure(UWin*, UView*, UX_Event);
  void on_mpress(UWin*, UView*, UX_Event);
  void on_mrelease(UWin*, UView*, UX_Event);
  void on_mmove(UWin*, UView*, UX_Event);
  void on_kpress(UWin*, UView*, UX_Event);
  void on_krelease(UWin*, UView*, UX_Event);
  void on_enter(UWin*, UView*, UX_Event);
  void on_leave(UWin*, UView*, UX_Event);
  void on_focus(UWin*, UView*, UX_Event);
  void on_notify(UWin*, UView*, UX_Event);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

private:
  friend class UNatAppli;
  friend class UWinGraph;
  friend class UGraph;
  friend bool UNatWin::realizeMainFrame(UNatDisp*, UNatWin*, UWin *win);

  class UDisp &disp;   // abstract display counterpart
  const int   id;
  int stat;
  UX_Display  xdisplay;
  UX_Screen   xscreen;
  UX_Visual   xvisual;
  UX_Colormap xcmap;
  UX_Window   xwin;    // used for getting depth, etc for drawing pixmaps...
  int         xconnection;
  int         depth;
  unsigned long red_mask, green_mask, blue_mask;
  int red_shift, green_shift, blue_shift;
  int red_bits, green_bits, blue_bits;

  UX_Font** fontFamilyMap;
  unsigned long fontFamilyCount;

  //default Pixmap with same depth as UNatDisp
  UX_Pixmap defaultPixmap;

#ifdef WITH_GL
  GLXContext glxc;
  UX_VisualInfo glvisual;      // GL visual info
  GLuint** glFontFamilyMap;   // array of display list ids
#endif
  
  // used by UWinGraph objects for standard toolkit drawing
  UNatGraph* sharedGraph;
  // used by UGraph objects for appli. drawing
  UNatGraph* clientGraph;

  // the UStr where the Server selection must be inserted
  class UStr *server_selection_str;
  int server_selection_pos;
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
