/* ==================================================== ======== ======= *
 *
 *  umsflow.cpp : event flow of the UMS. 
 *
 *  Ubit Project [Elc::2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)umsflow.cpp	ubit:03.06.04"
#include <ubit/umsclient.hpp>
#include "umserver.hpp"
#include "umsflow.hpp"
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMSeventFlow::UMSeventFlow(class UMServer* _ums, int _flow_id) :
  ums(*_ums), id(_flow_id) {
}

UMSeventFlow::~UMSeventFlow() {}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMSmouseFlow::UMSmouseFlow(class UMServer* _ums, int _flow_id, 
                           bool create_pointer) :
  UMSeventFlow(_ums, _flow_id) {
  mx = 0, my = 0;
  state_mask = 0;
  pointer_win = None;
  last_entered_win = None;
  last_entered_ptr_in_uwin = false;
  last_entered_winsock = -1;
  if (create_pointer) createPointer();
}

UMSmouseFlow::~UMSmouseFlow() {}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSmouseFlow::pressMouse(const UMSbuttonMapping* bm) {
  if (!bm)
    cerr << "UMSmouseFlow::pressMouse: null button" <<endl;
  else
    pressMouse(bm->flow_button_mask, bm->flow_modifiers);
}

void UMSmouseFlow::pressMouse(u_int btn_maskid, u_int modifiers) {
  int n_mx, n_my;
  UMSneighbor* n = checkBordersAndNeighbors(n_mx, n_my);

  state_mask = state_mask | modifiers;

  if (n) {
    //cerr << "press in neighbor "<< endl;
    n->client->moveMouse(id, n_mx, n_my, true);
    n->client->pressMouse(id,btn_maskid);   // !! il manque modifiers !!
  }
  else  {
    UMSpos pos;
    if (!ums.locatePointer(*this, pos)) return;
    ums.sendButtonEvent(*this, pos, btn_maskid, true);
 
   // pour remettre la pseudo-mouse au 1er plan
    if (pointer_win != None) XMapRaised(ums.getDisplay(), pointer_win);
  } 

  // X oblige de rajouter le btn_mask APRES le mousePress
  state_mask = state_mask | btn_maskid;
}

/* ==================================================== ======== ======= */

void UMSmouseFlow::releaseMouse(const UMSbuttonMapping* bm) {
  if (!bm)
    cerr << "UMSmouseFlow::releaseMouse: null button" <<endl;
  else
    releaseMouse(bm->flow_button_mask, bm->flow_modifiers);
}

void UMSmouseFlow::releaseMouse(u_int btn_maskid, u_int modifiers)  {
  int n_mx, n_my;
  UMSneighbor* n = checkBordersAndNeighbors(n_mx, n_my);

  if (n) {
    //cerr << "release in neighbor "<< endl;
    n->client->moveMouse(id, n_mx, n_my, true);
    n->client->releaseMouse(id, btn_maskid);
  }
  else  {
    UMSpos pos;
    if (!ums.locatePointer(*this, pos)) return;
    ums.sendButtonEvent(*this, pos, btn_maskid, false);
  }

  state_mask = state_mask & ~(btn_maskid | modifiers);
}

/* ==================================================== ======== ======= */

void UMSmouseFlow::moveMouse(int x, int y, bool abs_coords) {
  if (id == 0) {   // native X mouse 
    if (abs_coords) {mx = x; my = y;} 
    else {
      //mx += x; my += y;
      
      // cette sequence permet de synchroniser mflow[0] avec la position
      // reelle du pointer X. c'est utile en cas de fusion, quand le ptr
      // est controle par plusieurs sources (p.ex. une "vraie" souris X
      // et une source de l'UMS)

      Window xroot = None, xchild = None;
      int root_x = 0, root_y = 0, win_x, win_y;
      unsigned int mask;
      XQueryPointer(ums.getDisplay(), ums.getRoot(), &xroot, &xchild,
		    &root_x, &root_y, &win_x, &win_y, &mask);
      mx = root_x + x;
      my = root_y + y;
    }
    XWarpPointer(ums.getDisplay(), ums.getRoot(), ums.getRoot(),
		 0, 0, 0, 0, mx, my);
    XFlush(ums.getDisplay());
  }

  else {          // id != 0 : alternate mouse flow
    if (abs_coords) {mx = x; my = y;} 
    else {mx += x; my += y;}

    /*
    if (mx < 0) mx = 0;
    else if (mx >= ums.getScreenWidth()) mx = ums.getScreenWidth() -1;
    if (my < 0) my = 0;
    else if (my >= ums.getScreenHeight()) my = ums.getScreenHeight() -1;
    */
    int n_mx, n_my;
    UMSneighbor* n = checkBordersAndNeighbors(n_mx, n_my);
    if (n) n->client->moveMouse(id, n_mx, n_my, true);

    if (pointer_win != None) {
      XMoveWindow(ums.getDisplay(), pointer_win, mx, my);
      XRaiseWindow(ums.getDisplay(), pointer_win);     // always on top !
      XFlush(ums.getDisplay());
    }
  }

  // generer les evenements (y compris dans cas pointer natif)
  mouseMoved(mx, my);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSmouseFlow::mouseMoved(int x, int y) {
  mx = x;
  my = y;
  UMSpos pos;
  if (!ums.locatePointer(*this, pos)) return;

  if (pos.win != last_entered_win) {
    if (last_entered_win != None) {                   // wx, wy corrects ?
      UMSpos leave_pos(pos.rx, pos.ry, 0, 0, 
                       last_entered_win, last_entered_ptr_in_uwin,
                       last_entered_winsock);
      ums.sendCrossingEvent(*this, leave_pos, false);
    }

    ums.sendCrossingEvent(*this, pos, true);
    last_entered_win = pos.win;
    last_entered_ptr_in_uwin = pos.ptr_in_uwin;
    last_entered_winsock = pos.winsock;
  }

  ums.sendMotionEvent(*this, pos);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMSneighbor* UMSmouseFlow::checkBordersAndNeighbors(int& n_mx, int& n_my) {
  int n_col  = 1;  // neighbor is itself
  int n_line = 1;

  if (mx < 0) n_col = 0;
  else if (mx >= ums.getScreenWidth()) n_col = 2;

  if (my < 0) n_line = 0;
  else if (my >= ums.getScreenHeight()) n_line = 2;

  bool send_to_n = false;
  if (n_line != 1 || n_col != 1) {

    UMSneighbor* n = ums.getNeighbor(n_line, n_col);

    if (n) {
      if (n->client->getStatus() > 0) send_to_n = true;
      
      else if (UMServer::getTime() - n->cnx_time 
	       > UMSneighbor::CNX_RETRY_DELAY) {
       
	if (n->client->open(n->host, n->port) > 0) 
	  send_to_n = true;
	else cerr << "can't connect to: " << n->host << endl;

	n->cnx_time = UMServer::getTime();
      }
    }

    if (send_to_n) {
      // !!! on a besoinde connaitre width du precedent !!!
      // !!! ou de pouvoir specifier le point de reference !!!
      if (n_col < 1) n_mx = ums.getScreenWidth() + mx;       // !! FAUX 
      else if (n_col > 1) n_mx = mx - ums.getScreenWidth();
      else n_mx = mx;
      
      if (n_line < 1) n_my = ums.getScreenHeight() + my;     // !! FAUX 
      else if (n_line > 1) n_my = my - ums.getScreenHeight();
      else n_my = my;
      
      return n;
    }
    
      else {
	if (n_col < 1) mx = 0; 
	else if (n_col > 1) mx = ums.getScreenWidth() -1;

	if (n_line < 1) my = 0;
	else if (n_line > 1) my = ums.getScreenHeight() -1;
      }
    }

  return null;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSmouseFlow::setPointer(const char* fgcolor_name, 
	   	              const char* bgcolor_name) {
  if (!pointer_win) return;

  Display* xdisplay = ums.getDisplay();
  Screen* xscreen = ums.getScreen();

  long bgcolor = WhitePixelOfScreen(xscreen);
  long fgcolor = BlackPixelOfScreen(xscreen);

  XColor color, exact_color;
  if (fgcolor_name 
      && XAllocNamedColor(xdisplay, DefaultColormapOfScreen(xscreen),
			  fgcolor_name, &color, &exact_color)) {
    fgcolor = color.pixel;
  }
  if (bgcolor_name 
      && XAllocNamedColor(xdisplay, DefaultColormapOfScreen(xscreen),
			  bgcolor_name, &color, &exact_color)) {
    bgcolor = color.pixel;
  }

  static char mouse_bits[] = {      
    //0x7f, 0x3f, 0x1f, 0x1f, 0x3f, 0x73, 0x61       // 7x7
    0x00, 0xfe, 0x7e, 0x3e, 0x3e, 0x7e, 0xe6, 0xc2   // 8x8
  };

  Pixmap pix = XCreatePixmapFromBitmapData(xdisplay, 
					   RootWindowOfScreen(xscreen), 
					   mouse_bits, 8, 8, 
					   fgcolor,
					   bgcolor,
					   ums.getScreenDepth());
  XSetWindowBackgroundPixmap(xdisplay, pointer_win, pix);
  XMapRaised(xdisplay, pointer_win);
}

void UMSmouseFlow::createPointer(const char* fgcolor_name, 
				 const char* bgcolor_name) {
  Display* xdisplay = ums.getDisplay();

  // create cursor window 
  XSetWindowAttributes wattr;
  unsigned long wattr_mask = 0;

  // window pas geree par le WM (no borders...)
  wattr.override_redirect = True;
  wattr_mask |= CWOverrideRedirect;

  //wattr.background_pixmap = pix;
  //wattr_mask |= CWBackPixmap;

  //wattr.background_pixel = WhitePixelOfScreen(xscreen);
  //wattr_mask |= CWBackPixel;

  //wattr.border_pixel = bgcolor;
  //wattr_mask |= CWBorderPixel;

  pointer_win = XCreateWindow(xdisplay,
			      ums.getRoot(), // fenetre mere
			      0,0,      // x,y position
			      8, 8,     // width , height (0,0 crashes!)
			      0,	// border_width
			      ums.getScreenDepth(),
			      InputOutput,  // Class
			      CopyFromParent,
			      wattr_mask, &wattr);
  setPointer(fgcolor_name, bgcolor_name);
  XMapRaised(xdisplay, pointer_win);
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
