/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "wf_sort.h"
#include "defs.h"


/* is_sorted() is a non standard function */
template<typename _ForwardIter>
bool
my_is_sorted(_ForwardIter __first, _ForwardIter __last) {
  if (__first == __last)
    return true;
  
  _ForwardIter __next = __first;
  for (++__next; __next != __last; __first = __next, ++__next) {
    if (*__next < *__first)
      return false;
  }
  
  return true;
}

/* Operator needed by lower_bound(). */
bool
operator<(const wf_sort_availspecs& a, const string& str) {
  return (a.spec < str);
}

/* Operator needed by binary_search(). */
bool
operator<(const string& str, const wf_sort_availspecs& a) {
  return (str < a.spec);
}


wf_sort_criteria::wf_sort_criteria(int (*cmpfunc)(const void* a, const void* b),
				   bool reverse) :
  _cmpfunc(cmpfunc),
  _reverse(reverse)
{}


wf_sort::wf_sort() :
  criterias(),
  _available(NULL),
  _available_count(0)
{}


/* Must be implemented for my_is_sorted(). */
bool
operator<(const wf_sort_availspecs& a, const wf_sort_availspecs& b) {
  return (a.spec < b.spec);
}

bool
wf_sort::available_set(const struct wf_sort_availspecs available[],
		       int count) {
  /* the array must be sorted in order to use binary search */
  if (my_is_sorted(available, available + count) == false)
    return false;

  _available = available;
  _available_count = count;
  return true;
}

ostream&
wf_sort::available_print(ostream& os) const {
  int i;
  for (i = 0; i < _available_count - 1; i++)
    os << _available[i].spec << endl;
  return os;
}

void
wf_sort::criteria_add(const wf_sort_criteria& criteria) {
  criterias.push_back(criteria);
}

bool
wf_sort::criteria_add(const char* spec) {
  bool reverse = false;

  if (*spec == '-') {
    reverse = true;
    spec++;
  }
  else if (*spec == '+')
    spec++;

  if (binary_search(_available, _available + _available_count, spec) == false)
    return false; /* bad, spec not found */

  const struct wf_sort_availspecs* ptr =
    lower_bound(_available, _available + _available_count, spec);

  wf_sort_criteria newsortcrit(ptr->cmpfunc, reverse);
  criteria_add(newsortcrit);
  return true;
}

/*
 * Parse long sorting options.
 * Add corresponding sort criterias to the list.
 */
/* make it less C-ish ALL@@3 */
const char*
wf_sort::parse(const char* sort_arg) {
  char *buf;                   /* temp copy of arg to hack on */
  char *sep_loc;               /* separator location: " \t,\n" */
  char *walk;
  const char *err;       /* error code that could or did happen */
//  sort_node *snode;
  int items;
  int need_item;

  if (sort_arg == NULL)
    return _("no sort argument.");

  /*** prepare to operate ***/
  buf = strdup(sort_arg);
  
  /*** sanity check and count items ***/
  need_item = 1; /* true */
  items = 0;
  walk = buf;
  err = _("improper sort specifier list.");
  do {
    switch (*walk) {
    case ' ': case ',': case '\t': case '\n': case '\0':
      if (need_item) {
        free(buf);
        return _("improper sort list.");
      }
      need_item = 1;
      break;
    default:
      if (need_item) items++;
      need_item = 0;
    }
  } while (*++walk);

  if (!items) {
    free(buf);
    return _("empty sort list.");
  }
#ifdef STRICT_LIST
  if (need_item) {   /* can't have trailing deliminator */
    free(buf);
    return _("improper sort list.");
  }
#else
  if (need_item)     /* allow 1 trailing deliminator */
    *--walk = '\0';  /* remove the trailing deliminator */
#endif

  /*** actually parse the list ***/
  walk = buf;
  while (items--) {
    sep_loc = strpbrk(walk, " ,\t\n");
    if (sep_loc)
      *sep_loc = '\0';

    // cerr << "arg " << walk << endl;
   
   if (!strcmp(walk, "none")) {
//      output_immediateprint = 1; RV@@9
      return NULL;
    }

   if (criteria_add(walk) == false) {
     free(buf);
     return _("unknown sort specifier.");
   }

    walk = sep_loc + 1; /* point to next item, if any */
  }
  free(buf);

  return NULL;
}

void
wf_sort::criterias_clear() {
  criterias.clear();
}

/*
 * Operator surcharge, representing "lower than" comparison.
 * So the wf_sort class can be used by the STL sort method.
 */
bool
wf_sort::operator()(const void* a, const void* b) const {
  list<wf_sort_criteria>::const_iterator first = criterias.begin(),
    last = criterias.end();
  for (; first != last; ++first) {
    int result = first->_cmpfunc(a, b);
    if (result != 0) {
      if (first->_reverse)
	result = -result;
      return (result < 0);
    }
  }
  return false; /* no conclusion */
}
