/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <netdb.h>
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/types.h> /* <netinet/in.h> needs this */
#endif
#include <sys/socket.h>  /* for AF_INET */
#include <netinet/in.h>  /* for htons */

#include "dns.h"
#include "defs.h"


wf_dns::wf_dns() :
  cache(),
  verbose(0)
{}

wf_dns::wf_dns(unsigned char verb) :
  cache(),
  verbose(verb)
{}


/* If no name is found, we create the entry, but name will be "". */
wf_dns_entry*
wf_dns::resolv(const wf_ipaddr& ipaddr) {
  wf_dns_entry* dns_entry;

  dns_entry = cache_get(ipaddr);
  if (dns_entry != NULL) { /* found in cache */
    if (verbose > 1)
      fprintf(stderr, _("Found DNS info for %s in cache\n"),
	      ipaddr.tostr().c_str());
    return dns_entry;
  }

  /* Not found in cache. */

  dns_entry = new wf_dns_entry(); /* create a new cache entry */
  if (dns_entry == NULL)
    return NULL;
  dns_entry->ipaddr = ipaddr;

  if (verbose > 1)
    fprintf(stderr, _("Looking up DNS info for %s\n"),
	    ipaddr.tostr().c_str());

  string dnsname;
  if (ipaddr.dnsname(dnsname))
    dns_entry->name = dnsname;
  /* If no name is found, we create the entry, but name will be left empty. */

 /* code below is to keep trace of the checks and "double resolving" code */
#if 0 /* ALL@@2 add it some day... */
  if (reverse != NULL && reverse->h_name != NULL) {
    if ((unsigned int)reverse->h_length > sizeof(struct in_addr)) {
      fprintf(stderr, _("Wrong host name size\n"));
      reverse->h_length = sizeof(struct in_addr);
      reverse->h_name[reverse->h_length] = '\0';
    }

    pnt = reverse->h_name;
    while (*pnt != '\0') {
      if (isalnum((int)*pnt) || *pnt == '.' || *pnt == '-') {
	pnt++;
	continue;
      } else {
	*pnt = '_';
	pnt++;
      }
    }

    if (opt.verbose)
      fprintf(stderr, _("Resolving %s\n"), reverse->h_name);

    forward = gethostbyname(reverse->h_name);
    if (forward != NULL && forward->h_addr_list[0] != NULL) {
      if (strncmp(inet_ntoa(ip),
		  inet_ntoa(*(struct in_addr *)forward->h_addr_list[0]),
		  IPLEN) == 0) {
	xstrncpy(dns->fqdn, reverse->h_name, HOSTLEN);
      }
      else {
	snprintf(dns->fqdn, HOSTLEN, _("%s [forward lookup: %s]"),
		 reverse->h_name,
		 inet_ntoa(*(struct in_addr *)forward->h_addr_list[0]));
      }
    }
    else {
      snprintf(dns->fqdn, HOSTLEN, _("%s [forward lookup failed]"),
	       reverse->h_name);
    }
  } else {
    xstrncpy(dns->fqdn, "-", HOSTLEN);
  }
#endif

  cache_add(dns_entry);  /* Add entry to cache. */
  return dns_entry;
}

void
wf_dns::cache_add(const wf_dns_entry* dns_entry) {
  if (dns_entry != NULL)
    cache.push_back((wf_dns_entry*)dns_entry);
}

wf_dns_entry*
wf_dns::cache_get(const wf_ipaddr& ipaddr) const {
  if (ipaddr.isdefined() == false)
    return NULL;

  list<wf_dns_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first) {
    if ((*first)->ipaddr == ipaddr)
      return *first;
  }
  return NULL;
}

wf_dns_entry*
wf_dns::cache_get(const string& dnsname) const {
  if (dnsname.empty())
    return NULL;

  list<wf_dns_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first) {
    if ((*first)->name == dnsname)
      return *first;
  }
  return NULL;
}

ostream&
wf_dns::cache_debugprint(ostream& os) const {
  list<wf_dns_entry*>::const_iterator first = cache.begin(),
    last = cache.end();
  for (; first != last; ++first)
    (*first)->print(os) << endl;
  return os;
}

ostream&
wf_dns_entry::print(ostream& os) const {
  return os << ipaddr << ' ' << name;
}

ostream&
operator<<(ostream& os, const wf_dns_entry& dns_entry) {
  return dns_entry.print(os);
}
