#include "Entity.h"
#include "Application.h"
#include "World.h"
#include <Ark/ArkBuffer.h>
#include <Ark/ArkLexer.h>
#include <stdio.h>
#include <sstream>

void
Entry::Init (Ark::ClassDef *eclass,
	     const Ark::String &name,
	     const Ark::Entry &value)
{
   (*(Ark::Entry *) this) = (value);
   m_Def = eclass->FindEntryDef (name);
   m_Changed = false;
   m_EType = ENTRY_DEF;
   m_Name = name;

   if (m_Def == NULL)
      return;

   Ark::String desc = m_Def->m_Desc;

   if (desc[0] != '[')
      return;

   size_t endpos = desc.find ("]");
   if (endpos == Ark::String::npos)
      return;

   // Get the text between []
   desc = desc.substr (1, endpos - 1);

   if (desc == "")
      return;

   if (desc.substr (0,5) == "class")
   {
      assert (m_Def->m_Type == Ark::Entry::STRING);
      m_EType = ENTRY_CLASS;
      return;
   }
   else if (desc.substr (0,5) == "model")
   {
      assert (m_Def->m_Type == Ark::Entry::STRING);
      m_EType = ENTRY_MODEL;
      return;
   }
   else if (desc.substr (0,4) == "bool")
   {
      assert (m_Def->m_Type == Ark::Entry::INTEGER);
      m_EType = ENTRY_BOOL;
      return;
   }
   else if (desc.substr (0,4) == "enum")
   {
      assert (m_Def->m_Type == Ark::Entry::STRING);
      m_EType = ENTRY_ENUM;

      size_t token = 5;
      while (1)
      {
	 size_t tok = desc.find ("/", token);

	 if (tok == Ark::String::npos)
	    break;

	 m_EnumMembers.push_back (desc.substr(token, tok));
	 token = tok;
      }

      return;
   }
   else if (desc.substr (0,5) == "range")
   {
      assert (m_Def->m_Type == Ark::Entry::INTEGER);
      m_EType = ENTRY_RANGE;
      desc = desc.substr (6);

      m_RangeMin = 0;
      m_RangeMax = 10;
      sscanf (desc.c_str(), "%d/%d", &m_RangeMin, &m_RangeMax);
      return;
   }

}


Entity::Entity (World *world) : Ark::Entity (world)
{}

Entity::~Entity ()
{}


static void LoadTemplateEntries (const Ark::String &_class,
				 const Ark::String &_template,
				 Ark::EntryList &entries)
{
   Ark::String def = "implements ";
   def += _class;

   if (_template != "")
   {
      def += ", uses ";
      def += _template;
   }

   def += " {}";

   std::istringstream stream(def);
   Ark::Lexer lexer ("entity/settemplate/buffer", stream);
   g_Application->GetClasses()->Read (lexer, &entries);
}

void
Entity::SetTemplate (const Ark::String &_class,
		     const Ark::String &_template)
{

   Ark::EntryList entries;
   m_Class = g_Application->GetClasses()->Find (_class);

   LoadTemplateEntries (_class, _template, entries); 
   SetEntries (entries, true);
}

void
Entity::SetClass (Ark::ClassDef *d)
{
   SetTemplate(d->m_Name, "");
}

void
Entity::SetEntries(Ark::EntryList &entries, bool in_editor)
{
   m_Entries.clear();
//   for (map<Ark::String,Entry>::iterator it = m_Entries.begin();
//        it != m_Entries.end();
//        it++)
//   {
//      cout << it->first << " = " << it->second.ToString() << '\n';
//   }

   m_Class = g_Application->GetClasses()->Find (*entries["class"].d_str);

   for (Ark::EntryList::iterator it = entries.begin();
	it != entries.end(); it++)
   {
      EntryLI ent = m_Entries.find(it->first);

      if (ent != m_Entries.end() && in_editor
	  && (it->first == "name" ||
	      it->first == "model"||
	      it->first == "position" ||
	      it->first == "rotation" ||
	      it->first == "shortname"))
	 continue;

      Entry &entry = m_Entries[it->first];
      entry.Init (m_Class, it->first, it->second);
   }

   SetModel (*m_Entries["model"].d_str);
   SetPosition (*m_Entries["position"].d_vector3);

   Ark::Vector3 rot = *m_Entries["rotation"].d_vector3;
   m_MState.m_Rotation = Ark::Quat (rot.X, rot.Y, rot.Z);
}

void
Entity::SetEntry(const Ark::String &name, bool value)
{
   Entry &ent = m_Entries[name];
   assert(ent.m_EType == ENTRY_BOOL);

   ent.d_integer = static_cast<int>(value);
}

void
Entity::SetEntry(const Ark::String &name, int value)
{
   Entry &ent = m_Entries[name];
   assert(ent.m_Type == Ark::Entry::INTEGER);

   ent.d_integer = static_cast<int>(value);
}

void
Entity::SetEntry(const Ark::String &name, const Ark::String &value)
{
   Entry &ent = m_Entries[name];
   assert(ent.m_Type == Ark::Entry::STRING);

   delete ent.d_str; ent.d_str = new Ark::String (value);

   if (name == "model")
      SetModel(value);
}

// Convert this entity into a string. Try to do a "diff" against the template.
// FIXME: the 'diff' code is a bit clumsy
Ark::String
Entity::ToString () const
{
   Ark::EntryList tmplentries;
   bool hastemplate;

   std::ostringstream os;
   EntryList::const_iterator i = m_Entries.find ("class");
   
   Ark::String _class = i->second.ToString();
   os << "implements " << _class;

   i = m_Entries.find ("template");
   if (i != m_Entries.end())
   {
      Ark::String _template = i->second.ToString();
      hastemplate = true;
      
      LoadTemplateEntries (_class, _template, tmplentries);
      os << ", uses " << _template;
   }

   os << "\n{\n";

   for (i = m_Entries.begin(); i != m_Entries.end(); i++)
   {
      if (i->first == "class" || i->first == "template")
	 continue;
      
      Ark::String value = i->second.ToString ();

      if (hastemplate && i->first != "position" && i->first != "rotation")
      {
	 Ark::EntryList::iterator it = tmplentries.find (i->first);
	 if (it != tmplentries.end() && it->second.ToString() == value)
	    continue;
      }

      os << "  " << i->first << " = ";
      if (i->first == "position")
      {
	 os << "{"
	    << m_MState.m_Position.X << ", "
	    << m_MState.m_Position.Y << ", "
	    << m_MState.m_Position.Z << "};\n";
      }
      else if (i->first == "rotation")
      {
	 Ark::Quat q = m_MState.m_Rotation;
	 scalar y,p,r; q.GetEuler(&y,&p,&r);
	 os << "{" << y << ", " << p << ", " << r << "};\n";
      }
      else if (i->second.m_Type == Ark::Entry::STRING)
	 os << "\"" << value << "\";\n";
      else
	 os << value << ";\n";
   }

   os << "}\n";

   return os.str();
}
