// $Id: GLTerrainView.cpp,v 1.17 2003/02/28 22:42:36 zongo Exp $

#include <vector>
#include <string>
#include <iostream>
#include <algorithm>

#ifdef WIN32
# include <windows.h>
#endif
#include <gtk/gtksignal.h>

#include <Ark/ArkPrimBlock.h>
#include <Modules/Renderer/GLRenderer.h>

#include "../Application.h"
#include "../World.h"
#include "GLTerrainView.h"


// =========================================================================

class BoundingBoxRenderer
{
	Ark::VertexBuffer m_VertexBuffer;
	Ark::PrimitiveBlock m_VertexIndices;
	Ark::Material m_Material;

public:
	BoundingBoxRenderer();
	void Render(Ark::Renderer* renderer, const Ark::BBox &bb);
};


BoundingBoxRenderer::BoundingBoxRenderer() :
  m_Material("bbox/material")
{

	m_Material.m_Flags = Ark::MATERIAL_HAS_PASS1;
	{
		Ark::ShaderPass& pass = m_Material.m_Passes[0];
		pass.m_Flags |= Ark::PASS_HAS_BLENDING;
		pass.m_Flags &= ~Ark::PASS_HAS_DEPTHWRITE;
		pass.m_DepthTest = true;
		pass.m_DepthWrite = false;
		pass.m_BlendColor = Ark::Color (0.5f, 0.5f, 1.0f, 0.8f);
		pass.m_BlendSrc = Ark::BLEND_ONE_MINUS_SRC_ALPHA;
		pass.m_BlendDst = Ark::BLEND_SRC_ALPHA;
	}

	const int vertexCount = 8;
	const int vertexIndices[] = {5, 3, 7, 1, 0, 3, 2, 5, 4, 7, 6, 0, 4, 2};
	const int indexCount = sizeof(vertexIndices)/sizeof(int);

	m_VertexBuffer.SetFormat (Ark::VertexBuffer::VB_HAS_COORD);
	m_VertexBuffer.Resize(vertexCount);

	m_VertexIndices.SetType(Ark::PRIM_TRIANGLE_STRIP);
	m_VertexIndices.Resize(indexCount);
	m_VertexIndices.SetEnabledSize(indexCount);
   
	for (int i=0 ; i<indexCount ; ++i)
	{
		m_VertexIndices[i] = vertexIndices[i];
	}

}

void 
BoundingBoxRenderer::Render(Ark::Renderer* renderer, const Ark::BBox &bb)
{
	if (bb.m_Max.X < bb.m_Min.X)
		return;

	const int vertexCount = 8;
	Ark::Vector3 vs[vertexCount];
	bb.GetCorners (vs);

	for (int i=0 ; i<vertexCount ; ++i)
	{
		m_VertexBuffer.Coord(i) = vs[i];
	}

	renderer->SetActiveVB(m_VertexBuffer);
	renderer->LockVB(0, vertexCount);

	const int size = m_VertexIndices.EnabledSize();
	renderer->RenderBlock(m_Material, m_VertexIndices.Type(), &m_VertexIndices[0], size);
	
	renderer->UnlockVB();
}


GLTerrainView::GLTerrainView(TerrainView* terrain) : 
  GLView(),
  m_Terrain (terrain)
{
	m_BoundingBox = new BoundingBoxRenderer();
}

GLTerrainView::~GLTerrainView ()
{
	if (m_BoundingBox)
	{
		delete m_BoundingBox;
	}
}


// FIXME: put the following 2 functions somewhere in the engine, 'cause
// they are used both by the client and the world editor.
static
Ark::Vector3 
UnProject (scalar sX, scalar sY, scalar mz,
	   Ark::Matrix44 invmat)
{
   Ark::Vector3 R = invmat.Transform(Ark::Vector3 (sX, sY, mz));
   scalar w = (sX * invmat.M(0,3) +
	       sY * invmat.M(1,3) +
	       mz * invmat.M(2,3) +
	       invmat.M(3,3));
   
   R.Scale (1.f/w);
   return R;
}
  

Ark::Vector3
GLTerrainView::TraceMousePos (const Ark::Vector3 &pov,
			      int mx, int my)
{      
   Ark::Matrix44 projmat, modelmat;
   int w = GTK_WIDGET(m_Area)->allocation.width;
   int h = GTK_WIDGET(m_Area)->allocation.height;
   
   glGetFloatv (GL_PROJECTION_MATRIX, &projmat.M(0,0));
   glGetFloatv (GL_MODELVIEW_MATRIX, &modelmat.M(0,0));
   
   modelmat.Multiply (projmat);
   modelmat.Invert();
   
   scalar sX = 2.f * (scalar(mx) / scalar(w)) - 1.f;
   scalar sY = 2.f * (scalar(h-my) / scalar(h)) - 1.f;
   
   Ark::Vector3 lookat = UnProject (sX, sY, 1.f, modelmat);
   
   Ark::Vector3 dir = lookat - pov;
   dir.Scale (10000.0);
   
   lookat = pov + dir;
   std::vector<Ark::Collision> collisions;
   m_Terrain->GetWorld()->RayTrace (Ark::Ray(pov, lookat),
				    Ark::Collision::POSITION |
				    Ark::Collision::ENTITY |
				    Ark::Collision::WORLD,
				    collisions);
   if (collisions.size())
   {
      m_LastHit = collisions[0];

/*
      m_Terrain->GetWorld()->RayTrace (Ark::Ray(pov, lookat),
				       Ark::Collision::POSITION |
				       Ark::Collision::WORLD,
				       collisions);
*/
      return collisions[0].m_Pos;
   }
   else
   {
      m_LastHit.m_Flags = 0;
      m_LastHit.m_Entity = NULL;
   }
   
   return Ark::Vector3 (0.0, 0.0, 0.0);
}


bool
GLTerrainView::Render()
{
  int w = GTK_WIDGET(m_Area)->allocation.width;
  int h = GTK_WIDGET(m_Area)->allocation.height;

  m_Renderer->SetViewport(0, 0, w, h);
  glClear(GL_DEPTH_BUFFER_BIT|GL_COLOR_BUFFER_BIT);

  // Call to cwterrain -> height field Render functions
  m_Terrain->Render();

#if 0
  // Render bbox for selected entity.
  Ark::World* world = m_Terrain->GetWorld();
  Ark::EntityList& list = world->GetEntities();

  if (!list.empty())
  {
	  // XXX choose the selected entity
	  Ark::Entity* entity = list[0];
	  m_BoundingBox->Render(m_Renderer, entity->GetBBox());
  }
#endif

  return true;
}



static bool
ValidVector(World* w, const Ark::Vector3& vec)
{
  const int mx = w->m_SizeX;
  if ((vec.X < 0) || (mx <= vec.X)) return false;
  
  const int mz = w->m_SizeZ;
  if ((vec.Z < 0) || (mz <= vec.Z)) return false;

  return true;
}


bool
GLTerrainView::Motion(int x, int y, int state, int lastx, int lasty)
{
  bool redraw= true;
  
  const int Left = GDK_BUTTON1_MASK;
  const int Middle = GDK_BUTTON2_MASK;
  const int Right = GDK_BUTTON3_MASK;
  const int CtrlRight = (Right | GDK_CONTROL_MASK);
  const int ShiftRight = (Right | GDK_SHIFT_MASK);

  const scalar w = (float)m_Width;
  const scalar h = (float)m_Height;
  
  // Zooming using Control+Right
  if ((state & CtrlRight) == CtrlRight )
  {
    // Distance change
    m_Terrain->Zoom( 8.f * (lasty - y) / h );

  }
  // Angle changes using Shift+Right or Middle button
  else if ( (state & Middle) || (state & ShiftRight) == ShiftRight )
  {
    // Vertical angle change
    m_Terrain->Rotate( 180.f * (lastx - x) / w );
    
    // Pitch angle change
    m_Terrain->Pitch( 90.f * (lasty - y) / h );
    
  }
  // Moving using right button
  else if (state & Right) 
   m_Terrain->Drag( (x-lastx) / w, (y-lasty) / h);
    
  // Use default action for this move
  else if (state & Left)
  {
    Ark::Vector3 v = GetGLPos(x, y);

    if (!ValidVector(m_Terrain->GetWorld(), v))
      redraw = false;
    else 
    {
       TraceMousePos(m_Terrain->GetCamera()->m_PointOfView, x, y);
       m_Terrain->Change(v, m_LastHit);
    }

  }
  else
    redraw = false;


  if (redraw)
  {
    Redraw();
    return true;
  }

  return false;
}


bool
GLTerrainView::ButtonPress(int x, int y, int button, int state)
{
   if (button == 1)
   {
      Ark::Vector3 v = GetGLPos(x, y);
      TraceMousePos(m_Terrain->GetCamera()->m_PointOfView, x, y);

      if (!ValidVector(m_Terrain->GetWorld(), v))
	 return false;
      
      m_Terrain->Change(v, m_LastHit);
      
      Redraw();
   }
   
   m_LX = x;
   m_LY = y;

   return true;
}


bool
GLTerrainView::ButtonRelease(int x, int y, int button, int state)
{

   // Distance on wheel !
   if ((button == 4) || (button == 5))
   {
     
     m_Terrain->Zoom( (button == 4) ? 1.f : -1.f );
     Redraw();
     
     return true;
   }
   
   if (button == 1)
   {
     m_Terrain->UpdateTool();
   }

   return false;
}


bool
GLTerrainView::Change(int fromx, int fromy, int tox, int toy)
{
  /*
  World* world = m_Terrain->GetWorld();
  
  Ark::Vector3 vfrom = GetGLPos(fromx, fromy);
  if (!ValidVector(world, vfrom)) return false;
        
  Ark::Vector3 vto = GetGLPos(tox, toy);
  if (!ValidVector(world, vto)) return false;

  Ark::Vector3 diff = vto - vfrom;

  const int length = static_cast<int>( diff.GetMagnitude() ) + 2;
  const scalar scale = 1.0 / static_cast<scalar>( length );

  diff.Scale( scale );

  for (int i = 0 ; i < length ; ++i) {
    m_Terrain->Change( vfrom );
    vfrom += diff;
  }
  */
  return true;
}
