/* misc.c -- Some tricky functions. */

/* Copyright (C) 1996-1997  Janos Farkas

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.  */


#include "config.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <utmp.h>
#include <sys/stat.h>

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "misc.h"
#include "finger.h"
#include "util.h"
#include "broken.h"

/* return a fresh string (NULL only if you pass it, but the result
   may be empty), which is P stripped from trailing and leading blanks */

char *
normalize (const char *name)
{
  const char *s;
  char *p, *norm;

  if (!name)
    return NULL;

  /* skip leading blanks */
  for (s = name; *s && isspace (*s); s++)
    /* empty */ ;

  /* strip trailing blanks */
  norm = xstrdup (s);
  for (p = norm + strlen (norm) - 1; p >= norm && isspace (*p); p--)
    *p = 0;

  return norm;
}

char *
sanitize (const char *name)
{
  char *p, *save;

  save = xstrdup (name);
  /* convert any `_' and `,' to dots (maybe more) */
  for (p = save; p && *p; p++)
    if (*p == '_' || *p == ',' || *p == '/' || *p == ' ')
      *p = '.';

   return save;
}

char *
strlwr (char *str)
{
  char *p;

  for (p = str ; *p; p++) {
    if (isupper (*p))
      *p = tolower (*p);
  }
  return str;
}

/* check for illegal characters in NAME
   currently only alphanumerics and `.-_' are allowed
   this may be too strict, but should cause rarely problems
   also note that the NAME should already be sanitized */

int
is_illegal (const char *name)
{
  const char *p;

  for (p = name; *p; p++)
    if (!isalnum (*p) && *p != '.' && *p != '-' && *p != '_')
      return 1;

  return 0;
}

/* check if the given string is empty (or NULL) */

int
is_empty (const char *name)
{
  if (name == NULL || *name == 0)
    return 1;

  return 0;
}

/* print LEVEL number of spaces */

int
blanks (int level)
{
  /* Hmm.. may be ugly, but I like it.. :) */
  if (level > 0)
    return printf ("%*s", level, "");

  return 0;
}

/* just a stub */

int
display_start (void)
{
  return 0;
}

/* finish a display_field() series, move to a new line */

int
display_end (int start_pos)
{
  if (start_pos > 0)
    printf ("\n");

  return 0;
}

/* print the fields on the left or right side of the screen,
   depending on START_POS and return the position we left
   the cursor at */

int
display_field (int start_pos, const char *name, const char *field)
{
  char *p;

  p = xmalloc (strlen (name) + strlen (field) + 1);
  strcpy (p, name);
  strcat (p, field);

  /* devote a full line for too long fields */
  if (strlen (p) >= FIELD_WIDTH)
    start_pos = display_end (start_pos);

  /* if not at the left side, then do some padding until we
     reach the middle */
  if (start_pos > 0 && start_pos < FIELD_WIDTH) {
    start_pos += blanks (FIELD_WIDTH - start_pos);
  }

  start_pos += printf ("%s", p);
  free (p);

  /* if next field wouldn't fit here then CR */
  if (start_pos >= FIELD_WIDTH) {
    printf ("\n");
    start_pos = 0;
  }

  return start_pos;
}

static int newblock = 0;
static int blocklf = 0;

/* start a new `block' (paragraph), but only mark it.
   fill_block () will start a new line if BIF is set. */

void
start_block (int bif)
{
  if (!newblock)
    blocklf = bif;
  newblock = 1;
}

void
fill_block (void)
{
  if (newblock) {
    newblock = 0;
    if (blocklf) {
      printf("\n");
    }
  }
}

void
end_block (void)
{
  newblock = 0;
}

/* format MYTIME in a `nice' way, at least legible;
   return a static buffer */

char *
nicedate (time_t *mytime)
{
  const struct tm *p;
  const char *tz;
  static char wdtab[] = {"SunMonTueWedThuFriSat"};
  static char mytab[] = {"JanFebMarAprMayJunJulAugSepOctNovDec"};
  static char datebuf[40];

  p = localtime (mytime);
  tz = tzname[p->tm_isdst];
  sprintf (datebuf, "%.3s %.3s %2d %2d:%02d:%02d %d (%s)",
           &wdtab[p->tm_wday*3], &mytab[p->tm_mon*3], p->tm_mday,
           p->tm_hour, p->tm_min, p->tm_sec,
           p->tm_year+1900, tz);
  return datebuf;
}

static void
vfputc (int ch, FILE *f)
{
  int ascii;

  ascii = 1;
  if (ch & 0x80) {
    fputc ('M', f); fputc ('-', f);
    ch &= 0x7f;
    ascii = 0;
  }
  if (isprint (ch) || (ascii && (ch == ' ' || ch == '\n' || ch == '\t')))
    fputc (ch, f);
  else {
    fputc ('^', f);
    fputc (ch != 0x7f ? ch^0x40 : '?', f);
  }
}

int
secureopen (const char *fn, uid_t uid)
{
  int fd, ok;
  struct stat sb;

  ok = 0; fd = -1;
  if (lstat(fn, &sb) == 0) {
    /* Not a link */
    if (sb.st_uid == uid && S_ISREG(sb.st_mode) && (sb.st_mode & 004)) {
      /* World readable */

      /* Simpleminded timeout, in case he tries to race into a strange file. */
      alarm (1);
      if ((fd = open (fn, O_RDONLY|O_NONBLOCK|O_NOCTTY)) >= 0) {
        alarm (0);
        /* Now recheck everything */
        if (fstat (fd, &sb) == 0) {
          if (sb.st_uid == uid && S_ISREG(sb.st_mode) && (sb.st_mode & 004)) {
	    ok = 1;
          }
        }
      }
    }
  }
  if (!ok && fd >= 0) {
    close(fd);
    fd = -1;
  }

  return fd;
}

/* display the FILE; optionally removing control characters
   (if UID (the presumed owner of the file) nonzero) */

char *basename (const char *);

int
display_file (const char *header, const char *file, uid_t uid)
{
  int fd;
  FILE *f;
  struct stat sb;
  char buf[256];
  int shown;

  shown = 0;
  if (!stat (file, &sb)) {
    fd = secureopen(file, uid);
    if (fd >= 0) {
      f = fdopen (fd, "r");
      if (f) {
        fill_block ();
        shown = 1;

        if (header && *header)
          printf ("%s", header);

        if (!uid) {
	  /* root/root file */
          while (!feof (f)) {
    	    if (fgets (buf, sizeof (buf), f))
	      fputs (buf, stdout);
          }
        } else {
          while (!feof (f)) {
    	    int c;
	    if ((c = fgetc (f)) != EOF)
	      vfputc (c, stdout);
          }
        }
        fclose (f);
      }
    } else {
      fill_block ();
      printf("Won't show `%s', user is lame.\n", basename((char *)file));
    }
  }
  return shown;
}

/* format HOSTNAME to display a login location;
   return a static buffer that is overwritten on each call */

char *
if_origin (struct fng_info *fi, const char *hostname)
{
  static char conbuf[UT_HOSTSIZE+8];

  if (fi->origin) /* XXX */
    sprintf (conbuf, " from %s", *hostname ? hostname : "console");
  else
    *conbuf = 0;
  return conbuf;
}
