// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_pb_single_editor.cc
// *
// * Purpose: A single list of phonebook entries that can be edited
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 28.10.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_pb_single_editor.h"
#include "xgsm_multi_editor.h"
#include "xgsm_util.h"
#include "xgsm_dialogs.h"
#include "xgsm_pref.h"

#include <gtk--/base.h>
#include <gtk--/frame.h>
#include <gtk--/button.h>
#include <gtk--/main.h>
#include <gnome--/dialog.h>
#include <typeinfo>
#include <algorithm>
#include <gsmlib/gsm_util.h>
#include <limits.h>
#include <iostream>

extern "C" {
#include "interface.h"
#include "support.h"
}

using namespace std;
using namespace Xgsm;
using namespace SigC;
using namespace gsmlib;

// PbSingleEditor members

void PbSingleEditor::sensitive(bool deviceIdle)
{
  _textEntry->set_sensitive(deviceIdle && ! _dev.isnull());
  _numberEntry->set_sensitive(deviceIdle && ! _dev.isnull());
  _changeIndexMI->set_sensitive(rowsSelected() == 1 &&
                                deviceIdle && ! _dev.isnull());
  SingleEditor::sensitive(deviceIdle);
}

void PbSingleEditor::refresh()
{
  _textEntry->set_text("");
  _numberEntry->set_text("");
  if (_dev.isnull())
  {
    _clist->clear();
    ((Gtk::Entry*)_storeNameEntry->gtk_entry())->set_text("");
  }
  else
    _completionConnection =
      _dev->request(new GetPbEntriesRequest(_phonebook, _sortOrder,
                                            slot(this,
                                                 &PbSingleEditor::onRefresh)));
}

void PbSingleEditor::completeEdit(bool editorClosed)
{
  if (_rowEdited == -1) return;

  DEBUG_START(1);
  cout << "completeEdit called" << endl;
  DEBUG_END;
  if (_entries()[_rowEdited].pbText() != _clist->get_text(_rowEdited, 1) ||
      _entries()[_rowEdited].pbTelephone() != _clist->get_text(_rowEdited, 2))
  {
    DEBUG_START(1);
    cout << "completeEdit in progress" << endl;
    DEBUG_END;
    _entries()[_rowEdited].setPbText(_clist->get_text(_rowEdited, 1));
    _entries()[_rowEdited].setPbTelephone(_clist->get_text(_rowEdited, 2));
    _completionConnection =
      _dev->request(
        new UpdatePbEntryRequest(_phonebook,
                                 _entries()[_rowEdited],
                                 (slot(this, &SingleEditor::onEditDone))));
  }
  stopEdit();
}

void PbSingleEditor::onRefresh(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      (stringPrintf(_("Error reading %s \n("), _messageText.c_str()) +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    close();
  }
  else
  {
    assert(typeid(response()) == typeid(GetPbEntriesResponse));
    if (_destroyed) return;
    _entries = ((GetPbEntriesResponse&)response()).entries();
    _maxTextLength = ((GetPbEntriesResponse&)response()).maxTextLength();
    _maxTelephoneLength =
      ((GetPbEntriesResponse&)response()).maxTelephoneLength();
    _clist->freeze();
    _clist->clear();
    for (EntryList::iterator i = _entries->begin(); i != _entries->end(); ++i)
    {
      vector<string> newRow;
      newRow.push_back(stringPrintf("%d", i->pbIndex()));
      newRow.push_back(i->pbText());
      newRow.push_back(i->pbTelephone());
      _clist->append_row(Gtk::SArray(newRow));
    }
    _clist->thaw();
    stopEdit();
    sensitive(_deviceIdle);
  }
}

void PbSingleEditor::onTextChanged()
{
  string text = _textEntry->get_text();
  if (text.length() > _maxTextLength)
  {
    text = text.substr(0, _maxTextLength);
    _textEntry->set_text(text);
  }

  if (rowsSelected() == 1 && _rowEdited != -1)
    _clist->set_text(_clist->selection().begin()->get_row_num(), 1, text);
}

void PbSingleEditor::onNumberChanged()
{
  string number = _numberEntry->get_text();
  if (number.length() > _maxTelephoneLength)
  {
    number = number.substr(0, _maxTelephoneLength);
    _numberEntry->set_text(number);
  }

  if (rowsSelected() == 1 && _rowEdited != -1)
    _clist->set_text(_clist->selection().begin()->get_row_num(), 2, number);
}
    
void PbSingleEditor::onSelectRow(int row, int col, GdkEvent *event)
{
  completeEdit();
  if (rowsSelected() == 1)
  {
    _textEntry->set_text(_clist->get_text(row, 1));
    _numberEntry->set_text(_clist->get_text(row, 2));
    startEdit(row);
  }
  else
  {
    stopEdit();
    _textEntry->set_text("");
    _numberEntry->set_text("");
  }
  sensitive(true);
}

void PbSingleEditor::onUnselectRow(int row, int col, GdkEvent *event)
{
  completeEdit();
  if ((rowsSelected() == 2 && _rowEdited == row) || rowsSelected() == 1)
  {
    int r = _clist->selection().begin()->get_row_num();
    startEdit(r);
    _textEntry->set_text(_clist->get_text(r, 1));
    _numberEntry->set_text(_clist->get_text(r, 2));
  }
  else
  {
    stopEdit();
    _textEntry->set_text("");
    _numberEntry->set_text("");
  }
  sensitive(true);
}

void PbSingleEditor::onExtendSelection()
{
  // FIXME, remove
}

void PbSingleEditor::onSelectAll()
{
  // FIXME, remove
}

void PbSingleEditor::onUnselectAll()
{
  // FIXME, remove
}

void PbSingleEditor::onCut()
{
  // stop edit in progress, since the entry is deleted anyway
  stopEdit();

  DeletePbEntriesRequest *req =
    new DeletePbEntriesRequest(_phonebook,
                               slot(this, &SingleEditor::onCutDone));
  for (Gtk::CList::SelectionIterator i = _clist->selection().begin();
       i != _clist->selection().end(); ++i)
  {
    DEBUG_START(1);
    cout << "cutting " << i->get_row_num() << endl;
    DEBUG_END;
    req->addEntry(_entries()[i->get_row_num()]);
  }
  _completionConnection = _dev->request(req);
}

void PbSingleEditor::onAdd()
{
  EntryListRef e = new EntryList;
  PhonebookEntryBase newEntry(_numberEntry->get_text(),
                              _textEntry->get_text());
  e->push_back(newEntry);
  _completionConnection =
    _dev->request(new AddPbEntryRequest(AddPbEntryRequest::Add, _phonebook, e,
                                        slot(this, &SingleEditor::onAddDone)));
}

void PbSingleEditor::onColumnClicked(gint column)
{
  SortOrder oldSortOrder = _sortOrder;
  switch (column)
  {
  case 0:
    _sortOrder = ByIndex;
    break;
  case 1:
    _sortOrder = ByText;
    break;
  case 2:
    _sortOrder = ByTelephone;
    break;
  }
  if (_sortOrder != oldSortOrder)
    refresh();
}

#ifndef NDEBUG
void PbSingleEditor::dumpEntries()
{
  DEBUG_START(1);
  for (set<string>::iterator i = _storeNameHistory.begin();
       i != _storeNameHistory.end(); ++i)
    cout << "history: " << *i << endl;
  DEBUG_END;
}
#endif

PbSingleEditor::PbSingleEditor(MultiEditor *multiEditor, 
                               Gtk::Frame *frame, bool isLeft) :
  SingleEditor(multiEditor, 
               frame, isLeft ? "pb_left" : "pb_right",
               isLeft ? _("left phonebook") : _("right phonebook"),
               PHONEBOOK_FILE_EXTENSION, ByText),
  _maxTextLength(UINT_MAX), _maxTelephoneLength(UINT_MAX)
{
  // connect inherited widgets
  _clist->select_row.connect(slot(this, &PbSingleEditor::onSelectRow));
  _clist->unselect_row.connect(slot(this, &PbSingleEditor::onUnselectRow));
  _clist->click_column.connect(slot(this, &PbSingleEditor::onColumnClicked));
  _addButton->clicked.connect(slot(this, &PbSingleEditor::onAdd));
  _cutButton->clicked.connect(slot(this, &PbSingleEditor::onCut));
  _addMI->activate.connect(slot(this, &PbSingleEditor::onAdd));
  _cutMI->activate.connect(slot(this, &PbSingleEditor::onCut));

  // lookup and connect own widgets
  LOOKUP_WIDGET(_textEntry, _prefix + "_text_entry", GtkEntry,
                GTK_IS_ENTRY, frame);
  _textEntry->changed.connect(slot(this, &PbSingleEditor::onTextChanged));
  LOOKUP_WIDGET(_numberEntry, _prefix + "_number_entry", GtkEntry,
                GTK_IS_ENTRY, frame);
  LOOKUP_WIDGET(_changeIndexMI, _prefix + "_change_index", GtkMenuItem,
                GTK_IS_MENU_ITEM, _multiEditor);
  _numberEntry->changed.connect(slot(this, &PbSingleEditor::onNumberChanged));
}

void PbSingleEditor::open(string deviceName, string baudRate, 
                          string initString, bool swHandshake,
                          string phonebook)
{
  close();
  _phonebook = phonebook;
  _sourceName = _deviceName = deviceName;
  DeviceHelper::open(new OpenRequest(deviceName, baudRate,
                                     initString, swHandshake,
                                     slot(this, &DeviceHelper::onDeviceEvent),
                                     slot(this, &SingleEditor::onOpenDone)),
                     true);
}

void PbSingleEditor::open(string fileName)
{
  close();
  _phonebook = fileName;
  _sourceName = fileName;
  _deviceName = config.getPhoneDevice();
  DeviceHelper::open(new OpenRequest(fileName, PhonebookFile,
                                     slot(this, &DeviceHelper::onDeviceEvent),
                                     slot(this, &SingleEditor::onOpenDone)),
                     true);
}

void PbSingleEditor::init()
{
  _deviceName = config.getPhoneDevice();
  // just find out store names
  DeviceHelper::open(new OpenRequest("", PhonebookFile,
                                     slot(this, &DeviceHelper::onDeviceEvent),
                                     slot(this, &SingleEditor::onOpenDone)),
                     false);
}

void PbSingleEditor::copyStore(int operation, SingleEditorRef e)
{
  AddPbEntryRequest::Operation op;
  switch (operation)
  {
  case CopySelected:
    op = AddPbEntryRequest::Add;
    break;
  case CopyAll:
    op = AddPbEntryRequest::Add;
    break;
  case Synchronize:
    op = AddPbEntryRequest::Synchronize;
    break;
  }

  EntryListRef copyList = new EntryList;

  if (operation == CopySelected)
    for (Gtk::CList::SelectionIterator i = e->_clist->selection().begin();
         i != e->_clist->selection().end(); ++i)
      copyList->push_back(e->_entries()[i->get_row_num()]);
  else
    copyList() = e->_entries(); // deep copy

  _completionConnection =
    _dev->request(
      new AddPbEntryRequest(op, _phonebook, copyList,
                            slot(this, &SingleEditor::onAddDone)));
}
