// *************************************************************************
// * Xgsm - mobile phone manager
// *
// * File:    xgsm_send_sms.cc
// *
// * Purpose: Send SMS dialog
// *
// * Author:  Peter Hofmann (software@pxh.de)
// *
// * Created: 5.9.2000
// *************************************************************************

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xgsm_send_sms.h"
#include "xgsm_device.h"
#include "xgsm_main.h"
#include "xgsm_pref.h"

#include <gsmlib/gsm_sms.h>
#include <gsmlib/gsm_util.h>
#include <gtk--/style.h>
#include <gtk--/button.h>
#include <gtk--/text.h>
#include <gnome--/dialog.h>
#include <typeinfo>
#include <iostream>

extern "C" {
#include "interface.h"
#include "support.h"
}

using namespace std;
using namespace Xgsm;
using namespace SigC;
using namespace gsmlib;

// SendSMS members

void SendSMS::sendNextMessage()
{
  // check whether we are finished
  if (_nextMessageIndex < _textEntry->get_length())
  {
    sensitive(false);
    string number = ((Gtk::Entry*)_numberEntry->gtk_entry())->get_text();

    // output progess message
    if (_countSMS > 1)
      mainWindow->message(stringPrintf(_("Sending SMS #%d of %d to %s..."),
                                       ++_messageCount, _countSMS,
                                       number.c_str()), false);
    else
      mainWindow->message(stringPrintf(_("Sending SMS to %s..."),
                                       number.c_str()), false);

    // prepare and send message
    string message;

    if (_nextMessageIndex + _maxTextLength >  _textEntry->get_length())
    {
      message =
        _textEntry->get_chars(_nextMessageIndex, _textEntry->get_length());
      _nextMessageIndex = _textEntry->get_length();
    }
    else
    {
      message = _textEntry->get_chars(_nextMessageIndex,
                                      _nextMessageIndex + _maxTextLength);
      _nextMessageIndex += _maxTextLength;
    }
    
    Ref<SMSSubmitMessage> submitSMS =
      new SMSSubmitMessage(message, number);
    _dev->request(new SendSMSRequest(submitSMS,
                                     slot(this, &SendSMS::onSendDone)));
  }
  else
    sensitive(true);
}

void SendSMS::updateCounters()
{
  _charsLeftLabel->set_text(
    intToStr(_textEntry->get_length() == 0 ? 
             _maxTextLength :
             (_maxTextLength - _textEntry->get_length() % _maxTextLength)
             % _maxTextLength));

  int currentSMS = (_textEntry->get_position() + _maxTextLength - 1)  /
    _maxTextLength;
  _countSMS = (_textEntry->get_length() + _maxTextLength - 1) /
    _maxTextLength;
  if (currentSMS == 0 && _countSMS > 0)
    currentSMS = 1;

  _numberOfMessages->set_text(intToStr(currentSMS) + " / " +
                              intToStr(_countSMS));
}

void SendSMS::sensitive(bool deviceIdle)
{
  if (_sendButton == NULL)
    return;                     // not initialized yet

  _sendButton->set_sensitive(deviceIdle && ! _dev.isnull());
}

void SendSMS::onSendDone(ResponseRef response)
{
  if (typeid(response()) == typeid(ErrorResponse))
  {
    // handle error response
    Gnome::Dialogs::error
      (string(_("Error sending SMS \n(")) +
       ((ErrorResponse&)response()).exception().what() + ")")->run_and_close();
    closeDevice();
  }
  else
  {
    assert(typeid(response()) == typeid(SendSMSResponse));
    sendNextMessage();
  }
}

SendSMS::SendSMS() :
  Gtk::Widget(create_send_sms_window()), DeviceHelper("default device"),
  ToplevelHelper(this), _colorRefreshPosition(-1), _countSMS(0)
{
  // find out maximum SMS text length
  switch (config.getCharacterEncoding())
  {
  case DCS_DEFAULT_ALPHABET:
    _maxTextLength = 160;
    break;
  case DCS_EIGHT_BIT_ALPHABET:
    _maxTextLength = 140;
    break;
  case DCS_SIXTEEN_BIT_ALPHABET:
    _maxTextLength = 70;
    break;
  }

  // set sizes
  int width, height;
  if (config.getNewSmsSizes(width, height))
    ((Gtk::Window*)this)->set_default_size(width, height);

  // lookup widgets, connect callbacks
  CONNECT_SIGNAL("send_sms_phonebook_button", GtkButton,
                 GTK_IS_BUTTON, clicked, this, SendSMS::onPhonebookButton);

  LOOKUP_WIDGET(_sendButton, "send_sms_send_button", GtkButton,
                GTK_IS_BUTTON, this);
  _sendButton->clicked.connect(slot(this, &SendSMS::onSendButton));

  CONNECT_SIGNAL("send_sms_cancel_button", GtkButton,
                 GTK_IS_BUTTON, clicked, this, SendSMS::onCancelButton);
  CONNECT_SIGNAL("send_sms_help_button", GtkButton,
                 GTK_IS_BUTTON, clicked, this, SendSMS::onHelpButton);
  LOOKUP_WIDGET(_textEntry, "send_sms_text_entry", GtkText,
                GTK_IS_TEXT, this);
  _textEntry->insert_text.connect(slot(this, &SendSMS::onInsertText));
  _textEntry->delete_text.connect(slot(this, &SendSMS::onDeleteText));
  _textEntry->button_press_event.
    connect_after(slot(this, &SendSMS::onButtonPress));
  _textEntry->key_press_event.
    connect_after(slot(this, &SendSMS::onKeyPress));
  _textEntry->changed.connect(slot(this, &SendSMS::onTextChanged));

  LOOKUP_WIDGET(_numberEntry, "send_sms_number_entry", GnomeEntry,
                GNOME_IS_ENTRY, this);
  ((Gtk::Entry*)_numberEntry->gtk_entry())->
    insert_text.connect(slot(this, &SendSMS::onInsertNumber));
  _numberEntry->load_history();

  LOOKUP_WIDGET(_charsLeftLabel, "send_sms_characters_left", GtkLabel,
                GTK_IS_LABEL, this);

  LOOKUP_WIDGET(_numberOfMessages, "send_sms_number_of_messages", GtkLabel,
                GTK_IS_LABEL, this);

  configure_event.connect(slot(this, &SendSMS::onConfigureEvent));
  delete_event.connect(slot(this, &SendSMS::onDelete));

  updateCounters();

  // open device
  openDefaultDevice();

  show_all();
}

void SendSMS::onPhonebookButton()
{
  // FIXME: register for automatic entry pasting
  mainWindow->openDefaultPhonebook();
}

void SendSMS::onSendButton()
{
  _numberEntry->save_history();
  _nextMessageIndex = 0;
  _messageCount = 0;
  sendNextMessage();
}

void SendSMS::onCancelButton()
{
  destroy();
}

void SendSMS::onHelpButton()
{
  // FIXME
}

void SendSMS::onDestroy()
{
  DEBUG_START(1);
  cout << "SendSMS::onDestroy() called" << endl;
  DEBUG_END;
  _destroyed = true;
  delete this;
}

gint SendSMS::onDelete(GdkEventAny *ev)
{
  DEBUG_START(1);
  cout << "SendSMS::onDelete() called" << endl;
  DEBUG_END;
  _destroyed = true;
  delete this;
  return TRUE;
}

void SendSMS::onTextChanged()
{
  updateCounters();

  if (_colorRefreshPosition != -1)
  {
#ifndef NDEBUG
    unsigned int saveTextLen = _textEntry->get_length();
#endif
    Gtk::Text_Helpers::Context gc;
    gc.set_foreground(Gdk_Color(config.getLongSMSColor()));

    _textEntry->freeze();
    int savePos = _textEntry->get_position();

    // align to SMS boundary
    _colorRefreshPosition -= (_colorRefreshPosition % _maxTextLength);

    // fix colors after refresh point
    bool alternateColor = (_colorRefreshPosition / _maxTextLength) & 1;

    string text = _textEntry->get_chars(_colorRefreshPosition, -1);
    _textEntry->set_point(_colorRefreshPosition);
    _textEntry->forward_delete(_textEntry->get_length() -
                               _colorRefreshPosition);

    for (int i = 0; i < (int)text.length(); i += _maxTextLength)
    {
      if (alternateColor)
        _textEntry->insert(gc, text.substr(i, _maxTextLength));
      else
        _textEntry->insert(text.substr(i, _maxTextLength));
      alternateColor = ! alternateColor;
    }
    _textEntry->thaw();
    gtk_editable_set_position(((Gtk::Editable*)_textEntry)->gtkobj(), savePos);
    _textEntry->set_point(_textEntry->get_position());

    _colorRefreshPosition = -1;

    assert(saveTextLen == _textEntry->get_length());
  }
}

void SendSMS::onInsertNumber(const gchar *text, gint textLen, gint *position)
{
  try
  {
    checkTextAndTelephone("", string(text, textLen));
  }
  catch (GsmException &ge)
  {
    // user tried to insert invalid character(s)
    gdk_beep();
    if (textLen > 1)
      Gnome::Dialogs::error(ge.what())->run_and_close();
    mainWindow->message(ge.what(), true);

    // skip default handler
    gtk_signal_emit_stop_by_name((GtkObject*)
                                 _numberEntry->gtk_entry()->gtkobj(),
                                 "insert-text");
  }
}

gint SendSMS::onConfigureEvent(GdkEventConfigure* event)
{
  config.setNewSmsSizes(event->width, event->height);
  return TRUE;
}

void SendSMS::onInsertText(const gchar *text, gint textLen, gint *position)
{
  // skip default handler
  gtk_signal_emit_stop_by_name((GtkObject*)_textEntry->gtkobj(),
                               "insert-text");

  // ignore insertions that make the text too long
  if (! config.getLongSMS() &&
      (int)_textEntry->get_length() + textLen > _maxTextLength)
  {
    gdk_beep();
    string message =
      stringPrintf(_("Maximum SMS message length is %d characters"),
                   _maxTextLength);
    if (textLen > 1)
      Gnome::Dialogs::error(message)->run_and_close();
    mainWindow->message(message, true);
    return;
  }

  string t(text, textLen);

  // check for illegal characters if 7-bit default encoding is used
  if (config.getCharacterEncoding() == DCS_DEFAULT_ALPHABET)
  {
    t = gsmToLatin1(latin1ToGsm(t));
    
    if (string(text, textLen) != t) // check for non-gsm characters
      if (textLen == 1)
      {
        // if a single character is entered, ignore an invalid character
        gdk_beep();
        string s(text, textLen);
        string message = stringPrintf(_("Non-GSM character in '%s'"),
                                      s.c_str());
        if (textLen > 1)
          Gnome::Dialogs::error(message)->run_and_close();
        mainWindow->message(message, true);
        return;
      }
      else
         mainWindow->message(_("Warning: Inserting non-GSM characters"), true);
  }

  // insert text ourselves
  int startInsertPos = *position;

  int nextColorLen = (startInsertPos % _maxTextLength) + textLen;
  if (nextColorLen > _maxTextLength)
    nextColorLen -= _maxTextLength;
  else
    nextColorLen = 0;
  bool alternateColor = (startInsertPos / _maxTextLength) & 1;

  Gtk::Text_Helpers::Context gc;
  gc.set_foreground(Gdk_Color(config.getLongSMSColor()));

  _textEntry->set_point(startInsertPos);
  if (alternateColor)
  {
    _textEntry->insert(gc, t.substr(0, t.length() - nextColorLen));
    _textEntry->insert(t.substr(t.length() - nextColorLen, nextColorLen));
  }
  else
  {
    _textEntry->insert(t.substr(0, t.length() - nextColorLen));
    _textEntry->insert(gc, t.substr(t.length() - nextColorLen, nextColorLen));
  }

  // update position und color refresh position
  *position = _textEntry->get_point();
  if (textLen > _maxTextLength)
    _colorRefreshPosition = startInsertPos;
  else if (*position != (int)_textEntry->get_length())
    _colorRefreshPosition = *position;
}

void SendSMS::onDeleteText(const gint startPos, const gint endPos)
{
  // do not refresh the alternating colors if deleting at the end of the text
  if(endPos != -1 && endPos != (int)_textEntry->get_length())
    _colorRefreshPosition = startPos;
}

gint SendSMS::onButtonPress(GdkEventButton*)
{
  updateCounters();
  return TRUE;
}

gint SendSMS::onKeyPress(GdkEventKey*)
{
  updateCounters();
  return TRUE;
}
