#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xatom.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Dialog.h>

#include "config.h"

#include "toolbox.h"
#include "cfgfile.h"
#include "complete.h"
#include "x11.h"
#include "ipc.h"
#include "pcd.h"
#include "xpcd.h"

/* ---------------------------------------------------------------------- */

static void     close_all_AC(Widget, XEvent *, String *, Cardinal *);
static void     ipc_AC(Widget, XEvent *, String *, Cardinal *);

static void     handle_arg(char *arg);
static void     setviewer_CB(Widget, XtPointer, XtPointer);

/* ---------------------------------------------------------------------- */

Widget          app_shell;
XtAppContext    app_context;
Display        *dpy;

int             use_grays = 0, force_grays = 0, load_grays = 0;
int             tiny_turn = 0;
Widget          wtiff_compress,wuse_grays, wload_grays;
Widget          wtiny_turn, wswap_dragdir;

int             viewer_count = 2;
int             viewer_current = 0;
Widget          viewer_menu[12];
char           *viewer_title[12];
char           *viewer_cmd[12] =
{VIEWER_INTERN, VIEWER_GIMP};

int             jpeg_quality = 75;
int             tiff_compress = 1;
char           *cdrom;

Window          gimp_win = 0;
char            gimp_sock[256];

static XtActionsRec actionTable[] =
{
    {"CloseAll", close_all_AC},
    {"CloseFile", close_file_AC},
    {"Complete", CompleteAction},
    {"KbdScroll", kbd_scroll_viewport_AC},
    {"DragScroll", drag_scroll_viewport_AC},
    {"Help", help_AC},
    {"SetShadowWidth", set_shadowWidth_AC},
    {"Ipc", ipc_AC}
};

static String   fallback_res[] =
{
#if 0
#include "app-defaults.h"
#else
"xpcd.main.body.label: Oops: the application defaults are not installed",
#endif
    NULL
};

/* ---------------------------------------------------------------------- */

void            parse_viewer_config(char *cfg);

struct CONFIG_FILE config_file[] =
{
    {CFG_BOOL, "display_gray", &use_grays, NULL},
    {CFG_BOOL, "load_gray", &load_grays, NULL},
    {CFG_BOOL, "turn_thumbnails", &tiny_turn, NULL},
    {CFG_BOOL, "swap_dragdir", &swap_dragdir, NULL},
    {CFG_BOOL | CFG_RDONLY, "disable_color", &force_grays, NULL},
    {CFG_INT, "jpeg_quality", &jpeg_quality, NULL, 5, 100},
    {CFG_BOOL, "tiff_compress", &tiff_compress, NULL},
    {CFG_INT, "default_viewer", &viewer_current, NULL, 0, 11},
    {CFG_INT | CFG_RDONLY, "num_grays", &x11_grays, NULL, 4, 64},
    {CFG_STRING, "cdrom", &cdrom, NULL},
    {CFG_CALLBACK, "viewer", NULL, parse_viewer_config},
    {CFG_LAST, NULL, NULL, NULL}
};

void
parse_viewer_config(char *cfg)
{
    char            title[64], cmdline[128];

    if (2 != sscanf(cfg, " %63[^,], %127[^\n]", title, cmdline))
	return;
    viewer_title[viewer_count] = strdup(title);
    viewer_cmd[viewer_count] = strdup(cmdline);
    viewer_count++;
}

/* ---------------------------------------------------------------------- */

Boolean
exit_WP(XtPointer client_data)
{
    config_write(".xpcdrc", SYSRC, config_file);
    exit(0);
    return TRUE;
}

static void
close_all_AC(Widget widget, XEvent * event,
	     String * params, Cardinal * num_params)
{
    close_all_files();
    /* TODO: close viewer windows */
    XtAppAddWorkProc(app_context, exit_WP, NULL);
}

static void
ipc_AC(Widget widget, XEvent * event,
       String * params, Cardinal * num_params)
{
    Atom            type;
    int             format, i;
    unsigned long   nitems, bytesafter;
    unsigned char  *args = NULL;

    if (event->type == PropertyNotify &&
	event->xproperty.atom == xpcd_cmd &&
	Success == XGetWindowProperty(dpy,
				      event->xproperty.window,
				      event->xproperty.atom,
				      0, (65536 / sizeof(long)),
				      True, XA_STRING,
				    &type, &format, &nitems, &bytesafter,
				      &args) &&
	nitems != 0) {
	if (0 == strcmp(args, IPC_PASS_ARGV)) {
	    for (i = strlen(args) + 1; i < nitems; i += strlen(args + i) + 1) {
		handle_arg(args + i);
	    }
	} else {
	    fprintf(stderr, "ipc Oops: ");
	    for (i = 0; i < nitems; i += strlen(args + i) + 1)
		fprintf(stderr, "%s ", args + i);
	    fprintf(stderr, "\n");
	}
	XFree(args);
    }
}

void
gimp_on(Display * dpy, Window win)
{
    Atom            type;
    int             format;
    unsigned long   nitems, bytesafter;
    unsigned char  *sock = NULL;

    XSelectInput(dpy, win, StructureNotifyMask | PropertyChangeMask);
#if 0
    XSync(dpy, False);
#endif
    XGetWindowProperty(dpy, win, xpcd_gate,
		       0, (65536 / sizeof(long)),
		       False, XA_STRING,
		       &type, &format, &nitems, &bytesafter,
		       &sock);

    if (sock) {
#if 0
	fprintf(stderr, "gimp_on: xpcd-gate is listening on %s\n", sock);
#endif
	XtVaSetValues(viewer_menu[1], XtNsensitive, True, NULL);
	strcpy(gimp_sock, sock);
    }
#if 0
    else
	fprintf(stderr, "gimp_on: failure\n");
#endif
}

void
gimp_off()
{
#if 0
    fprintf(stderr, "gimp_off: xpcd-gate exited\n");
#endif
    XtVaSetValues(viewer_menu[1], XtNsensitive, False, NULL);
    gimp_win = 0;
    if (viewer_current == 1)
	setviewer_CB(viewer_menu[0], 0, NULL);
}

static int
x11_error_dev_null(Display * dpy, XErrorEvent * event)
{
    return 0;
}

void
ipc_rootwin(XEvent * event)
{
    Atom            type;
    int             format;
    unsigned long   nitems, bytesafter;
    unsigned char  *name = NULL;
    XtPointer       old_handler;

    old_handler = XSetErrorHandler(x11_error_dev_null);

    if (event->type == CreateNotify &&
	Success == XGetWindowProperty(event->xcreatewindow.display,
				      event->xcreatewindow.window,
				      client_name,
				      0, (65536 / sizeof(long)),
				      False, XA_STRING,
				    &type, &format, &nitems, &bytesafter,
				      &name) &&
	nitems != 0) {
#if 0
	fprintf(stderr, "new window: %s\n", name);
#endif

	if (0 == strcmp(name, "xpcd-gate")) {
	    gimp_win = event->xcreatewindow.window;
	    gimp_on(event->xcreatewindow.display, event->xcreatewindow.window);
	}
    }
    XSetErrorHandler(old_handler);
}

/* ---------------------------------------------------------------------- */

void
opencd_ok_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    char           *value, *h, *overview;
    Widget          dialog = XtParent(widget);

    value = XawDialogGetValueString(dialog);

    value = tilde_expand(value);
    if (cdrom)
	free(cdrom);
    cdrom = strdup(value);
    overview = malloc(strlen(value) + 25);
    strcpy(overview, value);
    free(value);
    h = overview + strlen(overview);
    if (*(h - 1) == '/')
	h--;
    strcpy(h, "/photo_cd/overview.pcd");

    XtDestroyWidget(XtParent(dialog));
    XtAppAddWorkProc(app_context, open_file_WP, (XtPointer) overview);
}

void
opencd_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    get_user_string(app_shell, "str_opencd_title", "str_opencd_label",
		    cdrom, opencd_ok_CB, NULL);
}

void
openfile_ok_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    char           *value, *filename;
    Widget          dialog = XtParent(widget);

    value = XawDialogGetValueString(dialog);

    filename = tilde_expand(value);

    XtDestroyWidget(XtParent(dialog));
    XtAppAddWorkProc(app_context, open_file_WP, (XtPointer) filename);
}

void
openfile_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    get_user_string(app_shell, "str_openfile_title", "str_openfile_label",
		    "", openfile_ok_CB, NULL);
}

void
setviewer_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    int             i, n = (int) client_data;

    viewer_current = n;
    for (i = 0; i < viewer_count; i++)
	XtVaSetValues(viewer_menu[i], XtNleftBitmap,
		      (viewer_menu[i] == widget) ? bm_yes : bm_no,
		      NULL);
}

void
toggle_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    int            *flag;

    flag = client_data;
    *flag = !*flag;
    XtVaSetValues(widget, XtNleftBitmap, *flag ? bm_yes : bm_no, NULL);
}

void
qjpeg_ok_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    char           *value;
    Widget          dialog = XtParent(widget);

    value = XawDialogGetValueString(dialog);

    jpeg_quality = atoi(value);

    XtDestroyWidget(XtParent(dialog));
}

void
qjpeg_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    char            h[10];

    sprintf(h, "%d", jpeg_quality);
    get_user_string(app_shell, "str_qjpeg_title", "str_qjpeg_label",
		    h, qjpeg_ok_CB, NULL);
}

void
help_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    XtCallActionProc(app_shell, "Help", NULL, NULL, 0);
}

void
about_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    tell_user(app_shell, "str_about_title", "str_about_label");
}

/* ---------------------------------------------------------------------- */

void
exit_CB(Widget widget, XtPointer client_data, XtPointer calldata)
{
    XtCallActionProc(app_shell, "CloseAll", NULL, NULL, 0);
}

/* ---------------------------------------------------------------------- */

void
create_mainwindow(Widget shell)
{
    Widget          main, menubar, menu, body;
    char            name[8];
    int             i;

    /* main window */
    main = XtVaCreateManagedWidget("main", panedWidgetClass, shell,
				   NULL);

    menubar = XtVaCreateManagedWidget("menu", boxWidgetClass, main,
				      XtNallowResize, False,
				      XtNshowGrip, False,
				      XtNskipAdjust, True,
				      XtNorientation, XtEhorizontal,
				      NULL);

    body = XtVaCreateManagedWidget("body", labelWidgetClass, main,
				   XtNshowGrip, False,
				   NULL);
#if 0
    stat = XtVaCreateManagedWidget("stat", labelWidgetClass, main,
				   XtNallowResize, False,
				   XtNshowGrip, False,
				   XtNskipAdjust, True,
				   NULL);
#endif

    /* File menu */
    menu = add_pulldown_menu(menubar, "file");
    add_menu_entry(menu, "opencd", opencd_CB, NULL);
    add_menu_entry(menu, "openf", openfile_CB, NULL);
    add_menu_sep(menu, "sep1");
    add_menu_entry(menu, "quit", exit_CB, NULL);

    /* for viewers */
    menu = add_pulldown_menu(menubar, "view");
    viewer_menu[0] = add_menu_entry(menu, "own", setviewer_CB, (XtPointer) 0);
    viewer_menu[1] = add_menu_entry(menu, "gimp", setviewer_CB, (XtPointer) 1);
    XtVaSetValues(viewer_menu[1], XtNsensitive, False, NULL);
    add_menu_sep(menu, "sep");

    strcpy(name, "ext?");
    for (i = 2; i < viewer_count; i++) {
	name[3] = '0' + i - 2;
	viewer_menu[i] =
	    add_menu_entry(menu, viewer_title[i], setviewer_CB, (XtPointer) i);
    }

    /* Options menu */
    menu = add_pulldown_menu(menubar, "opt");
#ifdef HAVE_LIBJPEG
    add_menu_entry(menu, "qjpeg", qjpeg_CB, NULL);
#endif
#ifdef HAVE_LIBTIFF
    wtiff_compress = add_menu_entry(menu, "ctiff", toggle_CB, &tiff_compress);
#endif
    wuse_grays = add_menu_entry(menu, "gray", toggle_CB, &use_grays);
    wload_grays = add_menu_entry(menu, "lgray", toggle_CB, &load_grays);
    wtiny_turn = add_menu_entry(menu, "tturn", toggle_CB, &tiny_turn);
    wswap_dragdir = add_menu_entry(menu, "swapd", toggle_CB, &swap_dragdir);

    /* help */
    menu = add_pulldown_menu(menubar, "help");
    add_menu_entry(menu, "help", help_CB, NULL);
    add_menu_entry(menu, "about", about_CB, NULL);
    XtInstallAllAccelerators(main, shell);
}

/* ---------------------------------------------------------------------- */

void
got_sigchild(int sig)
{
    int             stat, pid;

    for (;;) {
	pid = waitpid(0, &stat, WNOHANG);
	if (pid <= 0)
	    break;
	if (WIFSIGNALED(stat) && WTERMSIG(stat) != SIGTERM) {
	    fprintf(stderr, "%d: Oops, killed by signal %d (%s)\n",
		    pid, WTERMSIG(stat), strsignal(WTERMSIG(stat)));
	}
    }
    signal(sig, got_sigchild);
}

#ifdef __linux__
void
got_sigbus(int sig)
{
    fprintf(stderr,
	    "your fault (core dumped)\n"
	    "\n"
	  "Apply the kernel patch in the linux directory of the source\n"
	    "archive, recompile the kernel, reboot and try again\n"
	    "\n"
       "If this does'nt help, your PhotoCD has a MEDIUM ERROR. I can't\n"
	    "catch this, sorry\n");
    _exit(0);
}

#endif

static void
handle_arg(char *arg)
{
    struct stat     st;
    char           *overview, *h;

    if (-1 == stat(arg, &st)) {
	PERROR(arg);
	return;
    }
    if (S_ISREG(st.st_mode)) {
	XtAppAddWorkProc
	    (app_context, open_file_WP,
	     (XtPointer) strcpy(malloc(strlen(arg) + 1), arg));
    } else if (S_ISDIR(st.st_mode)) {
	overview = malloc(strlen(arg) + 25);
	strcpy(overview, arg);
	h = overview + strlen(overview);
	if (*(h - 1) == '/')
	    h--;
	strcpy(h, "/photo_cd/overview.pcd");
	XtAppAddWorkProc(app_context, open_file_WP, (XtPointer) overview);
    } else {
	fprintf(stderr, "Oops: what is %s?\n", arg);
    }
}

int
main(int argc, char *argv[])
{
    Window          running_xpcd, gate;
    int             i;

    app_shell = XtAppInitialize(&app_context, "Xpcd-2",
				NULL, 0,
				&argc, argv,
				fallback_res,
				NULL, 0);
    dpy = XtDisplay(app_shell);

    /* check if there is already a xpcd running... */
    ipc_init_atoms(dpy);
    if (0 != (running_xpcd =
	      ipc_find_window(dpy, xpcd_ver))) {
	ipc_pass_cmd(dpy, running_xpcd, IPC_PASS_ARGV,
		     argc - 1, argv + 1);
	XFlush(dpy);
	exit(0);
    }
    /* signal stuff  XXX: XtAppAddSignal */
    signal(SIGCHLD, got_sigchild);
#ifdef __linux__
    signal(SIGBUS, got_sigbus);
#endif

    /* read config file(s) */
    cdrom = strdup("/cdrom");
    config_read(".xpcdrc", SYSRC, config_file);

    /* set up Xaw/Xt/X11 */
    XtAppAddActions(app_context, actionTable,
		    sizeof(actionTable) / sizeof(XtActionsRec));
    x11_color_init(app_shell, &force_grays);

    create_mainwindow(app_shell);
    XSelectInput(dpy,
		 RootWindowOfScreen(XtScreen(app_shell)),
		 SubstructureNotifyMask | PropertyChangeMask);
    XtRealizeWidget(app_shell);
    create_pointers(app_shell);
    create_bitmaps(app_shell);
    XDefineCursor(dpy, XtWindow(app_shell), left_ptr);
    XSetWMProtocols(dpy, XtWindow(app_shell),
		    &wm_close, 1);
    XChangeProperty(dpy, XtWindow(app_shell),
		    xpcd_ver, XA_STRING,
		    8, PropModeReplace,
		    XPCD_VERSION, strlen(XPCD_VERSION) + 1);

    /* set options in menu */
    if (force_grays) {
	config_file[0].what |= CFG_RDONLY;
	use_grays = 1;
    }
    if (viewer_current < viewer_count)
	setviewer_CB(viewer_menu[viewer_current],
		     (XtPointer) viewer_current, NULL);
    else
	setviewer_CB(viewer_menu[0], 0, NULL);
#ifdef HAVE_LIBTIFF
    XtVaSetValues(wtiff_compress,
		  XtNleftBitmap, tiff_compress ? bm_yes : bm_no,
		  NULL);
#endif
    XtVaSetValues(wuse_grays,
		  XtNleftBitmap, use_grays ? bm_yes : bm_no,
		  XtNsensitive, force_grays ? False : True,
		  NULL);
    XtVaSetValues(wload_grays,
		  XtNleftBitmap, load_grays ? bm_yes : bm_no,
		  NULL);
    XtVaSetValues(wtiny_turn,
		  XtNleftBitmap, tiny_turn ? bm_yes : bm_no,
		  NULL);
    XtVaSetValues(wswap_dragdir,
		  XtNleftBitmap, swap_dragdir ? bm_yes : bm_no,
		  NULL);

    /* process args */
    for (i = argc - 1; i > 0; i--)
	handle_arg(argv[i]);

    /* look for xpcd-gate */
    if (0 != (gate = ipc_find_window(dpy, xpcd_gate)))
	gimp_on(dpy, gate);
    else
	gimp_off();

    /* main loop */
#if 0
    XtAppMainLoop(app_context);
#else
    for (;;) {
	XEvent          event;

	XtAppNextEvent(app_context, &event);
	if (event.type == CreateNotify &&
	    event.xcreatewindow.parent == RootWindowOfScreen(XtScreen(app_shell))) {
	    ipc_rootwin(&event);
	} else if (gimp_win && event.type == PropertyNotify &&
		   event.xproperty.window == gimp_win &&
		   event.xproperty.atom == xpcd_gate) {
	    gimp_on(dpy, event.xproperty.window);
	} else if (gimp_win && event.type == DestroyNotify &&
		   event.xdestroywindow.window == gimp_win) {
	    gimp_off();
	} else
	    XtDispatchEvent(&event);
    }
#endif
    /* keep compiler happy */
    return 0;
}
