/**
 * A client-side 802.1x implementation 
 *
 * This code is released under both the GPL version 2 and BSD licenses.
 * Either license may be used.  The respective licenses are found below.
 *
 * Copyright (C) 2002 Bryan D. Payne & Nick L. Petroni Jr.
 * All Rights Reserved
 *
 * --- GPL Version 2 License ---
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * --- BSD License ---
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  - All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *       This product includes software developed by the University of
 *       Maryland at College Park and its contributors.
 *  - Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*******************************************************************
 * File: cardif_linux_ssids.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux_ssids.c,v 1.6 2004/08/19 02:28:07 chessing Exp $
 * $Date: 2004/08/19 02:28:07 $
 * $Log: cardif_linux_ssids.c,v $
 * Revision 1.6  2004/08/19 02:28:07  chessing
 * First piece of WPA patch.  (The patch is growing fast, and this commit is to save what is done so far.)
 *
 * Revision 1.5  2004/08/16 23:36:28  chessing
 * Cosmetic fixes and trapping of an error that popped up.
 *
 * Revision 1.4  2004/08/16 01:56:44  chessing
 *
 * A few more updates to last nights patch.
 *
 * Revision 1.3  2004/08/15 04:29:35  chessing
 *
 * Completed support for scanning and joining wireless networks.  If your interface isn't configured, we will scan to gather information about all known wireless networks.  If a network in the list has a configuration, we will join it.  (Including turning on needed encryption.)  There are still a few rough edges that need to be hammered out.  But the overall user experience should be improved.
 *
 * Revision 1.2  2004/08/12 01:40:06  chessing
 *
 * Added "association" option to the global configuration file sections.  It can be set to either auto, or manual.  Setting "association" to auto means that XSupplicant will attempt to associate to a network with the lowest value in the "priority" setting.  The default for "association" is auto.  The other option added is "priority" which is a numeric value from 1..256.  A value of 1 is the highest priority, and will be the first network chosen.  (Assuming the network can be found in a scan.)  The default value for "priority" is 0, which means that network will be checked last.   --  Although the options are now in the config file, the actual functionality for auto association/priority is not yet complete.  (So, basically these options don't do anything yet. ;)
 *
 * Revision 1.1  2004/08/05 23:56:21  chessing
 *
 * Added basic support for scanning for broadcast SSIDs.  This is another step closer to WPA/11i support. ;)
 *
 *
 *******************************************************************/

#include <string.h>
#include <strings.h>

#include "../../profile.h"
#include "../../config.h"
#include "cardif.h"
#include "../../xsup_debug.h"
#include "../../xsup_err.h"
#include "cardif_linux_ssids.h"

/*******************************************************************
 *
 * Dump everything we know about the SSIDs in our catalog.
 *
 *******************************************************************/
void cardif_linux_ssids_dump(struct interface_data *intdata)
{
  struct ssids_list *curlist;

  debug_printf(DEBUG_INT, "Dumping SSID information structure.\n");

  curlist = (struct ssids_list *)intdata->ssid_list;

  while (curlist != NULL)
    {
      debug_printf(DEBUG_INT, "SSID : %s    Flags : %02X\n",
		   curlist->ssidName, curlist->flags);
      
      curlist = curlist->next;
    }
}

/*******************************************************************
 *
 * Clear out EVERYTHING that is included in the SSID list.
 *
 *******************************************************************/
void cardif_linux_ssids_clear(struct interface_data *intdata)
{
  struct ssids_list *curlist, *toclear;

  curlist = (struct ssids_list *)intdata->ssid_list;

  if (curlist == NULL)
    {
      debug_printf(DEBUG_EVERYTHING, "No SSIDs to clear!\n");
      return;
    }

  while (curlist != NULL)
    {
      toclear = curlist;
      curlist = curlist->next;

      if (toclear->ssidName != NULL)
	{
	  free(toclear->ssidName);
	  toclear->ssidName = NULL;
	}

      toclear->flags = 0;
      free(toclear);
    }

  intdata->ssid_list = NULL;
}

/*******************************************************************
 *
 * Add an SSID to the list.  (We will check to make sure that it doesn't
 * exist first.)
 *
 *******************************************************************/
void cardif_linux_ssids_add(struct interface_data *intdata, char *ssidName,
			    unsigned char flags, char *wpaie, int wpaielen,
			    char *rsnie, int rsnielen)
{
  struct ssids_list *curlist;

  curlist = (struct ssids_list *)intdata->ssid_list;

  if (cardif_linux_ssids_included(intdata, ssidName) == FALSE)
    {
      if (curlist == NULL)
	{
	  // This is our first addition.  So, it will be slightly different.
	  intdata->ssid_list = (void *)malloc(sizeof(struct ssids_list));
	  if (intdata->ssid_list == NULL)
	    {
	      debug_printf(DEBUG_EVERYTHING, "Memory allocation error!\n");
	      return;
	    }
	  curlist = (struct ssids_list *)intdata->ssid_list;
	} else {
	  // Find the end of the list, and work from there.
	  while (curlist->next != NULL)
	    {
	      curlist = curlist->next;
	    }

	  curlist->next = (struct ssids_list *)malloc(sizeof(struct ssids_list));
	  if (curlist->next == NULL)
	    {
	      debug_printf(DEBUG_EVERYTHING, "Memory allocation error!\n");
	      return;
	    }

	  curlist = curlist->next;
	}

      curlist->ssidName = (char *)malloc(strlen(ssidName)+1);
      if (curlist->ssidName == NULL)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for curlist->ssidName in %s!\n", __FUNCTION__);
	  curlist->flags = 0;
	  curlist->next = NULL;
	  return;
	}

      bzero(curlist->ssidName, strlen(ssidName)+1);

      strcpy(curlist->ssidName, ssidName);
      curlist->flags = flags;

      if (wpaie != NULL)
	{
	  curlist->wpa_ie_len = wpaielen;
	  curlist->wpa_ie = (char *)malloc(wpaielen);
	  if (curlist->wpa_ie == NULL)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for curlist->wpa_ie in %s!\n", __FUNCTION__);
	      return;
	    }

	  bzero(curlist->wpa_ie, wpaielen);
	  memcpy(curlist->wpa_ie, wpaie, wpaielen);
	  debug_printf(DEBUG_INT, "WPA IE = ");
	  debug_hex_printf(DEBUG_INT, curlist->wpa_ie, curlist->wpa_ie_len);
	}

      if (rsnie != NULL)
	{
	  curlist->rsn_ie_len = rsnielen;
	  curlist->rsn_ie = (char *)malloc(rsnielen);
	  if (curlist->rsn_ie == NULL)
	    {
	      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for curlist->rsn_ie in %s!\n", __FUNCTION__);
	      return;
	    }

	  bzero(curlist->rsn_ie, rsnielen);
	  memcpy(curlist->rsn_ie, rsnie, rsnielen);
	  debug_printf(DEBUG_INT, "RSN IE = ");
	  debug_hex_printf(DEBUG_INT, curlist->rsn_ie, curlist->rsn_ie_len);
	}

      curlist->next = NULL;
    }
}

/*******************************************************************
 *
 * Get information about the wireless network we are using.
 *
 *******************************************************************/
struct ssids_list *cardif_linux_ssids_find(struct interface_data *intdata)
{
  struct ssids_list *curlist;

  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid data passed to %s!\n", __FUNCTION__);
      return NULL;
    }

  curlist = (struct ssids_list *)intdata->ssid_list;

  while ((curlist != NULL) && (strcmp(intdata->cur_essid, curlist->ssidName) == 0))
    {
      curlist = curlist->next;
    }

  return curlist;
}

/*******************************************************************
 *
 * Check to see if the SSID named is already included in our list.
 * Return TRUE if it is, FALSE if it isn't.
 *
 *******************************************************************/
int cardif_linux_ssids_included(struct interface_data *intdata, char *ssidName)
{
  struct ssids_list *curlist = NULL;

  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data passed in to cardif_linux_ssids_included()!\n");
      return TRUE;
    }

  if (ssidName == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid ssidName in %s!\n", __FUNCTION__);
      return TRUE;
    }

  curlist = (struct ssids_list *)intdata->ssid_list;

  if (curlist == NULL) return FALSE;  

  while ((curlist != NULL) && (curlist->ssidName != NULL) &&
	 (strcmp(ssidName, curlist->ssidName) != 0))
    {
      curlist = curlist->next;
    }

  if ((curlist != NULL) && (curlist->ssidName != NULL) &&
      (strcmp(ssidName, curlist->ssidName) == 0))
    {
      debug_printf(DEBUG_EVERYTHING, "The SSID was already in the list!\n");
      return TRUE;
    } else {
      debug_printf(DEBUG_EVERYTHING, "The SSID was NOT already in the list!\n");
      return FALSE;
    }
}

/************************************************************************
 *
 *  A scan was completed, and we have some data about the SSIDs/APs that we
 *  can see.  So, find the SSID with the best priority and try to 
 *  associate to it.
 *
 ************************************************************************/
void cardif_linux_ssids_join_network(struct interface_data *intdata)
{
  struct ssids_list *best_choice=NULL, *curlist;
  int priority, highpri = 257;

  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data passed in to %s!\n",
		   __FUNCTION__);

      // Clear the scan done flag.
      intdata->flags &= (~SCAN_DONE);
      return;
    }

  curlist = (struct ssids_list *)intdata->ssid_list;

  if (curlist == NULL) 
    {
      intdata->flags &= (~SCAN_DONE);
      return;  
    }

  while (curlist != NULL)
    {
      priority = config_get_priority(curlist->ssidName);

      if ((priority < highpri) || ((highpri == 0) && (priority > 0)))
	{
	  if (!((priority == 0) && (highpri != 257)))
	    {
	      highpri = priority;
	      best_choice = curlist;
	      debug_printf(DEBUG_INT, "Highest priority = %d with ESSID = %s\n",
			   highpri, best_choice->ssidName);
	    }
	} 

      curlist = curlist->next;
    }

  if (best_choice == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No valid configurations for found networks!\n");
      intdata->flags &= (~SCAN_DONE);

      return;
    } else {
      if (!config_get_association())
	{
	  debug_printf(DEBUG_NORMAL, "Found configuration data for network %s.  Attempting to associate!\n",
		       best_choice->ssidName);
	} else {
	  debug_printf(DEBUG_INT, "Best wireless network is %s!\n", 
		       best_choice->ssidName);
	}
    }

  // Otherwise, attempt to associate.
  intdata->flags &= (~SCAN_DONE);

  if (!config_get_association()) 
    {
      debug_printf(DEBUG_INT, "Attempting to associate.\n");
      cardif_associate(intdata, best_choice);
    } else {
      debug_printf(DEBUG_INT, "Association is set to manual!\n");
    }
}
