// $Id: Calendar.cc,v 1.3 2003/02/04 21:45:39 flaterco Exp $
/*  Calendar  Manage construction, organization, and printing of calendars.

    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

Calendar::Calendar (Station *station, Timestamp start_tm, Timestamp end_tm) {
  for (unsigned a=0; a<calhashsize; a++)
    hash[a] = NULL;

  timezone = station->timeZone;
  settings = station->context->settings;
  isCurrent = station->isCurrent;

  start_day_start = start_tm;
  start_day_start.prev_day (timezone, settings);
  end_day_start = end_tm;
  end_day_start.prev_day (timezone, settings);

  Timestamp tm = start_tm;
  while (tm <= end_tm) {
    struct blurb *b = new blurb;
    Dstr unwanted_desc;
    station->predictExactTideEvent (tm, Station::forward, b->t_out,
      b->etype_out, b->etype_desc, unwanted_desc, b->pv_out);
    if (tm > end_tm) {
      delete b;
      break;
    }
    add_blurb (b);
  }
}

Calendar::~Calendar () {
  for (unsigned a=0; a<calhashsize; a++) {
    struct day *d = hash[a];
    while (d) {
      struct blurb *b = d->blurbs;
      while (b) {
        struct blurb *tb = b;
        b = b->next;
        delete tb;
      }
      struct day *td = d;
      d = d->next;
      delete td;
    }
  }
}

void Calendar::add_blurb (struct blurb *b) {
  Timestamp start = b->t_out;
  start.prev_day (timezone, settings);
  struct day *d = lookup_day (start);
  if (!d)
    d = add_day (start);
  add_blurb (b, d);
}

void Calendar::add_blurb (struct blurb *b, struct day *destination) {
  assert (b->t_out >= destination->start);
  if (!(destination->blurbs)) {
    destination->blurbs = b;
    b->next = NULL;
  } else if (b->t_out < destination->blurbs->t_out) {
    b->next = destination->blurbs;
    destination->blurbs = b;
  } else {
    struct blurb *prev_b = destination->blurbs;
    while (prev_b->next) {
      if (b->t_out < prev_b->next->t_out)
        break;
      prev_b = prev_b->next;
    }
    b->next = prev_b->next;
    prev_b->next = b;
  }
}

struct Calendar::day *Calendar::lookup_day (Timestamp start) {
  struct day *d = hash[start.timet() % calhashsize];
  while (d) {
    if (d->start == start)
      return d;
    d = d->next;
  }
  return NULL;
}

struct Calendar::day *Calendar::add_day (Timestamp start) {
  struct day *d = new day;
  d->start = start;
  d->blurbs = NULL;
  d->next = hash[start.timet() % calhashsize];
  hash[start.timet() % calhashsize] = d;
  return d;
}

void Calendar::add_month_banner (Dstr &text_out, Timestamp t, int html_form) {
  Dstr heading;
  t.printcalheading (heading, timezone, settings);
  if (html_form) {
    text_out += "<h2>";
    text_out += heading;
    text_out += "</h2>\n";
    text_out += "<table border>\n";
  } else {
    int numspaces = (settings->tw - heading.length()) / 2;
    for (int a=0; a<numspaces; a++)
      text_out += ' ';
    text_out += heading;
    text_out += "\n\n";
  }
}

void Calendar::flush_buf (Dstr &text_out, Dstr *buf, int buflen,
int html_form, int oldcal, int headers) {
  int a, isdone=1;

  // If it's totally null, don't write a blank line.
  for (a=0; a<buflen; a++)
    if (buf[a].length()) {
      isdone = 0;
      break;
    }
  if (isdone)
    return;

  if (html_form) {
    if (oldcal) {
      // Strip out the day headers into a separate row
      text_out += "<tr>";
      for (a=0; a<buflen; a++) {
        Dstr strip;
        text_out += "<th>";
        buf[a].getline (strip);
        text_out += strip;
        text_out += "</th>";
      }
      text_out += "</tr>\n";
    }

    // Do the rest
    text_out += "<tr>";
    for (a=0; a<buflen; a++) {
      text_out += (headers ? "<th>" : "<td>");
      text_out += "<small>";
      text_out += buf[a];
      buf[a] = (char *)NULL;
      text_out += "</small>";
      text_out += (headers ? "</th>" : "</td>");
    }
    text_out += "</tr>\n";

  } else {
    char fmt[80];
    int colwid = settings->tw / buflen;
    if (colwid < 2)
      return;
    char *tbuf = (char *) malloc (colwid+1);
    sprintf (fmt, "%%-%d.%ds ", colwid-1, colwid-1);
    isdone = 0;
    while (!isdone) {
      Dstr blankquash;
      isdone = 1;
      for (a=0; a<buflen; a++) {
	if (buf[a].length())
	  isdone = 0;
	Dstr strip;
	buf[a].getline (strip);
	sprintf (tbuf, fmt, strip.aschar());
	blankquash += tbuf;
      }
      blankquash += '\n';
      if (oldcal || headers || !isdone)
        text_out += blankquash;
    }
    free (tbuf);
  }
}

void Calendar::print (Dstr &text_out, int html_form, int oldcal) {
  int a;
  text_out = (char *)NULL;
  if (html_form)
    text_out += "<center>\n";
  if (oldcal) {
    Dstr weekbuf[7];
    int month = -1;
    Timestamp tm = start_day_start;
    while (tm <= end_day_start) {
      struct tm *tmtm = tm.get_tm (timezone, settings);
      int wday = tmtm->tm_wday;
      if (tmtm->tm_mon != month) {
        if (month != -1) {
          flush_buf (text_out, weekbuf, 7, html_form, 1);
          if (html_form)
            text_out += "</table>\n";
        }
	month = tmtm->tm_mon;
	add_month_banner (text_out, tm, html_form);
      } else if (wday == 0) {
        flush_buf (text_out, weekbuf, 7, html_form, 1);
      }

      Dstr temp;
      tm.printdayheading (temp, timezone, settings);
      weekbuf[wday] += temp;
      weekbuf[wday] += '\n';

      struct day *d = lookup_day (tm);
      if (d) {
	struct blurb *b = d->blurbs;
	while (b) {
	  weekbuf[wday] += b->etype_desc;
	  if (!(isSunMoonEvent (b->etype_out))) {
	    b->pv_out.printnp (temp);
	    weekbuf[wday] += ' ';
	    weekbuf[wday] += temp;
	  }
	  if (html_form)
	    weekbuf[wday] += "<br>";
	  weekbuf[wday] += '\n';
	  b->t_out.printtime (temp, timezone, settings);
	  weekbuf[wday] += temp;
	  if (html_form)
	    weekbuf[wday] += "<br>";
	  weekbuf[wday] += '\n';
	  b = b->next;
	}
      }
      tm.inc_day (timezone, settings);
    }
    flush_buf (text_out, weekbuf, 7, html_form, 1);

  } else {

    // nns = NOT nosunmoon
    // Value is number of columns needed
    int nns = 5;
    if (settings->ns != 'n')
      nns = 0;

    // Day ... [Mark transitions]? Surises Sunsets Moonphases
    // Tides: ... = High [Low High]+
    // Currents: ... = Slack Flood Slack [Ebb Slack Flood Slack]+
    // For Tides, X = max number of low tides in a day
    // For Currents, X = max number of max ebbs in a day
    // Exception:  ebbs with no intervening slack count as one

    // Find the value of "X" and check for mark transitions
    int X = 1;
    int havemarks = 0;
    for (a=0; a<calhashsize; a++) {
      Calendar::day *d = hash[a];
      while (d) {
        int tx = 0;
        int givemeslack = 0;
	Calendar::blurb *b = d->blurbs;
	while (b) {
          switch (b->etype_out) {
          case Station::min:
            if ((b->etype_desc != "Min Flood") && (!givemeslack)) {
              tx++;
              if (isCurrent)
                givemeslack = 1;
            }
            break;
          case Station::max:
            if ((b->etype_desc == "Min Ebb") && (!givemeslack)) {
              tx++;
              if (isCurrent)
                givemeslack = 1;
            }
            break;
          case Station::slackrise:
            givemeslack = 0;
            break;
          case Station::markrise:
          case Station::markfall:
            havemarks = 1;
          default:
            ;
          }
	  b = b->next;
	}
        if (tx > X)
          X = tx;
        d = d->next;
      }
    }

    int numcol = (isCurrent ? 3+X*4 : 1+X*2);
    // Col. 0 is day
    // Cols. 1 .. numcol are tides/currents
    // Col. numcol+1 is optionally mark 
    // numcol + havemarks + 1 is moon phase
    // numcol + havemarks + 2 is sunrise
    // numcol + havemarks + 3 is sunset
    // numcol + havemarks + 4 is moonrise
    // numcol + havemarks + 5 is moonset
    // The usually blank phase column makes a natural separator between
    // the tide/current times and the sunrise/sunset times.
    Dstr *colbuf = new Dstr[numcol+7];

    int month = -1;
    Timestamp tm = start_day_start;
    while (tm <= end_day_start) {
      struct tm *tmtm = tm.get_tm (timezone, settings);
      if (tmtm->tm_mon != month) {
        if (month != -1)
          text_out += (html_form ? "</table>\n" : "\n");
	month = tmtm->tm_mon;
	add_month_banner (text_out, tm, html_form);

        {
          int i = 0;
	  colbuf[i++] = "Day";
	  if (isCurrent) {
            colbuf[i] = (html_form ? "<br>\n" : "\n");
            colbuf[i++] += "Slack";
            colbuf[i++] = "Flood";
            colbuf[i] = (html_form ? "<br>\n" : "\n");
            colbuf[i++] += "Slack";
	  } else {
            colbuf[i++] = "High";
	  }
          for (a=0; a<X; a++) {
	    if (isCurrent) {
	      colbuf[i] = (html_form ? "<br>\n<br>\n" : "\n\n");
	      colbuf[i++] += "Ebb";
	      colbuf[i] = (html_form ? "<br>\n" : "\n");
	      colbuf[i++] += "Slack";
	      colbuf[i++] = "Flood";
	      colbuf[i] = (html_form ? "<br>\n" : "\n");
	      colbuf[i++] += "Slack";
	    } else {
	      colbuf[i] = (html_form ? "<br>\n" : "\n");
	      colbuf[i++] += "Low";
	      colbuf[i++] = "High";
	    }
          }
          if (havemarks)
            colbuf[i++] = "Mark";
          if (nns) {
            colbuf[i++] = "Phase";
            colbuf[i++] = "Sunrise";
            colbuf[i++] = "Sunset";
            colbuf[i++] = "Moonrise";
            colbuf[i++] = "Moonset";
          }
        }
        flush_buf (text_out, colbuf, numcol+havemarks+nns+1, html_form, 0, 1);
      }

      Dstr temp;
      tm.printdayheading (temp, timezone, settings);
      if (html_form)
        colbuf[0] += "<b>";
      colbuf[0] += temp;
      if (html_form)
        colbuf[0] += "</b>";

      // Tidecol X maps to colbuf element X+1
      int tidecol = 0;

      struct day *d = lookup_day (tm);
      if (d) {
	struct blurb *b = d->blurbs;
	while (b) {
  	  switch (b->etype_out) {

	  // For currents, we have the exception case of Min Floods and
	  // Min Ebbs to deal with.  The combination Max, Min, Max is
	  // crammed into one table cell.

	  // There is a lot of duplicated code here, but I find it more
	  // understandable when it's all spelled out like this.

          case Station::max:
            if (isCurrent) {
              if (b->etype_desc == "Min Ebb") {
                while ((tidecol+1) % 4)
                  tidecol++;
              } else {
                while ((tidecol+3) % 4)
                  tidecol++;
              }
	      assert (tidecol < numcol);
	      if (colbuf[tidecol+1].length())
		colbuf[tidecol+1] += (html_form ? "<br>\n" : "\n");
            } else {
              while (tidecol % 2)
                tidecol++;
              assert (tidecol < numcol);
              assert (!(colbuf[tidecol+1].length()));
            }
            b->t_out.printtime (temp, timezone, settings);
	    colbuf[tidecol+1] += temp;
	    colbuf[tidecol+1] += " / ";
	    b->pv_out.printnp (temp);
	    colbuf[tidecol+1] += temp;
            break;

          case Station::min:
            if (isCurrent) {
              if (b->etype_desc == "Min Flood") {
                while ((tidecol+3) % 4)
                  tidecol++;
              } else {
                while ((tidecol+1) % 4)
                  tidecol++;
              }
	      assert (tidecol < numcol);
	      if (colbuf[tidecol+1].length())
		colbuf[tidecol+1] += (html_form ? "<br>\n" : "\n");
            } else {
              while ((tidecol+1) % 2)
                tidecol++;
              assert (tidecol < numcol);
              assert (!(colbuf[tidecol+1].length()));
            }
            b->t_out.printtime (temp, timezone, settings);
	    colbuf[tidecol+1] += temp;
	    colbuf[tidecol+1] += " / ";
	    b->pv_out.printnp (temp);
	    colbuf[tidecol+1] += temp;
            break;

          case Station::slackrise:
            assert (isCurrent);
            while (tidecol % 4)
              tidecol++;
            assert (tidecol < numcol);
            assert (!(colbuf[tidecol+1].length()));
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[tidecol+1] += temp;
            break;

          case Station::slackfall:
            assert (isCurrent);
            while ((tidecol+2) % 4)
              tidecol++;
            assert (tidecol < numcol);
            assert (!(colbuf[tidecol+1].length()));
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[tidecol+1] += temp;
            break;

          case Station::markrise:
          case Station::markfall:
            assert (havemarks);
            if (colbuf[numcol+1].length())
              colbuf[numcol+1] += (html_form ? "<br>\n" : "\n");
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[numcol+1] += temp;
            break;

          case Station::sunrise:
            assert (nns);
            if (colbuf[numcol+havemarks+2].length())
              colbuf[numcol+havemarks+2] += (html_form ? "<br>\n" : "\n");
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[numcol+havemarks+2] += temp;
            break;

          case Station::sunset:
            assert (nns);
            if (colbuf[numcol+havemarks+3].length())
              colbuf[numcol+havemarks+3] += (html_form ? "<br>\n" : "\n");
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[numcol+havemarks+3] += temp;
            break;

          case Station::moonrise:
            assert (nns);
            if (colbuf[numcol+havemarks+4].length())
              colbuf[numcol+havemarks+4] += (html_form ? "<br>\n" : "\n");
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[numcol+havemarks+4] += temp;
            break;

          case Station::moonset:
            assert (nns);
            if (colbuf[numcol+havemarks+5].length())
              colbuf[numcol+havemarks+5] += (html_form ? "<br>\n" : "\n");
  	    b->t_out.printtime (temp, timezone, settings);
	    colbuf[numcol+havemarks+5] += temp;
            break;

          case Station::newmoon:
          case Station::firstquarter:
          case Station::fullmoon:
          case Station::lastquarter:
            assert (nns);
            assert (!(colbuf[numcol+havemarks+1].length()));
            colbuf[numcol+havemarks+1] += b->etype_desc;
            break;

          default:
            assert (0);
          }
	  b = b->next;
	}

        // Now print the day.
        flush_buf (text_out, colbuf, numcol+havemarks+nns+1, html_form, 0);
      }

      tm.inc_day (timezone, settings);
      for (a=0; a<numcol+5; a++)
        colbuf[a] = (char *)NULL;
    }
    delete [] colbuf;
  }

  if (html_form)
    text_out += "</table></center>\n";
}
