/*
 * cblasy.h
 * CBLAS routines used by yorick
 *
 * Basic Linear algebra Subprograms Technical (BLAST) Forum Standard
 * August 21, 2001  (http://www.netlib.org/blas/blast-forum/)
 * Appendix B: Legacy BLAS
 */

#undef INT_IN

/* BLAST forum standard does two things I dont like */
#ifdef USE_CBLAS
# define INT_IN const int
# include <stddef.h>
# ifndef CBLAS_INDEX
/* this may not be correct, size_t (in stddef.h) is just a guess
 * if so, define CBLAS_INDEX to correct type
 * NB- since size_t unsigned, cannot do integer divide on result */
#  define CBLAS_INDEX size_t
# endif

/* this is more consistent with rest of yorick */
#else
# define INT_IN long
# define CBLAS_INDEX long
#endif

/* these values are set in the standard */
enum CBLAS_ORDER {CblasRowMajor=101, CblasColMajor=102};
enum CBLAS_TRANSPOSE {CblasNoTrans=111, CblasTrans=112, CblasConjTrans=113};
enum CBLAS_UPLO {CblasUpper=121, CblasLower=122};
enum CBLAS_DIAG {CblasNonUnit=131, CblasUnit=132};
enum CBLAS_SIDE {CblasLeft=141, CblasRight=142};

/* level 1 */

extern double cblas_ddot(INT_IN n, const double *x, INT_IN incx,
                         const double *y, INT_IN incy);

extern double cblas_dnrm2(INT_IN n, const double *x, INT_IN incx);
extern double cblas_dasum(INT_IN n, const double *x, INT_IN incx);

extern CBLAS_INDEX cblas_idamax(INT_IN n, const double *x, INT_IN incx);

extern void cblas_dswap(INT_IN n, double *x, INT_IN incx, 
                        double *y, INT_IN incy);
extern void cblas_dcopy(INT_IN n, const double *x, INT_IN incx, 
                        double *y, INT_IN incy);
extern void cblas_daxpy(INT_IN n, const double alpha, const double *x,
                        INT_IN incx, double *y, INT_IN incy);

extern void cblas_drot(INT_IN n, double *x, INT_IN incx,
                       double *y, INT_IN incy,
                       const double c, const double  s);

extern void cblas_dscal(INT_IN n, const double alpha,
                        double *x, INT_IN incx);

/* level 2 */

extern void cblas_dgemv(const enum CBLAS_ORDER order,
                        const enum CBLAS_TRANSPOSE transa,
                        INT_IN m, INT_IN n,
                        const double alpha, const double *a, INT_IN lda,
                        const double *x, INT_IN incx, const double beta,
                        double *y, INT_IN incy);
extern void cblas_dtrmv(const enum CBLAS_ORDER order,
                        const enum CBLAS_UPLO uplo,
                        const enum CBLAS_TRANSPOSE transa,
                        const enum CBLAS_DIAG diag,
                        INT_IN n, const double *a, INT_IN lda, 
                        double *x, INT_IN incx);
extern void cblas_dtrsv(const enum CBLAS_ORDER order,
                        const enum CBLAS_UPLO uplo,
                        const enum CBLAS_TRANSPOSE transa,
                        const enum CBLAS_DIAG diag,
                        INT_IN n, const double *a, INT_IN lda,
                        double *x, INT_IN incx);

extern void cblas_dger(const enum CBLAS_ORDER order, INT_IN m, INT_IN n,
                       const double alpha, const double *x, INT_IN incx,
                       const double *y, INT_IN incy,
                       double *a, INT_IN lda);

/* level 3 */

extern void cblas_dgemm(const enum CBLAS_ORDER order,
                        const enum CBLAS_TRANSPOSE transa,
                        const enum CBLAS_TRANSPOSE transb,
                        INT_IN m, INT_IN n,
                        INT_IN k, const double alpha, const double *a,
                        INT_IN lda, const double *b, INT_IN ldb,
                        const double beta, double *c, INT_IN ldc);
extern void cblas_dtrmm(const enum CBLAS_ORDER order,
                        const enum CBLAS_SIDE side,
                        const enum CBLAS_UPLO uplo,
                        const enum CBLAS_TRANSPOSE transa,
                        const enum CBLAS_DIAG diag, INT_IN m, INT_IN n,
                        const double alpha, const double *a, INT_IN lda,
                        double *b, INT_IN ldb);
extern void cblas_dtrsm(const enum CBLAS_ORDER order,
                        const enum CBLAS_SIDE side,
                        const enum CBLAS_UPLO uplo,
                        const enum CBLAS_TRANSPOSE transa,
                        const enum CBLAS_DIAG diag, INT_IN m, INT_IN n,
                        const double alpha, const double *a, INT_IN lda,
                        double *b, INT_IN ldb);
